/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <dbus/dbus-glib.h>
#include <dbus/dbus-protocol.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-connection-manager.h"
#include "tpa-manager.h"
#include "tpa-account.h"
#include "tpa-ifaces.h"

#include "tpa-connection-manager-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-connection-manager-glue.h>
#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>
#include <tapioca/base/tpa-enums.h>

#define _DBUS_PATH "/org/freedesktop/Telepathy/ConnectionManager"

#define TPA_TYPE_PARAM (dbus_g_type_get_struct ("GValueArray", \
      G_TYPE_STRING, \
      G_TYPE_UINT, \
      G_TYPE_STRING, \
      G_TYPE_VALUE, \
      G_TYPE_INVALID))

G_DEFINE_TYPE_WITH_CODE(TpaManager, tpa_manager, G_TYPE_OBJECT, \
    G_IMPLEMENT_INTERFACE(TPA_TYPE_ICONNECTION_MANAGER, tpa_connection_manager_init))

struct _TpaManagerPrivate
{
    GPtrArray *accounts;
    DBusGConnection *dbus;

    gboolean registered;
    gboolean disposed;
};

#define TPA_MANAGER_GET_PRIVATE(o)     (G_TYPE_INSTANCE_GET_PRIVATE ((o), TPA_TYPE_MANAGER, TpaManagerPrivate))

void
tpa_manager_dispose (GObject *object)
{
    TpaManager *self = TPA_MANAGER (object);
    guint i;
    GObject* obj;

    VERBOSE ("(%p)", self);

    if (self->priv->disposed) {
        /* If dispose did already run, return. */
        return;
    }

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    /* unref connection objects */
    /* TODO: emit disconnection signal? */
    if (self->priv->accounts != NULL)
    {
        for (i = 0; i < self->priv->accounts->len; i++)
        {
            obj = G_OBJECT (g_ptr_array_index (self->priv->accounts, i));
            g_object_unref (obj);
        }

        g_ptr_array_free (self->priv->accounts, TRUE);
        self->priv->accounts = NULL;
    }

    G_OBJECT_CLASS (tpa_manager_parent_class)->dispose (object);
}

void
tpa_manager_finalize (GObject *object)
{
    VERBOSE ("(%p)", object);
    G_OBJECT_CLASS (tpa_manager_parent_class)->finalize (object);
}

/* type definition stuff */
static void
tpa_manager_class_init (TpaManagerClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    VERBOSE ("(%p)", klass);

    g_type_class_add_private (klass, sizeof (TpaManagerPrivate));

    object_class->dispose = tpa_manager_dispose;
    object_class->finalize = tpa_manager_finalize;

    dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
        &dbus_glib_connection_manager_object_info);
}

static void
tpa_manager_init (TpaManager *self)
{
    VERBOSE ("(%p)", self);

    self->priv = TPA_MANAGER_GET_PRIVATE (self);
    self->priv->accounts = g_ptr_array_new ();
    self->priv->dbus = NULL;
    self->priv->registered = FALSE;
}

/**
 * tpa_connetion_manager_new
 *
 * Create a new connection manager, if a it will try
 * to register it under org.freedesktop.Telepathy.Manager.(name)
 */
TpaManager *
tpa_manager_new (GType object_type,
                 const gchar *name,
                 const gchar **protocols)
{
    TpaIConnectionManager *iface = NULL;
    TpaManager *self = NULL;

    VERBOSE ("(%s, %p)", name, protocols);

    self = TPA_MANAGER (g_object_new (object_type, NULL));

    if (self && self->priv && name && g_str_equal (name, "") && protocols) {
        tpa_manager_register (self, name);
        iface = TPA_ICONNECTION_MANAGER (self);
        iface->protocols = g_strdupv ((gchar **) protocols);
    }

    return self;
}

/**
 * tpa_manager_unref
 *
 * Delete manager object.
 */
void
tpa_manager_unref (TpaManager *self)
{
    g_assert (TPA_IS_MANAGER (self));
    g_object_unref (self);
}

/**
 * tpa_manager_register
 *
 * Register connection managers on dbus, the DBUS path always
 * begins with "/org/freedesktop/Telepathy/Manager".
 */
gboolean
tpa_manager_register (TpaManager *self,
                      const gchar *name)
{
    TpaIConnectionManager *iface = NULL;
    DBusGProxy* driver_proxy = NULL;
    gchar *dbus_path = NULL;
    gchar *dbus_iface = NULL;
    gboolean ok = FALSE;
    GError* error = NULL;
    int request_ret = 0;

    VERBOSE ("(%p, %s)", self, name);

    if (self->priv->registered) {
        VERBOSE ("Service already registered");
        return TRUE;
    }

    self->priv->dbus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

    if (self->priv->dbus == NULL) {
        VERBOSE ("Unable to connect to dbus: %s", error->message);
        g_error_free (error);
    }

    g_assert (TPA_IS_MANAGER (self));
    g_return_val_if_fail (name != NULL, FALSE);
    g_return_val_if_fail (g_str_equal (name, "") != FALSE, FALSE);

    dbus_path = g_strdup_printf ("%s/%s", _DBUS_PATH, name);

    dbus_iface = g_strdup_printf ("%s.%s", DBUS_CONNECTION_MANAGER_IFACE, name);

    /* Register DBUS path */
    dbus_g_connection_register_g_object (self->priv->dbus,
        dbus_path,
        G_OBJECT (self));

    /* Register the service name, the constants here are defined in dbus-glib.h */
    driver_proxy = dbus_g_proxy_new_for_name (self->priv->dbus,
            DBUS_SERVICE_DBUS,
            DBUS_PATH_DBUS,
            DBUS_INTERFACE_DBUS);


    ok = dbus_g_proxy_call (driver_proxy, "RequestName", &error,
                            G_TYPE_STRING, dbus_iface,
                            G_TYPE_UINT, 0,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &request_ret, /* See tutorial for more infos about these */
                            G_TYPE_INVALID);

    if(!ok) {
        VERBOSE ("Unable to register service: %s\n", error->message);
        g_error_free (error);
    }
    else {
        VERBOSE ("interface registered: %s", dbus_iface);
        iface = TPA_ICONNECTION_MANAGER (self);
        iface->name = g_strdup (name);
        self->priv->registered = TRUE;
    }

    g_free (dbus_path);
    g_free (dbus_iface);
    g_object_unref (driver_proxy);
    return ok;
}

/**
 * tpa_manager_add_param
 *
 * Add String Parameter Specification - Helper Function
 */
void
tpa_manager_add_param (TpaManager *self,
                       const gchar *name,
                       guint flag,
                       const gchar *type,
                       gpointer value)
{
    TpaIConnectionManager *iface = NULL;

    VERBOSE ("(%p, %s, %d, %s, %p)", self, name, flag, type, value);

    if (name) {
        TpaConnectionManagerParam *parameter =
            (TpaConnectionManagerParam *) g_new (TpaConnectionManagerParam *, 1);
        parameter->name = g_strdup ((gchar *)name);
        parameter->flag = flag;
        parameter->type = g_strdup (type);
        parameter->value = value;

        iface = TPA_ICONNECTION_MANAGER (self);
        g_ptr_array_add (iface->parameters, parameter);
    }
    else
        VERBOSE ("cannot add %s param", name);
}

/**
 * tpa_manager_add_account
 */
void
tpa_manager_add_account (TpaManager *self,
                         GObject *account)
{
    g_assert (TPA_IS_MANAGER (self));
    g_assert (TPA_IS_ACCOUNT (account));
    VERBOSE ("(%p, %p)", self, account);

    if (self->priv->accounts) {
        g_ptr_array_add (self->priv->accounts, account);
        VERBOSE ("account %p added", account);
    }
}

/**
 * tpa_manager_remove_account
 */
void
tpa_manager_remove_account (TpaManager *self,
                            GObject *account)
{
    g_assert (TPA_IS_MANAGER (self));
    g_assert (TPA_IS_ACCOUNT (account));
    VERBOSE ("(%p, %p)", self, account);

    if (self->priv->accounts) {
        g_ptr_array_remove (self->priv->accounts, account);

// TODO: Quit if no more connections.
#if 0
        if (!g_ptr_array_index (self->priv->accounts, 0));
            tpa_manager_unref (G_Object (self));
#endif
    }
}
