/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "tpa-text.h"
#include "tpa-account.h"
#include "tpa-session.h"

#include "tpa-text-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

typedef struct _TpaMessage TpaMessage;

/* signal enum */
enum
{
    LOST_MESSAGE,
    RECEIVED,
    SEND_ERROR,
    SENT,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaMessage {
    guint id;
    time_t timestamp;
    guint handle;
    TpaTextChannelMessageType type;
    guint flags;
    gchar *text;
};

struct _TpaITextPrivate {
    GObject *object;
    GQueue *queue;
    guint handle;
};

/* we need to define the get_type function */
GType
tpa_text_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIText),
            NULL,	/* base init */
            NULL,	/* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIText",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_text_init (TpaIText *iface,
               gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->acknowledge_pending_messages = NULL;
    iface->list_pending_messages = NULL;
    iface->get_message_types = NULL;
    iface->send = NULL;

    /* Interface signals */
    signals[LOST_MESSAGE] = g_signal_new ("lost-message",
        G_OBJECT_CLASS_TYPE (iface),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE, 0);

    signals[RECEIVED] = g_signal_new ("received",
        G_OBJECT_CLASS_TYPE (iface),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        tpa_marshal_VOID__UINT_UINT_UINT_UINT_UINT_STRING,
        G_TYPE_NONE, 6, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

    signals[SEND_ERROR] = g_signal_new ("send-error",
        G_OBJECT_CLASS_TYPE (iface),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        tpa_marshal_VOID__UINT_UINT_UINT_STRING,
        G_TYPE_NONE, 4, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

    signals[SENT] = g_signal_new ("sent",
        G_OBJECT_CLASS_TYPE (iface),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        tpa_marshal_VOID__UINT_UINT_STRING,
        G_TYPE_NONE, 3, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

        iface->priv = g_new0 (TpaITextPrivate, 1);
        iface->priv->queue = g_queue_new ();
        VERBOSE ("private members %p", iface->priv);
}

void
tpa_text_finalize (GObject *obj)
{
    TpaIText *iface = TPA_ITEXT (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_text_acknowledge_pending_messages
 *
 * Implements DBus method ListPendingMessage
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_text_acknowledge_pending_messages (GObject *obj,
                                       const GArray *ids,
                                       GError **error)
{
    TpaIText *iface = TPA_ITEXT (obj);
    TpaError error_code = TPA_ERROR_NONE;
    guint id = 0;
    guint index = -1;
    guint c = 0;

    if (!iface)
        g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ids);

    if (iface->acknowledge_pending_messages)
        error_code = iface->acknowledge_pending_messages (obj, ids);
    else if (ids) {
        for (c = 0; c < ids->len; c++) {
            id = g_array_index (ids, guint, c);
            index = tpa_text_get_message_index (obj, id);
            if (index != -1)
                g_queue_pop_nth (iface->priv->queue ,index);
            else
                error_code = TPA_ERROR_INVALID_ARGUMENT;
        }
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_text_list_pending_messages
 *
 * Implements DBus method ListPendingMessage
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_text_list_pending_messages (GObject *obj,
                                gboolean clear,
                                GPtrArray **ret,
                                GError **error)
{
    TpaIText *iface = TPA_ITEXT (obj);
    TpaError error_code = TPA_ERROR_NONE;
    TpaMessage *msg = NULL;
    guint i = 0;
    guint len = 0;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %d, %p)", obj, clear, ret);

    if (iface->list_pending_messages)
        error_code = iface->list_pending_messages (obj, clear, ret);
    else if (ret) {
        len = g_queue_get_length (iface->priv->queue);
        *ret = g_ptr_array_sized_new (len);

        for (i = 0; i < len; i++) {
            GValue val = { 0, };

            if (clear)
                msg = g_queue_pop_nth (iface->priv->queue, i);
            else
                msg = g_queue_peek_nth (iface->priv->queue, i);
            if (msg) {
                g_value_init (&val, dbus_g_type_get_struct ("GValueArray",
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_STRING,
                    G_TYPE_INVALID));
                g_value_take_boxed (&val,
                    dbus_g_type_specialized_construct (dbus_g_type_get_struct ("GValueArray",
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_STRING,
                    G_TYPE_INVALID)));
                dbus_g_type_struct_set (&val,
                    0, msg->id,
                    1, msg->timestamp,
                    2, msg->handle,
                    3, msg->type,
                    4, msg->flags,
                    5, msg->text,
                    G_MAXUINT);

                g_ptr_array_add (*ret, g_value_get_boxed (&val));
            }
        }
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_text_get_message_types
 *
 * Implements DBus method GetMessageTypes
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_text_get_message_types (GObject *obj,
                            GArray **ret,
                            GError **error)
{
    TpaIText *iface = TPA_ITEXT (obj);
    TpaError error_code = TPA_ERROR_NONE;
    TpaTextChannelMessageType action = TPA_CHANNEL_TEXT_MESSAGE_TYPE_ACTION;
    TpaTextChannelMessageType auto_reply = TPA_CHANNEL_TEXT_MESSAGE_TYPE_AUTO_REPLY;
    TpaTextChannelMessageType normal = TPA_CHANNEL_TEXT_MESSAGE_TYPE_NORMAL;
    TpaTextChannelMessageType notice = TPA_CHANNEL_TEXT_MESSAGE_TYPE_NOTICE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    if (iface->get_message_types)
        error_code = iface->get_message_types (obj, ret);
    else if (ret) {
        *ret = g_array_new (FALSE, TRUE, sizeof (guint));
        g_array_append_val (*ret, action);
        g_array_append_val (*ret, auto_reply);
        g_array_append_val (*ret, normal);
        g_array_append_val (*ret, notice);
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_text_send
 *
 * Implements DBus method Send
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_text_send (GObject *obj,
               TpaTextChannelMessageType type,
               const gchar *text,
               GError **error)
{
    TpaIText *iface = TPA_ITEXT (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %d, %s)", obj, type, text);

    g_return_error_val_if_fail (iface->send != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->send (obj, type, text);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_text_signal_lost_message
 *
 * Implements DBus signal LostMessage
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 */
void
tpa_text_signal_lost_message (GObject *obj)
{
    TpaIText *iface = TPA_ITEXT (obj);

    if (!iface)
        return;

    VERBOSE ("(%p)", iface);

    g_signal_emit (obj, signals[LOST_MESSAGE], 0);
}

/**
 * tpa_text_get_message_index
 */
guint
tpa_text_get_message_index (GObject *obj, 
                            guint id)
{
    TpaIText *iface = TPA_ITEXT (obj);
    TpaMessage *msg;
    guint i = 0;
    guint len = g_queue_get_length (iface->priv->queue);

    for (i = 0; i < len; i++) {
        msg = g_queue_peek_nth (iface->priv->queue, i);
        if (msg && (msg->id == id))
            return i;
    }

    return -1;
}

void
tpa_chat_state_set_chat_state(GObject *obj,
                              uint state)
{
}

/**
 * tpa_text_signal_received
 *
 * Implements DBus signal Received
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 */
void
tpa_text_signal_received (GObject *obj,
                          guint id,
                          const gchar *contact,
                          TpaTextChannelMessageType type,
                          guint flags,
                          const gchar *text)
{
    TpaIText *iface = TPA_ITEXT (obj);
    TpaAccount *account = NULL;
    TpaMessage *msg = NULL;
    time_t timestamp = 0;
    TpaHandle *handle = NULL;

    if (!iface)
        return;

    VERBOSE ("(%p, %d, %s, %d, %d, %s)", iface, id, contact, type, flags, text);

    account = (TPA_SESSION (obj))->account;
    handle = tpa_connection_get_contact_handle (G_OBJECT (account), contact);
    timestamp = time (NULL);
    msg = g_new0 (TpaMessage, 1);
    msg->id = id;
    msg->timestamp = timestamp;
    msg->handle = handle->id;
    msg->type = type;
    msg->flags = flags;
    msg->text = g_strdup (text);
    g_signal_emit (obj, signals[LOST_MESSAGE], 0, id, timestamp, handle, type,
        flags, text);
}

/**
 * tpa_text_signal_send_error
 *
 * Implements DBus signal SendError
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 */
void
tpa_text_signal_send_error (GObject *obj,
                            guint error,
                            TpaTextChannelMessageType type,
                            const gchar *text)
{
    time_t timestamp = 0;

    g_assert (TPA_IS_ITEXT (obj));

    VERBOSE ("(%p, %d, %d, %s)", obj, error, type, text);

    timestamp = time (NULL);
    g_signal_emit (obj, signals[SEND_ERROR], 0, error, timestamp, type, text);
}

/**
 * tpa_text_signal_sent
 *
 * Implements DBus signal Sent
 * on interface org.freedesktop.Telepathy.Channel.Type.Text
 */
void
tpa_text_signal_sent (GObject *obj,
                      TpaTextChannelMessageType type,
                      const gchar *text)
{
    time_t timestamp = 0;

    g_assert (TPA_IS_ITEXT (obj));

    VERBOSE ("(%p, %d, %s)", obj, type, text);

    timestamp = time (NULL);
    g_signal_emit (obj, signals[SENT], 0, timestamp, type, text);
}
