/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include "QtTapioca/UserContact"
#include "QtTapioca/Handle"
#include "QtTapioca/Avatar"

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>
#include <QtTelepathy/Client/Connection>

namespace QtTapioca {

class UserContactPrivate {
public:
    UserContactPrivate(org::freedesktop::Telepathy::Connection *telepathyConn)
        :bus(QDBusConnection::sessionBus())
    {
        this->telepathyConn = telepathyConn;
    }
    ~UserContactPrivate()
    {
    }

    QDBusConnection bus;
    org::freedesktop::Telepathy::Connection *telepathyConn;
};

};

using namespace QtTapioca;

UserContact::UserContact(org::freedesktop::Telepathy::Connection *telepathyConn,
                           org::freedesktop::Telepathy::ConnectionAvatarsInterface  *iAvatar,
                           org::freedesktop::Telepathy::ConnectionPresenceInterface *iPresence,
                           org::freedesktop::Telepathy::ConnectionAliasingInterface *iAliasing,
                           org::freedesktop::Telepathy::ConnectionCapabilitiesInterface *iCapabilities,
                           Handle *handle,
                           QObject *parent)
    : ContactBase(telepathyConn, iAvatar, iPresence, iAliasing, iCapabilities, handle, parent),
      d(new UserContactPrivate(telepathyConn))
{
}

UserContact::~UserContact()
{
    delete d;
}

bool UserContact::setAvatar(Avatar *avatar)
{
    if ((!avatar) || (!telepathyIAvatar))
        return false;

    QString token = telepathyIAvatar->SetAvatar(avatar->data(), avatar->mimeType());

    avatar->setToken(token);

    return true;
}

bool UserContact::setPresence(ContactBase::Presence status)
{
    if (!telepathyIPresence)
        return false;

    QString presenceStr = presenceEnumToStr(status);

    setPresenceStatus(presenceStr, presenceMessage());

    m_presence = status;

    return true;
}

bool UserContact::setPresenceMessage(const QString &message)
{
    if (!telepathyIPresence)
        return false;

    QString presenceStr = presenceEnumToStr(presence());

    setPresenceStatus(presenceStr, message);

    m_presenceMessage = message;

    return true;
}

bool UserContact::setPresenceWithMessage(ContactBase::Presence status, const QString &message)
{
    if (!telepathyIPresence)
        return false;

    QString presenceStr = presenceEnumToStr(status);

    setPresenceStatus(presenceStr, message);

    m_presence = status;
    m_presenceMessage = message;

    return true;
}

bool UserContact::setAlias(const QString &alias)
{
    if (!telepathyIAliasing)
        return false;

    QMap<uint, QString> map;
    map.insert(handle()->id(), alias);

    telepathyIAliasing->SetAliases(map);

    return true;
}

bool UserContact::setCapabilities(ContactBase::Capability cap)
{
    if (!telepathyICapabilities)
        return false;

    QStringList remove;
    org::freedesktop::Telepathy::LocalCapabilityInfoList add;
    org::freedesktop::Telepathy::LocalCapabilityInfo capInfo;

    //Check if Text is supported
    if (cap & ContactBase::Text) {
        capInfo.channelType =  "org.freedesktop.Telepathy.Channel.Type.Text";
        capInfo.typeSpecificFlags = org::freedesktop::Telepathy::CONNECTION_CAPABILITY_FLAG_ALL;
        add << capInfo;
    }
    else {
        if (capabilities() & ContactBase::Text)
            remove << "org.freedesktop.Telepathy.Channel.Type.Text";
    }

    //Check if Audio is supported
    if (cap & ContactBase::Audio) {
        capInfo.channelType =  "org.freedesktop.Telepathy.Channel.Type.StreamedMedia";
        capInfo.typeSpecificFlags = org::freedesktop::Telepathy::CONNECTION_CAPABILITY_FLAG_ALL;
        add << capInfo;
    }
    else {
        //TODO - Check this conditional for video,
        //because audio and video uses the same telepathy interface(StreamedMedia).
        if (capabilities() & ContactBase::Audio)
            remove << "org.freedesktop.Telepathy.Channel.Type.StreamedMedia";
    }

    telepathyICapabilities->AdvertiseCapabilities(add, remove);

    return true;
}

void UserContact::setPresenceStatus(const QString &presenceStr, const QString &message)
{
    QMap<QString, QVariant> opt;
    org::freedesktop::Telepathy::PresenceState presenceStatus;

    opt.insert("message", message);

    presenceStatus.insert(presenceStr, opt);

    telepathyIPresence->SetStatus(presenceStatus);
}
