/*
 *	tardy - a tar post-processor
 *	Copyright (C) 1998, 1999, 2002 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate tar output unix std tars
 */

#include <ac/string.h>

#include <error.h>
#include <tar/format.h>
#include <tar/output/tar/posix.h>


tar_output_tar_posix::~tar_output_tar_posix()
{
}


tar_output_tar_posix::tar_output_tar_posix()
{
	::fatal("%s: %d: default contstructor is private (bug)",
		__FILE__, __LINE__);
}


tar_output_tar_posix::tar_output_tar_posix(file_output *arg)
	: tar_output_tar_base(arg)
{
}


tar_output_tar_posix::tar_output_tar_posix(const tar_output_tar_posix &)
{
	::fatal("%s: %d: copy constructor is private (bug)",
		__FILE__, __LINE__);
}


tar_output_tar_posix &
tar_output_tar_posix::operator = (const tar_output_tar_posix &)
{
	::fatal("%s: %d: assignment operator is private (bug)",
		__FILE__, __LINE__);
	return *this;
}


void
tar_output_tar_posix::write_header(const tar_header &h)
{
	char block[TBLOCK];
	header_ty *hp = (header_ty *)block;

	if (h.name.length() < 1)
		fatal("filename \"%s\" too short", h.name.to_c_string());

	if (h.name[h.name.length() - 1] == '/')
	{
		::fatal
		(
			"bug (%s:%d) name \"%s\" has slash",
			__FILE__,
			__LINE__,
			h.name.to_c_string()
		);
	}

	rcstring name = h.name;
	if (h.type == tar_header::type_directory)
	    name = h.name + "/";

	if (name.length() + 1 > sizeof(hp->name))
	{
	    //
	    // Write a bogus header block, indicating that the following
	    // data is a long file name.
	    //
	    memset(block, 0, sizeof(block));
	    memcpy(hp->magic, TMAGIC, sizeof(hp->magic));
	    hp->name_set("././@LongLink");
	    hp->mode_set(0);
	    hp->uid_set(0);
	    hp->uname_set("root");
	    hp->gid_set(0);
	    hp->gname_set("root");
	    hp->size_set(name.length() + 1);
	    hp->mtime_set(0);
	    hp->linkflag_set(LF_LONGNAME);
	    hp->chksum_set(hp->calculate_checksum());
	    write_data(block, sizeof(block));

	    //
	    // Write a bogus data block with the file name in it.
	    //
	    write_data(name.to_c_string(), name.length() + 1);
	    write_data_padding();
	}
	if
	(
	    (
	       	h.type == tar_header::type_link_hard
	    ||
	       	h.type == tar_header::type_link_symbolic
	    )
	&&
	    h.linkname.length() + 1 > sizeof(hp->linkname)
	)
	{
	    //
	    // Write a bogus header block, indicating that the following
	    // data is a long file name.
	    //
	    memset(block, 0, sizeof(block));
	    memcpy(hp->magic, TMAGIC, sizeof(hp->magic));
	    hp->name_set("././@LongLink");
	    hp->mode_set(0);
	    hp->uid_set(0);
	    hp->uname_set("root");
	    hp->gid_set(0);
	    hp->gname_set("root");
	    hp->size_set(name.length() + 1);
	    hp->mtime_set(0);
	    hp->linkflag_set(LF_LONGLINK);
	    hp->chksum_set(hp->calculate_checksum());
	    write_data(block, sizeof(block));

	    //
	    // Write a bogus data block with the file name in it.
	    //
	    write_data(h.linkname.to_c_string(), h.linkname.length() + 1);
	    write_data_padding();
	}

	//
	// Write the file header.
	//
	memset(block, 0, sizeof(block));
	memcpy(hp->magic, TMAGIC, sizeof(hp->magic));
	hp->name_set(name);

	hp->mode_set(h.mode);
	hp->uid_set(h.user_id);
	hp->gid_set(h.group_id);
	hp->size_set(h.size);
	hp->mtime_set(h.mtime);
	hp->uname_set(h.user_name);
	hp->gname_set(h.group_name);

	hp->linkflag_set(LF_NORMAL);
	switch (h.type)
	{
	case tar_header::type_normal:
		hp->linkflag_set(LF_NORMAL);
		break;

	case tar_header::type_normal_contiguous:
		hp->linkflag_set(LF_CONTIG);
		break;

	case tar_header::type_directory:
		hp->linkflag_set(LF_DIR);
		break;

	case tar_header::type_link_hard:
		hp->linkflag_set(LF_LINK);
		break;

	case tar_header::type_link_symbolic:
		hp->linkflag_set(LF_SYMLINK);
		break;

	case tar_header::type_fifo:
		hp->linkflag_set(LF_FIFO);
		break;

	case tar_header::type_device_block:
		hp->linkflag_set(LF_BLK);
		break;

	case tar_header::type_device_character:
		hp->linkflag_set(LF_CHR);
		break;

	case tar_header::type_socket:
		fatal
		(
			"\"%s\" named sockets not supported by this format",
			h.name.to_c_string()
		);
		break;
	}

	if
	(
		h.type == tar_header::type_link_hard
	||
		h.type == tar_header::type_link_symbolic
	)
	{
		if (h.linkname.length() < 1)
		{
			fatal
			(
				"linkname \"%s\" too short",
				h.linkname.to_c_string()
			);
		}
		hp->linkname_set(h.linkname);
	}

	if
	(
		h.type == tar_header::type_device_block
	||
		h.type == tar_header::type_device_character
	)
	{
		hp->devmajor_set(h.device_major);
		hp->devminor_set(h.device_minor);
	}
	else
	{
		hp->devmajor_set(0);
		hp->devminor_set(0);
	}

	hp->chksum_set(hp->calculate_checksum());

	write_data(block, sizeof(block));
}
