//
// tardy - a tar post-processor
// Copyright (C) 1991-1999, 2001, 2002, 2004, 2008, 2009 Peter Miller
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program. If not, see <http://www.gnu.org/licenses/>.
//

#include <libtardy/ac/stdio.h>
#include <libtardy/ac/stdlib.h>

#include <libtardy/version.h>
#include <libtardy/arglex.h>
#include <libtardy/error.h>
#include <libtardy/help.h>
#include <libtardy/progname.h>
#include <libtardy/rcstring.h>
#include <libtardy/trace.h>


static void
version_copyright(void)
{
    static const char *text[] =
    {
        ".so cr",
    };
    help(text, SIZEOF(text), (void (*)(void))0);
}


static void
version_license(void)
{
    static const char *text[] =
    {
#include <man1/tardy_license.h>
    };

    help(text, SIZEOF(text), (void (*)(void))0);
}


static void
version_usage(void)
{
    const char *progname = progname_get();
    fprintf(stderr, "usage: %s -VERSion [ <info-name> ]\n", progname);
    fprintf(stderr, "       %s -VERSion -Help\n", progname);
    quit(1);
}


struct table_ty
{
    const char *name;
    void (*func)(void);
};


static table_ty table[] =
{
    { "Copyright", version_copyright },
    { "License", version_license },
};


static void
version_main(void)
{
    void (*func)(void) = 0;
    trace(("version_main()\n{\n"//}
));
    const char *name = 0;
    while (arglex_token != arglex_token_eoln)
    {
        switch (arglex_token)
        {
        default:
            generic_argument(version_usage);
            continue;

        case arglex_token_string:
            if (name)
                  fatal("too many version information names given");
            name = arglex_value.alv_string;
            break;
        }
        arglex();
    }

    if (name)
    {
        int         nhit;
        table_ty    *tp;
        string_ty       *s1;
        string_ty       *s2;
        table_ty    *hit[SIZEOF(table)];
        int         j;

        nhit = 0;
        for (tp = table; tp < ENDOF(table); ++tp)
        {
            if (arglex_compare(tp->name, name))
                hit[nhit++] = tp;
        }
        switch (nhit)
        {
        case 0:
            fatal("version information name \"%s\" unknown", name);

        case 1:
            break;

        default:
            s1 = str_from_c(hit[0]->name);
            for (j = 1; j < nhit; ++j)
            {
                s2 = str_format("%s, %s", s1->str_text, hit[j]->name);
                str_free(s1);
                s1 = s2;
            }
            fatal
            (
                   "version information name \"%s\" ambiguous (%s)",
                name,
                s1->str_text
            );
        }
        arglex();
        func = hit[0]->func;
    }
    else
        func = version_copyright;

    func();
    trace((//{
"}\n"));
}


static void
version_help(void)
{
    static const char *text[] =
    {
".TH \"\\*(n) -VERSion\" 1 \\*(N)",
".SH NAME",
"\\*(n) -VERSion \\- give version information",
".SH SYNOPSIS",
".B \\*(n)",
".B -VERSion",
"[",
".IR info-name",
"]",
".br",
".B \\*(n)",
".B -VERSion",
".B -Help",
".SH DESCRIPTION",
"The",
".I \\*(n)",
".I -VERSion",
"command is used to",
"give version information",
"and conditions of use.",
".PP",
"There are a number of possible",
".IR info-name s,",
"as follow (abbreviations as for command line options):",
".TP 8n",
"Copyright",
"The copyright notice for the \\*(n) program.",
"Version information will also be printed.",
"This is the default.",
".TP 8n",
"Redistribution",
"Print the conditions of use and redistribution.",
".TP 8n",
"Warranty",
"Print the limited warranty.",
".SH OPTIONS",
"The following options are understood:",
".so o_help.so",
".so o__rules.so",
".SH RECOMMENDED ALIAS",
"The recommended alias for this command is",
".nf",
".ta 8n 16n",
"csh%\talias aev '\\*(n) -vers \\e!*'",
"sh$\taev(){\\*(n) -vers $*}",
".fi",
".SH ERRORS",
"It is an error if",
"the ",
".I info-name",
"given is unknown.",
".so z_exit.so",
".so z_cr.so",
    };

    help(text, SIZEOF(text), version_usage);
}


void
version(void)
{
    trace(("version()\n{\n"//}
));
    switch (arglex())
    {
    default:
        version_main();
        break;

    case arglex_token_help:
        version_help();
        break;
    }
    trace((//{
"}\n"));
}
