#!/usr/bin/perl
#
# var2fix.pl - Macro generator for variable parameters to fixed parameters
#              conversion
#
# Written 2001 by Werner Almesberger
# Copyright 2001 Network Robots
#


#
# We use a spectacularly ugly and restrictive macro language syntax, which
# just happens to be quite easy to parse in Perl.
#
# It goes as follow:
#
# #map <macro_name>(<free_params...>) <next_macro>
# #map <macro_name>(<fixed_params...>,,<free_params>) <next_macro>
#
# (notice the double comma between fixed_params and free_params)
#
# where <fixed_params> is <word>,...
# and <free_params> is <word>=<value>,... 
#
# <value> must not contain commas or parentheses.
#
# A #map directive can be continued on following lines by ending unfinished
# lines with a backslash.
#

#
# Known problems:
#
# - optional parameters are always provided to the underlying construct.
#   The only thing we can do is to set them to a reasonable default value.
#   This is sufficient if
#    - the parameter defaults to a well-known default anyway
#      (e.g. dsmark "mask" or "value")
#    - the parameter is ignored in the given situation, and making it
#      optional just relieves the user from having to provide this
#      parameter pro forma (e.g. "mtu" in meters if not using the "tc"
#      target)
# - error messages for missing, unrecognized, or duplicate parameters can
#   be a little obscure, i.e. they are "invalid type conversion", "syntax
#   error", and "number/handle not valid here" or "invalid type
#   conversion" respectively
# - what follows below almost certainly isn't valid CPP input, e.g a##b
#   with "b" containing whitespace is suspicious. Fortunately, most
#   versions of GNU CPP don't seem to mind.
#


#
# The structure of the macros generated by var2fix needs a bit of explaining.
#
# The parameters are processed by one macro per parameter, which stores the
# parameter value in a variable and then invokes the macros for the next
# parameter. Note that CPP normaly does not allow recursion. We work around
# this by having one separate macro for each parameter, so it just works out.
#
# Example:
#
# __set_SLB($var,cir 1Mbps,cbs 10kB);
#
# gets expanded in
# __m_set_SLB_set5(__VA_ARGS__);
# to
# __m_set_SLB_set5(cir 1Mbps,cbs 10kB);
#
# __m_set_SLB_set5 expands to
# __m_set_SLB_cir 1Mbps; __m_set_SLB_set4(cbs 10kB)
#
# __m_set_SLB_cir is simply
# $__set_SLB_cir =
# so we get
# $__set_SLB_cir = 1Mbps; __m_set_SLB_set4(cbs 10kB)
#
# "cir" is now handled, so the expansion proceeds in the same way for "cbs",
# i.e.
#
# __m_set_SLB_set4(cbs 10kB)
# __m_set_SLB_cbs = 10kB; __m_set_SLB_set3()
#
# __m_set_SLB_set3() expands to
# __m_set_SLB_; __m_set_SLB_set2()
# where __m_set_SLB_ is expanded to nothing, so any unused ...setN macros
# just yield a sequence of semicolons.
#


sub expand
{
    local ($line) = @_;

    if ($line =~ /^#map\s+(\S+)\s*\(([^)]*),,([^)]*)\)\s+(\S+)\s*$/) {
	($name,$fixed,$free,$next) = ($1,$2,$3,$4)
    }
    elsif ($line =~ /^#map\s+(\S+)\s*\(([^)]*)\)\s+(\S+)\s*$/) {
	($name,$fixed,$free,$next) = ($1,"",$2,$3)
    }
    else {
	die "unrecognized syntax";
    }
    $fixed =~ s/\s+//g;
    $fixed .= "," unless $fixed eq "";
    print "#define __m_${name}_\n\n";
    undef $defaults;
    undef $use;
    undef $pass;
    $set_num = 0;
    for (split(",",$free)) {
	($param,$value) = split("=",$_);
	$param =~ s/^\s+//;
	$param =~ s/\s+$//;
	print "#define __m_${name}_${param} \$__${name}_${param} =\n";
	$set_num++;
	$defaults .= "  \$__${name}_${param} = $value$semi \\\n";
	$use .= "  field __${name}_dummy = raw ".
	  "if \$__${name}_${param} == \$__${name}_${param}$semi \\\n";
	$pass .= "\$__${name}_${param},";
    }
    print "\n#define __m_${name}_set1(arg,...) __m_${name}_##arg$semi\n";
    for ($i = 2; $i <= $set_num; $i++) {
        print "#define __m_${name}_set$i(arg,...) \\\n".
	  "  __m_${name}_##arg$semi __m_${name}_set".($i-1).
	  "(__VA_ARGS__)\n";
    }
    print "\n";
    chop($pass);
    print "#define __$name($fixed...) \\\n";
    print "  /* set defaults */ \\\n$defaults";
    print "  /* \"use\" all variables */ \\\n$use";
    print "  /* copy parameters to variables */ \\\n";
    print "  __m_${name}_set$set_num(__VA_ARGS__)$semi \\\n";
    print "  /* invoke next macro */ \\\n";
    print "  $next($fixed$pass)\n\n";
    print "#define $name(...) __$name(__VA_ARGS__) /* expand macros */\n";
}


$semi = ";";

if ($ARGV[0] =~ /^-/) {
    print STDERR "usage: $0 [-S] [file ...]\n";
    exit(1);
}

print "/* MACHINE-GENERATED. DO NOT EDIT ! */\n\n";

while (<>) {
    if (/^#map\s/) {
	chop;
	$line = $_;
	while ($line =~ /\s*\\\s*$/) {
	    $more = <>;
	    last unless defined $more;
	    chop($more);
	    $line = $`." ".$more;
	}
	&expand($line);
    }
    else {
	print;
    }
}
exit(0);
