#!/usr/bin/perl -w

# Parses the bpf.h header file to generate the dlt_names.h header
# which maps the DLT types to the DLT string name

# run as:
# cat /usr/include/net/bpf.h | ./scripts/dlt2name.pl

use strict;
my $outfile = "dlt_names.h";

defined $ARGV[0] or die("Please specify an output file");

# open outfile
open(OUT, ">$ARGV[0]") or die("Unable to open $ARGV[0] for writing: $!");

# read STDIN

# some DLT types aren't in a format we can parse easily or just doesn't
# exist in my /usr/include/net/bpf.h file so we list them here
my %known = (107 => 'BSD/OS Frame Relay',
             108 => 'OpenBSD Loopback',
             113 => 'Linux Cooked Sockets',
             114 => 'Apple LocalTalk',
             115 => 'Acorn Econet',
             116 => 'OpenBSD IPFilter',
             117 => 'OpenBSD PF Log/SuSE 6.3 LANE 802.3',
             118 => 'Cisco IOS',
             119 => '802.11 Prism Header',
             120 => '802.11 Aironet Header',
             121 => 'Siemens HiPath HDLC',
             122 => 'IP over Fibre Channel'
            );
my @names = ( );

# put our known DLT types in names since the format of bpf.h is 
# inconsistent

foreach my $dlt (keys %known) {
  $names[$dlt] = { name => $known{$dlt} };
}

while (my $line = <STDIN>) {

  if ($line =~ /^\#define\s+(DLT_[a-zA-Z0-9_]+)\s+(\d+)/) {
    my $key = $1;
    my $dlt = $2;
    my $name = $names[$dlt]->{name} ? $names[$dlt]->{name} : "";
    if ($line =~ /\/\*\s+(.*)\s+\*\//) {
      $name = $1;
    }
    $names[$dlt] = { key => $key,
                     name => $name
                 };
  }

}

# prep the header
print OUT (<<HEADER);
/*
 * This file is generated by scripts/dlt2name.pl which converts your bpf.h
 * header file which comes with libpcap into a header file
 * which translates DLT values to their string names
 *
 * Hence DO NOT EDIT THIS FILE!
 * If your DLT type is not listed here, edit the %known hash in
 * scripts/dlt2name.pl
 */

/* DLT to descriptions */
char *dlt2desc[] = {
HEADER


for (my $i = 0; $i < $#names; $i ++) {
  if (! defined $names[$i]) {
    print OUT "\t\t\"Unknown\",\n";
  } else {
    print OUT "\t\t\"$names[$i]->{name}\",\n";
  }
}

print OUT "\t\tNULL\n};\n";
print OUT "\#define DLT2DESC_LEN $#names\n";
close OUT;

exit 0;
