/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * main.h:
 * Application entry point
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */


#include <stdlib.h>
#include <dbus/dbus-glib.h>
#include <gtk/gtk.h>
#include <string.h>
#include <glib/gprintf.h>

#include "ti-wnd-main.h"
#include "ti-signals-marshal.h"
#include "ti-config.h"

/* Function prototypes */
static void _ti_register_dbus_marshallers ();
static gboolean _ti_proccess_shell_arguments (int argc, char **argv);

/**
 * Application entry point
 */
int
main (int argc, char **argv)
{
    TIWndMain* wnd_main = NULL;
    gboolean keep_going;

    g_type_init ();

    g_set_prgname ("telepathy-inspector");
    g_set_application_name ("Telepathy Inspector");

    keep_going = _ti_proccess_shell_arguments (argc, argv);
    if (!keep_going)
        goto CLEAN_UP;

    gtk_init (&argc, &argv);

    dbus_g_type_specialized_init ();

    _ti_register_dbus_marshallers ();

    wnd_main = ti_wnd_main_new ();
    if (wnd_main == NULL)
        goto CLEAN_UP;

    ti_wnd_main_show (wnd_main);

    gtk_main ();

    CLEAN_UP:
    if (wnd_main != NULL)
        g_object_unref (wnd_main);

    return 0;
}

/**
 * Register D-Bus Marshallers
 */
static void _ti_register_dbus_marshallers ()
{
    GError* error;
    DBusGConnection* dbus_connection = NULL;

    // <ugly-hack>
    // It's *ugly* but I do have to get a dbus connection before calling any dbus_g_object_register_marshaller() function
    // even though that connection pointer never get used.
    // See bug #7826 at https://bugs.freedesktop.org
    error = NULL;
    dbus_connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (dbus_connection == NULL)
    {
        g_printerr ("Failed to open connection to bus: %s\n", error->message);
        g_error_free (error);
        exit (1);
    }
    // </ugly-hack>

    dbus_g_object_register_marshaller (ti_marshal_VOID__UINT_UINT_UINT_UINT_UINT_STRING,
                                       G_TYPE_NONE,
                                       G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);

    dbus_g_object_register_marshaller (ti_marshal_VOID__UINT_UINT_STRING,
                                       G_TYPE_NONE,
                                       G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
                                       
    dbus_g_object_register_marshaller (ti_marshal_VOID__UINT_UINT,
                                       G_TYPE_NONE,
                                       G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);

    dbus_g_object_register_marshaller (ti_marshal_VOID__STRING_BOXED_BOXED_BOXED_BOXED_UINT_UINT,
                                       G_TYPE_NONE,
                                       G_TYPE_STRING, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, DBUS_TYPE_G_UINT_ARRAY, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);
}

/**
 * Process Shell Arguments
 *
 * @return Whether the application should start.
 */
static gboolean
_ti_proccess_shell_arguments (int argc, char **argv)
{
    gboolean run_application = TRUE;
    GOptionContext* option_context = NULL;
    gboolean display_version = FALSE;
    GError* error = NULL;

    GOptionEntry entries[] = 
    {
        { "version", 0, 0, G_OPTION_ARG_NONE, &display_version, "Display application version.", NULL },
        { NULL }
    };

    option_context = g_option_context_new ("\n\nTelepathy Inspector is a telepathy client (GTK+) whose objective is to\nexpose all interfaces and functionalities implemented by a given\nconnection manager along with its connections, channels, etc.");
    g_option_context_add_main_entries (option_context, entries, NULL);
    g_option_context_add_group (option_context, gtk_get_option_group (TRUE));
    g_option_context_parse (option_context, &argc, &argv, &error);

    if (error == NULL)
    {
        if (display_version)
        {
            g_printf ("Telepathy Inspector, version %s\nCopyright (C) 2006 INdT - Instituto Nokia de Tecnologia\nThis software is released under the GNU Lesser General Public License\n", TI_VERSION_STRING);
            run_application = FALSE;
        }
    }
    else
    {
        g_printf ("%s\n", error->message);
        run_application = FALSE;
    }

    if (option_context != NULL)
        g_option_context_free (option_context);

    if (error != NULL)
        g_error_free (error);

    return run_application;
}
