/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-hold-handles.c:
 * Dialog for [...].Telepathy.Connection.HoldHandles() method.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-hold-handles.h"
#include "ti-util.h"
#include "ti-config.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgHoldHandles, ti_dlg_hold_handles, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgHoldHandlesPrivate {
    gboolean disposed;

    GtkWindow* parent;

    GladeXML* glade_xml;

    GtkWidget* dialog;

    GtkComboBox* combo_handle_type;
    GtkEntry* entry_handles;
};
typedef struct _TIDlgHoldHandlesPrivate TIDlgHoldHandlesPrivate;

#define TI_DLG_HOLD_HANDLES_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_HOLD_HANDLES, TIDlgHoldHandlesPrivate))

/* Function prototypes */
static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);

/**
 * Drop all references to other objects.
 */
static void
ti_dlg_hold_handles_dispose (GObject *object)
{
    TIDlgHoldHandles *dlg_hold_handles = TI_DLG_HOLD_HANDLES (object);
    TIDlgHoldHandlesPrivate *priv = TI_DLG_HOLD_HANDLES_GET_PRIVATE (dlg_hold_handles);

    if (priv->disposed) {
        return;
    } else {
        priv->disposed = TRUE;
    }

    if (priv->glade_xml != NULL) {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    G_OBJECT_CLASS (ti_dlg_hold_handles_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_hold_handles_class_init (TIDlgHoldHandlesClass *ti_dlg_hold_handles_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_hold_handles_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_hold_handles_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_hold_handles_class, sizeof (TIDlgHoldHandlesPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_hold_handles_init (TIDlgHoldHandles *ti_dlg_hold_handles)
{
    TIDlgHoldHandlesPrivate *priv = TI_DLG_HOLD_HANDLES_GET_PRIVATE (ti_dlg_hold_handles);

    priv->disposed = FALSE;
}

/**
 * Returns a new instance.
 */
TIDlgHoldHandles*
ti_dlg_hold_handles_new (GtkWindow* parent) 
{
    TIDlgHoldHandles* dialog = NULL;
    gchar* glade_file_path = NULL;
    TIDlgHoldHandlesPrivate* priv = NULL;

	dialog = g_object_new (TI_TYPE_DLG_HOLD_HANDLES, NULL);
    priv = TI_DLG_HOLD_HANDLES_GET_PRIVATE (dialog);

    priv->parent = parent;

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-hold-handles.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_hold_handles");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    priv->combo_handle_type = GTK_COMBO_BOX (glade_xml_get_widget (priv->glade_xml, "combo_handle_type"));
    g_assert (GTK_IS_COMBO_BOX (priv->combo_handle_type));

    priv->entry_handles = GTK_ENTRY (glade_xml_get_widget (priv->glade_xml, "entry_handles"));
    g_assert (GTK_IS_ENTRY (priv->entry_handles));
    
    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));
    gtk_window_set_position (GTK_WINDOW (priv->dialog), GTK_WIN_POS_CENTER_ON_PARENT);

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(_button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(_button_cancel_clicked), priv->dialog);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
gboolean
ti_dlg_hold_handles_run (TIDlgHoldHandles* dlg_hold_handles, guint* handle_type, GArray** handles)
{
    TIDlgHoldHandlesPrivate *priv = TI_DLG_HOLD_HANDLES_GET_PRIVATE (dlg_hold_handles);
    gint result;
    gint active_handle_type;

    g_assert (handle_type != NULL);
    g_assert (handles != NULL);

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    // Get handle type
    active_handle_type = gtk_combo_box_get_active (priv->combo_handle_type);
    if (active_handle_type < 0)
    {
        *handles = NULL;
        return FALSE;
    }

    *handle_type = (guint) active_handle_type;

    // Get handle numbers
    *handles = ti_parse_uint_list (gtk_entry_get_text (priv->entry_handles));

    if (*handles == NULL)
    {
        return FALSE;
    }
    else
    {
        return result == GTK_RESPONSE_OK;
    }
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}
