/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-request-conn.c:
 * Request Connection Dialog
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-request-conn.h"
#include "ti-config.h"
#include "ti-wdg-generic-conn-param.h"
#include "ti-wdg-boolean-conn-param.h"
#include "ti-util.h"

#include <glade/glade.h>
#include <string.h>
#include <stdlib.h>

G_DEFINE_TYPE (TIDlgRequestConn, ti_dlg_request_conn, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgRequestConnPrivate {
    gboolean disposed;

    GtkWindow* parent;

    GladeXML* glade_xml;

    GtkWidget* dialog;

    GtkTable* table_conn_params;

    GtkLabel* label_protocol_value;

    GPtrArray* array_param_widgets; // Stores TIWdgConnParam objects.
};
typedef struct _TIDlgRequestConnPrivate TIDlgRequestConnPrivate;

#define TI_DLG_REQUEST_CONN_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_REQUEST_CONN, TIDlgRequestConnPrivate))

/* Function prototypes */
void button_ok_clicked (GtkButton *button, gpointer user_data);
void button_cancel_clicked (GtkButton *button, gpointer user_data);
static gboolean _ti_dlg_request_conn_add_param_entries (TIDlgRequestConn* dlg_request_conn, GtkTreeModel* tree_model);
static void _ti_dlg_request_conn_remove_param_entries (TIDlgRequestConn* dlg_request_conn);
static void _ti_dlg_request_conn_fill_params_table (TIDlgRequestConn* dlg_request_conn);
static void _ti_dlg_request_conn_fetch_results (TIDlgRequestConn* dlg_request_conn, GHashTable** param_values);

/**
 * Drop all references to other objects.
 */
static void
ti_dlg_request_conn_dispose (GObject *object)
{
    TIDlgRequestConn *dlg_request_conn = TI_DLG_REQUEST_CONN (object);
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dlg_request_conn);

    if (priv->disposed)
        return;
    else
        priv->disposed = TRUE;

    if (priv->glade_xml != NULL)
    {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    G_OBJECT_CLASS (ti_dlg_request_conn_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_request_conn_class_init (TIDlgRequestConnClass *ti_dlg_request_conn_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_request_conn_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_request_conn_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_request_conn_class, sizeof (TIDlgRequestConnPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_request_conn_init (TIDlgRequestConn *ti_dlg_request_conn)
{
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (ti_dlg_request_conn);

    priv->disposed = FALSE;

    priv->array_param_widgets = NULL;
}

/**
 * Returns a new instance.
 */
TIDlgRequestConn*
ti_dlg_request_conn_new (GtkWindow* parent) 
{
    TIDlgRequestConn* dialog = NULL;
    TIDlgRequestConnPrivate* priv = NULL;
    gchar* glade_file_path = NULL;

	dialog = g_object_new (TI_TYPE_DLG_REQUEST_CONN, NULL);
    priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dialog);
    priv->parent = parent;

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-request-conn.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_request_connection");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));

    priv->label_protocol_value = GTK_LABEL (glade_xml_get_widget (priv->glade_xml, "label_protocol_value"));
    g_assert (GTK_IS_LABEL (priv->label_protocol_value));

    priv->table_conn_params = GTK_TABLE (glade_xml_get_widget (priv->glade_xml, "table_jabber_connection_params"));
    g_assert (GTK_IS_TABLE (priv->table_conn_params));

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(button_cancel_clicked), priv->dialog);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
gboolean
ti_dlg_request_conn_run (TIDlgRequestConn* dlg_request_conn, const gchar* protocol_name,
                         GtkTreeModel* tree_model, GHashTable** param_values)
{
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dlg_request_conn);
    gint result;
    gboolean ok;

    gtk_label_set_text (priv->label_protocol_value, protocol_name);

    ok = _ti_dlg_request_conn_add_param_entries (dlg_request_conn, tree_model);
    if (!ok)
        return FALSE;

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    _ti_dlg_request_conn_fetch_results (dlg_request_conn, param_values);
    _ti_dlg_request_conn_remove_param_entries (dlg_request_conn);

    return result == GTK_RESPONSE_OK;
}

/**
 * Add Parameter Entries
 * Adds UI data entries (and their respective labels) for the parameters specified in the GtkTreeModel.
 *
 * @return TRUE if succeeds and FALSE otherwise.
 */
static gboolean
_ti_dlg_request_conn_add_param_entries (TIDlgRequestConn* dlg_request_conn, GtkTreeModel* tree_model)
{
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dlg_request_conn);
    gboolean ok;
    GtkTreeIter iter;
    TIWdgConnParam* param_widgets;
    gchar* name = NULL;
    gchar* def_val = NULL;
    gchar* type_sig = NULL;

    g_assert (priv->array_param_widgets == NULL);
    priv->array_param_widgets = g_ptr_array_sized_new (10 /* yeah, a magic number */);

    ok = gtk_tree_model_get_iter_first (tree_model, &iter);
    g_return_val_if_fail(ok, FALSE);

    // Create widgets
    while (ok)
    {
        gtk_tree_model_get (tree_model, &iter,
                            0, &name,
                            2, &type_sig,
                            3, &def_val,
                            -1);

        if (g_str_equal (type_sig, "b"))
        {
            param_widgets = ti_wdg_boolean_conn_param_new (name, def_val);
        }
        else
        {
            param_widgets = ti_wdg_generic_conn_param_new (name, def_val, type_sig);
        }

        g_free (name);
        g_free (type_sig);
        g_free (def_val);

        g_ptr_array_add (priv->array_param_widgets, param_widgets);

        ok = gtk_tree_model_iter_next (tree_model, &iter);
    }

    _ti_dlg_request_conn_fill_params_table (dlg_request_conn);

    return TRUE;
}

/**
 * Fill Parameters Table
 * Fill connection params widget table with parameters' entries (and their respective labels).
 */
static void
_ti_dlg_request_conn_fill_params_table (TIDlgRequestConn* dlg_request_conn)
{
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dlg_request_conn);
    guint curr_row;
    guint i;
    TIWdgConnParam* param_widgets;
 
    gtk_table_resize (priv->table_conn_params,
                      priv->array_param_widgets->len + 1 /* there's already one row displaying the protocol name*/,
                      2 /* |label|entry| */);

    curr_row = 1;
    for (i = 0; i < priv->array_param_widgets->len; i++) {

        param_widgets = TI_WDG_CONN_PARAM (g_ptr_array_index (priv->array_param_widgets, i));

        gtk_table_attach (priv->table_conn_params,
                          GTK_WIDGET (param_widgets->label),
                          0,            /* left_attach */
                          1,            /* right_attach */
                          curr_row,     /* top_attach */
                          curr_row + 1, /* bottom_attach */
                          GTK_FILL,     /* xoptions */
                          GTK_FILL,     /* yoptions */
                          0,            /* xpadding */
                          0             /* ypadding */ );

        gtk_table_attach (priv->table_conn_params,
                          GTK_WIDGET (param_widgets->value),
                          1,            /* left_attach */
                          2,            /* right_attach */
                          curr_row,     /* top_attach */
                          curr_row + 1, /* bottom_attach */
                          GTK_FILL,     /* xoptions */
                          GTK_FILL,     /* yoptions */
                          0,            /* xpadding */
                          0             /* ypadding */ );

        curr_row++;
    }
}

/**
 * Fetch Results
 * Fill param_values with the values entered by the user.
 *
 * @param tree_model (in)
 * @param param_values (out)
 */
static void
_ti_dlg_request_conn_fetch_results (TIDlgRequestConn* dlg_request_conn, GHashTable** param_values)
{
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dlg_request_conn);
    TIWdgConnParam* param_widgets = NULL;
    guint i;
    GValue* value = NULL;
    gboolean is_active;
    const gchar* name = NULL;

    g_assert (param_values != NULL);

    *param_values = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, ti_value_destroy);

    for (i = 0; i < priv->array_param_widgets->len; i++)
    {
        param_widgets = TI_WDG_CONN_PARAM (g_ptr_array_index (priv->array_param_widgets, i));

        is_active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (param_widgets->label));
        if (!is_active)
            continue; // We will only add checked (active) parameters in the results.

        value = ti_wdg_conn_param_get_value (param_widgets);
        name = ti_wdg_conn_param_get_name (param_widgets);

        g_hash_table_insert (*param_values,
                             g_strdup (name),
                             value);
    }
}

/**
 * Remove Parameter Entries
 * Removes all existing UI data entries.
 */
static void
_ti_dlg_request_conn_remove_param_entries (TIDlgRequestConn* dlg_request_conn)
{
    TIDlgRequestConnPrivate *priv = TI_DLG_REQUEST_CONN_GET_PRIVATE (dlg_request_conn);
    GtkContainer* table_conn_params = GTK_CONTAINER (priv->table_conn_params);
    guint i;
    TIWdgConnParam* param_widgets;
    
    for (i = 0; i < priv->array_param_widgets->len; i++) {

        param_widgets = TI_WDG_CONN_PARAM (g_ptr_array_index (priv->array_param_widgets, i));

        gtk_container_remove (table_conn_params, param_widgets->label);
        gtk_container_remove (table_conn_params, param_widgets->value);

        g_object_unref (param_widgets);
    }

    g_ptr_array_free (priv->array_param_widgets, TRUE);
    priv->array_param_widgets = NULL;
}

/**
 * Called when OK button is clicked.
 */
void
button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
void
button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}
