/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-page-active-conns.c:
 * A GtkNotebook page that shows all currently active Telepathy connections.
 * i.e., all D-Bus services which represents Telepathy connections.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-page-active-conns.h"
#include "ti-page-priv.h"
#include "ti-message-bus.h"
#include "ti-wnd-connection.h"
#include "ti-config.h"

#include <glade/glade.h>
#include <string.h>

struct _TIPageActiveConnsClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageActiveConns, ti_page_active_conns, TI_TYPE_PAGE);

/* Function prototypes */
static void _ti_page_active_conns_setup_page (TIPage* page, GladeXML* glade_xml);
static void _ti_page_active_conns_build_treeview (TIPageActiveConns* page_active_conns, GladeXML* glade_xml);
static void _ti_page_active_conns_update_button_show_sensitivity (TIPageActiveConns* page_active_conns);
static void _ti_page_active_conns_show_selected_connection (TIPageActiveConns* page_active_conns);
static void _ti_page_active_conns_refresh_list (TIPageActiveConns* page_active_conns);

/**
 * Instance private data.
 */
struct _TIPageActiveConnsPrivate {
    gboolean disposed;

    TIMessageBus* message_bus;

    GPtrArray* conn_windows; // Array of TIWndConnection*

    GtkWindow* parent_wnd;

    GtkListStore* list_store;
    GtkTreeSelection* tree_selection;

    GtkButton* button_show;
    GtkButton* button_refresh;
};
typedef struct _TIPageActiveConnsPrivate TIPageActiveConnsPrivate;

#define TI_PAGE_ACTIVE_CONNS_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PAGE_ACTIVE_CONNS, TIPageActiveConnsPrivate))

/**
 * Drop all references to other objects.
 */
static void
ti_page_active_conns_dispose (GObject *object)
{
    TIPageActiveConns *page_active_conns = TI_PAGE_ACTIVE_CONNS (object);
    TIPageActiveConnsPrivate *priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);
    GObject* obj;
    guint i;

    if (priv->disposed)
        return;
    else
        priv->disposed = TRUE;

    if (priv->message_bus != NULL)
    {
        g_object_unref (priv->message_bus);
        priv->message_bus = NULL;
    }

    if (priv->conn_windows != NULL)
    {
        for (i = 0; i < priv->conn_windows->len; i++)
        {
            obj = G_OBJECT (g_ptr_array_index (priv->conn_windows, i));
            g_object_unref (obj);
        }

        g_ptr_array_free (priv->conn_windows, TRUE);
        priv->conn_windows = NULL;
    }

    G_OBJECT_CLASS (ti_page_active_conns_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_page_active_conns_class_init (TIPageActiveConnsClass *ti_page_active_conns_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_page_active_conns_class);
    TIPageClass* page_class = TI_PAGE_CLASS (ti_page_active_conns_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_page_active_conns_dispose;

    page_class->setup_page = _ti_page_active_conns_setup_page;
		
	/* Add private */
	g_type_class_add_private (ti_page_active_conns_class, sizeof (TIPageActiveConnsPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_page_active_conns_init (TIPageActiveConns *ti_page_active_conns)
{
    TIPageActiveConnsPrivate *priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (ti_page_active_conns);

    priv->disposed = FALSE;
    priv->message_bus = NULL;
    priv->conn_windows = g_ptr_array_new ();
}

/**
 * Returns a new instance.
 */
TIPageActiveConns*
ti_page_active_conns_new (GtkWindow* parent_wnd, GtkNotebook* parent_notebook) 
{
    TIPageActiveConns* page_active_conns;
    TIPageActiveConnsPrivate *priv;

	page_active_conns = g_object_new (TI_TYPE_PAGE_ACTIVE_CONNS, NULL);

    priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);

    priv->parent_wnd = parent_wnd;

    priv->message_bus = ti_message_bus_new ();
    if (priv->message_bus == NULL)
    {
        g_object_unref (page_active_conns);
        page_active_conns = NULL;
        return NULL;
    }

    _ti_page_new ((TIPage**)&page_active_conns, parent_notebook, "page-active-conns.xml");

    return page_active_conns;
}

/**
 * Setup Page - Helper Function
 */
static void
_ti_page_active_conns_setup_page (TIPage* page, GladeXML* glade_xml)
{
    TIPageActiveConns* page_active_conns = TI_PAGE_ACTIVE_CONNS (page);
    TIPageActiveConnsPrivate *priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);

    // Active connections' list
    _ti_page_active_conns_build_treeview (page_active_conns, glade_xml);

    // Button "Refresh"
    priv->button_refresh = GTK_BUTTON (glade_xml_get_widget (glade_xml, "button_refresh"));
    g_assert (GTK_IS_BUTTON (priv->button_refresh));
    g_signal_connect_swapped (priv->button_refresh, "clicked", G_CALLBACK (_ti_page_active_conns_refresh_list), page_active_conns);

    // Button "Show"
    priv->button_show = GTK_BUTTON (glade_xml_get_widget (glade_xml, "button_show"));
    g_assert (GTK_IS_BUTTON (priv->button_show));
    g_signal_connect_swapped (priv->button_show, "clicked", G_CALLBACK (_ti_page_active_conns_show_selected_connection), page_active_conns);
}

/**
 * Build Treeview
 */
static void
_ti_page_active_conns_build_treeview (TIPageActiveConns* page_active_conns, GladeXML* glade_xml)
{
    TIPageActiveConnsPrivate *priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);
    GtkWidget* treeview;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    treeview = glade_xml_get_widget(glade_xml, "treeview_connections");

    priv->list_store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
    gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), GTK_TREE_MODEL (priv->list_store));

    renderer = gtk_cell_renderer_text_new ();

    column = gtk_tree_view_column_new_with_attributes ("Conn. Man.",
                                                       renderer,
                                                       "text", 0,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

    column = gtk_tree_view_column_new_with_attributes ("Protocol",
                                                       renderer,
                                                       "text", 1,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

    column = gtk_tree_view_column_new_with_attributes ("Account",
                                                       renderer,
                                                       "text", 2,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

    priv->tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

    g_signal_connect_swapped (priv->tree_selection, "changed",
                              G_CALLBACK (_ti_page_active_conns_update_button_show_sensitivity),
                              page_active_conns);

    gtk_tree_selection_set_mode (priv->tree_selection, GTK_SELECTION_SINGLE);
}

/**
 * Update Button Show Sensitivity
 */
static void
_ti_page_active_conns_update_button_show_sensitivity (TIPageActiveConns* page_active_conns)
{
    TIPageActiveConnsPrivate *priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);
    gboolean has_selection;

    has_selection = gtk_tree_selection_get_selected (priv->tree_selection,  NULL, NULL);
    gtk_widget_set_sensitive (GTK_WIDGET (priv->button_show), has_selection);
}

/**
 * Show Selected Connection
 */
static void
_ti_page_active_conns_show_selected_connection (TIPageActiveConns* page_active_conns)
{
    TIPageActiveConnsPrivate* priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);
    TIWndConnection* wnd_connection = NULL;
    gchar* conn_man = NULL;
    gchar* protocol = NULL;
    gchar* account = NULL;
    GtkTreeIter iter;
    gboolean ok;

    ok = gtk_tree_selection_get_selected (priv->tree_selection, NULL, &iter);
    g_assert (ok);

    gtk_tree_model_get (GTK_TREE_MODEL (priv->list_store), &iter,
                        0, &conn_man,
                        1, &protocol,
                        2, &account,
                        -1);

    wnd_connection = ti_wnd_connection_new (priv->parent_wnd, conn_man, protocol, account);
    if (wnd_connection == NULL)
        return;

    g_ptr_array_add (priv->conn_windows, wnd_connection);

    ti_wnd_connection_show (wnd_connection);

    // Clean up
    g_free (conn_man);
    g_free (protocol);
    g_free (account);
}

/**
 * Refresh List
 */
static void
_ti_page_active_conns_refresh_list (TIPageActiveConns* page_active_conns)
{
    TIPageActiveConnsPrivate* priv = TI_PAGE_ACTIVE_CONNS_GET_PRIVATE (page_active_conns);
    GArray* active_conns = NULL;
    TIActiveConn active_conn;
    GtkTreeIter iter;
    guint i;
    GError* error = NULL;

    gtk_list_store_clear (priv->list_store);

    active_conns = ti_message_bus_list_active_conns (priv->message_bus, &error);
    if (error != NULL)
        goto CLEAN_UP;

    /* Fill the list. */
    for (i = 0; i < active_conns->len; i++)
    {
        active_conn = g_array_index (active_conns, TIActiveConn, i);
        
        gtk_list_store_append (priv->list_store, &iter);
        gtk_list_store_set (priv->list_store, &iter,
                            0, active_conn.conn_man,
                            1, active_conn.protocol,
                            2, active_conn.account,
                            -1);
    }

    CLEAN_UP:
    if (active_conns != NULL)
    {
        for (i = 0; i < active_conns->len; i++)
        {
            active_conn = g_array_index (active_conns, TIActiveConn, i);
            g_free (active_conn.conn_man);
            g_free (active_conn.protocol);
            g_free (active_conn.account);
        }

        g_array_free (active_conns, TRUE);
    }
}
