/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-page-aliasing.c:
 * A GtkNotebook page exposing
 * org.freedesktop.Telepathy.Connection.Interface.Aliasing functionality.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-page-aliasing.h"
#include "ti-page-priv.h"
#include "ti-preferences.h"
#include "ti-constants.h"
#include "ti-dlg-handles-list.h"
#include "ti-dlg-set-aliases.h"

#include <glade/glade.h>

enum {
    TI_COLUMN_ALIASES_LIST_HANDLE = 0,
    TI_COLUMN_ALIASES_LIST_NAME,
    TI_COLUMN_ALIASES_LIST_ALIAS
};


struct _TIPageAliasingClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageAliasing, ti_page_aliasing, TI_TYPE_PAGE);

/* Function prototypes */
static void _ti_page_aliasing_setup_page (TIPage* page, GladeXML* glade_xml);
static void _ti_page_aliasing_restart_page (TIPage* page);
static void _ti_page_aliasing_build_treeview (TIPageAliasing* self, GladeXML* glade_xml);
static void _ti_page_aliasing_request_aliases (TIPageAliasing* self);
static void _ti_page_aliasing_set_aliases (TIPageAliasing* self);
static void _ti_page_aliasing_contact_handle_display_mode_changed (TIPageAliasing* self, guint contact_handle_display_mode);
static void _ti_page_aliasing_aliases_changed (TIPageAliasing* self, GPtrArray* aliases);
static void _ti_page_aliasing_set_alias (TIPageAliasing* self, guint contact_handle, const gchar* alias);
static void _ti_page_aliasing_clear_aliases (TIPageAliasing* self);

/**
 * Instance private data.
 */
struct _TIPageAliasingPrivate {
    GtkWindow* parent_wnd;

    TIConnection* connection;
    TIPreferences* preferences;
    TIHandleMapper* handle_mapper;
    TIDlgHandlesList* dlg_handles_list;
    TIDlgSetAliases* dlg_set_aliases;

    GtkTreeView* tree_view;
    GtkListStore* list_store;
    GtkTreeSelection* tree_selection;
};
typedef struct _TIPageAliasingPrivate TIPageAliasingPrivate;

#define TI_PAGE_ALIASING_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PAGE_ALIASING, TIPageAliasingPrivate))

/**
 * Drop all references to other objects.
 */
static void
ti_page_aliasing_dispose (GObject *object)
{
    TIPageAliasing *page_aliasing = TI_PAGE_ALIASING (object);
    TIPageAliasingPrivate *priv = TI_PAGE_ALIASING_GET_PRIVATE (page_aliasing);

    if (priv->connection != NULL)
    {
        g_signal_handlers_disconnect_by_func(priv->connection,
                                             G_CALLBACK (_ti_page_aliasing_aliases_changed),
                                             page_aliasing);

        g_object_unref (priv->connection);
        priv->connection = NULL;
    }

    if (priv->preferences != NULL)
    {
        g_signal_handlers_disconnect_by_func(priv->preferences,
                                             G_CALLBACK (_ti_page_aliasing_contact_handle_display_mode_changed),
                                             page_aliasing);
        g_object_unref (priv->preferences);
        priv->preferences = NULL;
    }

    if (priv->handle_mapper != NULL)
    {
        g_object_unref (priv->handle_mapper);
        priv->handle_mapper = NULL;
    }

    if (priv->dlg_handles_list != NULL)
    {
        g_object_unref (priv->dlg_handles_list);
        priv->dlg_handles_list = NULL;
    }

    if (priv->dlg_set_aliases != NULL)
    {
        g_object_unref (priv->dlg_set_aliases);
        priv->dlg_set_aliases = NULL;
    }

    G_OBJECT_CLASS (ti_page_aliasing_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_page_aliasing_class_init (TIPageAliasingClass* page_aliasing_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (page_aliasing_class);
    TIPageClass* page_class = TI_PAGE_CLASS (page_aliasing_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_page_aliasing_dispose;

    page_class->setup_page = _ti_page_aliasing_setup_page;
    page_class->restart_page = _ti_page_aliasing_restart_page;
		
	/* Add private */
	g_type_class_add_private (page_aliasing_class, sizeof (TIPageAliasingPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_page_aliasing_init (TIPageAliasing* self)
{
    TIPageAliasingPrivate *priv = TI_PAGE_ALIASING_GET_PRIVATE (self);

    priv->connection = NULL;
    priv->preferences = ti_preferences_new ();
    priv->handle_mapper = NULL;
    priv->dlg_handles_list = NULL;
    priv->dlg_set_aliases = NULL;
}

/**
 * Returns a new instance.
 */
TIPageAliasing*
ti_page_aliasing_new (GtkWindow* parent_wnd, GtkNotebook* parent_notebook, TIConnection* connection, TIHandleMapper* handle_mapper) 
{
    TIPageAliasing* page_aliasing;
    TIPageAliasingPrivate *priv;

	page_aliasing = g_object_new (TI_TYPE_PAGE_ALIASING, NULL);

    priv = TI_PAGE_ALIASING_GET_PRIVATE (page_aliasing);

    priv->parent_wnd = parent_wnd;

    priv->connection = connection;
    g_object_ref (connection);

    priv->handle_mapper = handle_mapper;
    g_object_ref (handle_mapper);


    g_signal_connect_swapped (priv->connection, "aliases-changed",
                              G_CALLBACK (_ti_page_aliasing_aliases_changed), page_aliasing);

    g_signal_connect_swapped (priv->preferences, "contact-handle-display-changed",
                              G_CALLBACK (_ti_page_aliasing_contact_handle_display_mode_changed), page_aliasing);

    _ti_page_new ((TIPage**)&page_aliasing, parent_notebook, "page-aliasing.xml");

    return page_aliasing;
}

/**
 * Setup Page
 */
static void
_ti_page_aliasing_setup_page (TIPage* page, GladeXML* glade_xml)
{
    TIPageAliasing* self = TI_PAGE_ALIASING (page);
    TIPageAliasingPrivate* priv = TI_PAGE_ALIASING_GET_PRIVATE (self);
    GtkWidget* widget;
    gchar* str = NULL;
    guint flags;
    GError* error = NULL;

    priv->dlg_handles_list = ti_dlg_handles_list_new (priv->parent_wnd, priv->handle_mapper, "Request Aliases");
    priv->dlg_set_aliases = ti_dlg_set_aliases_new (priv->parent_wnd, priv->handle_mapper);

    // Alias Flags
    widget = glade_xml_get_widget (glade_xml, "label_flags_value");
    g_assert (widget != NULL && GTK_IS_LABEL (widget));
    flags = ti_connection_ialiasing_get_alias_flags (priv->connection, &error);
    if (error != NULL)
    {
        str = g_strdup ("error consulting CM");
    }
    else
    {
        str = g_strdup_printf ("%u", flags);
    }
    gtk_label_set_text (GTK_LABEL (widget), str);
    g_free (str);
    str = NULL;

    // Alias list
    _ti_page_aliasing_build_treeview (self, glade_xml);

    // Button "Request Aliases"
    widget = glade_xml_get_widget (glade_xml, "button_request_aliases");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_aliasing_request_aliases), self);

    // Button "Set Aliases"
    widget = glade_xml_get_widget (glade_xml, "button_set_aliases");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_aliasing_set_aliases), self);

    // Button "Clear"
    widget = glade_xml_get_widget (glade_xml, "button_clear");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_aliasing_clear_aliases), self);

    // Clean up
    if (error != NULL)
        g_error_free (error);
}

/**
 * Restart Page
 */
static void
_ti_page_aliasing_restart_page (TIPage* page)
{
    TIPageAliasing* self = TI_PAGE_ALIASING (page);
    TIPageAliasingPrivate* priv = TI_PAGE_ALIASING_GET_PRIVATE (self);

    gtk_list_store_clear (priv->list_store);
}

/**
 * Build Treeview
 */
static void
_ti_page_aliasing_build_treeview (TIPageAliasing* self, GladeXML* glade_xml)
{
    TIPageAliasingPrivate *priv = TI_PAGE_ALIASING_GET_PRIVATE (self);
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    priv->tree_view = GTK_TREE_VIEW (glade_xml_get_widget(glade_xml, "treeview_aliases"));
    g_assert (priv->tree_view != NULL && GTK_IS_TREE_VIEW (priv->tree_view));

    priv->list_store = gtk_list_store_new (3, G_TYPE_UINT,    // handle
                                              G_TYPE_STRING,  // name
                                              G_TYPE_STRING); // alias
    gtk_tree_view_set_model (priv->tree_view, GTK_TREE_MODEL (priv->list_store));

    renderer = gtk_cell_renderer_text_new ();

    if (ti_preferences_get_contact_handle_display_mode (priv->preferences) ==
        TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE)
    {
        column = gtk_tree_view_column_new_with_attributes ("Handle",
                                                           renderer,
                                                           "text", TI_COLUMN_ALIASES_LIST_HANDLE,
                                                           NULL);
    }
    else
    {
        column = gtk_tree_view_column_new_with_attributes ("Name",
                                                           renderer,
                                                           "text", TI_COLUMN_ALIASES_LIST_NAME,
                                                           NULL);
    }
    gtk_tree_view_append_column (priv->tree_view, column);

    column = gtk_tree_view_column_new_with_attributes ("Alias",
                                                       renderer,
                                                       "text", TI_COLUMN_ALIASES_LIST_ALIAS,
                                                       NULL);
    gtk_tree_view_append_column (priv->tree_view, column);

    priv->tree_selection = gtk_tree_view_get_selection (priv->tree_view);
    gtk_tree_selection_set_mode (priv->tree_selection, GTK_SELECTION_MULTIPLE);
}

/**
 * Request Aliases
 */
static void
_ti_page_aliasing_request_aliases (TIPageAliasing* self)
{
    TIPageAliasingPrivate *priv = TI_PAGE_ALIASING_GET_PRIVATE (self);
    gboolean ok;
    GArray* handles = NULL;
    gchar** aliases = NULL;
    GError* error = NULL;
    guint i;
    guint handle;
    const gchar* alias;

    ok = ti_dlg_handles_list_run (priv->dlg_handles_list, &handles);
    if (!ok)
    {
        goto CLEAN_UP;
    }

    aliases = ti_connection_ialiasing_request_aliases (priv->connection, handles, &error);
    if (error != NULL)
    {
        // TODO: Display a message or something.
        goto CLEAN_UP;
    }

    for (i = 0; i < handles->len; i++)
    {
        handle = g_array_index (handles, guint, i);
        alias = aliases[i];

        _ti_page_aliasing_set_alias (self, handle, alias);
    }

    CLEAN_UP:
    g_array_free (handles, TRUE);
    g_strfreev (aliases);

    if (error != NULL)
        g_error_free (error);
}

/**
 * Set Aliases
 */
static void
_ti_page_aliasing_set_aliases (TIPageAliasing* self)
{
    TIPageAliasingPrivate *priv = TI_PAGE_ALIASING_GET_PRIVATE (self);
    GHashTable* aliases = NULL;
    gboolean ok;
    GError* error;

    ok = ti_dlg_set_aliases_run (priv->dlg_set_aliases, &aliases);
    if (!ok)
    {
        goto CLEAN_UP;
    }

    ti_connection_ialiasing_set_aliases (priv->connection, aliases, &error);
    if (error != NULL)
    {
        // TODO: Display a message or something.
    }

    CLEAN_UP:

    if (aliases != NULL)
        g_hash_table_destroy (aliases);
}

/**
 * Contact Handle Display Mode Changed
 */
static void
_ti_page_aliasing_contact_handle_display_mode_changed (TIPageAliasing* self, guint contact_handle_display_mode)
{
    TIPageAliasingPrivate* priv = TI_PAGE_ALIASING_GET_PRIVATE (self);
    GtkTreeViewColumn* contact_handle_column;
    GtkCellRenderer* renderer;
    GList* renderers_list = NULL;

    g_assert (self != NULL && TI_IS_PAGE_ALIASING (self));

    contact_handle_column = gtk_tree_view_get_column (priv->tree_view, 0); // It's the first column.

    renderers_list = gtk_tree_view_column_get_cell_renderers (contact_handle_column);
    g_assert (g_list_length (renderers_list) == 1);

    renderer = GTK_CELL_RENDERER (renderers_list->data);

    if (contact_handle_display_mode == TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE)
    {
        gtk_tree_view_column_set_title (contact_handle_column, "Handle");

        gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
                                             "text", TI_COLUMN_ALIASES_LIST_HANDLE,
                                             NULL);
    }
    else // TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_NAME
    {
        gtk_tree_view_column_set_title (contact_handle_column, "Name");

        gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
                                             "text", TI_COLUMN_ALIASES_LIST_NAME,
                                             NULL);
    }

    // Clean up
    g_list_free (renderers_list);
}

/**
 * Aliases Changed
 */
static void
_ti_page_aliasing_aliases_changed (TIPageAliasing* self, GPtrArray* aliases)
{
    guint i;
    GValueArray* alias_change;
    guint contact_handle;
    const gchar* contact_alias;

    for (i = 0; i < aliases->len; i++)
    {
        alias_change = g_ptr_array_index (aliases, i);

        contact_handle = g_value_get_uint   (&(alias_change->values[0]));
        contact_alias  = g_value_get_string (&(alias_change->values[1]));

        _ti_page_aliasing_set_alias (self, contact_handle, contact_alias);
    }
}

/**
 * Set Alias
 */
static void
_ti_page_aliasing_set_alias (TIPageAliasing* self, guint contact_handle, const gchar* alias)
{
    TIPageAliasingPrivate* priv = TI_PAGE_ALIASING_GET_PRIVATE (self);
    gchar* contact_name;
    GtkTreeIter iter;
    gboolean ok;
    gboolean found = FALSE;
    guint curr_handle;

    // Look for the contact's row
    ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->list_store), &iter);
    while (ok && !found)
    {
        gtk_tree_model_get (GTK_TREE_MODEL (priv->list_store), &iter,
                            TI_COLUMN_ALIASES_LIST_HANDLE, &curr_handle,
                            -1);

        if (curr_handle == contact_handle)
        {
            found = TRUE;
        }
        else
        {
            ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (priv->list_store), &iter);
        }
    }

    if (!found)
    {
        // Create a new row for that handle
        gtk_list_store_append (priv->list_store, &iter);
    }

    contact_name = ti_handle_mapper_get_contact_handle_name (priv->handle_mapper, contact_handle);
    if (contact_name == NULL)
    {
        contact_name = g_strdup_printf ("%u", contact_handle);
    }

    gtk_list_store_set (priv->list_store, &iter,
                        TI_COLUMN_ALIASES_LIST_HANDLE, contact_handle,
                        TI_COLUMN_ALIASES_LIST_NAME, contact_name,
                        TI_COLUMN_ALIASES_LIST_ALIAS, alias,
                        -1);

    // Clean up
    g_free (contact_name);
}

/**
 * Clear Aliases
 */
static void
_ti_page_aliasing_clear_aliases (TIPageAliasing* self)
{
    TIPageAliasingPrivate* priv = TI_PAGE_ALIASING_GET_PRIVATE (self);

    gtk_list_store_clear (priv->list_store);
}
