/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-util.c:
 * General purpose utility functions.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-util.h"

#include <stdio.h>

#define TI_MAX_NUMBERS 100

/**
 * Parses a string containing a comma-separated uint list. e.g.:  "134,3423,343,343,434"
 */
GArray*
ti_parse_uint_list (const gchar* uint_list_str)
{
    GArray* numbers = NULL;
    gchar** number_str_array = NULL;
    guint number;
    int result;
    gboolean failed = FALSE;
    guint i;

    numbers = g_array_new (FALSE, FALSE, sizeof (guint));

    number_str_array = g_strsplit (uint_list_str, ",", TI_MAX_NUMBERS);

    for (i = 0; number_str_array[i] != NULL && !failed; i++)
    {
        result = sscanf (number_str_array[i], "%u", &number);
        failed = result != 1;

        g_array_append_val (numbers, number);
    }

    g_strfreev(number_str_array);

    if (failed)
    {
        g_array_free (numbers, TRUE);
        numbers = NULL;
    }

    return numbers;
}

/**
 * Helper function for ti_hash_table_to_array()
 */
static void
_ti_hash_table_to_array_iterator (gpointer key, gpointer value, GArray* array)
{
    TIHashEntry hash_entry;

    hash_entry.key = key;
    hash_entry.value = value;

    g_array_append_val (array, hash_entry);
}

/**
 * Hashtable to Array
 */
GArray*
ti_hash_table_to_array (GHashTable* hash_table)
{
    GArray* array;

    g_assert (hash_table != NULL);

    array = g_array_sized_new (FALSE, FALSE, sizeof (TIHashEntry), g_hash_table_size (hash_table));

    g_hash_table_foreach (hash_table, (GHFunc) _ti_hash_table_to_array_iterator, array);

    return array;
}

/**
 * Value to String
 */
gchar* ti_value_to_string (const GValue* value)
{
    gchar* string = NULL;

    switch (G_VALUE_TYPE (value))
    {
        case G_TYPE_UINT:
            string = g_strdup_printf ("%u", g_value_get_uint (value));
            break;

        case G_TYPE_INT:
            string = g_strdup_printf ("%u", g_value_get_int (value));
            break;

        case G_TYPE_STRING:
            string = g_value_dup_string (value);
            break;

        case G_TYPE_BOOLEAN:
            if (g_value_get_boolean (value) == TRUE)
            {
                string = g_strdup ("True");
            }
            else
            {
                string = g_strdup ("False");
            }
            break;

        default:
            g_critical ("ti_value_to_string: Unknown/unhandled GValue type.");
            string = g_strdup ("ti_value_to_string: Unknown/unhandled GValue type.");
    }

    return string;
}

/**
 * Value Destroy
 */
void
ti_value_destroy (gpointer data)
{
    GValue* value = (GValue*) data;

    g_assert (value != NULL);
    g_assert (G_IS_VALUE (value));

    g_value_unset (value);
    g_free (value);
}

/**
 * Remove Selected Elements
 */
void ti_remove_selected_elements (GtkTreeSelection* tree_selection)
{
    GList* selected_rows_list = NULL;
    GtkTreeModel* tree_model = NULL;
    GList* list_item = NULL;
    GtkTreePath* tree_path = NULL;
    GPtrArray* row_references_array = g_ptr_array_new ();
    GtkTreeRowReference* row_reference;
    guint i;
    GtkTreeIter iter;

    g_assert (tree_selection != NULL);
    g_assert (GTK_IS_TREE_SELECTION (tree_selection));

    // Build the row references array
    selected_rows_list = gtk_tree_selection_get_selected_rows (tree_selection, &tree_model);

    list_item = selected_rows_list;
    while (list_item != NULL)
    {
        tree_path = (GtkTreePath*) list_item->data;

        row_reference = gtk_tree_row_reference_new (tree_model, tree_path);
        g_ptr_array_add (row_references_array, row_reference);

        list_item = list_item->next;
    }

    // Remove the rows
    for (i = 0; i < row_references_array->len; i++)
    {
        row_reference = (GtkTreeRowReference*) g_ptr_array_index (row_references_array, i);

        tree_path = gtk_tree_row_reference_get_path (row_reference);

        gtk_tree_model_get_iter (tree_model, &iter, tree_path);

        if (GTK_IS_LIST_STORE (tree_model))
        {
            gtk_list_store_remove (GTK_LIST_STORE (tree_model), &iter);
        }
        else
        {
            g_assert (GTK_IS_TREE_STORE (tree_model));
            gtk_tree_store_remove (GTK_TREE_STORE (tree_model), &iter);
        }
    }

    // Clean up
    if (selected_rows_list != NULL)
    {
        g_list_foreach (selected_rows_list, (GFunc) gtk_tree_path_free, NULL);
        g_list_free (selected_rows_list);
    }

    if (row_references_array != NULL)
        g_ptr_array_free (row_references_array, TRUE);
}

/**
 * Get Selected Elements
 */
GArray*
ti_get_selected_elements (GtkTreeSelection* tree_selection, guint column, GType type)
{
    GArray* elements_array = NULL;
    GList* selected_rows_list = NULL;
    GtkTreeModel* tree_model = NULL;
    GtkTreePath* tree_path = NULL;
    GList* list_item = NULL;
    GtkTreeIter iter;
    guint elem_uint;
    gint elem_int;
    gchar* elem_string;

    g_assert (tree_selection != NULL);
    g_assert (GTK_IS_TREE_SELECTION (tree_selection));

    switch (type)
    {
        case G_TYPE_STRING:
            elements_array = g_array_new (FALSE, FALSE, sizeof (gchar*));
            break;

        case G_TYPE_UINT:
            elements_array = g_array_new (FALSE, FALSE, sizeof (guint));
            break;

        case G_TYPE_INT:
            elements_array = g_array_new (FALSE, FALSE, sizeof (gint));
            break;

        default:
            // TODO: implement the missing type, if that's the case.
            g_assert_not_reached();
            return NULL;
    }

    selected_rows_list = gtk_tree_selection_get_selected_rows (tree_selection, &tree_model);

    list_item = selected_rows_list;
    while (list_item != NULL)
    {
        tree_path = (GtkTreePath*) list_item->data;

        gtk_tree_model_get_iter (tree_model, &iter, tree_path);

        switch (type)
        {
            case G_TYPE_STRING:
                gtk_tree_model_get (tree_model, &iter,
                                    column, &elem_string,
                                    -1);
                g_array_append_val (elements_array, elem_string);
                break;

            case G_TYPE_UINT:
                gtk_tree_model_get (tree_model, &iter,
                                    column, &elem_uint,
                                    -1);
                g_array_append_val (elements_array, elem_uint);
                break;

            case G_TYPE_INT:
                gtk_tree_model_get (tree_model, &iter,
                                    column, &elem_int,
                                    -1);
                g_array_append_val (elements_array, elem_int);
                break;
        }

        list_item = list_item->next;
    }

    // Clean up
    if (selected_rows_list != NULL)
    {
        g_list_foreach (selected_rows_list, (GFunc) gtk_tree_path_free, NULL);
        g_list_free (selected_rows_list);
    }

    return elements_array;
}

/**
 * Get Tree Model Elements
 */
GArray*
ti_get_tree_model_elements (GtkTreeModel* tree_model, guint column, GType type)
{
    GtkTreeIter iter;
    gboolean ok;
    guint elem_uint;
    gint elem_int;
    gchar* elem_string;
    GArray* elements_array = NULL;

    g_assert (tree_model != NULL);
    g_assert (GTK_IS_TREE_MODEL (tree_model));

    switch (type)
    {
        case G_TYPE_STRING:
            elements_array = g_array_new (FALSE, FALSE, sizeof (gchar*));
            break;

        case G_TYPE_UINT:
            elements_array = g_array_new (FALSE, FALSE, sizeof (guint));
            break;

        case G_TYPE_INT:
            elements_array = g_array_new (FALSE, FALSE, sizeof (gint));
            break;

        default:
            // TODO: implement the missing type, if that's the case.
            g_assert_not_reached();
            return NULL;
    }

    ok = gtk_tree_model_get_iter_first (tree_model, &iter);
    while (ok)
    {
        switch (type)
        {
            case G_TYPE_STRING:
                gtk_tree_model_get (tree_model, &iter,
                                    column, &elem_string,
                                    -1);
                g_array_append_val (elements_array, elem_string);
                break;

            case G_TYPE_UINT:
                gtk_tree_model_get (tree_model, &iter,
                                    column, &elem_uint,
                                    -1);
                g_array_append_val (elements_array, elem_uint);
                break;

            case G_TYPE_INT:
                gtk_tree_model_get (tree_model, &iter,
                                    column, &elem_int,
                                    -1);
                g_array_append_val (elements_array, elem_int);
                break;
        }

        ok = gtk_tree_model_iter_next (tree_model, &iter);
    }

    return elements_array;
}
