/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-page-active-conns.c:
 * A GtkNotebook page that shows all currently active Telepathy connections.
 * i.e., all D-Bus services which represents Telepathy connections.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006-2007 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"
#include "page-active-conns.h"

#include <string.h>

#include <glade/glade.h>
#include <telepathy-glib/connection.h>
#include <telepathy-glib/dbus.h>

#include "page-priv.h"
#include "wnd-connection.h"

struct _TIPageActiveConnsClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageActiveConns, ti_page_active_conns, TI_TYPE_PAGE);

/* Function prototypes */
static void _ti_page_active_conns_setup_page (TIPage *page,
    GladeXML *glade_xml);
static void _ti_page_active_conns_build_treeview
    (TIPageActiveConns *page_active_conns, GladeXML *glade_xml);
static void _ti_page_active_conns_update_button_show_sensitivity
    (TIPageActiveConns *page_active_conns);
static void _ti_page_active_conns_show_selected_connection
    (TIPageActiveConns *page_active_conns);
static void _ti_page_active_conns_refresh_list
    (TIPageActiveConns *page_active_conns);
static void _ti_page_active_conns_on_conn_window_closed
    (TIPageActiveConns *page_active_conns, TIWndConnection *wnd_connection);
static void _ti_page_active_conns_restart_page (TIPage *page);

/**
 * Instance private data.
 */
struct _TIPageActiveConnsPrivate {
    gboolean disposed;

    TpDBusDaemon *bus_daemon;

    GPtrArray *conn_windows; // Array of TIWndConnection*

    GtkWindow *parent_wnd;

    GtkListStore *list_store;
    GtkTreeSelection *tree_selection;

    GtkButton *button_show;
    GtkButton *button_refresh;
};

static void
ti_page_active_conns_dispose (GObject *object)
{
  TIPageActiveConns *self = TI_PAGE_ACTIVE_CONNS (object);

  if (self->priv->disposed)
    return;

  self->priv->disposed = TRUE;

  if (self->priv->bus_daemon != NULL)
    {
      g_object_unref (self->priv->bus_daemon);
      self->priv->bus_daemon = NULL;
    }

  if (self->priv->conn_windows != NULL)
    {
      guint i;

      for (i = 0; i < self->priv->conn_windows->len; i++)
        {
          GObject *obj =
              G_OBJECT (g_ptr_array_index (self->priv->conn_windows, i));

          g_object_unref (obj);
        }

      g_ptr_array_free (self->priv->conn_windows, TRUE);
      self->priv->conn_windows = NULL;
    }

  G_OBJECT_CLASS (ti_page_active_conns_parent_class)->dispose (object);
}


static void
ti_page_active_conns_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_page_active_conns_parent_class)->constructed;
  TIPageActiveConns *self = TI_PAGE_ACTIVE_CONNS (object);

  if (chain_up != NULL)
    chain_up (object);

  (void) self;
}


static void
ti_page_active_conns_class_init (TIPageActiveConnsClass *cls)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (cls);
  TIPageClass *page_class = TI_PAGE_CLASS (cls);

  gobject_class->dispose = ti_page_active_conns_dispose;
  gobject_class->constructed = ti_page_active_conns_constructed;

  page_class->setup_page = _ti_page_active_conns_setup_page;
  page_class->restart_page = _ti_page_active_conns_restart_page;

  g_type_class_add_private (cls, sizeof (TIPageActiveConnsPrivate));
}

static void
ti_page_active_conns_init (TIPageActiveConns *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TI_TYPE_PAGE_ACTIVE_CONNS, TIPageActiveConnsPrivate);

  self->priv->disposed = FALSE;
  self->priv->bus_daemon = NULL;
  self->priv->conn_windows = g_ptr_array_new ();
}

TIPageActiveConns *
ti_page_active_conns_new (GtkWindow *parent_wnd,
                          GtkNotebook *parent_notebook,
                          TpDBusDaemon *dbus_daemon)
{
  TIPage *page;
  TIPageActiveConnsPrivate *priv;

  page = g_object_new (TI_TYPE_PAGE_ACTIVE_CONNS, NULL);

  priv = ((TIPageActiveConns *) page)->priv;

  priv->parent_wnd = parent_wnd;
  priv->bus_daemon = g_object_ref (dbus_daemon);

  _ti_page_new (&page, parent_notebook, "page-active-conns.xml");

  return (TIPageActiveConns *) page;
}

static void
_ti_page_active_conns_setup_page (TIPage *page,
                                  GladeXML *glade_xml)
{
  TIPageActiveConns *self = TI_PAGE_ACTIVE_CONNS (page);

  _ti_page_active_conns_build_treeview (self, glade_xml);

  self->priv->button_refresh = GTK_BUTTON (glade_xml_get_widget (glade_xml,
      "button_refresh"));
  g_assert (GTK_IS_BUTTON (self->priv->button_refresh));
  g_signal_connect_swapped (self->priv->button_refresh, "clicked",
      G_CALLBACK (_ti_page_active_conns_refresh_list), self);

  self->priv->button_show = GTK_BUTTON (glade_xml_get_widget (glade_xml,
      "button_show"));
  g_assert (GTK_IS_BUTTON (self->priv->button_show));
  g_signal_connect_swapped (self->priv->button_show, "clicked",
      G_CALLBACK (_ti_page_active_conns_show_selected_connection), self);
}

static void
_ti_page_active_conns_build_treeview (TIPageActiveConns *self,
                                      GladeXML *glade_xml)
{
  GtkWidget *treeview;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  treeview = glade_xml_get_widget (glade_xml, "treeview_connections");

  self->priv->list_store = gtk_list_store_new (3,
      G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview),
      GTK_TREE_MODEL (self->priv->list_store));

  renderer = gtk_cell_renderer_text_new ();

  column = gtk_tree_view_column_new_with_attributes ("Conn. Man.", renderer,
      "text", 0,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

  column = gtk_tree_view_column_new_with_attributes ("Protocol", renderer,
      "text", 1,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

  column = gtk_tree_view_column_new_with_attributes ("Connection", renderer,
      "text", 2,
      NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

  self->priv->tree_selection = gtk_tree_view_get_selection
      (GTK_TREE_VIEW (treeview));

  g_signal_connect_swapped (self->priv->tree_selection, "changed",
      G_CALLBACK (_ti_page_active_conns_update_button_show_sensitivity),
      self);

  gtk_tree_selection_set_mode (self->priv->tree_selection,
      GTK_SELECTION_SINGLE);
}

static void
_ti_page_active_conns_update_button_show_sensitivity (TIPageActiveConns *self)
{
  gtk_widget_set_sensitive (GTK_WIDGET (self->priv->button_show),
      gtk_tree_selection_get_selected (self->priv->tree_selection, NULL,
          NULL));
}

static void
_ti_page_active_conns_show_selected_connection (TIPageActiveConns *self)
{
  TpConnection *connection;
  TIWndConnection *wnd_connection;
  gchar *account = NULL;
  gchar *object_path;
  GtkTreeIter iter;
  GError *error = NULL;

  if (!gtk_tree_selection_get_selected (self->priv->tree_selection, NULL,
      &iter))
    {
      g_return_if_reached ();
    }

  gtk_tree_model_get (GTK_TREE_MODEL (self->priv->list_store), &iter,
      2, &account,
      -1);

  object_path = g_strdup_printf ("/%s", account);
  g_strdelimit (object_path, ".", '/');

  connection = tp_connection_new (self->priv->bus_daemon,
      account, object_path, &error);

  g_free (account);
  g_free (object_path);

  if (connection == NULL)
    {
      g_printerr ("Could not create connection: %s\n", error->message);
      g_error_free (error);
      return;
    }

  wnd_connection = ti_wnd_connection_new (self->priv->parent_wnd,
      connection);
  g_object_unref (connection);

  if (wnd_connection == NULL)
    return;

  g_ptr_array_add (self->priv->conn_windows, wnd_connection);
  g_signal_connect_swapped (wnd_connection, "closed",
      G_CALLBACK (_ti_page_active_conns_on_conn_window_closed), self);
  ti_wnd_connection_show (wnd_connection);
}

static void
got_connection_names (const gchar * const *names,
                      gsize n,
                      const gchar * const *cms,
                      const gchar * const *protocols,
                      const GError *error,
                      gpointer user_data,
                      GObject *weak_object)
{
  TIPageActiveConns *self = TI_PAGE_ACTIVE_CONNS (weak_object);
  GtkTreeIter iter;
  gsize i;

  gtk_list_store_clear (self->priv->list_store);

  if (error != NULL)
    return;

  /* Fill the list. */
  for (i = 0; i < n; i++)
    {
      gtk_list_store_append (self->priv->list_store, &iter);
      gtk_list_store_set (self->priv->list_store, &iter,
                          0, cms[i],
                          1, protocols[i],
                          2, names[i],
                          -1);
    }
}

static void
_ti_page_active_conns_refresh_list (TIPageActiveConns *self)
{
  tp_list_connection_names (self->priv->bus_daemon, got_connection_names,
      NULL, NULL, (GObject *) self);
}

static void
_ti_page_active_conns_on_conn_window_closed (TIPageActiveConns *self,
                                             TIWndConnection *wnd_connection)
{
  gboolean ok;

  ok = g_ptr_array_remove (self->priv->conn_windows, wnd_connection);
  g_assert (ok);

  g_object_unref (wnd_connection);
}

static void
_ti_page_active_conns_restart_page (TIPage *page)
{
    TIPageActiveConns *self = TI_PAGE_ACTIVE_CONNS (page);

    _ti_page_active_conns_refresh_list (self);
}
