/*
 * Hold page for channel notebook.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"

#include "page-hold.h"

#include <glade/glade.h>

#include "extensions/extensions.h"
#include "page-priv.h"

struct _TIPageHoldClass {
    TIPageClass parent;
    gpointer priv;
};


struct _TIPageHoldPrivate {
    TpChannel *channel;

    /* UI for telling the CM what to do */
    GtkToggleButton *hold_toggle;
    GtkButton *request_hold_button;

    /* UI for the CM telling us what's going on */
    GtkLabel *current_status_label;

    /* Event log */
    GtkTextView *event_log;
    GtkTextBuffer *event_buffer;
};


G_DEFINE_TYPE (TIPageHold, ti_page_hold, TI_TYPE_PAGE);

enum {
    PROP_CHANNEL = 1
};


static void
ti_page_hold_init (TIPageHold *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, TI_TYPE_PAGE_HOLD,
      TIPageHoldPrivate);
}


static void
ti_page_hold_dispose (GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (self->priv->channel != NULL)
    {
      g_object_unref (self->priv->channel);
      self->priv->channel = NULL;
    }

  G_OBJECT_CLASS (ti_page_hold_parent_class)->dispose (object);
}


static void
ti_page_hold_get_property (GObject *object,
                           guint property_id,
                           GValue *value,
                           GParamSpec *pspec)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_value_set_object (value, self->priv->channel);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
ti_page_hold_set_property (GObject *object,
                           guint property_id,
                           const GValue *value,
                           GParamSpec *pspec)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  switch (property_id)
    {
    case PROP_CHANNEL:
      g_return_if_fail (self->priv->channel == NULL);
      self->priv->channel = TP_CHANNEL (g_value_dup_object (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}


static void
log_event (TIPageHold *self,
           const gchar *text)
{
  gtk_text_buffer_insert_at_cursor (self->priv->event_buffer, text, -1);
  gtk_text_buffer_insert_at_cursor (self->priv->event_buffer, "\n", -1);
}


static void
requested_hold (TpChannel *channel G_GNUC_UNUSED,
                const GError *error,
                gpointer unused G_GNUC_UNUSED,
                GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (error != NULL)
    {
      gchar *text = g_strdup_printf ("RequestHold failed: %s", error->message);
      log_event (self, text);
      g_free (text);
      return;
    }

  log_event (self, "RequestHold succeeded");
}


static void
request_hold_clicked (TIPageHold *self)
{
  gboolean want = gtk_toggle_button_get_active (self->priv->hold_toggle);

  ti_cli_channel_interface_hold_call_request_hold (self->priv->channel, -1,
      want, requested_hold, NULL, NULL, (GObject *) self);
}


static void
ti_page_hold_setup_page (TIPage *base,
                         GladeXML *glade_xml)
{
  TIPageHold *self = TI_PAGE_HOLD (base);
  GtkWidget *page = glade_xml_get_widget (glade_xml, "page");
  GList *kids;
  GtkBox *hbox;
  GtkWidget *widget, *frame;

  g_assert (GTK_IS_VBOX (page));

  /* get rid of placeholder content in the Glade file */
  kids = gtk_container_get_children (GTK_CONTAINER (page));
  g_list_foreach (kids, (GFunc) gtk_widget_destroy, NULL);
  g_list_free (kids);

  /* The CM telling us what's going on */
  widget = gtk_label_new ("???");
  self->priv->current_status_label = GTK_LABEL (widget);
  frame = gtk_frame_new ("Current status");
  gtk_container_add (GTK_CONTAINER (frame), widget);
  gtk_box_pack_start (GTK_BOX (page), frame, FALSE, FALSE, 0);

  /* Telling the CM what to do */
  hbox = GTK_BOX (gtk_hbox_new (FALSE, 0));

  widget = gtk_check_button_new_with_label ("Put other user on hold");
  self->priv->hold_toggle = GTK_TOGGLE_BUTTON (widget);
  gtk_box_pack_start (hbox, widget, FALSE, FALSE, 0);

  widget = gtk_button_new_with_label ("Call RequestHold");
  self->priv->request_hold_button = GTK_BUTTON (widget);
  gtk_box_pack_end (hbox, widget, FALSE, FALSE, 0);

  g_signal_connect_swapped (self->priv->request_hold_button, "clicked",
      G_CALLBACK (request_hold_clicked), self);

  frame = gtk_frame_new ("Requested status");
  gtk_container_add (GTK_CONTAINER (frame), (GtkWidget *) hbox);
  gtk_box_pack_start (GTK_BOX (page), frame, FALSE, FALSE, 0);

  /* Event log */
  widget = gtk_text_view_new ();
  self->priv->event_log = GTK_TEXT_VIEW (widget);
  self->priv->event_buffer = gtk_text_view_get_buffer (self->priv->event_log);
  gtk_text_view_set_editable (self->priv->event_log, FALSE);
  gtk_text_view_set_cursor_visible (self->priv->event_log, FALSE);
  frame = gtk_frame_new ("Event log");
  gtk_container_add (GTK_CONTAINER (frame), widget);
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);
}


static void
hold_state_changed (TpChannel *channel G_GNUC_UNUSED,
                    gboolean hold_state,
                    gpointer unused G_GNUC_UNUSED,
                    GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);
  const gchar *text;

  if (hold_state)
    text = "Local user has placed the call on hold";
  else
    text = "Call is active";

  gtk_label_set_text (self->priv->current_status_label, text);
  log_event (self, text);
}


static void
got_hold_state (TpChannel *channel,
                gboolean hold_state,
                const GError *error,
                gpointer unused G_GNUC_UNUSED,
                GObject *object)
{
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (error == NULL)
    {
      hold_state_changed (channel, hold_state, NULL, object);
    }
  else
    {
      gchar *text = g_strdup_printf ("Error getting initial hold state: %s",
          error->message);

      gtk_label_set_text (self->priv->current_status_label, text);
      log_event (self, text);
      g_free (text);
    }
}


static void
ti_page_hold_constructed (GObject *object)
{
  void (*chain_up) (GObject *) =
    G_OBJECT_CLASS (ti_page_hold_parent_class)->constructed;
  TIPageHold *self = TI_PAGE_HOLD (object);

  if (chain_up != NULL)
    chain_up (object);

  ti_cli_channel_interface_hold_connect_to_hold_state_changed (
      self->priv->channel, hold_state_changed, NULL, NULL, object,
      NULL);

  ti_cli_channel_interface_hold_call_get_hold_state (self->priv->channel, -1,
      got_hold_state, NULL, NULL, object);
}


static void
ti_page_hold_class_init (TIPageHoldClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);
  GParamSpec *param_spec;

  object_class->constructed = ti_page_hold_constructed;
  object_class->dispose = ti_page_hold_dispose;
  object_class->get_property = ti_page_hold_get_property;
  object_class->set_property = ti_page_hold_set_property;

  page_class->setup_page = ti_page_hold_setup_page;

  g_type_class_add_private (klass, sizeof (TIPageHoldPrivate));

  param_spec = g_param_spec_object ("channel", "TpChannel",
      "This page's associated channel", TP_TYPE_CHANNEL,
      G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE |
      G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB | G_PARAM_STATIC_NICK);
  g_object_class_install_property (object_class, PROP_CHANNEL, param_spec);
}


TIPageHold *
ti_page_hold_new (GtkNotebook *parent_notebook,
                  TpChannel *channel)
{
  TIPage *page = NULL;

  g_return_val_if_fail (channel != NULL, NULL);

  page = g_object_new (TI_TYPE_PAGE_HOLD,
      "channel", channel,
      NULL);

  _ti_page_new (&page, parent_notebook, "page-hold.xml");

  return TI_PAGE_HOLD (page);
}
