/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-dlg-request-channel.c:
 * DlgRequestChannel - Request Channel Dialog
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "dlg-request-channel.h"

#include <glade/glade.h>
#include <telepathy-glib/enums.h>

#include "constants.h"
#include "preferences.h"
#include "util.h"

G_DEFINE_TYPE (TIDlgRequestChannel, ti_dlg_request_channel, G_TYPE_OBJECT);

struct _TIDlgRequestChannelPrivate {
    TIHandleMapper *handle_mapper;
    TIPreferences *preferences;

    GtkWindow *parent;

    GladeXML *glade_xml;

    /* Widgets created directly from code instead of from the glade file */
    GtkComboBox *combo_handle;
    GtkEntry *entry_handle;

    GtkWidget *dialog;

    GtkBin *bin;
    GtkComboBox *combo_type;
    GtkComboBox *combo_handle_type;
    GtkLabel *label_handle;
    GtkCheckButton *check_suppress;
};
typedef struct _TIDlgRequestChannelPrivate TIDlgRequestChannelPrivate;

#define TI_DLG_REQUEST_CHANNEL_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_REQUEST_CHANNEL, \
                                TIDlgRequestChannelPrivate))

static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);
static guint _ti_dlg_request_channel_get_handle (TIDlgRequestChannel *);
static guint _ti_dlg_request_channel_get_handle_type (TIDlgRequestChannel *);
static void _ti_dlg_request_channel_setup_handle_number_entry
    (TIDlgRequestChannel *self, guint handle_type);
static void _ti_dlg_request_channel_on_handle_type_changed
    (TIDlgRequestChannel *self);

static void
ti_dlg_request_channel_dispose (GObject *object)
{
  TIDlgRequestChannel *self = TI_DLG_REQUEST_CHANNEL (object);
  TIDlgRequestChannelPrivate *priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (self);

  TI_OBJ_UNREF (priv->handle_mapper);

  if (priv->preferences != NULL)
    {
      g_signal_handlers_disconnect_by_func (priv->preferences,
          G_CALLBACK (_ti_dlg_request_channel_on_handle_type_changed), self);
      g_object_unref (priv->preferences);
      priv->preferences = NULL;
    }

  TI_OBJ_UNREF (priv->glade_xml);
  TI_OBJ_UNREF (priv->combo_handle);
  TI_OBJ_UNREF (priv->entry_handle);

  G_OBJECT_CLASS (ti_dlg_request_channel_parent_class)->dispose (object);
}

static void
ti_dlg_request_channel_class_init (TIDlgRequestChannelClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = ti_dlg_request_channel_dispose;
  g_type_class_add_private (klass, sizeof (TIDlgRequestChannelPrivate));
}

static void
ti_dlg_request_channel_init (TIDlgRequestChannel *self)
{
  TIDlgRequestChannelPrivate *priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (self);

  priv->handle_mapper = NULL;
  priv->preferences = ti_preferences_new ();
  priv->glade_xml = NULL;

  priv->combo_handle = GTK_COMBO_BOX (gtk_combo_box_new ());
  g_object_ref_sink (priv->combo_handle);

  priv->entry_handle = GTK_ENTRY (gtk_entry_new ());
  gtk_entry_set_max_length (priv->entry_handle,
      12 /* Twelve is a magic number! */);
  g_object_ref_sink (priv->entry_handle);
}

TIDlgRequestChannel *
ti_dlg_request_channel_new (GtkWindow *parent,
                            TIHandleMapper *handle_mapper)
{
  TIDlgRequestChannel *dialog = NULL;
  TIDlgRequestChannelPrivate *priv = NULL;
  gchar *glade_file_path = NULL;
  GtkCellRenderer *renderer;

  dialog = g_object_new (TI_TYPE_DLG_REQUEST_CHANNEL, NULL);
  priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (dialog);

  priv->parent = parent;
  priv->handle_mapper = handle_mapper;
  g_object_ref (handle_mapper);

  glade_file_path = ti_get_glade_path ("dlg-request-channel.xml");
  priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);

  if (priv->glade_xml == NULL)
    {
      g_critical ("Error loading glade file \"%s\".", glade_file_path);
      g_object_unref (dialog);
      dialog = NULL;
      goto CLEAN_UP;
    }

  priv->dialog = glade_xml_get_widget (priv->glade_xml,
      "dialog_request_channel");
  g_assert (GTK_IS_DIALOG (priv->dialog));

  priv->combo_type = GTK_COMBO_BOX (glade_xml_get_widget (priv->glade_xml,
        "combo_type"));
  g_assert (GTK_IS_COMBO_BOX (priv->combo_type));

  priv->combo_handle_type = GTK_COMBO_BOX (glade_xml_get_widget
      (priv->glade_xml, "combo_handle_type"));
  g_assert (GTK_IS_COMBO_BOX (priv->combo_handle_type));

  priv->label_handle = GTK_LABEL (glade_xml_get_widget (priv->glade_xml,
        "label_handle"));
  g_assert (GTK_IS_LABEL (priv->label_handle));

  priv->check_suppress = GTK_CHECK_BUTTON (glade_xml_get_widget
      (priv->glade_xml, "checkbutton_suppress_handler"));
  g_assert (GTK_IS_CHECK_BUTTON (priv->check_suppress));

  priv->bin = GTK_BIN (glade_xml_get_widget (priv->glade_xml, "bin"));
  g_assert (GTK_IS_BIN (priv->bin));

  gtk_window_set_transient_for (GTK_WINDOW (priv->dialog),
      GTK_WINDOW (priv->parent));

  glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked",
      G_CALLBACK(_button_ok_clicked), priv->dialog);
  glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked",
      G_CALLBACK(_button_cancel_clicked), priv->dialog);

  g_signal_connect_swapped (priv->preferences, "handle-display-mode-changed",
      G_CALLBACK (_ti_dlg_request_channel_on_handle_type_changed), dialog);

  g_signal_connect_swapped (priv->combo_handle_type, "changed",
      G_CALLBACK (_ti_dlg_request_channel_on_handle_type_changed), dialog);

  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (priv->combo_handle), renderer,
      FALSE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (priv->combo_handle),
      renderer, "text", TI_HANDLES_LIST_COLUMN_NAME);
  gtk_combo_box_set_model (priv->combo_handle,
      GTK_TREE_MODEL (ti_handle_mapper_get_contact_handle_list_store
        (handle_mapper)));

  gtk_combo_box_set_active (priv->combo_handle_type, TP_HANDLE_TYPE_CONTACT);

CLEAN_UP:
  g_free (glade_file_path);

  return dialog;
}

gboolean
ti_dlg_request_channel_run (TIDlgRequestChannel *self,
                            gchar **type,
                            guint *handle_type,
                            guint *handle,
                            gboolean *suppress_handler)
{
  TIDlgRequestChannelPrivate *priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (self);
  gint result;

  gtk_widget_show_all (priv->dialog);
  result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
  gtk_widget_hide_all (priv->dialog);

  if (result != GTK_RESPONSE_OK)
    return FALSE;

  *type = gtk_combo_box_get_active_text (priv->combo_type);
  *handle_type = _ti_dlg_request_channel_get_handle_type (self);
  *handle = _ti_dlg_request_channel_get_handle (self);
  *suppress_handler = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
      (priv->check_suppress));

  return TRUE;
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button,
                    gpointer user_data)
{
  GtkDialog *dialog = GTK_DIALOG (user_data);

  gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button,
                        gpointer user_data)
{
  GtkDialog *dialog = GTK_DIALOG (user_data);

  gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}

static guint
_ti_dlg_request_channel_get_handle (TIDlgRequestChannel *self)
{
  TIDlgRequestChannelPrivate *priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (self);
  const gchar *handle_str = NULL;
  guint handle;
  GtkTreeModel *tree_model = NULL;
  GtkTreeIter iter;
  gboolean ok;

  if (_ti_dlg_request_channel_get_handle_type (self) == TP_HANDLE_TYPE_CONTACT &&
      ti_preferences_get_handle_display_mode (priv->preferences) == TI_PREFERENCES_HANDLE_DISPLAY_NAME)
    {
      tree_model = gtk_combo_box_get_model (priv->combo_handle);

      ok = gtk_combo_box_get_active_iter (priv->combo_handle, &iter);

      if (!ok)
        return 0;

      gtk_tree_model_get (tree_model, &iter,
          TI_HANDLES_LIST_COLUMN_NUMBER, &handle,
          -1);
    }
  else
    {
      handle_str = gtk_entry_get_text (priv->entry_handle);

      if (handle_str[0] == '\0')
        {
          handle = 0;
        }
      else if (sscanf (handle_str, "%u", &handle) != 1)
        {
          handle = 0;
          g_warning ("Not an unsigned integer: \"%s\"", handle_str);
        }
    }

  return handle;
}

static guint
_ti_dlg_request_channel_get_handle_type (TIDlgRequestChannel *self)
{
  TIDlgRequestChannelPrivate *priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (self);
  gint active_handle_type;

  active_handle_type = gtk_combo_box_get_active (priv->combo_handle_type);

  if (active_handle_type == -1)
    return 0;
  else
    return (guint) active_handle_type;
}

static void
_ti_dlg_request_channel_setup_handle_number_entry (TIDlgRequestChannel *self,
                                                   guint handle_type)
{
  TIDlgRequestChannelPrivate *priv = TI_DLG_REQUEST_CHANNEL_GET_PRIVATE (self);

  if (handle_type == TP_HANDLE_TYPE_CONTACT &&
      ti_preferences_get_handle_display_mode (priv->preferences) == TI_PREFERENCES_HANDLE_DISPLAY_NAME)
    {
      /* Remove the entry */
      if (gtk_bin_get_child (priv->bin) == GTK_WIDGET (priv->entry_handle))
        {
          gtk_container_remove (GTK_CONTAINER (priv->bin),
              GTK_WIDGET (priv->entry_handle));
        }

      /* Add the combo */
      if (gtk_bin_get_child (priv->bin) != GTK_WIDGET (priv->combo_handle))
        {
          gtk_label_set_text (priv->label_handle, "Name:");
          gtk_container_add (GTK_CONTAINER (priv->bin),
              GTK_WIDGET (priv->combo_handle));
          gtk_widget_show (GTK_WIDGET (priv->combo_handle));
        }
    }
  else
    {
      /* Remove the combo */
      if (gtk_bin_get_child (priv->bin) == GTK_WIDGET (priv->combo_handle))
        {
          gtk_container_remove (GTK_CONTAINER (priv->bin),
              GTK_WIDGET (priv->combo_handle));
        }

      /* Add the entry */
      if (gtk_bin_get_child (priv->bin) != GTK_WIDGET (priv->entry_handle))
        {
          gtk_label_set_text (priv->label_handle, "Number:");
          gtk_container_add (GTK_CONTAINER (priv->bin),
              GTK_WIDGET (priv->entry_handle));
          gtk_widget_show (GTK_WIDGET (priv->entry_handle));
        }
    }
}

static void
_ti_dlg_request_channel_on_handle_type_changed (TIDlgRequestChannel *self)
{
  guint handle_type;

  handle_type = _ti_dlg_request_channel_get_handle_type (self);
  _ti_dlg_request_channel_setup_handle_number_entry (self, handle_type);
}
