/*

    File: hfs.c

    Copyright (C) 1998-2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "hfs.h"
#include "intrf.h"
#include "fnctdsk.h"
static int set_HFS_info(t_partition *partition, const hfs_mdb_t *hfs_mdb);

int check_HFS(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+0x400)!=0)
  { return 1; }
  if(test_HFS(disk_car,(hfs_mdb_t *)&buffer,partition,debug,0)!=0)
    return 1;
  set_HFS_info(partition,(hfs_mdb_t *)&buffer);
  return 0;
}

int recover_HFS(t_param_disk *disk_car, const hfs_mdb_t *hfs_mdb,t_partition *partition,const int debug, const int dump_ind, const int backup)
{
  uint64_t part_size;
  if(test_HFS(disk_car,hfs_mdb,partition,debug,dump_ind)!=0)
    return 1;
  /* The extra 0x400 bytes are for the backup MDB */
  part_size=(uint64_t)be16(hfs_mdb->drNmAlBlks)*be32(hfs_mdb->drAlBlkSiz)+be16(hfs_mdb->drAlBlSt)*512+0x400;
  if(backup>0)
  {
    if(partition->part_offset+2*disk_car->sector_size<part_size)
      return 1;
    partition->boot_sector=(part_size-0x400)/disk_car->sector_size;
    partition->part_offset=partition->part_offset+2*disk_car->sector_size-part_size;
  }
  partition->part_size=part_size;
  set_HFS_info(partition,hfs_mdb);
  partition->part_type=(unsigned char)P_HFS;
  if(debug!=0)
  {
    ecrit_rapport("part_size %lu\n",(long unsigned)(partition->part_size/disk_car->sector_size));
  }
  return 0;
}

int test_HFS(t_param_disk *disk_car, const hfs_mdb_t *hfs_mdb,t_partition *partition,const int debug, const int dump_ind)
{
  if (hfs_mdb->drSigWord==be16(HFS_SUPER_MAGIC))
  {
    if(debug>0 || dump_ind!=0)
    {
      ecrit_rapport("\nHFS magic value at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    }
    if(dump_ind!=0)
    {
      /* There is a little offset ... */
      dump(stdscr,hfs_mdb,DEFAULT_SECTOR_SIZE);
    }
    if(debug>1)
    {
      ecrit_rapport("drNmAlBlks %u\n",(unsigned) be16(hfs_mdb->drNmAlBlks));
      ecrit_rapport("drAlBlkSiz %u\n",(unsigned) be32(hfs_mdb->drAlBlkSiz));
      ecrit_rapport("drAlBlSt %u\n",(unsigned) be16(hfs_mdb->drAlBlSt));
      ecrit_rapport("drFreeBks %u\n",(unsigned) be16(hfs_mdb->drFreeBks));
    }
    if(be32(hfs_mdb->drAlBlkSiz)%512!=0)
      return 1;
    partition->upart_type=UP_HFS;
    return 0;
  }
  return 1;
}

static int set_HFS_info(t_partition *partition, const hfs_mdb_t *hfs_mdb)
{
  unsigned int name_size=sizeof(hfs_mdb->drVN)-1;
  partition->info[0]='\0';
  if(name_size>hfs_mdb->drVN[0])
    name_size=hfs_mdb->drVN[0];
  memcpy(partition->name,&hfs_mdb->drVN[0]+1,name_size);
  snprintf(partition->info,sizeof(partition->info),"HFS");
  return 0;
}

