/*

    File: ufs.c

    Copyright (C) 2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 59
    Temple Place - Suite 330, Boston MA 02111-1307, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "ufs.h"
#include "fnctdsk.h"
#include "intrf.h"
extern const t_arch_fnct arch_sun;

static int set_ufs_info(const t_param_disk *disk_car, const struct ufs_super_block *sb,t_partition *partition, const int debug, const int dump_ind);
static int test_ufs(const t_param_disk *disk_car, const struct ufs_super_block *sb,t_partition *partition,const int debug, const int dump_ind);

int check_ufs(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[2048];
  const struct ufs_super_block*sb=(const struct ufs_super_block*)buffer;
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset+UFS_SBLOCK)!=0)
  { return 1; }
  if(test_ufs(disk_car,sb,partition,debug,0)!=0)
    return 1;
  set_ufs_info(disk_car,sb,partition,debug,0);
  return 0;
}

static int test_ufs(const t_param_disk *disk_car, const struct ufs_super_block *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if ((sb->fs_magic==UFS_MAGIC) || (sb->fs_magic==UFS_CIGAM))
  {
    partition->upart_type = UP_UFS;
    if(debug!=0)
      ecrit_rapport("\nUFS Marker at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    return 0;
  }
  else if ((sb->fs_magic==UFS2_MAGIC) || (sb->fs_magic==UFS2_CIGAM))
  {
    partition->upart_type = UP_UFS2;
    if(debug!=0)
      ecrit_rapport("\nUFS2 Marker at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
    return 0;
  }
  return 1;
}

int recover_ufs(t_param_disk *disk_car, const struct ufs_super_block *sb, t_partition *partition,const int debug, const int dump_ind)
{
  if(test_ufs(disk_car,sb,partition,debug,dump_ind)!=0)
    return 1;
  if(debug!=0 || dump_ind!=0)
  {
    ecrit_rapport("\nrecover_ufs\n");
    if(dump_ind!=0)
    {
      dump(stdscr,sb,sizeof(*sb));
    }
  }
  switch(le32(sb->fs_magic))
  {
    case UFS2_MAGIC:
      partition->part_size = (uint64_t)le64(sb->fs_u11.fs_u2.fs_size)*le32(sb->fs_fsize);
      if(debug>0)
      {
	ecrit_rapport("fs_size %lu, fs_fsize %lu\n", (long unsigned)le64(sb->fs_u11.fs_u2.fs_size), (long unsigned)le32(sb->fs_fsize));
      }
      break;
    case UFS2_CIGAM:
      partition->part_size = (uint64_t)be64(sb->fs_u11.fs_u2.fs_size)*be32(sb->fs_fsize);
      if(debug>0)
      {
	ecrit_rapport("fs_size %lu, fs_fsize %lu\n", (long unsigned)be64(sb->fs_u11.fs_u2.fs_size), (long unsigned)be32(sb->fs_fsize));
      }
      break;
    case UFS_MAGIC:
      partition->part_size = (uint64_t)le32(sb->fs_size)*le32(sb->fs_fsize);
      if(debug>0)
      {
	ecrit_rapport("fs_size %lu, fs_fsize %lu\n",(long unsigned)le32(sb->fs_size),(long unsigned)le32(sb->fs_fsize));
      }
      break;
    default:
      partition->part_size = (uint64_t)be32(sb->fs_size)*be32(sb->fs_fsize);
      if(debug>0)
      {
	ecrit_rapport("fs_size %lu, fs_fsize %lu\n",(long unsigned)be32(sb->fs_size),(long unsigned)be32(sb->fs_fsize));
      }
      break;
  }
  set_ufs_info(disk_car,sb,partition,debug,dump_ind);
  partition->arch=&arch_sun;
  if(strcmp(partition->name,"/")==0)
    partition->part_type = (unsigned char)PSUN_ROOT;
  else if(strcmp(partition->name,"/var")==0)
    partition->part_type = (unsigned char)PSUN_VAR;
  else if(strcmp(partition->name,"/usr")==0)
    partition->part_type = (unsigned char)PSUN_USR;
  else if(strcmp(partition->name,"/export/home")==0)
    partition->part_type = (unsigned char)PSUN_HOME;
  else
    partition->part_type = (unsigned char)PSUN_ROOT;
  return 0;
}

static int set_ufs_info(const t_param_disk *disk_car,const struct ufs_super_block *sb,t_partition *partition, const int debug, const int dump_ind)
{
  partition->info[0]='\0';
  switch(partition->upart_type)
  {
    case UP_UFS:
    default:
      set_part_name(partition,(const char*)sb->fs_u11.fs_u1.fs_fsmnt,sizeof(partition->name));
      strncpy(partition->info,"UFS1",sizeof(partition->info));
      break;
    case UP_UFS2:
      set_part_name(partition,(const char*)sb->fs_u11.fs_u2.fs_fsmnt,sizeof(partition->name));
      strncpy(partition->info,"UFS2",sizeof(partition->info));
      break;
  }
  return 0;
}
