/**
 * ntfs_fix.c - Part of the TestDisk project.
 *
 * Copyright (c) 2006 Christophe Grenier
 *
 * Original version comes from the Linux-NTFS project.
 * Copyright (c) 2000-2005 Anton Altaparmakov.
 * Copyright (c) 2002-2005 Szabolcs Szakacsits.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_LIBNTFS
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <errno.h>
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_MACHINE_ENDIAN_H
#include <machine/endian.h>
#endif
#include <stdarg.h>
#include <ntfs/attrib.h>
#include <ntfs/mft.h>
#include "types.h"
#include "common.h"
#include "intrf.h"
#include "ntfs.h"
#include "ntfs_dir.h"
#include "dir.h"
#include "io_redir.h"

extern struct ntfs_device_operations ntfs_device_testdisk_io_ops;

static const char *FAILED    = "FAILED\n";

static int fix_mftmirr(ntfs_volume *vol)
{
  s64 l, br;
  unsigned char *m, *m2;
  int i, ret = -1; /* failure */
  BOOL done;

  ecrit_rapport("\nProcessing $MFT and $MFTMirr...\n");

  /* Load data from $MFT and $MFTMirr and compare the contents. */
  m = (u8*)malloc(vol->mftmirr_size << vol->mft_record_size_bits);
  if (!m) {
    ecrit_rapport("Failed to allocate memory");
    return -1;
  }
  m2 = (u8*)malloc(vol->mftmirr_size << vol->mft_record_size_bits);
  if (!m2) {
    ecrit_rapport("Failed to allocate memory");
    free(m);
    return -1;
  }

  ecrit_rapport("Reading $MFT... ");
  l = ntfs_attr_mst_pread(vol->mft_na, 0, vol->mftmirr_size,
      vol->mft_record_size, m);
  if (l != vol->mftmirr_size) {
    ecrit_rapport(FAILED);
    if (l != -1)
      errno = EIO;
    ecrit_rapport("Failed to read $MFT");
    goto error_exit;
  }
  ecrit_rapport(OK);

  ecrit_rapport("Reading $MFTMirr... ");
  l = ntfs_attr_mst_pread(vol->mftmirr_na, 0, vol->mftmirr_size,
      vol->mft_record_size, m2);
  if (l != vol->mftmirr_size) {
    ecrit_rapport(FAILED);
    if (l != -1)
      errno = EIO;
    ecrit_rapport("Failed to read $MFTMirr");
    goto error_exit;
  }
  ecrit_rapport(OK);

  /*
   * FIXME: Need to actually check the $MFTMirr for being real. Otherwise
   * we might corrupt the partition if someone is experimenting with
   * software RAID and the $MFTMirr is not actually in the position we
   * expect it to be... )-:
   * FIXME: We should emit a warning it $MFTMirr is damaged and ask
   * user whether to recreate it from $MFT or whether to abort. - The
   * warning needs to include the danger of software RAID arrays.
   * Maybe we should go as far as to detect whether we are running on a
   * MD disk and if yes then bomb out right at the start of the program?
   */

  ecrit_rapport("Comparing $MFTMirr to $MFT... ");
  done = FALSE;
  for (i = 0; i < vol->mftmirr_size; ++i) {
    const char *ESTR[12] = { "$MFT", "$MFTMirr", "$LogFile",
      "$Volume", "$AttrDef", "root directory", "$Bitmap",
      "$Boot", "$BadClus", "$Secure", "$UpCase", "$Extend" };
    const char *s;

    if (i < 12)
      s = ESTR[i];
    else if (i < 16)
      s = "system file";
    else
      s = "mft record";

    if (ntfs_is_baad_recordp(m + i * vol->mft_record_size)) {
      ecrit_rapport("FAILED");
      ecrit_rapport("$MFT error: Incomplete multi sector "
	  "transfer detected in %s.\nCannot "
	  "handle this yet. )-:\n", s);
      goto error_exit;
    }
    if (!ntfs_is_mft_recordp(m + i * vol->mft_record_size)) {
      ecrit_rapport("FAILED");
      ecrit_rapport("$MFT error: Invalid mft record for "
	  "%s.\nCannot handle this yet. )-:\n",
	  s);
      goto error_exit;
    }
    if (ntfs_is_baad_recordp(m2 + i * vol->mft_record_size)) {
      ecrit_rapport("FAILED");
      ecrit_rapport("$MFTMirr error: Incomplete multi "
	  "sector transfer detected in %s.\n", s);
      goto error_exit;
    }
    if (memcmp((u8*)m + i * vol->mft_record_size, (u8*)m2 +
	  i * vol->mft_record_size,
	  ntfs_mft_record_get_data_size((MFT_RECORD*)(
	      (u8*)m + i * vol->mft_record_size)))) {
      if (!done) {
	done = TRUE;
	ecrit_rapport(FAILED);
	ecrit_rapport("Correcting differences in $MFTMirr...");
      }
      br = ntfs_mft_record_write(vol, i, (MFT_RECORD*)(m +
	    i * vol->mft_record_size));
      if (br) {
	ecrit_rapport(FAILED);
	ecrit_rapport("Error correcting $MFTMirr");
	goto error_exit;
      }
    }
  }
  ecrit_rapport(OK);
  ecrit_rapport("Processing of $MFT and $MFTMirr completed successfully.\n");
  ret = 0;
error_exit:
  free(m);
  free(m2);
  return ret;
}

int repair_MFT(t_param_disk *disk_car, t_partition *partition, const int debug)
{
  int ret=-1;
  t_my_data my_data;
  static struct ntfs_device *dev;
  ntfs_volume *vol=NULL;
  my_data.partition=partition;
  my_data.disk_car=disk_car;
  my_data.offset=0;
  if(check_NTFS(disk_car, partition, debug, 0)!=0)
  {
    display_message("Boot sector not valid, can't repair MFT.\n");
    return -1;
  }
  dev = ntfs_device_alloc("/", 0, &ntfs_device_testdisk_io_ops, NULL);
  if (dev)
  {
    dev->d_private=&my_data;
    /* Call ntfs_device_mount() to do the actual mount. */
    vol = ntfs_volume_startup(dev, 0);
  }
  if(!vol)
  {
    display_message("Volume is corrupt. You should run chkdsk.\n");
    ntfs_device_free(dev);
    return -1;
  }
  if (fix_mftmirr(vol) < 0)
    display_message("Failed to correct the MFT");
  else
  {
    ret=0;
    display_message("MFT are ok");
  }
  if (ntfs_umount(vol, 0))
    ntfs_umount(vol, 1);
  return ret;
}
#else
#include "types.h"
#include "common.h"
#include "intrf.h"

int repair_MFT(t_param_disk *disk_car, t_partition *partition, const int debug)
{
  aff_buffer(BUFFER_RESET,"Q");
  aff_copy(stdscr);
  wmove(stdscr,4,0);
  aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
  aff_part_rapport(disk_car,partition);
  aff_buffer(BUFFER_ADD,"Recompile with ntfsprogs library");
  screen_buffer_display(stdscr,"",NULL);
  return 0;
}
#endif
