/*

    File: file_pdf.c

    Copyright (C) 1998-2006 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <stdio.h>
#include "types.h"
#include "photorec.h"

static const char* header_check_pdf(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery);
static void file_check_pdf(t_file_recovery *file_recovery);


const t_file_hint file_hint_pdf= {
  .extension="pdf",
  .description="Portable Document Format, Adobe Illustrator",
  .min_header_distance=0,
  .min_filesize=0,
  .max_filesize=PHOTOREC_MAX_FILE_SIZE,
  .recover=1,
  .header_check=&header_check_pdf,
  .data_check=NULL,
  .file_check=&file_check_pdf
};
static inline const unsigned char *find_in_mem(const unsigned char *haystack, const unsigned char * haystack_end,
    const unsigned char *needle, const unsigned int needle_length);

static inline const unsigned char *find_in_mem(const unsigned char *haystack, const unsigned char * haystack_end,
    const unsigned char *needle, const unsigned int needle_length)
{
  while(haystack!=NULL)
  {
    haystack=memchr(haystack,needle[0],haystack_end-haystack);
    if(haystack!=NULL && haystack<=(haystack_end-needle_length))
    {
      if(memcmp(haystack,needle,needle_length)==0)
	return haystack;
      haystack++;
    }
    else
      haystack=NULL;
  };
  return NULL;
}

static const char* header_check_pdf(const unsigned char *buffer, const unsigned int buffer_size, const unsigned int safe_header_only,  t_file_recovery *file_recovery)
{
  const unsigned char pdf_header[]  = { '%','P','D','F','-','1'};
  if(memcmp(buffer,pdf_header,sizeof(pdf_header))==0)
  {
    const unsigned char sig_illustrator[11]={'I','l','l','u','s','t','r','a','t','o','r'};
    if(find_in_mem(buffer, buffer+512, sig_illustrator,sizeof(sig_illustrator)) != NULL)
    {
      return "ai";
    }
    return file_hint_pdf.extension;
  }
  return NULL;
}

static void file_check_pdf(t_file_recovery *file_recovery)
{
  const unsigned int read_size=4096;
  unsigned char buffer[read_size+3];
  int64_t pdf_size=file_recovery->file_size;
  int i;
  buffer[read_size]=0;
  buffer[read_size+1]=0;
  buffer[read_size+2]=0;
  do
  {
    int taille;
    pdf_size-=read_size;
    if(pdf_size<0)
    {
      pdf_size=0;
      buffer[read_size]=0;
      buffer[read_size+1]=0;
      buffer[read_size+2]=0;
    }
    if(fseek(file_recovery->handle,pdf_size,SEEK_SET)<0)
      return;
    taille=fread(buffer,1,read_size,file_recovery->handle);
    for(i=taille-1;i>=0;i--)
    {
      if(buffer[i]=='%' && buffer[i+1]=='E' && buffer[i+2]=='O' && buffer[i+3]=='F')
      {
	file_recovery->file_size=pdf_size+i+4;
	if(i+4<=read_size+2 && buffer[i+4]=='\r')
	{
	  file_recovery->file_size++;
	  if(i+5<=read_size+2 && buffer[i+5]=='\n')
	    file_recovery->file_size++;
	}
	else if(i+4<=read_size+2 && buffer[i+4]=='\n')
	  file_recovery->file_size++;
	return;
      }
    }
    buffer[read_size]=buffer[0];
    buffer[read_size+1]=buffer[1];
    buffer[read_size+2]=buffer[2];
  } while(pdf_size>0);
  file_recovery->file_size=0;
}
