/*
 * dvifontlist.c - display list of fonts in a dvi file
 * by Hirotsugu Kakugawa
 *
 *  27 Feb 1997  First implementation.
 *   5 Mar 1997  Upgraded for DVIlib 2.3
 *   3 Jun 1997  Added vflibcap parameterlization feature for vflibcap.
 *  13 Nov 1997  Added font existence checking.
 *  27 Nov 1998  Added -q, -dpi, -mode, -cx, -sparcptr, -ljfour options. 
 *  17 Dec 1998  Added print format feature.
 *
 */
/*
 * Copyright (C) 1997-1998  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <varargs.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_STRING_H
#  include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#  include <strings.h>
#endif
#include <ctype.h>

#include "dvi-2_6.h"
#include "defs.h"
#include "compiled.h"
#include "../params.h"

#define   PROGRAM         "dvifontlist"
#define   VERSION         "1.3.2"
#define   DEFAULT_FORMAT  \
          "% 4n %f scaled %.4m, design size %.3dpt, scaled size %.3xpt\\n"

int            param_dpi           = DEFAULT_DPI;
char          *param_kpathsea_mode = DEFAULT_KPATHSEA_MODE;
char          *param_vflibcap      = DEFAULT_VFLIBCAP;
char          *param_dvi_file      = NULL;
DVI_PROPERTY   param_dvi_prop      = NULL;
char          *param_format        = DEFAULT_FORMAT;
int            param_print_missing = 0;
int            param_bm_scale      = 0;


char  *parameters[3];
char   param_value_dpi[128];
char   param_value_mode[128];


DVI_DEVICE create_dev(void);
void   dev_print_fonts(DVI_DEVICE,DVI,char*,long,double,
		       long,double,long,double);

void  parse_args(int,char**);
void  usage(void);
void  license(void);


#define PR1(s1)     fprintf(stderr, s1);
#define PR2(s1,s2)  fprintf(stderr, s1,s2);


int
main(int argc, char **argv)
{
  DVI          dvi  = NULL;
  DVI_DEVICE   dev  = NULL;
  char         params[1024], tmp[1024];

  parse_args(argc, argv);

  sprintf(params, "%s=%s", PARAM_NAME_PROG, "dvifontlist");
  if (param_dpi > 0){
    sprintf(tmp, ", %s=%d", PARAM_NAME_DPI,  param_dpi);
    strcat(params, tmp);
  }
  if (param_kpathsea_mode != NULL){
    sprintf(tmp, ", %s=%s", PARAM_NAME_MODE, param_kpathsea_mode);
    strcat(params, tmp);
  }

  if (DVI_INIT(param_vflibcap, params) < 0){
    fprintf(stderr, "%s: Can't initialize DVI interpreter\n", PROGRAM);
    exit(1);
  }

  if ((dev = create_dev()) == NULL){
    fprintf(stderr, "%s: Can't initialize device\n", PROGRAM);
    exit(1);
  }

  param_dvi_prop = DVI_PROPERTY_ALLOC_DEFAULT();
  if (param_print_missing == 0){
    DVI_PROPERTY_SET(param_dvi_prop, DVI_PROP_LIST_FONTS);
    DVI_PROPERTY_UNSET(param_dvi_prop, DVI_PROP_LIST_MISSING_FONTS);
    DVI_PROPERTY_SET(param_dvi_prop, DVI_PROP_DELAYED_FONT_OPEN);
  } else {
    DVI_PROPERTY_UNSET(param_dvi_prop, DVI_PROP_LIST_FONTS);
    DVI_PROPERTY_SET(param_dvi_prop, DVI_PROP_LIST_MISSING_FONTS);
  }
  if (param_bm_scale == 0){
    DVI_PROPERTY_UNSET(param_dvi_prop, DVI_PROP_SCALE_FONT_IF_NOT_EXIST);
  } else {
    DVI_PROPERTY_SET(param_dvi_prop, DVI_PROP_SCALE_FONT_IF_NOT_EXIST);
  }


  if ((dvi = DVI_CREATE(dev, param_dvi_file, param_dvi_prop)) == NULL){
    fprintf(stderr, "Can't open a DVI file: %s\n", param_dvi_file);
    exit(1);
  }

  (void) DVI_OPEN_FONT(dvi, dev);
  
  return 0;
}


void
parse_args(int argc, char **argv)
{
  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
      exit(0);
    } else if (strcmp(*argv, "-license") == 0){
      license();
      exit(0);
    } else if ((strcmp(*argv, "-v") == 0)  && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if (strcmp(*argv, "-cx") == 0){
      param_dpi = 300;  
      param_kpathsea_mode = "cx";
    } else if (strcmp(*argv, "-sparcptr") == 0){
      param_dpi = 400;  
      param_kpathsea_mode = "sparcptr";
    } else if (strcmp(*argv, "-ljfour") == 0){
      param_dpi = 600;  
      param_kpathsea_mode = "ljfour";
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if (strcmp(*argv, "-missing") == 0){
      param_print_missing = 1;
    } else if (strcmp(*argv, "-scale") == 0){
      param_bm_scale = 1;
    } else if ((strcmp(*argv, "-format") == 0) && (argc > 1)){
      argc--;  argv++;
      param_format = *argv;
    } else if ((strcmp(*argv, "-f") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dvi_file = *argv;
    } else if (**argv != '-'){
      param_dvi_file = *argv;
    } else {
      fprintf(stderr, "%s: Unknow option: %s\n", PROGRAM, *argv);
      exit(1);
    }
  }

  if (param_dvi_file == NULL){
    fprintf(stderr, "%s: No dvi file.\n", PROGRAM);
    exit(1);
  }
}


DVI_DEVICE  
create_dev(void)
{
  DVI_DEVICE  dev;
  
  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = dev->v_dpi   = param_dpi;

  if (param_print_missing == 0){
    dev->font_list          = dev_print_fonts;
  } else {
    dev->font_list_missing  = dev_print_fonts;
  }
  dev->message_error = NULL;
  dev->message_warning = NULL;

  return dev;
}


void  
usage(void)
{
  PR2("%s --- display font list in a DVI file.\n", PROGRAM);
  fprintf(stderr, "version %s of %s on %s\n", 
	  VERSION, COMPILED_DATE, COMPILED_HOST);
  PR2("Usage: %s [Options] DVI-FILE\n", PROGRAM);
  PR1("Options: (Value enclosed by [ ] is the default)\n");
  PR1(" -missing     Print only missing fonts.\n");
  PR1(" -format FMT  Print format\n");
  PR2("      [%s]\n", DEFAULT_FORMAT);
  PR1("      %%f for font name, %%d for design size, %%m for magnification\n");
  PR1("      factor, %%x for scaled size, and %%n for font number.\n");
  PR2(" -v VFLIBCAP  A full path name for vflibcap [%s]\n", 
      DEFAULT_VFLIBCAP);
  PR2(" -dpi DPI     Device resolution in DPI [%d]\n",
      DEFAULT_DPI);
  PR2(" -mode MODE   Device mode name for kpathsea library [%s]\n",
      DEFAULT_KPATHSEA_MODE);
  PR1(" -cx          Same as `-dpi 300 -mode cx'\n");
  PR1(" -sparcptr    Same as `-dpi 400 -mode sparcptr'\n");
  PR1(" -ljfour      Same as `-dpi 600 -mode ljfour'\n");
  PR1(" -license     Print software license\n");
  PR1(" -h           Print this\n");
}

void  
license(void)
{
  PR1("Copyright (C) 1997-1998 Hirotsugu Kakugawa.\n");
  PR1("All rights reserved.\n");
  PR1("This program is free software; you can redistribute it and/or\n");
  PR1("modify it under the terms of the GNU General Public License as\n");
  PR1("published by the Free Software Foundation; either version 2, or \n");
  PR1("(at your option) any later version.\n");
  PR1("\n");
  PR1("This program is distributed in the hope that it will be useful,\n");
  PR1("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
  PR1("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
  PR1("GNU General Public License for more details.\n");
  PR1("\n");
  PR1("This program uses DVIlib and VFlib which are distributed under the\n");
  PR1("terms of GNU Lribrary General Public License\n");
}

void  
dev_print_fonts(DVI_DEVICE dev, DVI dvi, 
		char *font,   /* font name */
		long fontid,  /* font number in DVI */
		double mag,   /* = dsx/ds */
		long dsx,     /* scaled size, unit: sp (= 1^{-16} pt) */
		double fdsx,  /* scaled size, unit: point */
		long ds,      /* design size, unit: sp */
		double fds)   /* design size, unit: point */
{
  char  *s;
  char   f[1024];
  int    i;

  for (s = param_format; *s != '\0'; s++){
    if (*s == '\\'){
      s++;
      switch (*s){
      case '\0':
      case '\\': putchar('\\');	break;
      case '\'': putchar('\''); break;
      case '\"': putchar('\"'); break;
      case 'n':  putchar('\n'); break;
      case 't':  putchar('\t');	break;
      case 'v':  putchar('\v');	break;
      case 'b':  putchar('\b');	break;
      case 'r':  putchar('\r');	break;
      case 'f':  putchar('\f');	break;
      case '0':  putchar('\0');	break;
      }
    } else if (*s == '%'){
      s++;
      switch (*s){
      case '\0':
      case '%':
	putchar('%');
	break;
      case 'f':
	printf("%s", font);
	break;
      case ' ': case '.':
      case '0': case '1': case '2':case '3': case '4':
      case '5': case '6': case '7':case '8': case '9': 
      case 'n': case 'm': case 'd': case 'D': case 'x': case 'X': 
	i = 0;
	f[i++] = '%';
	if (*s == ' '){            /* space part */
	  f[i++] = ' ';
	  s++;
	}
	while (isdigit((int)*s)){  /* int. part */
	  f[i++] = *s;
	  s++;
	}
	if (*s == '.'){            /* point */
	  f[i++] = *s;
	  s++;
	}
	while (isdigit((int)*s)){  /* frac. part */
	  f[i++] = *s;
	  s++;
	}
	switch (*s){          /* format */
	case 'n':  strcpy(&f[i], "ld"); printf(f, fontid);  break;
	case 'm':  strcpy(&f[i], "f");  printf(f, mag);     break;
	case 'd':  strcpy(&f[i], "f");  printf(f, fds);     break;
	case 'D':  strcpy(&f[i], "ld"); printf(f, ds);      break;
	case 'x':  strcpy(&f[i], "f");  printf(f, fdsx);    break;
	case 'X':  strcpy(&f[i], "ld"); printf(f, dsx);     break;
	default:   i = 0; break;
	}
      }
    } else {
      putchar(*s);
    }
  }
}


/*EOF*/
