/*  XGDVI
 *  Copyright (C) 1999-2001  Hirotsugu Kakugawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Edition History
 *  12 Dec 2001   First implementation.
 *
 */

#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#ifdef HAVE_SYS_STAT_H
#ifdef HAVE_SYS_TIME_H
#include <sys/stat.h>
#include <sys/time.h>
#endif
#endif

#include <glib.h>
#include <gtk/gtk.h>

#include "libdvi29.h"
#include "defs.h"
#include "xgdvi.h"
#include "imgcache.h"




static void  wr_compress1(FILE *fp, guchar *buf, int width);
static void  rd_compress1(FILE *fp, guchar *buf, int width);
static void  wr_compress3(FILE *fp, guchar *buf, int width);
static void  rd_compress3(FILE *fp, guchar *buf, int width);


#ifndef DEBUG_CMPR

static int       imgcache_find_free(IMGCACHE ic);
static void      imgcache_put(IMGCACHE ic, long key, long mag, char *cfname,
			      long width, long height, long pos_x, long pos_y);
static void      imgcache_cf_write(IMGCACHE ic, long key, long mag, 
				   FILE *fp, int type, guchar *img,
				   long width, long height);
static guchar   *imgcache_cf_read(IMGCACHE ic, long key, long mag, 
				  FILE *fp, int *type_p);

#define TO_INT_MAG(m)  ((long)((m)*100L))




IMGCACHE
imgcache_new(void)
{
  IMGCACHE  ic;

  if ((ic = (IMGCACHE)g_malloc(sizeof(struct s_imgcache))) == NULL){
    fprintf(stderr, "No memory\n");
    exit(0);
  }

  ic->cftbl_size = 0;
  ic->cftbl      = NULL;
  ic->mag        = -1;

  return ic;
}


void
imgcache_dispose(IMGCACHE ic)
{
  if (ic != NULL){
    if (ic->cftbl != NULL){
      imgcache_purge(ic);
      g_free(ic->cftbl);
      ic->cftbl = NULL;
    }
    g_free(ic);
  }
}


void
imgcache_purge(IMGCACHE ic)
{
  int   i;

  if (ic == NULL)
    return;

  if (ic->cftbl != NULL){
    for (i = 0; i < ic->cftbl_size; i++){
      if (ic->cftbl[i].cfname == NULL)
	continue;
      unlink(ic->cftbl[i].cfname);
      g_free(ic->cftbl[i].cfname);
      ic->cftbl[i].cfname = NULL;
    }
  }
}


static int
imgcache_find_free(IMGCACHE ic)
{
  int  i;

  if (ic == NULL){
    fprintf(stderr, "Internal error in xgdvi:imgcache_find_free()\n");
    abort();
  }

  if (ic->cftbl != NULL){
    for (i = 0; i < ic->cftbl_size; i++){
      if (ic->cftbl[i].cfname == NULL)
	return i;  /* found */
    }
  }

  return -1;  /* not found */
}


int
imgcache_search(IMGCACHE ic, long key, long mag)
{
  int  i;

  if (ic == NULL){
    fprintf(stderr, "Internal error in xgdvi:imgcache_search()\n");
    abort();
  }

  if (ic->cftbl != NULL){
    for (i = 0; i < ic->cftbl_size; i++){
      if ((ic->cftbl[i].key == key) && (ic->cftbl[i].mag == mag)){
	return i;  /* found */
      }
    }
  }

  return -1;  /* not found */
}


static void
imgcache_put(IMGCACHE ic, long key, long mag, char *cfname,
	     long width, long height, long pos_x, long pos_y)
{
  int   t, i, osize, x;

  if (ic == NULL){
    fprintf(stderr, "Internal error in xgdvi:imgcache.c:imgcache_put()\n");
    abort();
  }

  t = imgcache_find_free(ic);

  if (t < 0){
    osize = ic->cftbl_size;
    ic->cftbl_size += 256; 
    x = ic->cftbl_size * sizeof(struct s_imgcache_cftbl);
    if ((ic->cftbl = (IMGCACHE_CFTBL)g_realloc(ic->cftbl, x)) == NULL){
      fprintf(stderr, "No memory\n");
      exit(0);
    }
    for (i = osize; i < ic->cftbl_size; i++){
      ic->cftbl[i].key    = -1; 
      ic->cftbl[i].mag    = -1; 
      ic->cftbl[i].cfname = NULL;
      ic->cftbl[i].pos_x  = 0;
      ic->cftbl[i].pos_y  = 0;
    }

    t = imgcache_find_free(ic);
  }

  if (t < 0){
    fprintf(stderr, "Internal error in xgdvi:imgcache.c:imgcache_put()\n");
    abort();
  }

  ic->cftbl[t].key    = key; 
  ic->cftbl[t].mag    = mag; 
  ic->cftbl[t].cfname = strdup(cfname); 
  ic->cftbl[t].width  = width;
  ic->cftbl[t].height = height;
  ic->cftbl[t].pos_x  = pos_x;
  ic->cftbl[t].pos_y  = pos_y;
}


guchar*
imgcache_is_cached(IMGCACHE ic, long key, double m, int *type_p, 
		   long *width_p, long *height_p, 
		   long *pos_x_p, long *pos_y_p)
{
  int     t;
  long    mag;
  guchar *img;
  FILE   *fp;

  if (ic == NULL){
    fprintf(stderr, "Internal error in xgdvi:imgcache_is_cached()\n");
    abort();
  }

  mag = TO_INT_MAG(m);
  t = imgcache_search(ic, key, mag);
  if (t < 0){
    if (width_p  != NULL)
      *width_p  = 0;
    if (height_p != NULL)
      *height_p = 0;
    if (pos_x_p  != NULL)
      *pos_x_p  = 0;
    if (pos_y_p  != NULL)
      *pos_y_p  = 0;
    return NULL;
  }

#if 0
  fprintf(stderr, "Open Cache File: %s\n", ic->cftbl[t].cfname);
#endif

  if ((fp = fopen(ic->cftbl[t].cfname, "rb")) == NULL)
    return NULL;

  img = imgcache_cf_read(ic, key, mag, fp, type_p);

  fclose(fp);
  fp = NULL;

  if (width_p  != NULL)
    *width_p  = ic->cftbl[t].width;
  if (height_p != NULL)
    *height_p = ic->cftbl[t].height;
  if (pos_x_p  != NULL)
    *pos_x_p  = ic->cftbl[t].pos_x;
  if (pos_y_p  != NULL)
    *pos_y_p  = ic->cftbl[t].pos_y;
 
  return img;
}


void
imgcache_do_cache(IMGCACHE ic, long key, double m,
		  int  type, guchar *img, 
		  long width, long height,
		  long pos_x, long pos_y)
{
  char    cfname[1024];
  long    mag;
  int     fd;
  FILE   *fp;

  if (ic == NULL){
    fprintf(stderr, "Internal error in xgdvi:imgcache_do_cache()\n");
    abort();
  }

  mag = TO_INT_MAG(m);
  sprintf(cfname, ".xgdvi-%ld-%ld.XXXXXX", mag, key);

  fd = mkstemp(cfname);
  fp = fdopen(fd, "wb");
#if 0
  fprintf(stderr, "Created Cache File: %s\n", cfname);
#endif

  if (fp == NULL){
    close(fd);
    return;
  }

  mag = TO_INT_MAG(m);
  imgcache_cf_write(ic, key, mag, fp, type, img, width, height);
  imgcache_put(ic, key, mag, cfname, width, height, pos_x, pos_y);

  fclose(fp);
}

void
imgcache_free_img(guchar *img, int type)
{
  if (img != NULL)
    g_free(img);
}

static void
imgcache_cf_write(IMGCACHE ic, long key, long mag, 
		  FILE *fp, int type, guchar *img,
		  long width, long height)
{
  long  y;
  
  switch (type){
  default:
    fprintf(stderr, "Internal error in xgdvi:imgcache_cf_write()\n");
    abort();
    break;
  case IMGTYPE_BITMAP:
    fprintf(fp, "#bitmap\n");
    fprintf(fp, "%d %ld %ld, %d\n", type, width, height, 1);
    for (y = 0; y < height; y++)
      wr_compress1(fp, &img[y*width], (width+7)/8);
    break;
  case IMGTYPE_GRAYMAP:
    fprintf(fp, "#graymap\n");
    fprintf(fp, "%d %ld %ld\n", type, width, height);
    for (y = 0; y < height; y++)
      wr_compress1(fp, &img[y*width], width);
    break;
  case IMGTYPE_PIXMAP_RGB:
    fprintf(fp, "#pixmap_rgb\n");
    fprintf(fp, "%d %ld %ld\n", type, width, height);
    for (y = 0; y < height; y++)
      wr_compress3(fp, &img[y*width*3], width);
    break;
  }
}

static guchar*
imgcache_cf_read(IMGCACHE ic, long key, long mag, FILE *fp, int *type_p)
{
  guchar  *img = NULL;   
  int      type;
  long     w, h, y;
  char   buff[BUFSIZ];

  if (fp == NULL)
    return NULL;

  /* header */
  if (fgets(buff, sizeof(buff), fp) == NULL)  
    return NULL;
  if (buff[0] != '#')
    return NULL;
  if (fgets(buff, sizeof(buff), fp) == NULL)  
    return NULL;
  if (sscanf(buff, "%d%ld%ld", &type, &w, &h) != 3)
    return NULL;

  /* body */
  switch (type){
  case IMGTYPE_BITMAP:
    if ((img = g_malloc(w*h)) == NULL)
      return NULL;
    for (y = 0; y < h; y++)
      rd_compress1(fp, &img[y*w], w);
    break;
  case IMGTYPE_GRAYMAP:
    if ((img = g_malloc(w*h)) == NULL)
      return NULL;
    for (y = 0; y < h; y++)
      rd_compress1(fp, &img[y*w], (w+7)/8);
    break;
  case IMGTYPE_PIXMAP_RGB:
    if ((img = g_malloc(3*w*h)) == NULL)
      return NULL;
    for (y = 0; y < h; y++)
      rd_compress3(fp, &img[y*w*3], w);
    break;
  }

  *type_p = type;

  return  img;
}


#endif /*DEBUG_CMPR*/






#ifdef DEBUG_CMPR

int            x_r = 0;
int            x_w = 0;
unsigned char  data_c[300];    /* compressed */
unsigned char  data_u[300];  /* uncompressed */

int main(int argc, char **argv)
{
  int  n, i;
  unsigned char  data[300];

  argc--; argv++;
  n = 0;
#if 0
  while (argc > 0){
    data[3*n+0] = data[3*n+1] = data[3*n+2] = atoi(*argv);
    n++;
    argc--; argv++;
  }
  printf("Orig: ");
  for (i = 0; i < n; i++)
    printf("%d ", data[3*i]);
  printf("\n");
  printf("Cmpr: ");
  wr_compress3(NULL, data, n);
  printf("\n");
  rd_compress3(NULL, data_u, n);
  printf("Uncp: ");
  for (i = 0; i < n; i++)
    printf("%d ", data_u[i*3+0]);
  printf("\n");
#else
  while (argc > 0){
    data[n] = atoi(*argv);
    n++;
    argc--; argv++;
  }
  printf("Orig: ");
  for (i = 0; i < n; i++)
    printf("%d ", data[i]);
  printf("\n");
  printf("Cmpr: ");
  wr_compress1(NULL, data, n);
  printf("\n");
  rd_compress1(NULL, data_u, n);
  printf("Uncp: ");
  for (i = 0; i < n; i++)
    printf("%d ", data_u[i]);
  printf("\n");
#endif
}

#endif /*DEBUG_CMPR*/


#ifndef DEBUG_CMPR

#  define shipout1(fp,c)           fputc(c,fp)
#  define shipin1(fp)              fgetc(fp)
#  define shipout3_1(fp,c)         fputc(c,fp)
#  define shipout3_3(fp,c1,c2,c3)  fprintf(fp,"%c%c%c",c1,c2,c3)
#  define shipin3(fp)              fgetc(fp)

#else

#  define shipout1(fp,c)         { printf("%d ", c); data_c[x_w++]=c;}
#  define shipin1(fp)            data_c[x_r++]
#  define shipout3_1(fp,c)         \
       { printf("%d ", c); data_c[x_w++]=c;}
#  define shipout3_3(fp,c1,c2,c3)  \
       { printf("%d ",c1); data_c[x_w++]=c1;data_c[x_w++]=c2;data_c[x_w++]=c3;}
#  define shipin3(fp)              \
        data_c[x_r++]

#endif




static void
wr_compress1(FILE *fp, unsigned char *buf, int width)
{
  int            c;
  unsigned int   r, i;
  int            prev = -1;
  
  i = 0;
  while (i < width){ 
    shipout1(fp, buf[i]);
    c = (i > 0) && (prev == buf[i]);
    prev = buf[i]; 
    i++;
    if (!c)
      continue;
    r = 0;
    while (i < width){ 
      if (prev != buf[i])
	break;
      r++;
      if (r == 65535)
	break;
      i++;
    }
    if (r < 255){
      shipout1(fp, r); 
    } else {
      shipout1(fp, 255); 
      shipout1(fp, r/0x100); 
      shipout1(fp, r%0x100); 
    }
  }
}

static void
rd_compress1(FILE *fp, unsigned char *buf, int width)
{
  unsigned int   r, i, j;
  
  i = 0;
  while (i < width){ 
    buf[i] = shipin1(fp);
    i++;
    if ((i > 1) && (buf[i-1] == buf[i-2])){
      r = shipin1(fp);
      if (r == 255){
	r = shipin1(fp) * 0x100;
	r = r + shipin1(fp);
      }
      j = i-1;
      while ((r > 0) && ( i < width)){
	buf[i] = buf[j];
	r--;
	i++;
      }
    }
  }
}




#define CMPR3(p,s)   (((p)[0]==(s)[0])&&((p)[1]==(s)[1])&&((p)[2]==(s)[2]))

static void
wr_compress3(FILE *fp, unsigned char *buf, int width)
{
  int            c;
  unsigned int   r, i;
  unsigned char  prev[3];
  
  i = 0;
  while (i < width){ 
    shipout3_3(fp, buf[3*i+0], buf[3*i+1], buf[3*i+2]);
    c = (i > 0) && CMPR3(prev,buf+3*i);
    prev[0] = buf[3*i+0]; prev[1] = buf[3*i+1]; prev[2] = buf[3*i+2];
    i++;
    if (!c)
      continue;
    r = 0;
    while (i < width){ 
      if (!CMPR3(prev,buf+3*i))
	break;
      r++;
      if (r == 65535)
	break;
      i++;
    }
    if (r < 255){
      shipout3_1(fp, r); 
    } else {
      shipout3_1(fp, 255); 
      shipout3_1(fp, r/0x100); 
      shipout3_1(fp, r%0x100); 
    }
  }
}

static void
rd_compress3(FILE *fp, unsigned char *buf, int width)
{
  unsigned int   r, i, j;
  
  i = 0;
  while (i < width){ 
    buf[3*i+0] = shipin3(fp);
    buf[3*i+1] = shipin3(fp);
    buf[3*i+2] = shipin3(fp);
    i++;
    if ((i > 1) && CMPR3(buf+3*(i-1),buf+3*(i-2))){
      r = shipin3(fp);
      if (r == 255){
	r = shipin3(fp) * 0x100;
	r = r + shipin3(fp);
      }
      j = i-1;
      while ((r > 0) && ( i < width)){
	buf[3*i+0] = buf[3*j+0];
	buf[3*i+1] = buf[3*j+1];
	buf[3*i+2] = buf[3*j+2];
	r--;
	i++;
      }
    }
  }
}



/*EOF*/
