/*
 *  wordtree.c - Routines to manage word tree and synonimous list
 *
 *  Copyright (C) 2003 Giuseppe Modugno
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "wordtree.h"


/* Internal function prototypes. */
void write_bigend( FILE *f, unsigned int n );
void wordtree_output_idx( WTelem *tree, FILE *idx );
void wordtree_output_dat( WTelem *tree, FILE *dat );



WTelem *
wordtree_add( char *word, WTelem **tree )
{
  /* Add an element in the tree, maintaining alphabetic order of words.
   * Return the new allocated element in the tree.
   * If word exists, return the tree element present in tree without
   * allocating another element. */
  int cmp;
  
  if( *tree==NULL ) {
    /* Create a new element in the tree. */
    if( (*tree=(WTelem *)malloc(sizeof(WTelem)))==NULL )
      return(NULL);
    /* Copy string. */
    (*tree)->word = strdup( word );
    /* Set default status for the word. */
    (*tree)->synlist = NULL;
    (*tree)->num_syn = 0;
    (*tree)->left = NULL;
    (*tree)->right = NULL;
    (*tree)->isword = 0;
    return(*tree);
  }
  
  if( (cmp=strcmp(word,(*tree)->word))>0 )
    return wordtree_add(word,&(*tree)->right);
  else if( cmp<0 )
    return wordtree_add(word,&(*tree)->left);
  else
    return(*tree);		/* Word is present in the tree. */
}



SLelem *
synlist_add( WTelem *word, WTelem *syn )
{
  /* Add synonimous syn to the word word. */
  SLelem **list=&(word->synlist);
  SLelem *new;
  int cmp;

  /* Check if synonimous <syn> is the first of the list. */
  if( (*list==NULL) || (cmp=strcmp( syn->word, (*list)->syn->word ))<0 ) {
    /* Add synonimous <syn> as the first element of the list. */
    if( (new=(SLelem *)malloc(sizeof(SLelem)))==NULL )
      return(NULL);
    new->syn = syn;
    new->next = *list;
    word->num_syn++;
    return( *list=new );
  }
  if( !cmp ) {
    /* First element of synonimous list is equal to the new synonimous. */
    fprintf( stderr, "Warning: synonimous %s yet added to word %s\n", syn->word, word->word );
    return( *list );
  }

  /* Go down to the list up to the element alphabetically previous of new
   * synonimous <syn>. */
  while( ((*list)->next!=NULL) && (cmp=strcmp(syn->word,(*list)->next->syn->word))>0 )
    list = &((*list)->next);
  if( !cmp ) {
    /* Found the same synonimous in the list. */
    fprintf( stderr, "Warning: synonimous %s yet added to word %s\n", syn->word, word->word );
    return( (*list)->next );
  }

  /* Allocate new element list. */
  if( (new=(SLelem *)malloc(sizeof(SLelem)))==NULL )
    return(NULL);
  new->syn = syn;
  new->next = (*list)->next;
  word->num_syn++;

  /* Add new element to the list. */
  return( (*list)->next=new );
}



void
wordtree_output_idx( WTelem *tree, FILE *idx )
{
  /* Write output .idx files. */
  static unsigned int index=0;
  static unsigned int dat_offset=0;
  
  if( tree->left!=NULL )
    wordtree_output_idx( tree->left, idx );
  
  tree->index = index++;
  /* Write word and offset of dat file to idx file. */
  fprintf( idx, "%s,%u\n", tree->word, dat_offset );
  dat_offset += 2+(tree->num_syn<<1);
  
  if( !tree->isword )
    fprintf( stderr, "Warning: Word %s is only a synonimous\n", tree->word );
  if( !tree->num_syn )
    fprintf( stderr, "Warning: Word %s has no synonimous\n", tree->word );

  if( tree->right!=NULL )
    wordtree_output_idx( tree->right, idx );
  
}



void
wordtree_output_dat( WTelem *tree, FILE *dat )
{
  /* Write output .dat files. */
  SLelem *sle;
  static unsigned int dat_offset = 0;
  
  if( tree->left!=NULL )
    wordtree_output_dat( tree->left, dat );
  
  write_bigend( dat, tree->num_syn );
  dat_offset += 2;
  sle = tree->synlist;
  while( sle!=NULL ) {
    write_bigend( dat, sle->syn->index );
    dat_offset += 2;
    sle = sle->next;
  }
  
  if( tree->right!=NULL )
    wordtree_output_dat( tree->right, dat );
  
}


void
wordtree_output( WTelem *tree, FILE *idx, FILE *dat )
{
  /* Write output .idx and .dat files. */
  
  /* It's important to write .idx file first because
   * write_output_idx() enumerate words too and index
   * is used by write_output_dat() routine. */
  wordtree_output_idx( tree, idx );
  wordtree_output_dat( tree, dat );
}


void
write_bigend( FILE *f, unsigned int n )
{
  /* Write number n to file f as a 16-bit Big Endian format. */
  fprintf( f, "%c", (unsigned char)((n&0xFF00)>>8) );
  fprintf( f, "%c", (unsigned char)(n&0x00FF) );
}



void
wordtree_free( WTelem *tree )
{
  /* Free tree memory. */
  SLelem *sle, *stmp;
  
  if( tree->left!=NULL )
    wordtree_free( tree->left );
  
  if( tree->right!=NULL )
    wordtree_free( tree->right );
  
  /* Free synonimous list of that word. */
  sle = tree->synlist;
  while( sle!=NULL ) {
    stmp = sle->next;
    free(sle);
    sle = stmp;
  }
  
  /* Free tree element. */
  free(tree);
  
}
