/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2008 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <float.h>
#include <ctype.h>
#include "DLTutils.h"
#include "Error.h"
#include "pdbMalloc.h"
#include "Coords.h"
#include "PDBCoords.h"
#include "Coords.h"
#include "PDBCoords.h"
#include "DLTmath.h"


Seq2PDB
*Seq2pdbInit(void)
{
    Seq2PDB        *seq2pdb;

    seq2pdb = malloc(sizeof(Seq2PDB));

    seq2pdb->pdbfile_name = NULL;
    seq2pdb->seqname = NULL;
    seq2pdb->map = NULL;
    seq2pdb->pdbfile_name_space = NULL;
    seq2pdb->seqname_space = NULL;

    return(seq2pdb);
}


void
Seq2pdbAlloc(Seq2PDB *seq2pdb, const int seqnum)
{
    int             i;

    seq2pdb->seqnum = seqnum;

    seq2pdb->pdbfile_name = (char **) malloc(seqnum * sizeof(char *));
    seq2pdb->seqname = (char **) malloc(seqnum * sizeof(char *));
    seq2pdb->map = (int *) calloc(seqnum, sizeof(int));

    /* allocate space for the fields in total */
    seq2pdb->pdbfile_name_space = (char *) malloc(FILENAME_MAX * seqnum * sizeof(char));
    seq2pdb->seqname_space = (char *) malloc(128 * seqnum * sizeof(char));

    if (   seq2pdb->pdbfile_name == NULL
        || seq2pdb->pdbfile_name == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr, "\n  ERROR73: could not allocate memory in function Seq2pdbAlloc(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    /*  now 'point' the pointers */
    for (i = 0; i < seqnum; ++i)
    {
        seq2pdb->pdbfile_name[i] = seq2pdb->pdbfile_name_space  + (i * FILENAME_MAX);
        seq2pdb->seqname[i] = seq2pdb->seqname_space  + (i * 128);
    }
}


void
Seq2pdbDestroy(Seq2PDB **seq2pdb_ptr)
{
    Seq2PDB *seq2pdb = *seq2pdb_ptr;

    free(seq2pdb->pdbfile_name_space);
    free(seq2pdb->seqname_space);

    free(seq2pdb->pdbfile_name);
    free(seq2pdb->seqname);
    free(seq2pdb->map);

    MSAfree(&(seq2pdb->msa));

    free(seq2pdb);
    *seq2pdb_ptr = NULL;
}


void
AlgorithmDestroy(Algorithm *algo)
{
    int         i;
    /* printf("\n AlgorithmDestroy(%p)", algo);fflush(NULL); */
    if (algo->argv != NULL)
    {
        for (i = 0; i < algo->argc; ++i)
            if (algo->argv[i] != NULL)
                free(algo->argv[i]);

        free(algo->argv);
    }

    free(algo);
}


Algorithm
*AlgorithmInit(void)
{
    Algorithm      *algo = (Algorithm *) malloc (sizeof(Algorithm));
    if (algo == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR6: could not allocate memory in function CoordsArrayInit(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }
    /* printf("\n AlgorithmInit(%p)", algo);fflush(NULL); */
    algo->argv         = NULL;
    mystrcpy(algo->rootname, "theseus");
    algo->weight       = 0;
    algo->verbose      = 0; /* verbose behavior = 1 */
    algo->method       = 3; /* 7: LAPACK SVD, 10: my classic Jacobi SVD */
    algo->print_trans  = 0;
    algo->write_file   = 1;
    algo->print_weight = 0;
    algo->precision    = 1e-7;
    algo->iterations   = 200;
    algo->selection    = NULL;
    algo->atomslxn     = NULL;
    algo->revsel       = 0;
    algo->atoms        = 0;
    algo->reflection   = 0;
    algo->embedave     = 0; /* 2 = biased ML embedded structure */
    algo->jackknife    = 0;
    algo->writestats   = 1;
    algo->random       = 0;
    algo->pca          = 0;
    algo->fullpca      = 0;
    algo->cormat       = 1;
    algo->tenberge     = 0;
    algo->morph        = 0;
    algo->stats        = 0;
    algo->constant     = -1e-40; /* If negative, finds min var empirically */
    algo->info         = 0;
    algo->princaxes    = 1;
    algo->nullrun      = 0;
    algo->binary       = 0;
    algo->modelpca     = 0;
    algo->raxes[0] =
    algo->raxes[1] = 
    algo->raxes[2] =   1.0;
    algo->mbias        = 0;
    algo->notrans      = 0;
    algo->norot        = 0;
    algo->alignment    = 0;
    algo->fmodel       = 0;
    algo->dimweight    = 0;
    algo->covweight    = 0;
    algo->varweight    = 1;
    algo->LedoitWolf   = 0;
    algo->hierarch     = 1;
    algo->leastsquares = 0;
    algo->filenum      = 0;
    algo->infiles      = NULL;
    algo->noave        = 0;
    algo->noinnerloop  = 0;
    algo->htrans       = 0;
    algo->rounds       = 0;
    algo->innerrounds  = 0;
    algo->fasta        = 0;
    algo->olve         = 0;
    algo->abort        = 0;
    algo->seed         = 0;
    algo->mixture      = 1;
    algo->threads      = 0;
    algo->printlogL    = 0;
    algo->bfact        = 0;
    algo->convlele     = 0;
    algo->param[0] = algo->param[1] = 1.0;
    algo->radii[0] = algo->radii[1] = algo->radii[2] = 50.0;
    algo->ssm          = 0;
    algo->lele5        = 0;
    algo->bayes        = 0;
    algo->ipmat        = 0;
    algo->commandeur   = 0;
    algo->missing      = 0;
    algo->scale        = 0;
    algo->instfile     = 0;

    return(algo);
}


Statistics
*StatsInit(void)
{
    Statistics     *stats;
    int             i;

    stats = (Statistics *) malloc(sizeof(Statistics));
    if (stats == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR6: could not allocate memory in function StatsInit(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    for (i = 0; i < 4; ++i)
        stats->skewness[i] = stats->kurtosis[i] = 0.0;

    stats->hierarch_p1 = stats->hierarch_p2 = 1.0;

    return(stats);
}


CoordsArray
*CoordsArrayInit(void)
{
    CoordsArray *cdsA;

    cdsA = (CoordsArray *) malloc(sizeof(CoordsArray));
/* printf("\ncdsA malloc(%p)", (void *) cdsA); */
/* fflush(NULL); */
    if (cdsA == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR6: could not allocate memory in function CoordsArrayInit(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    strncpy(cdsA->outfile_name, "theseus.pdb", 11);
    cdsA->anchorf_name = NULL;
    cdsA->mapfile_name = NULL;
    cdsA->msafile_name = NULL;
    cdsA->pdbA = NULL;
    cdsA->scratchA = NULL;

    cdsA->coords = NULL;
    cdsA->avecoords = NULL;
    cdsA->tcoords = NULL;
    cdsA->jkcoords = NULL;

    cdsA->w = NULL;
    cdsA->var = NULL;
    cdsA->df = NULL;
    cdsA->S2 = NULL;

    cdsA->algo = AlgorithmInit();
    cdsA->stats = StatsInit();

    cdsA->residuals = NULL;

    cdsA->Var_matrix = NULL;
    cdsA->Dij_matrix = NULL;
    cdsA->distmat = NULL;
    cdsA->CovMat = NULL;
    cdsA->WtMat = NULL;
    cdsA->FullCovMat = NULL;
    cdsA->MVCovMat = NULL;
    cdsA->SCovMat = NULL;

    cdsA->AxCovMat = MatInit(3, 3);
    cdsA->AxWtMat  = MatInit(3, 3);

    cdsA->pcamat = NULL;
    cdsA->pcavals = NULL;
    cdsA->modpcamat = NULL;
    cdsA->modpcavals = NULL;

    cdsA->tmpmat1 = NULL;
    cdsA->tmpmat2 = NULL;
    cdsA->tmpmatKK1 = NULL;
    cdsA->tmpmatKK2 = NULL;
    cdsA->tmpvecK = NULL;
    cdsA->tmpmat3K = NULL;
    cdsA->tmpmatK3a = NULL;
    cdsA->tmpmatK3b = NULL;

    cdsA->tmpmat3a = MatInit(3, 3);
    cdsA->tmpmat3b = MatInit(3, 3);
    cdsA->tmpmat3c = MatInit(3, 3);
    cdsA->tmpmat3d = MatInit(3, 3);

    return(cdsA);
}


void
CoordsArrayAllocNum(CoordsArray *cdsA, const int cnum)
{
    cdsA->cnum = cnum;

    cdsA->coords = (Coords **) malloc(cnum * sizeof(Coords *));
    if (cdsA->coords == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR6: could not allocate memory in function CoordsArrayAlloc(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }
}


void
CoordsArrayAllocLen(CoordsArray *cdsA, const int vlen)
{
    int             i;

    cdsA->vlen = vlen;

    for (i = 0; i < cdsA->cnum; ++i)
    {
        cdsA->coords[i] = CoordsInit();
        CoordsAlloc(cdsA->coords[i], vlen);
    }

    cdsA->avecoords = CoordsInit();
    CoordsAlloc(cdsA->avecoords, vlen);
    cdsA->tcoords = CoordsInit();
    CoordsAlloc(cdsA->tcoords, vlen);
    cdsA->jkcoords = CoordsInit();
    CoordsAlloc(cdsA->jkcoords, vlen);

    cdsA->var = (double *) calloc(vlen, sizeof(double));
    cdsA->w   = (double *) calloc(vlen, sizeof(double));
    cdsA->df  = (int *) calloc(vlen, sizeof(int));
}


void
CoordsArrayAlloc(CoordsArray *cdsA, const int cnum, const int vlen)
{
    CoordsArrayAllocNum(cdsA, cnum);
    CoordsArrayAllocLen(cdsA, vlen);
}


void
CoordsArraySetup(CoordsArray *cdsA)
{
    memsetd(cdsA->var, 1.0, cdsA->vlen);
    memsetd(cdsA->w, 1.0, cdsA->vlen);
    cdsA->AxCovMat[0][0] = cdsA->AxCovMat[1][1] = cdsA->AxCovMat[2][2] = 1.0;
    cdsA->AxWtMat[0][0] = cdsA->AxWtMat[1][1] = cdsA->AxWtMat[2][2] = 1.0;
    memsetd(&cdsA->axesw[0], 1.0, 3);
}


void
CoordsArrayDestroy(CoordsArray **cdsA_ptr)
{
    int             i;

    CoordsArray *cdsA = *cdsA_ptr;

    free(cdsA->msafile_name);
    cdsA->msafile_name = NULL;
    free(cdsA->mapfile_name);
    cdsA->mapfile_name = NULL;

    for (i = 0; i < cdsA->cnum; ++i)
    {
        CoordsDestroy(&(cdsA->coords[i]));
    }

    free(cdsA->coords);
    cdsA->coords = NULL;

    CoordsDestroy(&(cdsA->avecoords));
    CoordsDestroy(&(cdsA->tcoords));
    CoordsDestroy(&(cdsA->jkcoords));

    free(cdsA->stats);
    cdsA->stats = NULL;

    if (cdsA->algo->selection != NULL)
    {
        free(cdsA->algo->selection);
        cdsA->algo->selection = NULL;
    }

    if (cdsA->algo->atomslxn != NULL)
    {
        free(cdsA->algo->atomslxn);
        cdsA->algo->atomslxn = NULL;
    }

    AlgorithmDestroy(cdsA->algo);

    DistMatsDestroy(cdsA);
    CovMatsDestroy(cdsA);

    if (cdsA->residuals != NULL)
    {
        free(cdsA->residuals);
        cdsA->residuals = NULL;
    }

    free(cdsA->w);
    cdsA->w = NULL;
    free(cdsA->var);
    cdsA->var = NULL;
    free(cdsA->df);
    cdsA->df = NULL;

    if (cdsA->S2 != NULL)
    {
        free(cdsA->S2);
        cdsA->S2 = NULL;
    }

    free(cdsA);
    *cdsA_ptr = NULL;
}


void
DistMatsAlloc(CoordsArray *cdsA)
{
    /* cdsA->Var_matrix = MatInit(cdsA->vlen, cdsA->vlen); */
    cdsA->Dij_matrix = MatInit(cdsA->vlen, cdsA->vlen);
    /* cdsA->distmat    = Mat3DInit(cdsA->cnum, cdsA->vlen, cdsA->vlen); */
}


void
DistMatsDestroy(CoordsArray *cdsA)
{
/*     if (cdsA->Var_matrix != NULL) */
/*         MatDestroy(&(cdsA->Var_matrix)); */
    if (cdsA->Dij_matrix != NULL)
        MatDestroy(&(cdsA->Dij_matrix));
/*     if (cdsA->distmat != NULL) */
/*         Mat3DDestroy(&(cdsA->distmat)); */
}


void
CovMatsDestroy(CoordsArray *cdsA)
{
    if (cdsA->CovMat != NULL)
        MatDestroy(&(cdsA->CovMat));
    if (cdsA->FullCovMat != NULL)
        MatDestroy(&(cdsA->FullCovMat));
    if (cdsA->WtMat != NULL)
        MatDestroy(&(cdsA->WtMat));
    if (cdsA->tmpmatKK1 != NULL)
        MatDestroy(&(cdsA->tmpmatKK1));
    if (cdsA->tmpmatKK1 != NULL)
        MatDestroy(&(cdsA->tmpmatKK2));
    if (cdsA->tmpvecK != NULL)
        free(cdsA->tmpvecK);
    if (cdsA->tmpmat1 != NULL)
        MatDestroy(&(cdsA->tmpmat1));
    if (cdsA->tmpmat2 != NULL)
        MatDestroy(&(cdsA->tmpmat2));
    if (cdsA->tmpmat3a != NULL)
        MatDestroy(&(cdsA->tmpmat3a));
    if (cdsA->tmpmat3b != NULL)
        MatDestroy(&(cdsA->tmpmat3b));
    if (cdsA->tmpmat3c != NULL)
        MatDestroy(&(cdsA->tmpmat3c));
    if (cdsA->tmpmat3d != NULL)
        MatDestroy(&(cdsA->tmpmat3d));
    if (cdsA->MVCovMat != NULL)
        MatDestroy(&(cdsA->MVCovMat));
    if (cdsA->AxCovMat != NULL)
        MatDestroy(&(cdsA->AxCovMat));
    if (cdsA->AxWtMat != NULL)
        MatDestroy(&(cdsA->AxWtMat));
    if (cdsA->SCovMat != NULL)
        MatDestroy(&(cdsA->SCovMat));
    if (cdsA->tmpmatK3a != NULL)
        MatDestroy(&(cdsA->tmpmatK3a));
    if (cdsA->tmpmatK3b != NULL)
        MatDestroy(&(cdsA->tmpmatK3b));
    if (cdsA->tmpmat3K != NULL)
        MatDestroy(&(cdsA->tmpmat3K));
}


void
PCAAlloc(CoordsArray *cdsA)
{
    cdsA->pcamat = MatInit(cdsA->vlen, cdsA->vlen);
    cdsA->pcavals = malloc(cdsA->vlen * sizeof(double));
}


void
PCADestroy(CoordsArray *cdsA)
{
    if (cdsA->pcamat != NULL)
        MatDestroy(&(cdsA->pcamat));
    if (cdsA->pcavals != NULL)
    {
        free(cdsA->pcavals);
        cdsA->pcavals = NULL;
    }
}


Coords
*CoordsInit(void)
{
    int             i, j;
    Coords         *coords;

    coords = (Coords *) malloc(sizeof(Coords));

    coords->resName    = NULL;
    coords->resName_space = NULL;
    coords->chainID    = NULL;
    coords->resSeq     = NULL;
    coords->x          = NULL;
    coords->y          = NULL;
    coords->z          = NULL;
    coords->o          = NULL;
    coords->b          = NULL;
    coords->residual_x = NULL;
    coords->residual_y = NULL;
    coords->residual_z = NULL;
    coords->covx       = NULL;
    coords->covy       = NULL;
    coords->covz       = NULL;

    coords->innerprod = NULL;
    coords->vlen = 0;
    coords->innerprod2 = NULL;

    coords->matrix = MatInit(3, 3);
    coords->last_matrix = MatInit(3, 3);
    coords->evecs = MatInit(4,4);
    coords->jackmat = MatInit(3, 3);

    for (i = 0; i < 4; ++i)
        coords->evecs[i][0] = 1.0;

    for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
            coords->matrix[i][j] = coords->last_matrix[i][j] = 0.0;

    coords->tmpmat3a = MatInit(3, 3);
    coords->tmpmat3b = MatInit(3, 3);
    coords->tmpmat3c = MatInit(3, 3);
    coords->tmpmat3d = MatInit(3, 3);

    coords->bfact_c = 1.0;
    coords->scale = 1.0;

    return(coords);
}


void
CoordsAlloc(Coords *coords, const int vlen)
{
    int             i;

    coords->vlen = vlen;

    coords->resName    = (char **)  calloc(vlen, sizeof(char *));
    coords->resName_space = (char *)  calloc(4 * vlen, sizeof(char));
    coords->chainID    = (char *)   calloc(vlen, sizeof(char));
    coords->resSeq     = (int *)    calloc(vlen, sizeof(int));
    coords->x          = (double *) calloc(vlen, sizeof(double));
    coords->y          = (double *) calloc(vlen, sizeof(double));
    coords->z          = (double *) calloc(vlen, sizeof(double));
    coords->o          = (double *) calloc(vlen, sizeof(double));
    coords->b          = (double *) calloc(vlen, sizeof(double));
    coords->prvar      = (double *) calloc(vlen, sizeof(double));
    coords->residual_x = (double *) calloc(vlen, sizeof(double));
    coords->residual_y = (double *) calloc(vlen, sizeof(double));
    coords->residual_z = (double *) calloc(vlen, sizeof(double));
    coords->covx       = (double *) calloc(vlen, sizeof(double));
    coords->covy       = (double *) calloc(vlen, sizeof(double));
    coords->covz       = (double *) calloc(vlen, sizeof(double));

    if (   coords->resName       == NULL
        || coords->resName_space == NULL
        || coords->chainID       == NULL
        || coords->resSeq        == NULL
        || coords->x             == NULL
        || coords->y             == NULL
        || coords->z             == NULL
        || coords->o             == NULL
        || coords->b             == NULL
        || coords->prvar         == NULL
        || coords->residual_x    == NULL
        || coords->residual_y    == NULL
        || coords->residual_z    == NULL
        || coords->covx          == NULL
        || coords->covy          == NULL
        || coords->covz          == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr, "\n  ERROR5: could not allocate memory in function CoordsAlloc(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    for (i = 0; i < vlen; ++i)
        coords->resName[i] = coords->resName_space + (i * 4); /* 3 */

    CoordsSetup(coords);
}


void
CoordsSetup(Coords *coords)
{
    int             i;

    memsetd(coords->translation, 0.0, 3);
    memsetd(coords->jktranslation, 0.0, 3);
    memsetd(coords->transsum, 0.0, 3);

    for (i = 0; i < 4; ++i)
        coords->evecs[i][0] = 1.0;

    for (i = 0; i < 3; ++i)
        coords->matrix[i][i] = coords->last_matrix[i][i] = 1.0;
}


void
CoordsDestroy(Coords **coords_ptr)
{
    Coords *coords = *coords_ptr;

    free(coords->resSeq);
    free(coords->resName_space);
    free(coords->chainID);
    free(coords->resName);
    free(coords->x);
    free(coords->y);
    free(coords->z);
    free(coords->o);
    free(coords->b);
    free(coords->prvar);
    free(coords->residual_x);
    free(coords->residual_y);
    free(coords->residual_z);
    free(coords->covx);
    free(coords->covy);
    free(coords->covz);
    MatDestroy(&(coords->matrix));
    MatDestroy(&(coords->last_matrix));
    MatDestroy(&(coords->jackmat));

    if (coords->innerprod != NULL)
        MatDestroy(&(coords->innerprod));

    if (coords->innerprod2 != NULL)
        MatDestroy(&(coords->innerprod2));

    MatDestroy(&(coords->evecs));

    MatDestroy(&(coords->tmpmat3a));
    MatDestroy(&(coords->tmpmat3b));
    MatDestroy(&(coords->tmpmat3c));
    MatDestroy(&(coords->tmpmat3d));

    free(coords);
    *coords_ptr = NULL;
}


PDBCoordsArray
*PDBCoordsArrayInit(void)
{
    PDBCoordsArray *pdbA; 

    pdbA = (PDBCoordsArray *) malloc(sizeof(PDBCoordsArray));
    if (pdbA == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR6: could not allocate memory in function PDBCoordsArrayAlloc(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    pdbA->coords = NULL;
    pdbA->avecoords = NULL;
    pdbA->upper = NULL;
    pdbA->lower = NULL;
    pdbA->scratchA = NULL;
    pdbA->cdsA = NULL;
    pdbA->seq2pdb = NULL;

    return(pdbA);
}


void
PDBCoordsArrayAlloc(PDBCoordsArray *pdbA, const int cnum, const int vlen)
{
    PDBCoordsArrayAllocNum(pdbA, cnum);
    PDBCoordsArrayAllocLen(pdbA, vlen);
}


void
PDBCoordsArrayAllocNum(PDBCoordsArray *pdbA, const int cnum)
{
    int             i;

    pdbA->coords = (PDBCoords **) malloc(cnum * sizeof(PDBCoords *));
    pdbA->cnum = cnum;

    for (i = 0; i < cnum; ++i)
        pdbA->coords[i] = PDBCoordsInit();

    pdbA->avecoords = PDBCoordsInit();
}


void
PDBCoordsArrayAllocLen(PDBCoordsArray *pdbA, const int vlen)
{
    int             i;

    if (pdbA == NULL
        || pdbA->coords == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR6: could not allocate memory in function PDBCoordsArrayAlloc(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    pdbA->vlen = vlen;

    for (i = 0; i < pdbA->cnum; ++i)
    {
        if (pdbA->coords[i]->vlen == 0) /* only allocate if it needs it */
            PDBCoordsAlloc(pdbA->coords[i], vlen);
    }

    PDBCoordsAlloc(pdbA->avecoords, vlen);
}


void
PDBCoordsArrayDestroy(PDBCoordsArray **pdbA_ptr)
{
    PDBCoordsArray *pdbA = *pdbA_ptr;
    int             i;

    for (i = 0; i < pdbA->cnum; ++i)
        PDBCoordsDestroy(&(pdbA->coords[i]));

    PDBCoordsDestroy(&(pdbA->avecoords));

    if (pdbA->upper != NULL)
        free(pdbA->upper);

    if (pdbA->lower != NULL)
        free(pdbA->lower);

    if (pdbA->seq2pdb != NULL)
        Seq2pdbDestroy(&(pdbA->seq2pdb));

    free(pdbA->coords);
    free(pdbA);

    *pdbA_ptr = NULL;
}


PDBCoords
*PDBCoordsInit(void)
{
    PDBCoords      *pdbcoords;

    pdbcoords = (PDBCoords *) malloc(sizeof(PDBCoords));
    pdbcoords->vlen = 0; /* so that we know later if this has been allocated or not */
    pdbcoords->translation = calloc(3, sizeof(double));
    pdbcoords->matrix = MatInit(3, 3);
    memset(&pdbcoords->matrix[0][0], 0, 9 * sizeof(double));

    return(pdbcoords);
}


void
PDBCoordsAlloc(PDBCoords *pdbcoords, const int vlen)
{
    int             i;

    pdbcoords->vlen = vlen;

    /* allocate memory for the pointers to the fields */
    pdbcoords->record     = (char **)  calloc(vlen, sizeof(char *));
    pdbcoords->serial     = (unsigned int *) calloc(vlen, sizeof(unsigned int));
    pdbcoords->Hnum       = (char *)   calloc(vlen, sizeof(char));
    pdbcoords->name       = (char **)  calloc(vlen, sizeof(char *));
    pdbcoords->altLoc     = (char *)   calloc(vlen, sizeof(char));
    pdbcoords->resName    = (char **)  calloc(vlen, sizeof(char *));
    pdbcoords->xchainID   = (char *)   calloc(vlen, sizeof(char));
    pdbcoords->chainID    = (char *)   calloc(vlen, sizeof(char));
    pdbcoords->resSeq     = (int *)    calloc(vlen, sizeof(int));
    pdbcoords->iCode      = (char *)   calloc(vlen, sizeof(char));
    pdbcoords->x          = (double *) calloc(vlen, sizeof(double));
    pdbcoords->y          = (double *) calloc(vlen, sizeof(double));
    pdbcoords->z          = (double *) calloc(vlen, sizeof(double));
    pdbcoords->occupancy  = (double *) calloc(vlen, sizeof(double));
    pdbcoords->tempFactor = (double *) calloc(vlen, sizeof(double));
    pdbcoords->segID      = (char **)  calloc(vlen, sizeof(char *));
    pdbcoords->element    = (char **)  calloc(vlen, sizeof(char *));
    pdbcoords->charge     = (char **)  calloc(vlen, sizeof(char *));

    /* allocate space for the fields in total */
    pdbcoords->record_space  = (char *)  calloc(8 * vlen, sizeof(char));
    pdbcoords->name_space    = (char *)  calloc(4 * vlen, sizeof(char));
    pdbcoords->resName_space = (char *)  calloc(4 * vlen, sizeof(char));
    pdbcoords->segID_space   = (char *)  calloc(8 * vlen, sizeof(char));
    pdbcoords->element_space = (char *)  calloc(4 * vlen, sizeof(char));
    pdbcoords->charge_space  = (char *)  calloc(4 * vlen, sizeof(char));

    if (   pdbcoords->record        == NULL
        || pdbcoords->serial        == NULL
        || pdbcoords->Hnum          == NULL
        || pdbcoords->name          == NULL
        || pdbcoords->altLoc        == NULL
        || pdbcoords->resName       == NULL
        || pdbcoords->xchainID      == NULL
        || pdbcoords->chainID       == NULL
        || pdbcoords->resSeq        == NULL
        || pdbcoords->iCode         == NULL
        || pdbcoords->x             == NULL
        || pdbcoords->y             == NULL
        || pdbcoords->z             == NULL
        || pdbcoords->occupancy     == NULL
        || pdbcoords->tempFactor    == NULL
        || pdbcoords->segID         == NULL
        || pdbcoords->element       == NULL
        || pdbcoords->charge        == NULL
        || pdbcoords->record_space  == NULL
        || pdbcoords->name_space    == NULL
        || pdbcoords->resName_space == NULL
        || pdbcoords->segID_space   == NULL
        || pdbcoords->element_space == NULL
        || pdbcoords->charge_space  == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr, "\n  ERROR7: could not allocate memory in function PDBCoordsAlloc(). \n");
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    /*  now 'point' the pointers */
    for (i = 0; i < pdbcoords->vlen; ++i)
    {
        pdbcoords->record[i]  = pdbcoords->record_space  + (i * 8); /* 6 */
        pdbcoords->name[i]    = pdbcoords->name_space    + (i * 4); /* 3 */
        pdbcoords->resName[i] = pdbcoords->resName_space + (i * 4); /* 3 */
        pdbcoords->segID[i]   = pdbcoords->segID_space   + (i * 8); /* 4 */
        pdbcoords->element[i] = pdbcoords->element_space + (i * 4); /* 2 */
        pdbcoords->charge[i]  = pdbcoords->charge_space  + (i * 4); /* 2 */
    }
}


void
PDBCoordsDestroy(PDBCoords **pdbcoords_ptr)
{
    PDBCoords *pdbcoords = *pdbcoords_ptr;

    free(pdbcoords->record_space);
    free(pdbcoords->name_space);
    free(pdbcoords->resName_space);
    free(pdbcoords->segID_space);
    free(pdbcoords->element_space);
    free(pdbcoords->charge_space);

    MatDestroy(&(pdbcoords->matrix));
    free(pdbcoords->translation);

    free(pdbcoords->record);
    free(pdbcoords->serial);
    free(pdbcoords->Hnum);
    free(pdbcoords->name);
    free(pdbcoords->altLoc);
    free(pdbcoords->resName);
    free(pdbcoords->xchainID);
    free(pdbcoords->chainID);
    free(pdbcoords->resSeq);
    free(pdbcoords->iCode);
    free(pdbcoords->x);
    free(pdbcoords->y);
    free(pdbcoords->z);
    free(pdbcoords->occupancy);
    free(pdbcoords->tempFactor);
    free(pdbcoords->segID);
    free(pdbcoords->element);
    free(pdbcoords->charge);

    free(pdbcoords);
    *pdbcoords_ptr = NULL;
}
