/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2009 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include <float.h>
#include <unistd.h>
#include <ctype.h>
#include "DLTmath.h"
#include "gamma_dist.h"
#include "normal_dist.h"
#include "invgauss_dist.h"
#include "statistics.h"

#define EPS1 0.001
#define EPS2 1.0e-8


double
RoundInt(const double x)
{
    double      tmp;

	if (modf(x, &tmp) < 0.5)
		return(tmp);
	else
		return(tmp + 1.0);
}


double
average(const double *data, const int dim)
{
    double          m = 0.0;
    int             i = dim;

    while(i-- > 0)
        m += *data++;

    return(m / (double) dim);
}


double
Variance(const double *data, const int dim, const double mean)
{
    double          v = 0.0, tmpv;
    int             i = dim;

    while(i-- > 0)
    {
        tmpv = *data++ - mean;
        v += (tmpv * tmpv);
    }

    return(v / dim);
}


/* calculate the average standard error of a vector of variances
   see eqn 54 from http://mathworld.wolfram.com/NormalDistribution.html
   and http://mathworld.wolfram.com/SampleVarianceDistribution.html
   This calculation is parametric, and assumes a normal distribution.
   It is based on the fact that the variance of a variance is given by
   Var(var) =  2 * var^2 * (N-1) / N^2, where N is the sample size.
   Here N (sdim) is in the denom since this is the standard error of the mean,
   i.e. we are interested in the variance of the estimate, not the
   distribution.*/
double
VarVariance(const double *data, const int ddim, const int sdim)
{
    double          v = 0.0, tmpv;
    int             i = ddim;

    while(i-- > 0)
    {
        tmpv = *data++;
        v += (tmpv * tmpv);
    }

    return(2.0 * v * (sdim - 1) / (sdim * sdim * sdim * ddim));
    /* return(2.0 * v / (ddim * sdim * sdim)); */
}


int
moments(const double *data, const int dim, /* data array and length */
        double *ave, double *median,       /* mean and median */
        double *adev, double *mdev,        /* ave dev from mean and median */
        double *sdev, double *var,         /* std deviation, variance */
        double *skew, double *kurt,        /* skewness and kurtosis */
        double *hrange, double *lrange)    /* range of data, high and low */
{
    int             i;
    double          verror, sum, temp;
    double         *x;
    double          ddim = (double) dim;

    if (dim <= 1)
    {
        fprintf(stderr, "\nERROR: #data points in \'moments\' is less than 2 \n");
        return(0);
    }

    x = malloc((dim + 1) * sizeof(double));
    memcpy(x, data, dim * sizeof(double));
    x[dim] = DBL_MAX; /* required sentinel for quicksort */
    /* quicksort(x, dim); */
    qsort(x, dim, sizeof(double), dblcmp);

    if (dim % 2 == 0)
        *median = (x[dim / 2] + x[(dim - 1) / 2]) / 2.0;
    else
        *median = x[(dim - 1) / 2];

    *hrange = x[dim - 1];
    *lrange = x[0];

    sum = 0.0;
    for (i = 0; i < dim; ++i)
        sum += x[i];

    *ave = sum / ddim;

    (*adev) = (*mdev) = (*var) = (*skew) = (*kurt) = verror = 0.0;
    for (i = 0; i < dim; ++i)
    {
        sum = x[i] - (*ave);
        *adev += fabs(sum);
        *mdev += fabs(x[i] - (*median));
        verror += sum;
        temp = sum * sum;
        *var += temp;
        temp *= sum;
        *skew += temp;
        temp *= sum;
        *kurt += temp;
    }

    free(x);

    *adev /= ddim;
    *mdev /= ddim;
    *var = (*var - (verror * verror / ddim)) / (ddim - 1.0);
    *sdev = sqrt(*var);

    if (*var)
    {
        *skew /= (ddim * (*var) * (*sdev));
        *kurt = (*kurt) / (ddim * (*var) * (*var)) - 3.0;
        return(1);
    }
    else
        return(0);
}


/* chi_sqr_adapt() calculates the reduced chi^2 value for a fit
   of data to a distribution, and the log likelihood for the given
   parameters.

   You must specify the distribution's pdf, its ln(pdf), and a
   function for integrating the pdf between two values. For
   compatibility, all functions here take two parameters for the
   distribution, usu. a location and a shape parameter. If your
   dist has only one parameter, make a second dummy parameter and
   pass it a NULL value. The following function assumes that the
   distribution is approximately correct, and assigns equal numbers
   of data points to the bins (using the limits of that specific
   binned data for the integration). Chi^2 tests only work well
   when there are > 6 expected data points per bin. Thus, if the
   dist does not describe the data well, this might not be true,
   but in that case you should get a large chi^2 value anyway.

   double  *data            array of your data
   int      num             number of data points - must be >= 25
   int      nbins           number of bins for chi^2 fit
                            if you pass 0, the default is sqrt(num)
   double  *logL            pointer to log likelihood value
   double   ave             first distribution parameter, usu. location
   double   lambda          second dist param, usu. scale
   double  *dist_pdf        function pointer, calcualtes pdf for dist
   double  *dist_lnpdf      function pointer, calculates log of pdf for dist
   double  *dist_int        function pointer, integrates pdf between x and y
                            the function integrate_romberg() can be used to
                            integrate a pdf if the cdf or sdf are unknown
*/
double
chi_sqr_adapt(const double *data, const int num, int nbins, double *logL,
              const double ave, const double lambda,
              double (*dist_pdf)(double x, double param1, double param2),
              double (*dist_lnpdf)(double x, double param1, double param2),
              double (*dist_int)(double x, double y, double param1, double param2))
{
    int             i, j, bincount;
    double          chisq, temp, expect;
    double         *bins, *array;
/* *logL = dist_logL(dist_lnpdf, ave, lambda, data, num); */
/* if (1) return(0); */
    temp = expect = chisq = 0;

    if (nbins == 0)
    {
/*         if (num < 25) */
/*         { */
/*             fprintf(stderr, "\n\n ERROR: not enough data points in chi_sqr_adapt \n"); */
/*             exit(EXIT_FAILURE); */
/*         } */

        nbins = floor(sqrt((double) num));
    }

    bins  = calloc(nbins, sizeof(double));
    array = calloc(num,   sizeof(double));

    for (j = 0; j < num; ++j)
        array[j] = data[j];

    qsort(array, num, sizeof(double), dblcmp);

    chisq = 0.0;
    for (i = 0; i < nbins - 1; ++i)
    {
        expect = dist_int(array[i * nbins], array[(i+1) * nbins], ave, lambda);

        if (expect == 0.0)
            continue;
 
        expect *= (double) num;

        bincount = nbins;
        temp = bincount - expect;
        temp = mysquare(temp) / expect;
        chisq += temp;
/*         printf("\n%-7d range [%10.3f to %10.3f (%10.3f)] %-8.3f %-8.3f %-8.3f %-14.5f", */
/*                 i+1, array[i * nbins], array[(i+1) * nbins], array[(i+1) * nbins] - array[i * nbins],  */
/*                 (double) bincount, expect, (double) num * dist_int(array[i * nbins], array[(i+1) * nbins], ave, lambda), temp); */
/*         fflush(NULL); */
    }

    expect = dist_int(array[i * nbins], array[num - 1], ave, lambda);

    if (expect != 0.0)
    {
        expect *= (double) num;
    
        bincount = num - (nbins * (nbins - 1));
        temp = bincount - expect;
        temp = mysquare(temp) / expect;
        chisq += temp;
/*         printf("\n%-7d range [%10.3f to %10.3f] %-8.3f %-8.3f %-8.3f %-14.5f", */
/*                 i+1, array[i * nbins], array[num - 1], */
/*                 (double) bincount, expect, (double) num * dist_int(array[i * nbins], array[num - 1], ave, lambda), temp); */
        fflush(NULL);
    }

    *logL = dist_logL(dist_lnpdf, ave, lambda, data, num);

    free(bins);
    free(array);
/* printf("\n%f", chisq/nbins); */
    return (chisq / (double) nbins);
}


double
chi_sqr_adapt_mix(const double *data, const int num, int nbins, double *logL,
                  const double *p1, const double *p2, const double *mixp, const int nmix,
                  double (*dist_pdf)(double x, double param1, double param2),
                  double (*dist_lnpdf)(double x, double param1, double param2),
                  double (*dist_int)(double x, double y, double param1, double param2))
{
    int             i, j, bincount;
    double          chisq, temp, expect;
    double         *bins, *array;

    temp = expect = chisq = 0;

    if (nbins == 0)
        nbins = floor(sqrt((double) num));

    bins  = calloc(nbins, sizeof(double));
    array = calloc(num,   sizeof(double));

    for (j = 0; j < num; ++j)
        array[j] = data[j];

    qsort(array, num, sizeof(double), dblcmp);

    chisq = 0.0;
    for (i = 0; i < nbins - 1; ++i)
    {
        expect = 0.0;
        for (j = 0; j < nmix; ++j)
            expect += mixp[j] * dist_int(array[i * nbins], array[(i+1) * nbins], p1[j], p2[j]);

        if (expect == 0.0)
            continue;
 
        expect *= (double) num;

        bincount = nbins;
        temp = bincount - expect;
        temp = mysquare(temp) / expect;
        chisq += temp;
/*         printf("\n%-7d range [%10.3f to %10.3f (%10.3f)] %-8.3f %-8.3f %-8.3f %-14.5f", */
/*                 i+1, array[i * nbins], array[(i+1) * nbins], array[(i+1) * nbins] - array[i * nbins],  */
/*                 (double) bincount, expect, (double) num * dist_int(array[i * nbins], array[(i+1) * nbins], ave, lambda), temp); */
/*         fflush(NULL); */
    }

	expect = 0.0;
	for (j = 0; j < nmix; ++j)
		expect += mixp[j] * dist_int(array[i * nbins], array[num - 1], p1[j], p2[j]);

    if (expect != 0.0)
    {
        expect *= (double) num;
    
        bincount = num - (nbins * (nbins - 1));
        temp = bincount - expect;
        temp = mysquare(temp) / expect;
        chisq += temp;
/*         printf("\n%-7d range [%10.3f to %10.3f] %-8.3f %-8.3f %-8.3f %-14.5f", */
/*                 i+1, array[i * nbins], array[num - 1], */
/*                 (double) bincount, expect, (double) num * dist_int(array[i * nbins], array[num - 1], ave, lambda), temp); */
        fflush(NULL);
    }

    /* *logL = dist_logL(dist_lnpdf, ave, lambda, data, num); */

    free(bins);
    free(array);
/* printf("\n%f", chisq/nbins); */
    return (chisq / (double) nbins);
}


/* Calculates the log likelihood for a general distribution given
   (1) the data array, (2) a function that calculates the log of the pdf
   of your distribution, and (3) specified parameters.
   See chi_sqr_adapt() for more detail.
*/
double
dist_logL(double (*dist_lnpdf)(const double x, const double param1, const double param2),
          double param1, double param2, const double *data, const int num)
{
    int             i, outliers;
    double          logL, temp;

    logL = 0.0;
    outliers = 0;
    for (i = 0; i < num; ++i)
    {
        temp = dist_lnpdf(data[i], param1, param2);
/*         printf("\n%3d % e", i, temp); */
/*         fflush(NULL); */

        if (temp == DBL_MAX || temp == -DBL_MAX)
        {
            ++outliers;
            continue;
        }
        else
        {
            logL += temp;
            /* logL2 += temp * temp; */
        }
    }

    /* SCREAMF(sqrt((logL2/(double) num) - mysquare(logL/(double)num))); */

    if (outliers > 0)
    {
/*         for (i = 0; i < num; ++i) */
/*             printf("%3d % e\n", i, data[i]); */

        printf(" --vv logL calculated without %d outliers \n", outliers);
        fflush(NULL);
    }

    return(logL);
}


double
chi_sqr_hist(double *data, double *freq, int num, double *logL, double ave, double lambda,
             double (*dist_pdf)(double x, double param1, double param2),
             double (*dist_lnpdf)(double x, double param1, double param2),
             double (*dist_int)(double x, double y, double param1, double param2))
{
    int             i;
    double          chisq, temp, expect, norm;

    temp = expect = chisq = 0;

    norm = 0.0;
    for (i = 0; i < num; ++i)
        norm += freq[i];

SCREAMF(norm);

    chisq = 0.0;
    for (i = 0; i < num; ++i)
    {
        expect = dist_pdf(data[i], ave, lambda);
        expect = dist_int(data[i]-0.015811/2, data[i]+0.015811/2, ave, lambda);
        expect *= norm;

        if (expect == 0.0)
            continue;

        temp = freq[i] - expect;
        temp = mysquare(temp)/*  / expect */;
        chisq += temp;
        printf("%-7d range [%11.3f] %-8.3f %-8.3f %-14.5f\n",
                i+1, data[i], freq[i], expect, temp);
        fflush(NULL);
    }

    *logL = dist_logL(dist_lnpdf, ave, lambda, data, num);

    return (chisq / (double) num);
}


double
chi_sqr_one(double *data, int num, int nbins, double *prob, double ave, double lambda,
            double (*dist)(double x, double param1, double param2))
{
    int             i, j;
    double          chisq, temp, binrange, expect;
/*     double          x; */
    double         *bins, *array;
/*     FILE           *igfile = NULL; */

    if (nbins == 0)
    {
        nbins = num / 10;

        if (nbins > 100)
            nbins = 100;

        if (nbins < 5)
            nbins = 5;
    }

    bins  = (double *) calloc(nbins, sizeof(double));
    array = (double *) calloc(num,   sizeof(double));

    for (j = 0; j < num; ++j)
        array[j] = data[j];

    qsort(array, num, sizeof(double), dblcmp);

/*     for (i = 0; i < num; ++i) */
/*         printf("\n%-6d %-12.6 ", i, array[i]); */

    binrange = (array[num-1] - array[0]) / (double) nbins;

SCREAMF(binrange);

    j = 0;
    while (array[j] <= binrange + array[0])
        ++j;

SCREAMD(j);

    for (i = 1; i < nbins; ++i)
    {
        if (j == num)
            break;

        bins[i] = 0;
        while (array[j] <= (double)(i+1) * binrange + array[0])
        {
            ++bins[i];
            ++j;

            if (j == num)
                break;
        }
    }

SCREAMD(nbins);
SCREAMD(num);

    chisq = 0.0;
    for (j = 0; j < nbins; ++j)
    {
        expect = integrate_romberg(dist, ave, lambda, (double)j * binrange + array[0], (double)(j+1) * binrange + array[0]);
        /* printf("\n--> %-8d %-14.5 ", j, expect); */

        if (expect == 0.0)
            continue;
 
        expect *= (double) num;
        temp = bins[j] - expect;
        temp = mysquare(temp) / expect;
        chisq += temp;
        printf("%-8d %-14.5f %-14.5f %-14.5f\n", j+1, bins[j], expect, temp);
    }
    fflush(NULL);

    *prob = 0.0;
    for (j = 0; j < nbins; ++j)
        *prob += bins[j];

SCREAMD((int) *prob);
SCREAMD((int) num);

/*     igfile = fopen("invgamma_dist.txt", "w"); */
/*  */
/*     fprintf(igfile, "num = %-4d, binrange = %8.3e, ave = %8.3e, lambda = %8.3e \n", num, binrange, ave, lambda); */
/*  */
/*     for (i = 2; i <= nbins; ++i) */
/*     { */
/*         x = invgauss_pdf(binrange * (double)i, ave, lambda); */
/*         fprintf(igfile, "%-8.3 %-16.5e \n", (double) i * binrange, x); */
/*     } */
/*  */
/*     fclose(igfile); */
    free(bins);
    free(array);

    return (chisq / (double) nbins);
}


int
dblcmp(const void *dbl1, const void *dbl2)
{
    double *first  = (double *) dbl1;
    double *second = (double *) dbl2;

    if (*first < *second)
        return (-1);
    else if (*first == *second)
        return (0);
    else /* if (*first > *second) */
        return (1);
}


int
dblcmp_rev(const void *dbl1, const void *dbl2)
{
    double *first  = (double *) dbl1;
    double *second = (double *) dbl2;

    if (*first > *second)
        return (-1);
    else if (*first == *second)
        return (0);
    else /* if (*first < *second) */
        return (1);
}


double
kstwo(double       *data1,
      int           n1,
      double       *data2,
      int           n2)
{
    int     j1 = 1, j2 = 1;
    double  d, d1, d2, dt, en1, en2, en, fn1 = 0.0, fn2 = 0.0;
    double  prob;

    quicksort (data1, n1);
    quicksort (data2, n2);
    en1 = n1;
    en2 = n2;
    d = 0.0;
/*     printf("\n N1 = %d", n1); */
/*     printf("\n N2 = %d", n2); */

    while (j1 <= n1 && j2 <= n2)
    {
        if ((d1 = data1[j1]) <= (d2 = data2[j2]))
            fn1 = j1++ / en1;
        if (d2 <= d1)
            fn2 = j2++ / en2;
        if ((dt = fabs(fn2 - fn1)) > d)
            d = dt;
    }

/*     printf("\n Dmax = %f", d); */
    en = en1 * en2 / (en1 + en2);
/*     printf("\n Ne = %d", (int) en); */
    prob = 1.0 - KS(d, en);

/*     en = sqrt(en1 * en2 / (en1 + en2)); */
/*     prob = probks((en + 0.12 + 0.11 / en) * d); */

    return (prob);
}


double
ksone(double *data, int n, double (*func)(double))
{
    int             j;
    double          d, dt, ff, fn, fo = 0.0;
    double          prob;
    double          ave, stddev;

    quicksort(data, n);

    ave = 0.0;
    for (j = 0; j < n; ++j)
        ave += data[j];
    ave /= (double)n;

    stddev = 0.0;
    for (j = 0; j < n; ++j)
        stddev += mysquare(data[j] - ave);
    stddev /= (double)(n - 1);
    stddev = sqrt(stddev);

    for (j = 0; j < n; ++j)
        data[j] = (data[j] - ave)/stddev;

    d = 0.0;
    for (j = 0; j < n; ++j)
    {
        fn = j / (double) n;
        ff = (*func)(data[j]);
        /* printf("\n ff = %f", ff); */
        /* printf("\n data[%5d] = %f", j, data[j]); */
        dt = mymaxdbl(fabs(fo - ff), fabs(fn - ff));
        if (dt > d)
            d = dt;
        fo = fn;
    }

    printf(" Dmax = %f\n", d);
/*     printf("\n Ne = %d", (int) en); */
    prob = 1.0 - KS(d, n);

    return (prob);
}


double 
probks(double alam)
{
    int             j;
    double          a2, fac = 2.0, sum = 0.0, term, termbf = 0.0;

    a2 = -2.0 * alam * alam;
    for (j = 1; j <= 100; j++)
    {
        term = fac * exp(a2 * j * j);
        sum += term;
        if (fabs(term) <= EPS1 * termbf || fabs(term) <= EPS2 * sum)
            return (sum);
        fac = -fac;
        termbf = fabs(term);
    }
    return (1.0);
}


#undef EPS1
#undef EPS2

void
mMultiply(double *A, double *B, double *C, int m)
{
    int             i, j, k;
    double          s;

    for (i = 0; i < m; i++)
        for (j = 0; j < m; j++)
        {
            s = 0.;
            for (k = 0; k < m; k++)
                s += A[i * m + k] * B[k * m + j];
            C[i * m + j] = s;
        }
}


/* George Marsaglia, Wai Wan Tsang, and Jingbo Wang 
   "Evaluating Kolmogorov's Distribution"
   Volume 8, 2003, Issue 18
   good to 7 digits *everywhere*, better if you uncomment the line */
void
mPower(double *A, int eA, double *V, int *eV, int m, int n)
{
    double         *B;
    int             eB, i;

    if (n == 1)
    {
        for (i = 0; i < m * m; i++)
            V[i] = A[i];

        *eV = eA;

        return;
    }

    mPower(A, eA, V, eV, m, n / 2);
    B = (double *) malloc((m * m) * sizeof(double));
    mMultiply(V, V, B, m);
    eB = 2 * (*eV);

    if (n % 2 == 0)
    {
        for (i = 0; i < m * m; i++)
            V[i] = B[i];

        *eV = eB;
    }
    else
    {
        mMultiply(A, B, V, m);
        *eV = eA + eB;
    }

    if (V[(m / 2) * m + (m / 2)] > 1e140)
    {
        for (i = 0; i < m * m; i++)
            V[i] = V[i] * 1e-140;

        *eV += 140;
    }

    free(B);
}


double 
KS(double d, int n)
{
    int             k, m, i, j, g, eH, eQ;
    double          h, s, *H, *Q;

    /* OMIT NEXT TWO LINEs IF YOU REQUIRE > 7 DIGIT ACCURACY IN THE RIGHT TAIL */
    s = d * d * n;

    if (s > 7.24 || (s > 3.76 && n > 99))
        return (1 - 2 * exp(-(2.000071 + .331 / sqrt(n) + 1.409 / n) * s));

    k = (int) (n * d) + 1;
    m = 2 * k - 1;
    h = k - n * d;
    H = (double *) malloc((m * m) * sizeof(double));
    Q = (double *) malloc((m * m) * sizeof(double));

    for (i = 0; i < m; i++)
        for (j = 0; j < m; j++)
            if (i - j + 1 < 0)
                H[i * m + j] = 0;
            else
                H[i * m + j] = 1;

    for (i = 0; i < m; i++)
    {
        H[i * m] -= pow(h, i + 1);
        H[(m - 1) * m + i] -= pow(h, (m - i));
    }

    H[(m - 1) * m] += (2 * h - 1 > 0 ? pow(2 * h - 1, m) : 0);

    for (i = 0; i < m; i++)
        for (j = 0; j < m; j++)
            if (i - j + 1 > 0)
                for (g = 1; g <= i - j + 1; g++)
                    H[i * m + j] /= g;

    eH = 0;
    mPower(H, eH, Q, &eQ, m, n);
    s = Q[(k - 1) * m + k - 1];

    for (i = 1; i <= n; i++)
    {
        s = s * i / n;

        if (s < 1e-140)
        {
            s *= 1e140;
            eQ -= 140;
        }
    }

    s *= pow(10., eQ);

    free(H);
    free(Q);

    return s;
}


double
nlogn(double n)
{
    return(n * log(n));
}


double
Factorial(long unsigned int N)
{
    int             i;
    double          F;

    if (N == 1 || N == 0)
        return(1.0);

    F = 1.0;
    for (i = 1; i <= N; ++i)
        F *= (double) i;

    return (F);
}


double
Gosper(long unsigned int N)
{
    double          F;
    double          dblN = (double)N;

    if (N == 1 || N == 0)
        return(1.0);

    F = sqrt((2.0*dblN + 1.0/3.0) * MY_PI);
    F *= pow((dblN / MY_E), dblN);

    return(F);
}


double
Stirling(long unsigned int N)
{
    double          F;
    double          dblN = (double)N;
    double          series;
    double          sqrN, cubeN, pow4N, pow5N;

    if (N == 1 || N == 0)
        return(1.0);

    sqrN = mysquare(dblN);
    cubeN = sqrN*dblN;
    pow4N = mysquare(sqrN);
    pow5N = pow4N*dblN;

    F = sqrt(2.0 * MY_PI * dblN);
    F *= pow((dblN / MY_E), dblN);
    series = 1.0 + 1.0/(12.0 * dblN);
    series += 1.0/(288.0 * sqrN);
    series -= 139.0/(51840.0 * cubeN);
    series -= 571.0/(2488320.0 * pow4N);
    series += 163879.0/(209018880.0 * pow5N);
    F *= series;

    return(F);
}


/* http://mathworld.wolfram.com/StirlingsSeries.html */
double
LnFactorial(long unsigned int N)
{
    double          S, cubeN, pow5N;

    if (N == 1 || N == 0)
        return(0.0);

    cubeN = mycube(N);
    pow5N = mycube(N) * mysquare(N);
    /*pow7N = pow5N * mysquare(N);*/

    S = log(2.0 * MY_PI)/2.0;
    S += log(N)/2.0;
    S += (nlogn(N) - N);
    S += (1.0 / (12.0 * N));
    S += (1.0 / (360.0 * cubeN));
    S += (1.0 / (1260.0 * pow5N));
    /*S -= (1.0 / (1680.0 * pow7N)); this term makes no difference */

    return (S);
}


double
LnFactorialPlus(long unsigned int N)
{
    double          S, cubeN, pow5N, pow7N;
    double          dblN = (double)N;

    if (N == 1 || N == 0 || N == 1)
        return(0.0);

    cubeN = mycube(dblN);
    pow5N = mycube(dblN) * mysquare(dblN);
    pow7N = pow5N * mysquare(dblN);

    S = log(2.0 * MY_PI)/2.0;
    S += log(dblN)/2.0;
    S += (nlogn(dblN) - dblN);
    S += (1.0 / (12.0 * dblN));
    S += (1.0 / (360.0 * cubeN));
    S += (1.0 / (1260.0 * pow5N));
    S -= (1.0 / (1680.0 * pow7N));

    return (S);
}


double
DblFactorialNaive(long unsigned int N)
{
    int             i;
    double          DF;

    if (N == -1 || N == 0 || N == 1)
        return(1.0);

    DF = 1.0;
    for (i = N; i > 0; i=i - 2)
        DF *= (double) i;

    return (DF);
}


double
DblFactorial(long unsigned int N)
{
    double          DF, dblN = (double) N;
    double          twoPI = 2.0 / MY_PI;

    if (N == -1 || N == 0 || N == 1)
        return(1.0);

    DF = pow(twoPI, (1.0 - cos(MY_PI * dblN))/4.0);
    DF *= sqrt(MY_PI);
    DF *= pow(dblN, (double)(N+1)/2.0);
    DF *= exp(-dblN/2.0);
    DF *= (1.0 + 1.0/(6.0 * dblN) + 1.0/(72.0 * mysquare(dblN)) - 139.0/(6480.0 * mycube(dblN)));

    return(DF);
}


/* http://functions.wolfram.com/GammaBetaErf/Factorial2/06/02/0002/ */
double
LnDblFactorial(long unsigned int N)
{
    double          DF, dblN = (double) N;
    double          twoPI = 2.0 / MY_PI;
    double          sqrN = mysquare(dblN);
    double          cubeN = sqrN*dblN;
    double          quadN = cubeN * dblN;
    /* double          pentN = quadN * dblN; */
    double          series;

    if (N == -1 || N == 0 || N == 1)
        return(0.0);

    DF = log(twoPI) * ((1.0 - cos(MY_PI*N))/4.0);
    DF += log(MY_PI)/2.0;
    DF += log(dblN) * ((double)(N+1)/2.0);
    DF -= N/2.0;
    series = 1.0 + 1.0/(6.0 * dblN);
    series += 1.0 / (72.0 * sqrN);
    series -= 139.0 / (6480.0 * cubeN);
    series -= 571.0 / (155520.0 * quadN);
    /*series -= 163879.0 / (6531840.0 * pentN);*/
    DF += log(series);
    return(DF);
}


/************************************************************************
* Combination                                                           *
*                                                                       *
* This defines an iterative combination without needing three           *
* factorials.                                                           *
*                                                                       *
* The number of combinations is returned for N objects taken M at a     *
* time, without regard for order.                                       *
************************************************************************/
double
Combination(double N, double M)
{
    int             i;
    double          C;
    double          top_term = 1.0;

    if (N == M)
        return (1.0);
    else if ((N/M) < 2.0)
    {
        for (i = (M + 1.0); i <= N; ++i)
            top_term *= (double) i;
        C = top_term / Factorial(N - M);
    }
    else
    {
        for (i = (N - M + 1.0); i <= N; ++i)
            top_term *= (double) i;
        C = top_term / Factorial(M);
    }
    return (C);
}


double
MultinomCoeff(long unsigned int tries, long unsigned int *wins, int nclasses)
{
    int             i;
    double          denom, mc;

    denom = 0.0;
    for (i = 0; i < nclasses; ++i)
    {
        if (wins[i] < 170)
            denom += log(Factorial(wins[i]));
        else
            denom += LnFactorialPlus(wins[i]);
    }

    if (tries < 170)
        mc = log(Factorial(tries));
    else
        mc = LnFactorialPlus(tries);

    mc = exp(mc-denom);

    return (mc);
}


double
LnMultinomCoeff(long unsigned int tries, long unsigned int *wins, int nclasses)
{
    int             i, tmp;
    double          denom, mc;

    tmp = 0;
    for (i = 0; i < nclasses; ++i)
        tmp += wins[i];

    if (tmp != tries)
    {
        printf("\n\n %3d %3d \n\n", tmp, (int) tries);
        exit(EXIT_FAILURE);
    }

    denom = 0.0;
    for (i = 0; i < nclasses; ++i)
    {
        if (wins[i] < 170)
            denom += log(Factorial(wins[i]));
        else
            denom += LnFactorialPlus(wins[i]);
    }

    if (tries < 170)
        mc = log(Factorial(tries));
    else
        mc = LnFactorialPlus(tries);

    mc -= denom;

    return (mc);
}


double
LnCombinationStirling(double N, double M)
{
    double          C;
    double          lnfactN, lnfactM, lnfactNM;
    
    if (N < 170)
        lnfactN = log(Factorial(N));
    else
        lnfactN = LnFactorialPlus(N);

    if (M < 170)
        lnfactM = log(Factorial(M));
    else
        lnfactM = LnFactorialPlus(M);

    if (N-M < 170)
        lnfactNM = log(Factorial(N-M));
    else
        lnfactNM = LnFactorialPlus(N-M);

    C = lnfactN - lnfactM - lnfactNM;

    return (C);
}


/************************************************************************
* Binomial_P                                                            *
*                                                                       *
* Calculates the specific binomial probability.                         *
*                                                                       *
* Takes N number of tries, M number of wins, for an individual          *
* probability p of success for each win.                                *
*                                                                       *
************************************************************************/
double
Binomial_P(long unsigned int n, long unsigned int m, double p)
{
    double          binom_p;
    double          t1, t2;

    t1 = Combination(n, m);
    t2 = (double) pow((double) p, (double) m) * (double) pow((double) (1 - p), (double) (n - m));
    binom_p = t1 * t2;

    return (binom_p);
}


/* This works */
double
Multinomial_P(long unsigned int tries, long unsigned int *wins, double *p, int nclasses)
{
    int             i;
    double          multinom_p;
    double          t1, t2;

    int tmp = 0;
    for (i = 0; i < nclasses; ++i)
        tmp += wins[i];

    if (tmp != tries)
        printf("\n  Multinomial_P BONK! %3d %3d \n", tmp, (int)tries);

    t1 = LnMultinomCoeff(tries, wins, nclasses);

    t2 = 0.0;
    for (i = 0; i < nclasses; ++i)
    {
        if (p[i] == 0.0 || wins[i] == 0.0)
            continue;
        else
            t2 += ((double) wins[i] * log(p[i]));
    }

    multinom_p = t1 + t2;

    return (exp(multinom_p));
}


/************************************************************************
* Binomial_sum                                                          *
*                                                                       *
* Sums the individual binomial probabilities for m wins **or more**     *
*                                                                       *
************************************************************************/
double
Binomial_sum(long unsigned int n, long unsigned int m, double p)
{
    double          bsum;
    int             i;

    bsum = 0.0;
    for (i = m; i <= n; ++i)
        bsum += Binomial_P(n, i, p);

    return (bsum);
}


double
student_t(double t, double df)
{
    return(InBeta((0.5 * df), 0.5, (df / (df + mysquare(t)))));
}


/* returns a random variate from the standard normal */
/* distribution. Taken from Numerical Recipes in C. */ 
double
stdnormal(void)
{
    double          fac, r, v1, v2;
 
    do
    {
            v1 = 2.0 * genrand_real2() - 1.0;
            v2 = 2.0 * genrand_real2() - 1.0;
            r = mysquare(v1) + mysquare(v2);
    }
    while (r >= 1.0);

    fac = sqrt(-2.0 * log(r) / r);

    return (v2 * fac);
}
