
/*********************************************************************
 *                
 * Filename:      smapi_core.c
 * Description:   smapi is a kernel module that serves to interface
 *                with the SMAPI BIOS in IBM ThinkPads
 * Author:        Thomas Hood
 * Created:       19 July 1999 
 *
 * Please report bugs to the author ASAP.
 * 
 *     Copyright (c) 1999 J.D. Thomas Hood, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *     
 ********************************************************************/

#include "thinkpad_driver.h"

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/version.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include "thinkpad_common.h"
#include "smapi.h"
#include "smapi_call.h"


/****** definitions ******/


/****** declarations ******/

extern struct proc_dir_entry *thinkpad_ppde;


/****** variables ******/

static const char _szMyName[] = "smapi";
static const char _szImName[] = "smapi_do";
static const char _szMyVersion[] = "5.0";
static const char _szProcfile[] = "driver/thinkpad/smapi";

#ifdef MODULE
MODULE_AUTHOR( "Thomas Hood" );
MODULE_DESCRIPTION( "Driver for the SMAPI BIOS on IBM ThinkPads" );
MODULE_LICENSE( "GPL" );
#endif


/****** functions *******/


static flag_t smapi_req_need_w( byte bFunc, byte bSubFunc )
{

	switch ( bFunc ) {
		case 0x00: return 0;
		case 0x10: if ( bSubFunc == 0x00 ) return 0; break;
		case 0x11: if ( bSubFunc == 0x02 ) return 0; break;
		case 0x13: if ( bSubFunc == 0x02 ) return 0; break;
		case 0x22:
			if ( bSubFunc == 0x00 ) return 0;
			if ( bSubFunc == 0x02 ) return 0;
			break;
		case 0x30: if ( bSubFunc == 0x00 ) return 0; break;
		case 0x31: if ( bSubFunc == 0x00 ) return 0; break;
		case 0x32:
			if ( bSubFunc == 0x00 ) return 0;
			if ( bSubFunc == 0x02 ) return 0;
			if ( bSubFunc == 0x06 ) return 0;
			break;
		case 0x33: if ( bSubFunc == 0x00 ) return 0; break;
		case 0x34:
			if ( bSubFunc == 0x00 ) return 0;
			if ( bSubFunc == 0x02 ) return 0;
			break;
		default:
			return 1;
	}

	return 1;
}


static byte byte_of_bcd8( bcd8_t bcd8The )
{
	byte bTens, bUnits;

	bUnits = (byte)bcd8The & 0xF;
	bTens = (byte)(bcd8The & 0xF0) >> 4;

	if ( bUnits > 9 || bTens > 9 ) {
		return 0xFF;
	}

	return bUnits + (bTens * 10);
}


static int smapi_read_proc(
	char *pchBuf, 
	char **ppchStart, 
	off_t off,
	int intCount, 
	int *pintEof, 
	void *data
) {
	int intRtn;
	bcd8_t bcd8SysMgmtHigh, bcd8SysMgmtLow;
	bcd8_t bcd8SMAPIIfaceHigh, bcd8SMAPIIfaceLow;
	word wSysMgmtBiosRevMajor, wSysMgmtBiosRevMinor;
	word wSmapiBiosIfaceRevMajor, wSmapiBiosIfaceRevMinor;
	smb_inparm_t inparmMy;
	smb_outparm_t outparmMy;

	inparmMy.bFunc = (byte) 0;
	inparmMy.bSubFunc = (byte) 0;

	intRtn = SMAPI_BIOS_do( &inparmMy, &outparmMy );
	if ( intRtn ) {
		return snprintf(
			pchBuf, intCount,
			"%s module ver. %s interfacing to 32-bit protected mode system management BIOS:\n"
			"   error getting BIOS information\n",
			_szMyName, _szMyVersion
		);
	}

	bcd8SysMgmtHigh = (bcd8_t)( (outparmMy.dwParm4 >> 8) & 0xFF );
	bcd8SysMgmtLow =  (bcd8_t)(  outparmMy.dwParm4       & 0xFF );
	wSysMgmtBiosRevMajor = (word) byte_of_bcd8( bcd8SysMgmtHigh );
	wSysMgmtBiosRevMinor = (word) byte_of_bcd8( bcd8SysMgmtLow );

	bcd8SMAPIIfaceHigh =   (bcd8_t)( (outparmMy.dwParm5 >> 8) & 0xFF );
	bcd8SMAPIIfaceLow =    (bcd8_t)(  outparmMy.dwParm5       & 0xFF );
	wSmapiBiosIfaceRevMajor = (word) byte_of_bcd8( bcd8SMAPIIfaceHigh );
	wSmapiBiosIfaceRevMinor = (word) byte_of_bcd8( bcd8SMAPIIfaceLow );

	return snprintf(
		pchBuf, intCount,
		"%s module ver. %s interfacing to 32-bit protected mode System-Management-API BIOS ver. %d.%d at address 0x%lx through interface version %d.%d\n",
		_szMyName, _szMyVersion,
		wSysMgmtBiosRevMajor, wSysMgmtBiosRevMinor,
		(unsigned long) _dwAddressSmapiBios,
		wSmapiBiosIfaceRevMajor, wSmapiBiosIfaceRevMinor
	);
}

int smapi_do( 
	unsigned long ulongIoctlArg,
	flag_t fCallerHasWritePerm
) {
	smb_inparm_t inparmMy;
	smb_outparm_t outparmMy;
	unsigned long ulRtnCopy;
	int intRtnDo;

	ulRtnCopy = copy_from_user(
		&inparmMy,
		(byte *)ulongIoctlArg,
		sizeof( inparmMy )
	);
	if ( ulRtnCopy ) return -EFAULT;

	if ( smapi_req_need_w( inparmMy.bFunc, inparmMy.bSubFunc ) && !fCallerHasWritePerm ) {
		printk(KERN_ERR
			"%s: Caller does not have permission to make this request of SMAPI\n",
			_szMyName
		);
		return -EACCES;
	}

	intRtnDo = SMAPI_BIOS_do( &inparmMy , &outparmMy );

	if ( intRtnDo != outparmMy.bRc ) {
		static unsigned char count; /* = 0 */
		if ( !count++ )
			printk(KERN_INFO "%s: SMAPI BIOS return codes differ!\n", _szMyName );
	}

	ulRtnCopy = copy_to_user(
		(byte *)ulongIoctlArg,
		&outparmMy,
		sizeof( inparmMy )
	);
	if ( ulRtnCopy ) return -EFAULT;

	return outparmMy.bRc ? -ETHINKPAD_SUBDRIVER : 0;
}


/*
 * Returns a pointer to the first SMAPI BIOS header found
 * with correct checksum
 * Returns NULL if no header found with correct checksum
 */
static smb_header_t * locate_smb_header( void )
{
	smb_header_t *pheaderSearch, *pheaderLim;

	/*** Search for the sig  ***/
	pheaderSearch = (smb_header_t *) phys_to_virt(0xF0000);
	pheaderLim = (smb_header_t *) phys_to_virt(0xFFFF0);

	while ( 1 ) {

		if (
			( pheaderSearch->bSig[0] == '$') &&
			( pheaderSearch->bSig[1] == 'S') &&
			( pheaderSearch->bSig[2] == 'M') &&
			( pheaderSearch->bSig[3] == 'B')
		) {
			/* We found a signature */
			/*** check the header checksum ***/
			byte chksum;
			int cnt;
	
			chksum = 0;
			for ( cnt=0; cnt < pheaderSearch->bLen; cnt++ ) {
				chksum += (byte)(pheaderSearch->bSig[cnt]);
			}
			if ( chksum == 0 ) /* good checksum */
				return pheaderSearch;
	
			/* bad checksum */
		}

		/*** Keep searching ***/
		pheaderSearch++ ;
		if ( pheaderSearch > pheaderLim ) {
			/* search space exhausted */
			return (smb_header_t *) NULL;
		}

	} /* while */
	
}


static int __init smapi_init( void )
{
	smb_header_t * pheaderFound;

	/*** locate SMAPI BIOS header ***/
	pheaderFound = locate_smb_header();
	if ( pheaderFound == NULL ) {
		printk(KERN_ERR
			"%s: No SMAPI BIOS header was found. :-(\n",
			_szMyName
		);
		return -ETHINKPAD_HW_NOT_FOUND;
	}
	/* a header was located */

	if ( ! (pheaderFound->wInfo & 4) ) {
		printk(KERN_ERR
			"%s: The SMAPI BIOS does not support 32-bit protected mode. :-(\n",
			_szMyName
		);
		return -ETHINKPAD_HW_NOT_FOUND;
	}
	/* a header was located and it supports 32-bit protected mode */

	_dwAddressSmapiBios = (dword) phys_to_virt(
		pheaderFound->dwP32_base +
		pheaderFound->dwP32_offset
	);

	printk(KERN_INFO
		"%s: 32-bit protected mode SMAPI BIOS found. :-)\n",
		_szMyName
	);

	if ( !thinkpad_ppde || !create_proc_read_entry(
		_szProcfile,
		S_IFREG | S_IRUGO,
		NULL,
		smapi_read_proc,
		NULL
	) ) {
		printk(KERN_ERR "%s: Could not create_proc_read_entry() /proc/%s\n", _szMyName, _szProcfile );
	}
	/* proc entry created */

	inter_module_register( _szImName, THIS_MODULE, &smapi_do );

	return 0;
}


static void __exit smapi_exit( void )
{

	inter_module_unregister( _szImName );

	remove_proc_entry( _szProcfile, NULL );

	return;
}

module_init(smapi_init);
module_exit(smapi_exit);
