/***************************************************************************
                         th-job-cropping-dialog.c
                         ------------------------
    begin                : Tue Nov 23 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-utils.h"
#include "th-job-cropping-dialog.h"

#include <gtk/gtk.h>
#include <gst/gst.h>
#include <glib/gi18n.h>
#include <string.h>

/* factor by which to scale down snapshot images */
#define CROP_SNAPSHOT_SCALE   (2) 

enum
{
	TH_RESPONSE_RETRIEVE_ERROR = 9312
};

struct _ThJobCroppingDialogPrivate
{
	ThJob            *job;
	guint             job_num_chapters;
	guint             job_len_secs;

	gulong            response_id;
	
	GtkWidget        *image;
	GtkWidget        *spin_top;
	GtkWidget        *spin_left;
	GtkWidget        *spin_right;
	GtkWidget        *spin_bottom;
	GtkWidget        *frame_scale;
	
	GtkWidget        *progressbar;
	GtkWidget        *progress_box;
	GtkWidget        *cropping_box;

	GtkWidget        *cancel_button;
	GtkWidget        *ok_button;

	const GList      *frames;        /* list of GstBuffers, list and buffers are owned by the job   */
	guint             num_frames;    /* length of frames list                                       */
	guint             cur_frame_num; /* which n-th frame of the list we are currently showing       */
	GdkPixbuf        *cur_pixbuf;    /* pixbuf version of current frame (uncropped, we hold no ref) */

	guint             in_width, in_height;
	guint             out_width, out_height;

	/* for snapshot retrieval */
	GstElement       *snapshot_pipeline;
	gulong            snapshot_iterate_id;
	GList            *snapshot_buffers;    /* retrieved frames as GstBuffers */
	gchar            *snapshot_err_msg;
};

static void             job_cropping_dialog_class_init    (ThJobCroppingDialogClass *klass);

static void             job_cropping_dialog_instance_init (ThJobCroppingDialog *cp);

static void             job_cropping_dialog_finalize      (GObject *object);

static gboolean         jcd_snapshot_pipeline_iterate_cb  (ThJobCroppingDialog *jcd);

static gboolean         jcd_setup_snapshot_pipeline       (ThJobCroppingDialog *jcd, GError **err);

static void             jcd_free_snapshot_buffers         (ThJobCroppingDialog *jcd);

static void             jcd_set_current_frame             (ThJobCroppingDialog *jcd, guint frame_number);

static void             jcd_update_frame_cropping         (ThJobCroppingDialog *jcd);

static gchar           *jcd_format_frame_scale_value      (ThJobCroppingDialog *jcd, gdouble val);

static void             jcd_frame_scale_value_changed     (ThJobCroppingDialog *jcd, GtkRange *frame_scale);

static void             jcd_snapshot_set_dimensions       (ThJobCroppingDialog *jcd, guint snapshot_width, guint snapshot_height);

/* variables */

GObjectClass           *jcd_parent_class; /* NULL */

/***************************************************************************
 *
 *   job_cropping_dialog_class_init
 *
 ***************************************************************************/

static void
job_cropping_dialog_class_init (ThJobCroppingDialogClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	jcd_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = job_cropping_dialog_finalize;
}

/***************************************************************************
 *
 *   job_cropping_dialog_instance_init
 *
 ***************************************************************************/

static void
job_cropping_dialog_instance_init (ThJobCroppingDialog *jcd)
{
	jcd->priv = g_new0 (ThJobCroppingDialogPrivate, 1);

	g_signal_connect (jcd, "delete-event",
	                  G_CALLBACK (gtk_widget_hide_on_delete),
	                  NULL);
	
	gtk_window_set_icon_from_file (GTK_WINDOW (jcd),
	                               DATADIR "/ui/icons/thoggen.png",
	                               NULL);

	jcd->priv->in_width = 0;
	jcd->priv->in_height = 0;
	jcd->priv->out_width = 0;
	jcd->priv->out_height = 0;
}

/***************************************************************************
 *
 *   job_cropping_dialog_finalize
 *
 ***************************************************************************/

static void
job_cropping_dialog_finalize (GObject *object)
{
	ThJobCroppingDialog *jcd;

	jcd = (ThJobCroppingDialog*) object;

	g_print("ThJobCroppingDialog: finalize\n");

	jcd_free_snapshot_buffers (jcd);
	g_free (jcd->priv->snapshot_err_msg);

	memset (jcd->priv, 0xab, sizeof (ThJobCroppingDialogPrivate));
	g_free (jcd->priv);
	jcd->priv = NULL;

	/* chain up */
	jcd_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_job_cropping_dialog_get_type
 *
 ***************************************************************************/

GType
th_job_cropping_dialog_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThJobCroppingDialogClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) job_cropping_dialog_class_init,
			NULL, NULL,
			sizeof (ThJobCroppingDialog),
			0,
			(GInstanceInitFunc) job_cropping_dialog_instance_init
		};

		type = g_type_register_static (GTK_TYPE_DIALOG, "ThJobCroppingDialog", &info, 0);
	}

	return type;
}


/***************************************************************************
 *
 *   th_job_cropping_dialog_new
 *
 ***************************************************************************/

GtkWidget *
th_job_cropping_dialog_new (void)
{
	ThJobCroppingDialog *jcd;
	GtkWidget           *glade_window = NULL;
	GtkWidget           *toplevel_vbox;

	jcd = (ThJobCroppingDialog *) g_object_new (TH_TYPE_JOB_CROPPING_DIALOG, NULL);
	
	if (!th_utils_ui_load_interface ("th-job-cropping-dialog.glade", 
	                                 FALSE,
	                                 "th-job-cropping-dialog", &glade_window,
	                                 "th-toplevel-vbox",       &toplevel_vbox,
	                                 "th-crop-image",          &jcd->priv->image,
	                                 "th-top-spin",            &jcd->priv->spin_top,
	                                 "th-left-spin",           &jcd->priv->spin_left,
	                                 "th-right-spin",          &jcd->priv->spin_right,
	                                 "th-bottom-spin",         &jcd->priv->spin_bottom,
	                                 "th-progressbar",         &jcd->priv->progressbar,
	                                 "th-progress-box",        &jcd->priv->progress_box,
	                                 "th-cropping-box",        &jcd->priv->cropping_box,
	                                 "th-frame-scale",         &jcd->priv->frame_scale,
                                   NULL))
	{
		g_printerr ("th_utils_ui_load_interface (\"th-ui-job-cropping-dialog.glade\") failed.\n");
		if (glade_window)
			gtk_widget_destroy (glade_window);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		return NULL;
	}
	
	g_object_ref (toplevel_vbox);
	gtk_container_remove (GTK_CONTAINER (glade_window), toplevel_vbox);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (jcd)->vbox), toplevel_vbox);
	g_object_unref (toplevel_vbox);

	jcd->priv->cancel_button = gtk_dialog_add_button (GTK_DIALOG (jcd),
	                                                  GTK_STOCK_CANCEL,
	                                                  GTK_RESPONSE_CANCEL);

	/* makes cancel button stick to the left; let's see
	 *  how long it takes until some HIGian complains ;) */
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (GTK_DIALOG(jcd)->action_area),
	                                    jcd->priv->cancel_button,
	                                    TRUE);

	jcd->priv->ok_button = gtk_dialog_add_button (GTK_DIALOG (jcd),
	                                              GTK_STOCK_OK,
	                                              GTK_RESPONSE_ACCEPT);
	

	g_signal_connect_swapped (jcd->priv->spin_top, "value-changed",
	                          G_CALLBACK (jcd_update_frame_cropping), jcd);

	g_signal_connect_swapped (jcd->priv->spin_left, "value-changed",
	                          G_CALLBACK (jcd_update_frame_cropping), jcd);

	g_signal_connect_swapped (jcd->priv->spin_right, "value-changed",
	                          G_CALLBACK (jcd_update_frame_cropping), jcd);

	g_signal_connect_swapped (jcd->priv->spin_bottom, "value-changed",
	                          G_CALLBACK (jcd_update_frame_cropping), jcd);

	g_signal_connect_swapped (jcd->priv->frame_scale, "format-value",
	                          G_CALLBACK (jcd_format_frame_scale_value), jcd);

	g_signal_connect_swapped (jcd->priv->frame_scale, "value-changed",
	                          G_CALLBACK (jcd_frame_scale_value_changed), jcd);

	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_top), 
	                                CROP_SNAPSHOT_SCALE, CROP_SNAPSHOT_SCALE * 10.0);
	
	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_left), 
	                                CROP_SNAPSHOT_SCALE, CROP_SNAPSHOT_SCALE * 10.0);
	
	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_right), 
	                                CROP_SNAPSHOT_SCALE, CROP_SNAPSHOT_SCALE * 10.0);
	
	gtk_spin_button_set_increments (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), 
	                                CROP_SNAPSHOT_SCALE, CROP_SNAPSHOT_SCALE * 10.0);

	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_top), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_left), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_right), TRUE);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), TRUE);
	
	/* spin button ranges are set in jcd_snapshot_set_dimensions()
	 *  once we know the size of the input and of the snapshots */
	
	return GTK_WIDGET (jcd);
}

/***************************************************************************
 *
 *   jcd_snapshot_pipeline_set_progress
 *
 ***************************************************************************/

static void
jcd_snapshot_pipeline_set_progress (ThJobCroppingDialog *jcd, gdouble fraction)
{
	gchar *s = g_strdup_printf ("%.0f%%", fraction*100.0);
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (jcd->priv->progressbar), fraction);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (jcd->priv->progressbar), s);
	g_free (s);
}

/***************************************************************************
 *
 *   jcd_format_frame_scale_value
 *
 ***************************************************************************/

static gchar *
jcd_format_frame_scale_value (ThJobCroppingDialog *jcd, gdouble val)
{
	return g_strdup_printf (_("Frame %.0f of %u"), val + 1.0, jcd->priv->num_frames);
}

/***************************************************************************
 *
 *   jcd_frame_scale_value_changed
 *
 ***************************************************************************/

static void
jcd_frame_scale_value_changed (ThJobCroppingDialog *jcd, GtkRange *frame_scale)
{
	guint v;
	
	v = (guint) gtk_range_get_value (GTK_RANGE (frame_scale));
	
	g_return_if_fail (v < jcd->priv->num_frames);
	
	jcd_set_current_frame (jcd, v);
}

/***************************************************************************
 *
 *   jcd_get_spin_value_as_int
 *
 *   Gets spin button value and makes sure it's a multiple of 
 *    CROP_SNAPSHOT_SCALE and sets a new value if required. 
 *    Returns FALSE if the value has been adjusted (in which
 *    case another value-changed signal will have been fired).
 *   
 ***************************************************************************/

static gboolean
jcd_get_spin_value_as_int (GtkWidget *spinbutton, gint *p_val)
{
	gint v = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinbutton));
	
	if (v % CROP_SNAPSHOT_SCALE != 0)
	{
		v = (v / CROP_SNAPSHOT_SCALE) * CROP_SNAPSHOT_SCALE;
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (spinbutton), (gdouble) v);
		*p_val = v;
		return FALSE;
	}
	
	*p_val = v;
	return TRUE;
}

/***************************************************************************
 *
 *   jcd_update_frame_cropping
 *
 ***************************************************************************/

static void
jcd_update_frame_cropping (ThJobCroppingDialog *jcd)
{
	GdkPixbuf  *subpixbuf;
	gint        left, top, right, bottom, w, h;

	g_return_if_fail (jcd->priv->cur_pixbuf != NULL);

	if (!jcd_get_spin_value_as_int (jcd->priv->spin_top, &top)
	 || !jcd_get_spin_value_as_int (jcd->priv->spin_left, &left)
	 || !jcd_get_spin_value_as_int (jcd->priv->spin_right, &right)
	 || !jcd_get_spin_value_as_int (jcd->priv->spin_bottom, &bottom))
	{
		return;
	}
	
	top /= CROP_SNAPSHOT_SCALE;
	left /= CROP_SNAPSHOT_SCALE;
	right /= CROP_SNAPSHOT_SCALE;
	bottom /= CROP_SNAPSHOT_SCALE;
	
	w =  gdk_pixbuf_get_width (jcd->priv->cur_pixbuf) - left - right;
	h =  gdk_pixbuf_get_height (jcd->priv->cur_pixbuf) - top - bottom;

	/* note: subpixbufs hold a reference to their 'parent' pixbuf */
	subpixbuf = gdk_pixbuf_new_subpixbuf (jcd->priv->cur_pixbuf, left, top, w, h);
	g_return_if_fail (subpixbuf != NULL);

	gtk_image_set_from_pixbuf (GTK_IMAGE (jcd->priv->image), subpixbuf);
	g_object_unref (subpixbuf); /* image holds reference */
}

/***************************************************************************
 *
 *   jcd_set_current_frame
 *
 ***************************************************************************/

static void
jcd_set_current_frame (ThJobCroppingDialog *jcd, guint frame_number)
{
	GstBuffer *buf;
	GdkPixbuf *pixbuf;

	frame_number = frame_number % jcd->priv->num_frames;
	
	if (frame_number == jcd->priv->cur_frame_num)
		return;

	buf = GST_BUFFER (g_list_nth_data ((GList*)jcd->priv->frames, frame_number));
	g_return_if_fail (buf != NULL);

	pixbuf = th_pixbuf_from_yuv_i420_buffer (buf, jcd->priv->out_width, jcd->priv->out_height);
	
	g_return_if_fail (pixbuf != NULL);

	g_object_add_weak_pointer (G_OBJECT (pixbuf), (gpointer*) &jcd->priv->cur_pixbuf);

	gtk_image_set_from_pixbuf (GTK_IMAGE (jcd->priv->image), pixbuf);
	g_object_unref (pixbuf); /* image holds reference */

	/* previous pixbuf should have been finalized now */
	g_assert (jcd->priv->cur_pixbuf == NULL);

	jcd->priv->cur_pixbuf = pixbuf;
	jcd->priv->cur_frame_num = frame_number;
	
	jcd_update_frame_cropping (jcd);

	gtk_range_set_value (GTK_RANGE (jcd->priv->frame_scale), (gdouble) frame_number);
}

/***************************************************************************
 *
 *   job_cropping_dialog_run_with_frames
 *
 ***************************************************************************/

static gboolean
job_cropping_dialog_run_with_frames (ThJobCroppingDialog *jcd)
{
	gint ret, t, l, r, b;
	
	gtk_widget_hide (jcd->priv->progress_box);
	gtk_widget_show (jcd->priv->cropping_box);

	gtk_widget_show (jcd->priv->ok_button);
	gtk_window_present (GTK_WINDOW (jcd));
	
	jcd->priv->frames = th_job_get_snapshot_frames (jcd->priv->job);
	jcd->priv->num_frames = g_list_length ((GList*) jcd->priv->frames);
	jcd->priv->cur_frame_num = G_MAXUINT;

	gtk_range_set_range (GTK_RANGE (jcd->priv->frame_scale), 0.0, jcd->priv->num_frames);
	jcd_set_current_frame (jcd, 0);

	g_print ("job_cropping_dialog_run_with_frames(): got %u frames\n", jcd->priv->num_frames);

	g_object_get (jcd->priv->job, 
	              "crop-top", &t, "crop-left", &l, 
	              "crop-right", &r, "crop-bottom", &b, 
	              NULL);
	
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_top), t);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_left), l);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_right), r);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), b);
	
	ret = gtk_dialog_run (GTK_DIALOG (jcd));

	jcd->priv->frames = NULL;
	jcd->priv->num_frames = 0;
	jcd->priv->cur_frame_num = 0;

	if (ret == GTK_RESPONSE_ACCEPT)
	{
		gint t = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_top));
		gint l = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_left));
		gint r = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_right));
		gint b = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (jcd->priv->spin_bottom));

		g_object_set (jcd->priv->job, 
		              "crop-top", t, 
		              "crop-left", l, 
		              "crop-right", r, 
		              "crop-bottom", b, 
		              NULL);
	}
	
	gtk_widget_destroy (GTK_WIDGET (jcd));
	
	return (ret == GTK_RESPONSE_ACCEPT);
}

/***************************************************************************
 *
 *   th_job_cropping_dialog_run_for_job
 *
 *   Returns TRUE if 'OK' was pressed, FALSE otherwise.
 *
 ***************************************************************************/

gboolean
th_job_cropping_dialog_run_for_job (ThJob *job, GtkWindow *parent_window)
{
	ThJobCroppingDialog *jcd;
	GError              *err = NULL;
	gint                 ret;

	g_return_val_if_fail (TH_IS_JOB (job), FALSE);

	jcd = (ThJobCroppingDialog*) th_job_cropping_dialog_new ();

	if (parent_window)
	{
		gtk_window_set_transient_for (GTK_WINDOW (jcd), GTK_WINDOW (parent_window));
	}

	jcd->priv->job = job;

	gtk_window_set_type_hint (GTK_WINDOW (jcd), GDK_WINDOW_TYPE_HINT_NORMAL);
	gtk_window_set_position (GTK_WINDOW (jcd), GTK_WIN_POS_CENTER);
	
	gtk_widget_hide (jcd->priv->ok_button);
	
	/* Already got frames? do the cropping dialog then */
	if (th_job_get_snapshot_frames (job))
	{
		/* FIXME: this needs to be solved differently once we have
		 *        other inputs than DVD video with known size */
		jcd->priv->in_width = 720;
		jcd->priv->in_height = 576;

		jcd_snapshot_set_dimensions (jcd, jcd->priv->in_width / CROP_SNAPSHOT_SCALE, 
		                             jcd->priv->in_height / CROP_SNAPSHOT_SCALE);

		return job_cropping_dialog_run_with_frames (jcd);
	}

	/* Retrieve a couple of frames first ... */
	gtk_widget_show (jcd->priv->progress_box);
	gtk_widget_hide (jcd->priv->cropping_box);
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (jcd->priv->progressbar), 0.0);
	
	if (!jcd_setup_snapshot_pipeline (jcd, &err))
	{
		GtkWidget *dlg;
		
		dlg = gtk_message_dialog_new (GTK_WINDOW (jcd),
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("Failed to setup frame preview pipeline.\n%s\n"),
		                              (err) ? err->message : _("unknown error"));
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		if (err)
			g_error_free (err);
		return FALSE;
	}

	/* make sure dialog is visible before
	 *  accessing the dvd drive blocks */
	gtk_window_present (GTK_WINDOW (jcd));
	while (gtk_events_pending ())
		gtk_main_iteration ();

	jcd->priv->snapshot_iterate_id = 
		g_idle_add ((GSourceFunc) jcd_snapshot_pipeline_iterate_cb, jcd);
	
	gst_element_set_state (GST_ELEMENT (jcd->priv->snapshot_pipeline), GST_STATE_PLAYING);
	
	/* run dialog until we got all frames, 
	 *  encountered an error, or the user
	 *  pressed Cancel */

	ret = gtk_dialog_run (GTK_DIALOG (jcd));
	
	g_source_remove (jcd->priv->snapshot_iterate_id);
	jcd->priv->snapshot_iterate_id = 0;

	gst_element_set_state (jcd->priv->snapshot_pipeline, GST_STATE_NULL);
	gst_object_unref (GST_OBJECT (jcd->priv->snapshot_pipeline));
	jcd->priv->snapshot_pipeline = NULL;

	if (ret == TH_RESPONSE_RETRIEVE_ERROR)
	{
		GtkWidget *dlg;
		
		dlg = gtk_message_dialog_new (GTK_WINDOW (jcd),
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("Error while extracting preview frames for cropping.\n%s\n"),
		                              jcd->priv->snapshot_err_msg);
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (GTK_WIDGET (dlg));
		
		/* continue if we managed to get at least a couple of frames */
		if (g_list_length (jcd->priv->snapshot_buffers) < 5)
		{
			jcd_free_snapshot_buffers (jcd);
			gtk_widget_destroy (GTK_WIDGET (jcd));
			return FALSE;
		}
	}
	
	if (ret == GTK_RESPONSE_CANCEL  
	 || ret == GTK_RESPONSE_DELETE_EVENT
	 || jcd->priv->snapshot_buffers == NULL)
	{
		jcd_free_snapshot_buffers (jcd);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		return FALSE;
	}

	th_job_set_snapshot_frames (jcd->priv->job, jcd->priv->snapshot_buffers);
	jcd->priv->snapshot_buffers = NULL; /* job takes ownership of list + bufs */

	g_assert (ret == GTK_RESPONSE_OK);

	return job_cropping_dialog_run_with_frames (jcd);
}

/***************************************************************************
 *
 *   jcd_snapshot_set_dimensions
 *
 ***************************************************************************/

static void
jcd_snapshot_set_dimensions (ThJobCroppingDialog *jcd, guint snapshot_width, guint snapshot_height)
{
	jcd->priv->out_width = snapshot_width;
	jcd->priv->out_height = snapshot_height;
	
	/* The remaining picture should be at least 16x16 in size */
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_right), 
	                           0.0, (jcd->priv->in_width / 2) - 8.0);
	
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_left), 
	                           0.0, (jcd->priv->in_width / 2) - 8.0);
	
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_top), 
	                           0.0, (jcd->priv->in_height / 2) - 8.0);
	
	gtk_spin_button_set_range (GTK_SPIN_BUTTON (jcd->priv->spin_bottom), 
	                           0.0, (jcd->priv->in_height / 2) - 8.0);

	gtk_widget_set_size_request (jcd->priv->image, snapshot_width, snapshot_height);
}

/***************************************************************************
 *
 *   jcd_free_snapshot_buffers
 *
 ***************************************************************************/

static void
jcd_free_snapshot_buffers (ThJobCroppingDialog *jcd)
{
	g_list_foreach (jcd->priv->snapshot_buffers, (GFunc) gst_data_unref, NULL);
	g_list_free (jcd->priv->snapshot_buffers);
	jcd->priv->snapshot_buffers = NULL;
}

/***************************************************************************
 *
 *   jcd_snapshot_pipeline_eos_cb
 *
 ***************************************************************************/

static void
jcd_snapshot_pipeline_eos_cb (ThJobCroppingDialog *jcd, GstElement *el)
{
	g_print ("===== EOS ==== \n");
	jcd_snapshot_pipeline_set_progress (jcd, 1.0);
	gtk_dialog_response (GTK_DIALOG (jcd), GTK_RESPONSE_OK);
}

/***************************************************************************
 *
 *   jcd_snapshot_pipeline_error_cb
 *
 ***************************************************************************/

static void
jcd_snapshot_pipeline_error_cb (ThJobCroppingDialog *jcd, 
                                GstElement          *el, 
                                GError              *err, 
                                const gchar         *dbg, 
                                GstElement          *pipeline)
{
	g_print ("===== ERROR: %s ==== \n", err->message);
	jcd->priv->snapshot_err_msg = g_strdup (err->message);
	gtk_dialog_response (GTK_DIALOG (jcd), TH_RESPONSE_RETRIEVE_ERROR);
}

/***************************************************************************
 *
 *   jcd_snapshots_pipeline_handoff_cb
 *
 ***************************************************************************/

#define TH_SNAPSHOT_INTERVAL            (10)   /* in seconds */
#define TH_SNAPSHOT_PERIOD(timestamp)   (timestamp/(TH_SNAPSHOT_INTERVAL*GST_SECOND))

static void
jcd_snapshot_pipeline_handoff_cb (ThJobCroppingDialog *jcd, GstBuffer *buf, GstPad *pad, GstElement *fakesink)
{
	gdouble  fract = 0.0;
	guint64  timestamp, last_timestamp = 0;
	GList   *l;
	
	if ((l = g_list_last (jcd->priv->snapshot_buffers)))
		last_timestamp = GST_BUFFER_TIMESTAMP(GST_BUFFER(l->data));

	timestamp = GST_BUFFER_TIMESTAMP(buf);

	/* ignore buffer within the same TH_SNAPSHOT_INTERVAL seconds as the previous one */
	if (TH_SNAPSHOT_PERIOD (last_timestamp) == TH_SNAPSHOT_PERIOD (timestamp))
		return;

	gst_buffer_ref (buf);
	if (l != NULL)
		l->next = g_list_append (NULL, buf);
	else
		jcd->priv->snapshot_buffers = g_list_append (NULL, buf);

	/* if title doesn't have that many chapters, just keep
	 *  decoding and add a snapshot every 10 seconds */
	if (jcd->priv->job_num_chapters <= 10)
	{
		if (jcd->priv->job_len_secs < ((25+1) * TH_SNAPSHOT_INTERVAL))
		{
			fract = (timestamp * 1.0) / ((jcd->priv->job_len_secs+1.0) * GST_SECOND);
		}
		else
		{
			guint num_shots = g_list_length (jcd->priv->snapshot_buffers);
			fract = MIN (1.0, num_shots / 25.0);
		}
	}
	else
	{
		GstElement *pipeline, *src;
		guint num_shots = g_list_length (jcd->priv->snapshot_buffers);
		pipeline = GST_ELEMENT (gst_object_get_parent (GST_OBJECT (fakesink)));
		src = gst_bin_get_by_name (GST_BIN (pipeline), "src");
		fract = MIN (1.0, (num_shots * 1.0) / (jcd->priv->job_num_chapters * 1.0));
		g_object_set (src, "chapter", num_shots, NULL);
	}

	jcd_snapshot_pipeline_set_progress (jcd, fract);
	
	if (fract >= 1.0)
	{
		GstElement *pipeline;
		pipeline = GST_ELEMENT (gst_object_get_parent (GST_OBJECT (fakesink)));
		gst_element_set_eos (pipeline);
	}
}

/***************************************************************************
 *
 *   jcd_snapshot_decoder_caps_notify_cb
 *
 ***************************************************************************/

static void
jcd_snapshot_decoder_caps_notify_cb (ThJobCroppingDialog *jcd,
                                     GParamSpec          *spec,
                                     GstPad              *pad)
{
	const GstCaps *caps;
	GstStructure  *s;
	GstElement    *vdecoder, *pipeline, *scaler, *sink;
	gint           w, h;

	if (pad == NULL || !GST_PAD_IS_SRC (pad))
		return;

	if (!gst_pad_is_negotiated (pad))
		return;

	caps = gst_pad_get_negotiated_caps (pad);
	s = gst_caps_get_structure (caps, 0);

	if (gst_structure_get_int (s, "width", &w)
	 && gst_structure_get_int (s, "height", &h))
	{
		if (jcd->priv->in_width != w || jcd->priv->in_height != h)
		{
			GstCaps *filtercaps;
			
			th_log ("cropping input dimensions: %d x %d\n", w,h);
			
			jcd->priv->in_width = (guint) w;
			jcd->priv->in_height = (guint) h;

			jcd_snapshot_set_dimensions (jcd, w / CROP_SNAPSHOT_SCALE, h / CROP_SNAPSHOT_SCALE);

			vdecoder = GST_ELEMENT (gst_object_get_parent (GST_OBJECT (pad)));
			pipeline = GST_ELEMENT (gst_element_get_parent (vdecoder));
			scaler = gst_bin_get_by_name (GST_BIN (pipeline), "scaler");
			sink = gst_bin_get_by_name (GST_BIN (pipeline), "fakesink");
			g_assert (scaler && sink);
			
			gst_element_set_state (pipeline, GST_STATE_PAUSED);
			
			gst_element_unlink (scaler, sink);
			filtercaps = gst_caps_new_simple ("video/x-raw-yuv",
			                                  "width", G_TYPE_INT, w / CROP_SNAPSHOT_SCALE,
			                                  "height", G_TYPE_INT, h / CROP_SNAPSHOT_SCALE,
			                                  NULL);
			gst_element_link_filtered (scaler, sink, filtercaps);
			gst_caps_free (filtercaps);
			
			gst_element_set_state (pipeline, GST_STATE_PLAYING);
		}
	}
}

/***************************************************************************
 *
 *   jcd_snapshot_probe_cb
 *
 *   Called whenever data (a buffer or an event) 
 *    passes through the source pad of the video decoder
 *
 ***************************************************************************/

static gboolean
jcd_snapshot_probe_cb (GstProbe *probe, GstData **p_data, ThJobCroppingDialog *jcd)
{
	GstBuffer *buf;
	guint64    timestamp, last_timestamp = 0;
	GList     *l;
	
	if (!p_data || *p_data == NULL || !GST_IS_BUFFER (*p_data))
		return TRUE; /* pass through */
	
	buf = GST_BUFFER (*p_data);
	
	if ((l = g_list_last (jcd->priv->snapshot_buffers)))
		last_timestamp = GST_BUFFER_TIMESTAMP (GST_BUFFER (l->data));

	timestamp = GST_BUFFER_TIMESTAMP (buf);

	/* ignore buffers within the same TH_SNAPSHOT_INTERVAL 
	 *  seconds of the last saved one */
	if (TH_SNAPSHOT_PERIOD (last_timestamp) == TH_SNAPSHOT_PERIOD (timestamp))
		return FALSE; /* not interested, discard buffer */

	return TRUE; /* do not remove data from stream */
}

/***************************************************************************
 *
 *   jcd_snapshot_pipeline_iterate_cb
 *
 ***************************************************************************/

static gboolean
jcd_snapshot_pipeline_iterate_cb (ThJobCroppingDialog *jcd)
{
	gst_bin_iterate (GST_BIN (jcd->priv->snapshot_pipeline));
	return TRUE; /* call again */
}

/***************************************************************************
 *
 *   jcd_setup_snapshot_pipeline
 *
 ***************************************************************************/

static gboolean
jcd_setup_snapshot_pipeline (ThJobCroppingDialog *jcd, GError **err)
{
	GstElement  *pipeline, *fakesink, *vdecoder, *src;
	GstProbe    *probe;
	GstPad      *src_pad;
	ThJob       *j;
	gchar       *desc, *device;
	guint        title_num;

	g_return_val_if_fail (TH_IS_JOB_CROPPING_DIALOG (jcd), FALSE);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);
	g_return_val_if_fail (jcd->priv->snapshot_pipeline == NULL, FALSE);

	j = jcd->priv->job;

	g_object_get (jcd->priv->job, 
	              "device", &device, 
	              "title-num", &title_num, 
	              "num-chapters", &jcd->priv->job_num_chapters,
	              "title-length", &jcd->priv->job_len_secs,
	              NULL);

	th_job_set_snapshot_frames (jcd->priv->job, NULL);
	
	desc = g_strdup_printf ("dvdreadsrc title=%u name=src"
	                        " ! dvddemux"
	                        " ! mpeg2dec name=decoder"
	                        " ! thyuvscale name=scaler"
	                        " ! fakesink name=fakesink signal-handoffs=true ",
	                        title_num);
	
	g_print ("%s: Launching pipeline:\n  %s\n\n", G_STRLOC, desc);

	pipeline = gst_parse_launch (desc, err);
	
	g_free (desc); desc = NULL;
	
	if (err && *err)
	{
		if (pipeline)
			gst_object_unref (GST_OBJECT (pipeline));
		g_free (device);
		return FALSE;
	} 
	
	if (pipeline == NULL)
	{
		g_free (device);
		return FALSE;
	}
	
	fakesink = gst_bin_get_by_name (GST_BIN (pipeline), "fakesink");
	vdecoder = gst_bin_get_by_name (GST_BIN (pipeline), "decoder");
	src = gst_bin_get_by_name (GST_BIN (pipeline), "src");
	
	/* use g_object_set() to avoid escaping issues when
	 * using a local directory path instead of a device */
	g_object_set (src, "device", device, NULL);
	
	g_signal_connect_swapped (pipeline, "eos", 
	                          G_CALLBACK (jcd_snapshot_pipeline_eos_cb),
	                          jcd);

	g_signal_connect_swapped (pipeline, "error", 
	                          G_CALLBACK (jcd_snapshot_pipeline_error_cb), 
	                          jcd);

	g_signal_connect_swapped (fakesink, "handoff", 
	                          G_CALLBACK (jcd_snapshot_pipeline_handoff_cb), 
	                          jcd);

	src_pad = gst_element_get_pad (vdecoder, "src");
	g_return_val_if_fail (GST_IS_PAD (src_pad), FALSE);

	/* so we know the size of the input video, and can set filtered caps
	 *  between the scaler and fakesink, so that we get a scaled-down
	 *  version of the input according to our fixed scale factor */
	g_signal_connect_swapped (src_pad, "notify::caps", 
	                          G_CALLBACK (jcd_snapshot_decoder_caps_notify_cb), 
	                          jcd);

	probe = gst_probe_new (FALSE, (GstProbeCallback) jcd_snapshot_probe_cb, jcd);
	
	gst_pad_add_probe (src_pad, probe);
	
	g_object_weak_ref (G_OBJECT (src_pad),
	                   (GWeakNotify) gst_probe_destroy, 
	                   probe);

	jcd->priv->snapshot_pipeline = pipeline;
	
	jcd_snapshot_pipeline_set_progress (jcd, 0.0);

	g_free (device);
	
	return TRUE;
}
