/***************************************************************************
                             th-preferences.c
                             ----------------
    begin                : Thu Apr 29 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : tim centricular net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-marshal.h"
#include "th-preferences.h"
#include "th-utils.h"

#include <glib/gi18n.h>

#include <string.h>
#include <stdio.h>
#include <errno.h>

#define TH_PREFERENCES_ERROR  (g_quark_from_static_string ("th-preferences-error-quark"))

enum
{
  PROP_FILENAME = 1
};

enum
{
	DEEP_NOTIFY = 0,
	LAST_SIGNAL
};

enum
{
  TH_PREFERENCES_ERROR_VALUE_INVALID = 1,
};

struct _ThPreferencesPrivate
{
	gchar     *filename;

	GList     *opp_list;
};

typedef struct _ThObjPropProxy  ThObjPropProxy;

struct _ThObjPropProxy
{
	GObject    *obj;
	gchar      *obj_prop_name;
	gchar      *prefs_name;
	gchar      *group;
	GValue      val;
	GParamSpec *pspec;
	gulong      notify_id; /* g_signal_connect() return value */
	guint       prop_num;
};

/* boilerplate stuff */

#define th_preferences_init        preferences_instance_init
#define th_preferences_class_init  preferences_class_init

G_DEFINE_TYPE (ThPreferences, th_preferences, G_TYPE_OBJECT)  

static void              preferences_class_init       (ThPreferencesClass *klass);


static void              preferences_instance_init    (ThPreferences *prefs);

static void              preferences_get_property     (GObject      *object,
                                                       guint         param_id,
                                                       GValue       *value,
                                                       GParamSpec   *pspec);
                                                          
static void              preferences_set_property     (GObject      *object,
                                                       guint         param_id,
                                                       const GValue *value,
                                                       GParamSpec   *pspec);

static gchar            *preferences_serialise        (ThPreferences *prefs);

static void              preferences_deserialise      (ThPreferences *prefs, 
                                                       GKeyFile      *keyfile);


static gboolean          preferences_parse_uint       (ThPreferences *prefs, 
                                                       const gchar   *val_str,
                                                       guint         *p_uint,
                                                       GError       **error);

static gboolean          preferences_parse_uint64     (ThPreferences *prefs, 
                                                       const gchar   *val_str, 
                                                       guint64       *p_uint64,
                                                       GError       **error);

static guint             prefs_signals[LAST_SIGNAL];   /* all 0 */

/***************************************************************************
 *
 *   preferences_find_opp_from_obj
 *
 ***************************************************************************/

static ThObjPropProxy *
preferences_find_opp_from_obj (ThPreferences *prefs, GObject *obj)
{
	GList *node;

	for (node = prefs->priv->opp_list;  node;  node = node->next)
	{
		ThObjPropProxy *opp = (ThObjPropProxy *) node->data;

		if (opp->obj == obj)
			return opp;
	}
	
	return NULL;
}

/***************************************************************************
 *
 *   preferences_find_opp_from_pspec
 *
 ***************************************************************************/

static ThObjPropProxy *
preferences_find_opp_from_pspec (ThPreferences *prefs, GParamSpec *pspec)
{
	GList *node;

	for (node = prefs->priv->opp_list;  node;  node = node->next)
	{
		ThObjPropProxy *opp = (ThObjPropProxy *) node->data;

		if (opp->pspec == pspec)
			return opp;
	}
	
	return NULL;
}

/***************************************************************************
 *
 *   preferences_find_opp_from_prefs_name
 *
 ***************************************************************************/

static ThObjPropProxy *
preferences_find_opp_from_prefs_name (ThPreferences *prefs, 
                                      const gchar   *prefs_name,
                                      const gchar   *group)
{
	GList *node;

	for (node = prefs->priv->opp_list;  node;  node = node->next)
	{
		ThObjPropProxy *opp = (ThObjPropProxy *) node->data;

		if (g_str_equal (opp->prefs_name, prefs_name)
		 && (group == NULL || g_str_equal (opp->group, group)))
			return opp;
	}
	
	return NULL;
}

/***************************************************************************
 *
 *   preferences_weak_notify
 *
 ***************************************************************************/

static void
preferences_weak_notify (ThPreferences *prefs, GObject *old_obj_addr)
{
	ThObjPropProxy *opp;

	while ((opp = preferences_find_opp_from_obj (prefs, old_obj_addr)))
	{
		g_free (opp->obj_prop_name);
		g_free (opp->prefs_name);
		g_free (opp->group);
		
		g_value_unset (&opp->val);
		
		memset (opp, 0x00, sizeof (ThObjPropProxy));

		prefs->priv->opp_list = g_list_remove (prefs->priv->opp_list, opp);

		g_free (opp);
	}
}

/***************************************************************************
 *
 *   preferences_notify_cb
 *
 ***************************************************************************/

static void
preferences_notify_cb (ThPreferences *prefs, GParamSpec *pspec, GObject *obj)
{
	ThObjPropProxy *opp;
	GQuark          detail;
	
	opp = preferences_find_opp_from_pspec (prefs, pspec);
	g_return_if_fail (opp != NULL);

	g_value_reset (&opp->val);

	g_object_get_property (G_OBJECT (obj), pspec->name, &opp->val);

	detail = g_quark_from_string (opp->prefs_name);
	g_signal_emit (prefs, prefs_signals[DEEP_NOTIFY], detail, obj, pspec);
}

/***************************************************************************
 *
 *   th_preferences_register_proxy
 *
 ***************************************************************************/

void
th_preferences_register_proxy (ThPreferences *prefs,
                               GObject       *obj,
                               const gchar   *obj_prop_name,
                               const gchar   *prefs_name,
                               const gchar   *group_name)
{
	ThObjPropProxy *opp;
	GParamSpec     *pspec;
	gchar          *notifysignal;

	g_return_if_fail (TH_IS_PREFERENCES (prefs));
	g_return_if_fail (G_IS_OBJECT (obj));
	g_return_if_fail (obj_prop_name != NULL);
	g_return_if_fail (prefs_name != NULL);

	if (group_name == NULL)
		group_name = PACKAGE;

	opp = g_new0 (ThObjPropProxy, 1);

	opp->obj           = obj;
	opp->obj_prop_name = g_strdup (obj_prop_name);
	opp->prefs_name    = g_strdup (prefs_name);
	opp->group         = g_strdup (group_name);

	pspec = g_object_class_find_property (G_OBJECT_GET_CLASS (obj), obj_prop_name);
	g_return_if_fail (pspec != NULL);

	opp->pspec = pspec;
	g_value_init (&opp->val, pspec->value_type);

	g_object_weak_ref (obj, (GWeakNotify) preferences_weak_notify, prefs);

	prefs->priv->opp_list = g_list_append (prefs->priv->opp_list, opp);

	notifysignal = g_strdup_printf ("notify::%s", obj_prop_name);
	
	opp->notify_id = g_signal_connect_swapped (obj, 
	                                           notifysignal, 
	                                           G_CALLBACK (preferences_notify_cb), 
	                                           prefs);
	
	g_object_get_property (G_OBJECT (obj), obj_prop_name, &opp->val);

	g_free (notifysignal);
}
                           
/***************************************************************************
 *
 *   preferences_finalize
 *
 ***************************************************************************/

static void
preferences_finalize (GObject *obj)
{
	ThPreferences *prefs = (ThPreferences*) obj;
 
	while (prefs->priv->opp_list)
	{
		ThObjPropProxy *opp = (ThObjPropProxy*) prefs->priv->opp_list->data;
		
		g_signal_handler_disconnect (opp->obj, opp->notify_id);
		
		g_object_weak_unref (opp->obj, 
		                     (GWeakNotify) preferences_weak_notify, 
		                     prefs);
		
		preferences_weak_notify (prefs, opp->obj);
	}
	
	/* free and poison */
	memset (prefs->priv, 0xAB, sizeof(ThPreferencesPrivate));
	g_free (prefs->priv);
	prefs->priv = NULL;
  
	G_OBJECT_CLASS (th_preferences_parent_class)->finalize (obj);
}


/***************************************************************************
 *
 *   preferences_class_init
 *
 ***************************************************************************/

static void              
preferences_class_init (ThPreferencesClass *klass)
{
	GObjectClass  *object_class = (GObjectClass*) klass;
  
	object_class->finalize     = preferences_finalize;
	object_class->set_property = preferences_set_property;
	object_class->get_property = preferences_get_property;

	g_object_class_install_property (object_class, PROP_FILENAME, 
	                                 g_param_spec_string ("filename", 
	                                                      "filename", 
	                                                      "filename", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	prefs_signals[DEEP_NOTIFY] = g_signal_new ("deep-notify",
	                                           G_TYPE_FROM_CLASS (object_class),
	                                           G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
	                                           G_STRUCT_OFFSET (ThPreferencesClass, deep_notify),
	                                           NULL, NULL,
	                                           th_marshal_VOID__OBJECT_POINTER,
	                                           G_TYPE_NONE, 2, G_TYPE_OBJECT, G_TYPE_POINTER);
}


/***************************************************************************
 *
 *   preferences_instance_init
 *
 ***************************************************************************/

static void              
preferences_instance_init (ThPreferences *prefs)
{
	prefs->priv = g_new0 (ThPreferencesPrivate, 1);
}

/***************************************************************************
 *
 *   preferences_get_property
 *
 ***************************************************************************/

static void
preferences_get_property (GObject      *object,
                          guint         param_id,
                          GValue       *value,
                          GParamSpec   *pspec)
{
	ThPreferences  *prefs = (ThPreferences*) object;

	g_return_if_fail (TH_IS_PREFERENCES (prefs));
	
	switch (param_id)
	{
		case PROP_FILENAME:
		{
			g_value_set_string (value, prefs->priv->filename);
		}
		break;
			
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}


/***************************************************************************
 *
 *   preferences_set_property
 *
 ***************************************************************************/

static void
preferences_set_property (GObject      *object,
                          guint         param_id,
                          const GValue *value,
                          GParamSpec   *pspec)
{
	ThPreferences  *prefs = (ThPreferences*) object;

	g_return_if_fail (TH_IS_PREFERENCES (object));

	switch (param_id)
	{
		case PROP_FILENAME:
		{
			g_free (prefs->priv->filename);
			prefs->priv->filename = NULL;
			g_return_if_fail (g_path_is_absolute (g_value_get_string (value)));
			prefs->priv->filename = g_value_dup_string (value);
		}
		break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   preferences_deserialise_opp
 *
 ***************************************************************************/

static void
preferences_deserialise_opp (ThPreferences  *prefs, 
                             ThObjPropProxy *opp, 
                             GKeyFile       *keyfile)
{
	GType type;

	type = G_VALUE_TYPE (&opp->val);

	if (G_TYPE_IS_ENUM (type))
		type = G_TYPE_ENUM;

	switch (type)
	{
	    case G_TYPE_STRING:
		{
			gchar *s;
	        s = g_key_file_get_string (keyfile, opp->group, opp->prefs_name, NULL);
			g_value_take_string (&opp->val, s);
		}
		break;

		case G_TYPE_BOOLEAN:
		{
			gboolean b;
			b = g_key_file_get_boolean (keyfile, opp->group, opp->prefs_name, NULL);
			g_value_set_boolean (&opp->val, b);
		}
		break;

	    case G_TYPE_INT:
		{
			gint i;
			i = g_key_file_get_integer (keyfile, opp->group, opp->prefs_name, NULL);
			g_value_set_int (&opp->val, i);
		}
		break;

		case G_TYPE_ENUM:
		{
			gint i;
			i = g_key_file_get_integer (keyfile, opp->group, opp->prefs_name, NULL);
			g_value_set_enum (&opp->val, i);
		}
		break;

		case G_TYPE_UINT:
		{
			gchar *valstr;
			guint  u;

			valstr = g_key_file_get_value (keyfile, opp->group, opp->prefs_name, NULL);

			if (valstr && preferences_parse_uint (prefs, valstr, &u, NULL))
				g_value_set_uint (&opp->val, u);

			g_free (valstr);
		}
		break;

		case G_TYPE_UINT64:
		{
			guint64  u;
			gchar   *valstr;

			valstr = g_key_file_get_value (keyfile, opp->group, opp->prefs_name, NULL);

			if (valstr && preferences_parse_uint64 (prefs, valstr, &u, NULL))
				g_value_set_uint64 (&opp->val, u);

			g_free (valstr);
		}
		break;

		default:
		{
			g_warning ("ThPreferences: Unexpected type '%s' for preference '%s'.\n",
			           g_type_name (type), opp->prefs_name);
		}
		break;
	}
}

/***************************************************************************
 *
 *   preferences_serialise_opp
 *
 ***************************************************************************/

static void
preferences_serialise_opp (ThPreferences  *prefs, 
                           ThObjPropProxy *opp, 
                           GKeyFile       *keyfile)
{
	GType type;

	type = G_VALUE_TYPE (&opp->val);

	if (G_TYPE_IS_ENUM (type))
		type = G_TYPE_ENUM;

	switch (type)
	{
	    case G_TYPE_STRING:
		{
			const gchar *txt = g_value_get_string (&opp->val);

	        g_key_file_set_string (keyfile, 
			                       opp->group, 
			                       opp->prefs_name, 
			                       (txt) ? txt : "");
		}
		break;

		case G_TYPE_BOOLEAN:
		{
			g_key_file_set_boolean (keyfile, 
			                        opp->group, 
			                        opp->prefs_name, 
			                        g_value_get_boolean (&opp->val));
		}
		break;

	    case G_TYPE_INT:
		{
			g_key_file_set_integer (keyfile, 
			                        opp->group, 
			                        opp->prefs_name, 
			                        g_value_get_int (&opp->val));
		}
		break;

		case G_TYPE_ENUM:
		{
			g_key_file_set_integer (keyfile, 
			                        opp->group, 
			                        opp->prefs_name, 
			                        (gint) g_value_get_enum (&opp->val));
		}
		break;

		case G_TYPE_UINT:
		{
			gchar *valstr = g_strdup_printf ("%u", g_value_get_uint (&opp->val));

			g_key_file_set_value (keyfile, 
			                      opp->group, 
			                      opp->prefs_name, 
			                      valstr);

			g_free (valstr);
		}
		break;

		case G_TYPE_UINT64:
		{
			gchar *valstr = g_strdup_printf ("%" G_GUINT64_FORMAT, g_value_get_uint64 (&opp->val));

			g_key_file_set_value (keyfile, 
			                      opp->group, 
			                      opp->prefs_name, 
			                      valstr);

			g_free (valstr);
		}
		break;

		default:
		{
			g_warning ("ThPreferences: Unexpected type '%s' for preference '%s'.\n",
			           g_type_name (type), opp->prefs_name);
		}
		break;
	}
}

/***************************************************************************
 *
 *   preferences_serialise
 *
 ***************************************************************************/

static gchar *
preferences_serialise (ThPreferences *prefs)
{
	GKeyFile *keyfile;
	GList    *l;
	gchar    *str;

	keyfile = g_key_file_new ();

	for (l = prefs->priv->opp_list;  l;  l = l->next)
	{
		ThObjPropProxy *opp = (ThObjPropProxy*) l->data;

		/* make sure the value is up-to-date */
		g_object_get_property (G_OBJECT (opp->obj), 
		                       opp->obj_prop_name, 
		                       &opp->val);

		preferences_serialise_opp (prefs, opp, keyfile);
	}

	str = g_key_file_to_data (keyfile, NULL, NULL);

	g_key_file_free (keyfile);

	return str;
}


/***************************************************************************
 *
 *   prefs_str_to_uint64
 *
 *   Returns TRUE if the string could successfully 
 *    be parsed into an integer and *p_num is set.
 *    Accepts negative values as well (will just
 *    assume they fit then)
 *
 ***************************************************************************/

static gboolean
prefs_str_to_uint64 (const gchar *attr_val, guint64 *p_num, gboolean *p_is_negative)
{
	gboolean is_negative;
	guint64  num;
	gchar   *endptr = NULL;
  
	g_return_val_if_fail (attr_val != NULL, FALSE);
  
	is_negative = (*attr_val == '-');
  
	if (is_negative)
		++attr_val;

	if (p_is_negative)
		*p_is_negative = is_negative;

	if (p_num)
		*p_num = 0;
  
	num = g_ascii_strtoull (attr_val, &endptr, 10);
  
	if (endptr == NULL  ||  *endptr != 0x00)
		return FALSE;
  
	if (num == G_MAXUINT64  && errno == ERANGE)
		return FALSE;
  
	if (is_negative  &&  num >= G_MAXINT64)
		return FALSE;
    
	if (is_negative)
		num = (guint64)0 - num;

	if (p_num)
		*p_num = num;

	return TRUE;
}

/***************************************************************************
 *
 *   preferences_parse_uint
 *
 ***************************************************************************/

static gboolean
preferences_parse_uint (ThPreferences *prefs, 
                        const gchar   *val_str,
                        guint         *p_uint,
                        GError       **error)
{
	gboolean is_negative;
	guint64  num;

	if (!prefs_str_to_uint64 (val_str, &num, &is_negative))
	{
		g_set_error (error, TH_PREFERENCES_ERROR, 
		             TH_PREFERENCES_ERROR_VALUE_INVALID,
		             _("Unable to parse uint attribute value '%s'."), val_str);
		return FALSE;
	}
  
	if (num <= G_MAXUINT  &&  !is_negative)
	{
		if (p_uint)
			*p_uint = (guint) num;
		return TRUE;
	}
    
	if (is_negative)
	{
		g_set_error (error, TH_PREFERENCES_ERROR, 
		             TH_PREFERENCES_ERROR_VALUE_INVALID,
		             _("uint attribute '%s' is negative."),
		             val_str);
		return FALSE;
	}
    
	g_set_error (error, TH_PREFERENCES_ERROR, 
	             TH_PREFERENCES_ERROR_VALUE_INVALID,
	             _("uint attribute '%s' too large"),
	             val_str);

	return FALSE;
}


/***************************************************************************
 *
 *   preferences_parse_uint64
 *
 ***************************************************************************/

static gboolean
preferences_parse_uint64 (ThPreferences *prefs, 
                          const gchar   *val_str, 
                          guint64       *p_uint64,
                          GError       **error)
{
	gboolean is_negative;
	guint64  num;

	if (!prefs_str_to_uint64 (val_str, &num, &is_negative))
	{
		g_set_error (error, TH_PREFERENCES_ERROR, 
		             TH_PREFERENCES_ERROR_VALUE_INVALID,
		             _("Unable to parse uint64 attribute value '%s'."), val_str);
		return FALSE;
	}
  
	if (is_negative)
	{
		g_set_error (error, TH_PREFERENCES_ERROR, 
		             TH_PREFERENCES_ERROR_VALUE_INVALID,
		             _("uint64 attribute '%s' is negative."),
		             val_str);
		return FALSE;
	}
    
	if (p_uint64)
		*p_uint64 = num;

	return TRUE;
}


/***************************************************************************
 *
 *   preferences_deserialise
 *
 ***************************************************************************/

static void
preferences_deserialise (ThPreferences *prefs, GKeyFile *keyfile)
{
	GList *l;

	for (l = prefs->priv->opp_list;  l;  l = l->next)
	{
		ThObjPropProxy *opp = (ThObjPropProxy*) l->data;

		preferences_deserialise_opp (prefs, opp, keyfile);

		/* update object property (TODO: only update if different?) */
		g_object_set_property (G_OBJECT (opp->obj), 
		                       opp->obj_prop_name, 
		                       &opp->val);
	}
}

/***************************************************************************
 *
 *   th_preferences_load
 *
 ***************************************************************************/

void
th_preferences_load (ThPreferences *prefs)
{
	GKeyFile *keyfile;
	GError   *err = NULL;

	g_return_if_fail (TH_IS_PREFERENCES (prefs));
	g_return_if_fail (prefs->priv->filename != NULL);

	keyfile = g_key_file_new ();

	if (!g_key_file_load_from_file (keyfile, prefs->priv->filename, G_KEY_FILE_NONE, &err))
	{
		if (err->code != G_FILE_ERROR_NOENT)
			g_warning (_("%s failed: %s."), __FUNCTION__, err->message);
		g_error_free (err);
		g_key_file_free (keyfile);
		return;
	}

	preferences_deserialise (prefs, keyfile);

	g_key_file_free (keyfile);
}

/***************************************************************************
 *
 *   th_preferences_save
 *
 ***************************************************************************/

void
th_preferences_save (ThPreferences *prefs)
{
	gchar  *content;
	FILE   *f;
	
	g_return_if_fail (TH_IS_PREFERENCES (prefs));
	g_return_if_fail (prefs->priv->filename != NULL);

	content = preferences_serialise (prefs);
	g_return_if_fail (content != NULL);

	f = fopen (prefs->priv->filename, "w");
	if (f == NULL)
	{
		g_warning (_("%s failed: %s."), __FUNCTION__, g_strerror (errno));
	}
	else
	{
		fprintf (f, "%s\n", content);
		fclose (f);
	}
	g_free (content);
}

/***************************************************************************
 *
 *   th_preferences_new
 *
 ***************************************************************************/

ThPreferences *
th_preferences_new (const gchar *filename)
{
	GObject  *obj;
	gchar    *fn;

	if (filename == NULL)
	{
		return (ThPreferences *) g_object_new (TH_TYPE_PREFERENCES, NULL);
	}
	
	if (g_path_is_absolute (filename))
	{
		return (ThPreferences *) g_object_new (TH_TYPE_PREFERENCES, 
		                                       "filename", filename, 
		                                       NULL);
	}
	
	fn = th_get_user_config_fn (filename);

	obj = g_object_new (TH_TYPE_PREFERENCES, "filename", fn, NULL);
	
	g_free (fn);

	return (ThPreferences *) obj;
}


/***************************************************************************
 *
 *   th_preferences_get_value
 *
 ***************************************************************************/

gboolean
th_preferences_get_value (ThPreferences *prefs,
                          const gchar   *prefs_name,
                          const gchar   *group_name,
                          GValue        *p_value)
{
	ThObjPropProxy *opp;

	g_return_val_if_fail (TH_IS_PREFERENCES (prefs), FALSE);

	opp = preferences_find_opp_from_prefs_name (prefs, prefs_name, group_name);

	if (opp == NULL)
		return FALSE;

	if (p_value)
	{
		g_value_init (p_value, G_VALUE_TYPE (&opp->val));
		g_value_copy (&opp->val, p_value);
	}

	return TRUE;
}


/***************************************************************************
 *
 *   th_preferences_set_value
 *
 ***************************************************************************/

void
th_preferences_set_value (ThPreferences *prefs,
                          const gchar   *prefs_name,
                          const gchar   *group_name,
                          const GValue  *value)
{
	ThObjPropProxy *opp;

	g_return_if_fail (TH_IS_PREFERENCES (prefs));

	opp = preferences_find_opp_from_prefs_name (prefs, prefs_name, group_name);

	g_return_if_fail (opp != NULL);

	g_object_set_property (opp->obj, opp->obj_prop_name, value);
}


