/***************************************************************************
                             th-title-preview.c
                             ------------------
    begin                : Tue Oct 03 2006
    copyright            : (C) 2006 by Tim-Philipp Müller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <gst/video/video.h>
#include <gdk/gdkx.h>

#include <glib/gi18n.h>

#include "th-title-preview.h"
#include "th-utils.h"

#define PREVIEW_DEFAULT_WIDTH  384
#define PREVIEW_DEFAULT_HEIGHT 288

static void th_title_preview_class_init (ThTitlePreviewClass *klass);
static void th_title_preview_init (ThTitlePreview * preview);

struct _ThTitlePreviewPrivate
{
  GstElement *pipeline;

  GtkWidget *picture_area;
  GtkWidget *seek_slider;
  GtkWidget *close_button;
  GtkWidget *next_button;
  GtkWidget *prev_button;
  GtkWidget *play_pause_button;
  GtkWidget *play_pause_img;
  GtkWidget *statusbar;
  guint      statusbar_id;             /* statusbar context ID           */

  guint      timer_id;

  guint      seek_id;                  /* timeout for delayed seeking    */
  gint64     seek_pos;                 /* position to seek to (nanosecs) */

  guint      num_titles;
  guint      cur_title;

  GTimeVal   last_seek_to_msg;         /* to block current position printing to
                                        * the statusbar for a few secs after
                                        * we've printed a 'Seek to' message */

  gulong     slider_value_changed_id;  /* so we can block the value-changed
                                        * signal from being emitted while we
                                        * update the seek slider ourselves */
};

G_DEFINE_TYPE (ThTitlePreview, th_title_preview, GTK_TYPE_WINDOW);

static void th_title_preview_update_widget_state (ThTitlePreview * preview);
static void th_title_preview_setup_query_time (ThTitlePreview * preview,
    guint timeout);
static void th_title_preview_statusbar_printf (ThTitlePreview * preview,
    const gchar * format, ...) G_GNUC_PRINTF (2,3);

static void
th_title_preview_init (ThTitlePreview * preview)
{
  preview->priv = g_new0 (ThTitlePreviewPrivate, 1);
  
  gtk_window_set_icon_from_file (GTK_WINDOW (preview),
      DATADIR "/ui/icons/thoggen.png", NULL);

  g_get_current_time (&preview->priv->last_seek_to_msg);
}

static void
th_title_preview_finalize (GObject * obj)
{
  ThTitlePreview *preview = TH_TITLE_PREVIEW (obj);

  th_title_preview_setup_query_time (preview, 0);

  if (preview->priv->seek_id)
    g_source_remove (preview->priv->seek_id);

  if (preview->priv->pipeline) {
    gst_element_set_state (preview->priv->pipeline, GST_STATE_NULL);
    gst_object_unref (preview->priv->pipeline);
    preview->priv->pipeline = NULL;
  }

  g_free (preview->priv);
  preview->priv = NULL;

  G_OBJECT_CLASS (th_title_preview_parent_class)->finalize (obj);
}


static void
th_title_preview_class_init (ThTitlePreviewClass *klass)
{
  G_OBJECT_CLASS (klass)->finalize = th_title_preview_finalize;
}

static void
th_title_preview_sync_msg (ThTitlePreview * preview, GstMessage * msg,
    GstBus * bus)
{
  GdkWindow *window;
  XID window_id;

  if (msg->type != GST_MESSAGE_ELEMENT || msg->structure == NULL ||
      !gst_structure_has_name (msg->structure, "prepare-xwindow-id"))
    return;

  g_return_if_fail (msg->src != NULL && GST_IS_X_OVERLAY (msg->src));

  GST_DEBUG ("Handling sync prepare-xwindow-id message");

  window = GTK_WIDGET (preview->priv->picture_area)->window;
  g_return_if_fail (window != NULL);

  window_id = GDK_WINDOW_XWINDOW (window);
  gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (msg->src), window_id);

  if (g_object_class_find_property (G_OBJECT_GET_CLASS (msg->src), "force-aspect-ratio")) {
    g_object_set (msg->src, "force-aspect-ratio", TRUE, NULL);
  }
}

static void
th_title_preview_statusbar_printf (ThTitlePreview * preview,
    const gchar * format, ...)
{
  GtkStatusbar *statusbar;
  gchar *s;

  if (format == NULL) {
    s = g_strdup ("");
  } else {
    va_list args;

    va_start (args, format);
    s = g_strdup_vprintf (format, args);
    va_end (args);
  }

  statusbar = GTK_STATUSBAR (preview->priv->statusbar);
  gtk_statusbar_pop (statusbar, preview->priv->statusbar_id);
  gtk_statusbar_push (statusbar, preview->priv->statusbar_id, s);
	
  g_free (s);
}

static gboolean
th_title_preview_query (ThTitlePreview * preview)
{
  GTimeVal cur_time;
  GstFormat format;
  gdouble percent;
  gchar *title_str;
  gint64 dur = 0, pos = 0;

  g_get_current_time (&cur_time);
  cur_time.tv_sec -= preview->priv->last_seek_to_msg.tv_sec;
  if (cur_time.tv_usec < preview->priv->last_seek_to_msg.tv_usec) {
    ++cur_time.tv_sec;
    cur_time.tv_usec += G_USEC_PER_SEC;
  }
  cur_time.tv_usec -= preview->priv->last_seek_to_msg.tv_usec;

  if (cur_time.tv_sec <= 2)
    goto done;

  format = GST_FORMAT_TIME;
  if (!gst_element_query_duration (preview->priv->pipeline, &format, &dur) ||
      !gst_element_query_position (preview->priv->pipeline, &format, &pos)) {
    th_log ("Could not query duration or position of preview pipeline\n");
    goto done;
  }

  title_str = g_strdup_printf (_("Title %u of %u"), 
      preview->priv->cur_title, preview->priv->num_titles);

  th_title_preview_statusbar_printf (preview,
      "%s - %" TH_TIME_FORMAT " / %" TH_TIME_FORMAT, title_str,
      TH_TIME_ARGS (pos), TH_TIME_ARGS (dur));

  g_free (title_str);

  /* update slider */
  g_signal_handler_block (preview->priv->seek_slider,
      preview->priv->slider_value_changed_id);
  percent = CLAMP ((gdouble) pos * 100.0 / (gdouble) dur, 0.0, 100.0);
  gtk_range_set_value (GTK_RANGE (preview->priv->seek_slider), percent);
  g_signal_handler_unblock (preview->priv->seek_slider,
      preview->priv->slider_value_changed_id);

done:
  return TRUE; /* call us again */
}

static void
th_title_preview_setup_query_time (ThTitlePreview * preview, guint timeout)
{
  if (preview->priv->timer_id > 0) {
    g_source_remove (preview->priv->timer_id);
    preview->priv->timer_id = 0;
  }

  if (timeout > 0) {
    preview->priv->timer_id =
        g_timeout_add (timeout, (GSourceFunc) th_title_preview_query, preview);
  }
}

static void
th_title_preview_async_msg (ThTitlePreview * preview, GstMessage * msg,
    GstBus * bus)
{
  switch (msg->type) {
    case GST_MESSAGE_STATE_CHANGED: {
      GstState old_state, new_state;

      /* only interested in the top-level pipeline */
      if (!GST_IS_PIPELINE (msg->src))
        break;

      gst_message_parse_state_changed (msg, &old_state, &new_state, NULL);

      if (new_state == GST_STATE_PAUSED) {
        th_title_preview_setup_query_time (preview, 250);

        if (old_state == GST_STATE_READY) {
          GstElement *src;
          GstFormat title_format = gst_format_get_by_nick ("title");
          gint64 num = 0;

          src = gst_bin_get_by_name (GST_BIN (preview->priv->pipeline), "src");
          if (gst_element_query_duration (src, &title_format, &num)) {
            preview->priv->num_titles = (guint) num;
          } else {
            preview->priv->num_titles = 0;
          }
          gst_object_unref (src);
        }
      } else if (new_state < GST_STATE_PAUSED) {
        th_title_preview_setup_query_time (preview, 0);
      }
      th_title_preview_update_widget_state (preview);
      break;
    }
    case GST_MESSAGE_ERROR: {
      GError *err = NULL;
      gchar *dbg = NULL;

      gst_message_parse_error (msg, &err, &dbg);

      th_title_preview_setup_query_time (preview, 0);
      if (preview->priv->seek_id) {
        g_source_remove (preview->priv->seek_id);
        preview->priv->seek_id = 0;
      }

      gst_element_set_state (preview->priv->pipeline, GST_STATE_NULL);
      th_title_preview_update_widget_state (preview);

      if (err != NULL) {
        GtkWidget *dlg;

        dlg = gtk_message_dialog_new (GTK_WINDOW (preview),
		                              GTK_DIALOG_MODAL,
		                              GTK_MESSAGE_ERROR,
		                              GTK_BUTTONS_OK,
		                              _("Error playing DVD title %u"),
		                              preview->priv->cur_title);

        gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dlg),
		    "%s", err->message);
        g_warning ("Error playing DVD title %u: %s [debug: %s]",
            preview->priv->cur_title, err->message, (dbg) ? dbg : "none");
        (void) gtk_dialog_run (GTK_DIALOG (dlg));
        gtk_widget_destroy (dlg);
        g_error_free (err);
      }
      g_free (dbg);
      break;
    }
    case GST_MESSAGE_EOS: {
      g_signal_handler_block (preview->priv->seek_slider,
          preview->priv->slider_value_changed_id);
      gtk_range_set_value (GTK_RANGE (preview->priv->seek_slider), 100.0);
      g_signal_handler_unblock (preview->priv->seek_slider,
          preview->priv->slider_value_changed_id);
      break;
    }
    default:
      break;
  }
}

static void
th_title_preview_picture_area_realize (ThTitlePreview * preview, GtkWidget * w)
{
  th_log ("preview window realized\n");
  gst_element_set_state (preview->priv->pipeline, GST_STATE_PLAYING);
}

static gboolean
th_title_preview_seek_delayed (ThTitlePreview * preview)
{
  gst_element_seek_simple (preview->priv->pipeline, GST_FORMAT_TIME,
      GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, preview->priv->seek_pos);
  preview->priv->seek_id = 0;
  return FALSE; /* don't call us again */
}

static void
th_title_preview_slider_value_changed (ThTitlePreview * preview,
    GtkRange * range)
{
  GstFormat tformat = GST_FORMAT_TIME;
  gdouble val;
  gint64 len;

  if (!gst_element_query_duration (preview->priv->pipeline, &tformat, &len)) {
    g_warning ("Cannot seek: failed to query preview pipeline duration!\n");
    return;
  }

  val = gtk_range_get_value (range);
  preview->priv->seek_pos = (gint64) (val * (gdouble) len / 100.0);
  th_title_preview_statusbar_printf (preview, _("Seek to %" TH_TIME_FORMAT),
      TH_TIME_ARGS (preview->priv->seek_pos));

  g_get_current_time (&preview->priv->last_seek_to_msg);

  if (preview->priv->seek_id)
    g_source_remove (preview->priv->seek_id);

  preview->priv->seek_id =
      g_timeout_add (500, (GSourceFunc) th_title_preview_seek_delayed, preview);
}

static void
th_title_preview_next (ThTitlePreview * preview, GtkWidget * button)
{
  if (preview->priv->cur_title >= preview->priv->num_titles)
    return;

  ++preview->priv->cur_title;

  th_log ("Seeking to next title %u", preview->priv->cur_title);

  /* seek counts titles from 0 */
  gst_element_seek_simple (preview->priv->pipeline,
      gst_format_get_by_nick ("title"),
      GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT,
      preview->priv->cur_title - 1);
  gst_element_seek_simple (preview->priv->pipeline, GST_FORMAT_TIME,
      GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, 0);

  /* force update even if there was a recent seek to message */
  g_get_current_time (&preview->priv->last_seek_to_msg);
  g_time_val_add (&preview->priv->last_seek_to_msg, -10 * 1000 * 1000);
}

static void
th_title_preview_prev (ThTitlePreview * preview, GtkWidget * button)
{
  GstFormat format;
  gint64 pos = 0;

  if (preview->priv->cur_title <= 1)
    return;

  format = GST_FORMAT_TIME;
  if (gst_element_query_position (preview->priv->pipeline, &format, &pos)) {
    if (pos > 0 && pos < (2 * GST_SECOND)) {
      gst_element_seek_simple (preview->priv->pipeline, GST_FORMAT_TIME,
          GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, 0);
      return;
    }
  }

  --preview->priv->cur_title;

  th_log ("Seeking to previous title %u", preview->priv->cur_title);

  /* seek counts titles from 0 */
  gst_element_seek_simple (preview->priv->pipeline,
      gst_format_get_by_nick ("title"),
      GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT,
      preview->priv->cur_title - 1);
  gst_element_seek_simple (preview->priv->pipeline, GST_FORMAT_TIME,
      GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT, 0);

  /* force update even if there was a recent seek to message */
  g_get_current_time (&preview->priv->last_seek_to_msg);
  g_time_val_add (&preview->priv->last_seek_to_msg, -10 * 1000 * 1000);
}

static void
th_title_preview_play_pause (ThTitlePreview * preview, GtkWidget * button)
{
  GstStateChangeReturn ret;
  GstState cur_state;

  ret = gst_element_get_state (preview->priv->pipeline, &cur_state, NULL, 0);
  if (ret == GST_STATE_CHANGE_SUCCESS) {
    switch (cur_state) {
        case GST_STATE_PLAYING:
          gst_element_set_state (preview->priv->pipeline, GST_STATE_PAUSED);
          break;
        case GST_STATE_PAUSED:
          gst_element_set_state (preview->priv->pipeline, GST_STATE_PLAYING);
          break;
        default:
          break;
    }
  }
}

static void
th_title_preview_update_widget_state (ThTitlePreview * preview)
{
  GstStateChangeReturn ret;
  GstState cur_state;

  ret = gst_element_get_state (preview->priv->pipeline, &cur_state, NULL, 0);
  if (ret == GST_STATE_CHANGE_SUCCESS) {
    if (cur_state == GST_STATE_PLAYING) {
      gtk_image_set_from_stock (GTK_IMAGE (preview->priv->play_pause_img),
          GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_BUTTON);
    } else {
      gtk_image_set_from_stock (GTK_IMAGE (preview->priv->play_pause_img),
          GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_BUTTON);
    }

    if (cur_state < GST_STATE_PAUSED) {
      th_title_preview_statusbar_printf (preview, "%s", "");
    }
    gtk_widget_set_sensitive (preview->priv->prev_button, TRUE);
    gtk_widget_set_sensitive (preview->priv->next_button, TRUE);
    gtk_widget_set_sensitive (preview->priv->play_pause_button, TRUE);

    if (preview->priv->cur_title <= 1 || preview->priv->num_titles == 0)
      gtk_widget_set_sensitive (preview->priv->prev_button, FALSE);

    if (preview->priv->cur_title >= preview->priv->num_titles)
      gtk_widget_set_sensitive (preview->priv->next_button, FALSE);
  } else if (ret == GST_STATE_CHANGE_FAILURE) {
    gtk_widget_set_sensitive (preview->priv->prev_button, FALSE);
    gtk_widget_set_sensitive (preview->priv->next_button, FALSE);
    gtk_widget_set_sensitive (preview->priv->play_pause_button, FALSE);
  } else {
    /* if async, leave widget state alone, probably seeking */
  }
}

/* FIXME: maybe use gconfvideosink instead? */
#define PREVIEW_PIPELINE \
  "dvdreadsrc name=src "                 \
  " ! dvddemux "                         \
  " ! mpeg2dec"             \
  " ! queue max-size-time=3000000000 "   \
  " ! ffmpegcolorspace "                 \
  " ! videoscale "                       \
  " ! autovideosink"

GtkWidget  *
th_title_preview_new (const gchar * device, const gchar * title_tag,
    guint title, GError ** error)
{
  ThTitlePreview *preview;
  GtkWidget *glade_window = NULL;
  GtkWidget *toplevel_vbox;
  GstElement *pipeline;
  GstBus *bus;

  preview = (ThTitlePreview *) g_object_new (TH_TYPE_TITLE_PREVIEW,
      "type", GTK_WINDOW_TOPLEVEL, NULL);

  if (!th_utils_ui_load_interface ("th-title-preview.glade", FALSE,
      "title-preview-dialog", &glade_window, "toplevel-vbox",
      &toplevel_vbox, "picture-area", &preview->priv->picture_area,
      "next-button", &preview->priv->next_button, "prev-button",
      &preview->priv->prev_button, "play-pause-button",
      &preview->priv->play_pause_button, "play-pause-image",
      &preview->priv->play_pause_img, "seek-slider",
      &preview->priv->seek_slider, "statusbar", &preview->priv->statusbar,
      "close-button", &preview->priv->close_button, NULL))
  {
    g_printerr ("th_utils_ui_load_interface (\"%s\") failed.\n",
        "th-title-preview.glade");
    if (glade_window)
      gtk_widget_destroy (glade_window);
    gtk_widget_destroy (GTK_WIDGET (preview));
    return NULL;
  }
	
  g_object_ref (toplevel_vbox);
  gtk_container_remove (GTK_CONTAINER (glade_window), toplevel_vbox);
  gtk_container_add (GTK_CONTAINER (preview), toplevel_vbox);
  g_object_unref (toplevel_vbox);
  gtk_widget_destroy (glade_window);

  if (title_tag == NULL)
    title_tag = "";

  pipeline = gst_parse_launch (PREVIEW_PIPELINE, error);

  if (pipeline == NULL || (error && *error != NULL)) {
    if (pipeline)
      gst_object_unref (pipeline);
    gtk_widget_destroy (GTK_WIDGET (preview));
    return NULL;
  }

  /* we don't specify the device in the pipeline above to avoid escaping
   * issues or issues with spaces when we are using a local directory path
   * with an .iso image or a directory instead of a DVD device */
  th_bin_set_child_properties (GST_BIN (pipeline), "src",
       "device", device, "title", title, NULL);

  preview->priv->cur_title = title;
  preview->priv->num_titles = 0;

  gtk_window_set_default_size (GTK_WINDOW (preview), PREVIEW_DEFAULT_WIDTH,
      PREVIEW_DEFAULT_HEIGHT);
	
  preview->priv->pipeline = pipeline;

  bus = GST_ELEMENT_BUS (pipeline);

  gst_bus_set_sync_handler (bus, gst_bus_sync_signal_handler, preview);

  g_signal_connect_swapped (bus, "sync-message::element",
      G_CALLBACK (th_title_preview_sync_msg), preview);

  gst_bus_add_signal_watch (bus);

  g_signal_connect_swapped (bus, "message",
      G_CALLBACK (th_title_preview_async_msg), preview);

  g_signal_connect_swapped (preview->priv->picture_area, "realize",
      G_CALLBACK (th_title_preview_picture_area_realize), preview);

  preview->priv->slider_value_changed_id =
    g_signal_connect_swapped (preview->priv->seek_slider, "value-changed",
        G_CALLBACK (th_title_preview_slider_value_changed), preview);

  th_title_preview_update_widget_state (preview);

  preview->priv->statusbar_id =
      gtk_statusbar_get_context_id (GTK_STATUSBAR (preview->priv->statusbar),
          "current-playback-position");

  g_signal_connect_swapped (preview->priv->close_button, "clicked",
      G_CALLBACK (gtk_widget_destroy), preview);

  g_signal_connect_swapped (preview->priv->prev_button, "clicked",
      G_CALLBACK (th_title_preview_prev), preview);

  g_signal_connect_swapped (preview->priv->next_button, "clicked",
      G_CALLBACK (th_title_preview_next), preview);

  g_signal_connect_swapped (preview->priv->play_pause_button, "clicked",
      G_CALLBACK (th_title_preview_play_pause), preview);

  return GTK_WIDGET (preview);
}
