/***************************************************************************
                             th-utils.c
                             ----------
    begin                : Tue Apr 06 2004
    copyright            : (C) 2004 by Tim-Philipp Müller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-utils.h"

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>

#ifdef HAVE_MNTENT_H
# include <mntent.h>
#endif

#ifdef HAVE_SYS_MNTTAB_H
# include <sys/mnttab.h>
#endif

#ifdef HAVE_SYS_STATFS_H
#include <sys/statfs.h>
#endif

#ifdef HAVE_SYS_STATVFS_H
# include <sys/statvfs.h>
#endif

#ifdef HAVE_VALGRIND
# include <valgrind/valgrind.h>
#endif

#ifdef HAVE_LIBINTL_H
# include <libintl.h> /* for dcgettext() */
#endif

#include <gst/gst.h>
#include <glade/glade.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#ifdef G_OS_WIN32
# include <Windows.h>

/* Mingw is missing declaration of NAME_MAX */
/* so here is small hack */
# ifndef NAME_MAX
#  define NAME_MAX FILENAME_MAX
# endif
#endif

#define TH_UTILS_ERROR  g_quark_from_static_string("thoggen-utils-error")

static gpointer        gui_thread;          /* NULL */
static gpointer        log_text_buffer;     /* NULL */
static GAsyncQueue    *pending_log_lines;   /* NULL */
static GHashTable     *iso_639_ht;          /* NULL */

/***************************************************************************
 *
 *   utils_free_iso_639_hashtable
 *
 ***************************************************************************/

static void
utils_free_iso_639_hashtable (void)
{
	g_hash_table_destroy (iso_639_ht);
	iso_639_ht = NULL;
}

/***************************************************************************
 *
 *   utils_init_iso_639_parse_start_tag
 *
 ***************************************************************************/

static void
utils_init_iso_639_parse_start_tag (GMarkupParseContext *ctx,
                                    const gchar         *element_name,
                                    const gchar        **attr_names,
                                    const gchar        **attr_values,
                                    gpointer             data,
                                    GError             **error)
{
	const gchar *ccode, *lang_name;

	if (!g_str_equal (element_name, "iso_639_entry")
	 || attr_names == NULL
	 || attr_values == NULL)
		return;

	ccode = NULL;
	lang_name = NULL;

	while (*attr_names && *attr_values)
	{
		if (g_str_equal (*attr_names, "iso_639_1_code"))
		{
			/* skip if empty */
			if (**attr_values)
			{
				g_return_if_fail (strlen (*attr_values) == 2);
				ccode = *attr_values;
			}
		}
		else if (g_str_equal (*attr_names, "name"))
		{
			lang_name = *attr_values;
		}

		++attr_names;
		++attr_values;
	}

	if (ccode && lang_name)
	{
		/* th_log ("iso_639_ht: added %s => %s\n", ccode, lang_name); */

		g_hash_table_insert (iso_639_ht, 
		                     g_strdup (ccode), 
		                     g_strdup (lang_name));
	}
}

/***************************************************************************
 *
 *   utils_init_iso_639_hashtable
 *
 ***************************************************************************/

static void
utils_init_iso_639_hashtable (void)
{
	GError  *err = NULL;
	gchar   *buf;
	gsize    buf_len;

	iso_639_ht = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

	g_atexit (utils_free_iso_639_hashtable);

	if (g_file_get_contents (ISO_639_XML_PATH, &buf, &buf_len, &err))
	{
		GMarkupParseContext *ctx;
		GMarkupParser        parser = { utils_init_iso_639_parse_start_tag, NULL, NULL, NULL, NULL };
		
		ctx = g_markup_parse_context_new (&parser, 0, NULL, NULL);

		if (!g_markup_parse_context_parse (ctx, buf, buf_len, &err))
		{
			g_warning ("Failed to parse '%s': %s\n", ISO_639_XML_PATH, err->message);
			g_error_free (err);
		}

		g_markup_parse_context_free (ctx);
		g_free (buf);
	}
	else
	{
		g_warning ("Failed to load '%s': %s\n", ISO_639_XML_PATH, err->message);
		g_error_free (err);
	}
}

/***************************************************************************
 *
 *   th_utils_get_language_name
 *
 *   Gets the language name from the ISO-639-1 language code
 *    (e.g. 'German' for 'de'). Returns NULL if not found.
 *
 ***************************************************************************/

const gchar *
th_utils_get_language_name (const gchar *lang)
{
	static gchar  lang_code[3];
	const gchar  *lang_name;
	
	g_return_val_if_fail (lang != NULL, NULL);
	
	if (strlen (lang) != 2)
		return NULL;
	
	if (iso_639_ht == NULL)
		utils_init_iso_639_hashtable ();

	lang_name = (const gchar*) g_hash_table_lookup (iso_639_ht, lang);

	if (lang_name)
	{
#ifdef HAVE_DCGETTEXT
		/* TODO: can we assume that there is dgettext() 
		 *  when we know that we have dcgettext()? */
		return dgettext (ISO_639_DOMAIN, lang_name);
#else
		return lang_name;
#endif
	}

	if (!g_utf8_validate (lang, -1, NULL))
		return NULL;

	/* if we have a country code but no language name,
	 *  at least return the country code - better than
	 *  nothing or 'Default language' */
	g_snprintf (lang_code, sizeof (lang_code), "%s", lang);

	return lang_code;
}

/***************************************************************************
 *
 *   utils_ui_retrieve_widget
 *
 ***************************************************************************/

static void
utils_ui_retrieve_widget (GladeXML      *xml, 
                          const gchar   *name, 
                          GtkWidget    **p_widget, 
                          gboolean      *p_allok)
{
	g_assert (GLADE_IS_XML (xml) && name && p_widget && p_allok);
	
	*p_widget = glade_xml_get_widget (xml, name);
	
	if (*p_widget == NULL)
	{
		g_warning ("Failed to load widget '%s' in '%s'\n", name, xml->filename);
		*p_allok = FALSE;
	}
	
	g_return_if_fail (GTK_IS_WIDGET (*p_widget));

//	g_print("Loaded '%s' of type '%s' ok.\n", widget_name1, g_type_name(G_OBJECT_TYPE(*p_widget1)));
}


/***************************************************************************
 *
 *   th_utils_ui_load_interface
 *
 *   returns TRUE if all went fine, FALSE on error 
 *
 *   Expects pairs of widget name string + GtkWidget ** pointers, 
 *    and a NULL parameter as terminator.
 *
 ***************************************************************************/

gboolean
th_utils_ui_load_interface (const gchar   *gladefn,  
                            gboolean       autoconnect_signals,
                            const gchar   *widget_name1,
                            GtkWidget    **p_widget1,
                            ...                        )
{
	const gchar  *name;
	GtkWidget   **p_widget;
	GladeXML    *xml;
	gboolean     allok;
	gchar       *fn;
	
	va_list  args;

	g_return_val_if_fail (gladefn      != NULL, FALSE);
	g_return_val_if_fail (widget_name1 != NULL, FALSE);
	g_return_val_if_fail (p_widget1    != NULL, FALSE);

	if (!g_path_is_absolute (gladefn))
		fn = g_build_path (G_DIR_SEPARATOR_S, DATADIR, "ui", gladefn, NULL);
	else
		fn = g_strdup (gladefn);
	
	if ((xml = glade_xml_new (fn, NULL, GETTEXT_PACKAGE)))
		goto allfine;

	/* libglade will spew a warning, we don't need to */

	/* If the path was absolute, that was it */
	if (g_path_is_absolute (gladefn) 
	 || strchr (gladefn, G_DIR_SEPARATOR))
		goto errorout;
		
	/* Try loading it from the current directory,
	 *  so thoggen runs uninstalled as well */
	if ((xml = glade_xml_new (gladefn, NULL, GETTEXT_PACKAGE)))
		goto allfine;
	
	g_free (fn);
	fn = g_strdup_printf ("./src/%s", gladefn);
	if ((xml = glade_xml_new (fn, NULL, GETTEXT_PACKAGE)))
		goto allfine;

errorout:
	
	g_free (fn);
	return FALSE;

allfine:

	allok = TRUE;

	utils_ui_retrieve_widget (xml, widget_name1, p_widget1, &allok);
	
	va_start (args, p_widget1);
	
	do
	{
		name = va_arg (args, const gchar *);
		
		if (name)
		{
			p_widget = va_arg (args, GtkWidget **);

			utils_ui_retrieve_widget (xml, name, p_widget, &allok);
		}
	} while (name != NULL);
	
	va_end (args);

	if (autoconnect_signals)
		glade_xml_signal_autoconnect (xml);

	g_object_unref (xml);
	g_free (fn);
	
	return allok;
}


/***************************************************************************
 *
 *   th_label_set_text
 *
 ***************************************************************************/

void              
th_label_set_text (gpointer label, const gchar *format, ...)
{
	va_list args;
	gchar   *s;

	g_return_if_fail (GTK_IS_LABEL (label));
	g_return_if_fail (format != NULL);

	if (format == NULL)
	{
		gtk_label_set_text ((GtkLabel*) label, "");
		return;
	}

	va_start (args, format);
	s = g_strdup_vprintf (format, args);
	va_end (args);
	
	gtk_label_set_text ((GtkLabel*) label, s);
	
	g_free (s);
}

/***************************************************************************
 *
 *   th_label_set_markup
 *
 ***************************************************************************/

void              
th_label_set_markup (gpointer label, const gchar *format, ...)
{
	va_list args;
	gchar   *s;

	g_return_if_fail (GTK_IS_LABEL (label));
	g_return_if_fail (format != NULL);

	if (format == NULL)
	{
		gtk_label_set_text ((GtkLabel*) label, "");
		return;
	}

	va_start (args, format);
	s = g_markup_vprintf_escaped (format, args);
	va_end (args);
	
	gtk_label_set_markup ((GtkLabel*) label, s);
	
	g_free (s);
}


/* these macros are adapted from videotestsrc, paint_setup_I420() */
#define ROUND_UP_2(x)  (((x)+1)&~1)
#define ROUND_UP_4(x)  (((x)+3)&~3)
#define ROUND_UP_8(x)  (((x)+7)&~7)

#define I420_Y_ROWSTRIDE(width) (ROUND_UP_4(width))
#define I420_U_ROWSTRIDE(width) (ROUND_UP_8(width)/2)
#define I420_V_ROWSTRIDE(width) ((ROUND_UP_8(I420_Y_ROWSTRIDE(width)))/2)

#define I420_Y_OFFSET(w,h) (0)
#define I420_U_OFFSET(w,h) (I420_Y_OFFSET(w,h)+(I420_Y_ROWSTRIDE(w)*ROUND_UP_2(h)))
#define I420_V_OFFSET(w,h) (I420_U_OFFSET(w,h)+(I420_U_ROWSTRIDE(w)*ROUND_UP_2(h)/2))

#define I420_SIZE(w,h) (I420_V_OFFSET(w,h)+(I420_V_ROWSTRIDE(w)*ROUND_UP_2(h)/2))

#define clip_8_bit(val)              \
{                                    \
  if (G_UNLIKELY (val < 0))          \
    val = 0;                         \
  else if (G_UNLIKELY (val > 255))   \
    val = 255;                       \
}

/***************************************************************************
 *
 *   yv12torgb
 *
 *   Code taken from:
 *   transcode Copyright (C) Thomas Oestreich - June 2001
 *   enix      enix.berlios.de
 *
 ***************************************************************************/

static guint8 *
yv12torgb (guint8 *src_y, guint8 *src_u, guint8 *src_v, int width, int height) 
{
	gint     row, j;
	guchar  *rgb;

	rgb = g_malloc0 (width * height * 3);

	for (row = 0; row < height; ++row) 
	{
	  for (j = 0; j < width; ++j) 
		{
			gint  y, u, v;
			gint  r, g, b;

			/***************************************************
			 *  Colour conversion from
			 *  http://www.poynton.com/notes/colour_and_gamma/ColorFAQ.html#RTFToC30
			 *  
			 *  Thanks to Billy Biggs <vektor@dumbterm.net>
			 *  for the pointer and the following conversion.
			 *  
			 *  R' = [ 1.1644         0    1.5960 ]   ([ Y' ]   [  16 ])
			 *  G' = [ 1.1644   -0.3918   -0.8130 ] * ([ Cb ] - [ 128 ])
			 *  B' = [ 1.1644    2.0172         0 ]   ([ Cr ]   [ 128 ])
			 *  
			 *  Where in xine the above values are represented as
			 *  Y' == image->y
			 *  Cb == image->u
			 *  Cr == image->v
			 *  
			 ***************************************************/

			/* Fixed row strides (Tim) */
			y = src_y[( row    * I420_Y_ROWSTRIDE(width)) +  j   ] - 16;
			u = src_u[((row/2) * I420_U_ROWSTRIDE(width)) + (j/2)] - 128;
			v = src_v[((row/2) * I420_V_ROWSTRIDE(width)) + (j/2)] - 128;

			r = (1.1644 * y) + (1.5960 * v);
			g = (1.1644 * y) - (0.3918 * u) - (0.8130 * v);
			b = (1.1644 * y) + (2.0172 * u);

			clip_8_bit (r);
			clip_8_bit (g);
			clip_8_bit (b);

			rgb[(row * width + j) * 3 + 0] = r;
			rgb[(row * width + j) * 3 + 1] = g;
			rgb[(row * width + j) * 3 + 2] = b;
		}
	}

	return rgb;
}

/***************************************************************************
 *
 *   th_pixbuf_from_yuv_i420_buffer
 *
 *   FIXME: this code makes ceratin assumptions about the row stride, maybe
 *          we should double-check our assumptions against the buffer size?
 *
 ***************************************************************************/

GdkPixbuf *
th_pixbuf_from_yuv_i420_buffer (GstBuffer *buf, guint width, guint height)
{
	GdkPixbuf *pixbuf;
	guint8    *rgb;

	g_return_val_if_fail (GST_IS_BUFFER (buf), NULL);
	g_return_val_if_fail (width > 0, NULL);
	g_return_val_if_fail (height > 0, NULL);

	/* probably happens if the rowstrides are different */
	g_return_val_if_fail (GST_BUFFER_SIZE (buf) >= I420_SIZE(width,height), NULL);
	
	rgb = yv12torgb (GST_BUFFER_DATA (buf) + I420_Y_OFFSET (width, height), 
	                 GST_BUFFER_DATA (buf) + I420_U_OFFSET (width, height),
	                 GST_BUFFER_DATA (buf) + I420_V_OFFSET (width, height),
	                 width, height);

	pixbuf = gdk_pixbuf_new_from_data (rgb,
	                                   GDK_COLORSPACE_RGB, FALSE,
	                                   8, width, height, 3 * width,
	                                   (GdkPixbufDestroyNotify) g_free, rgb);

	return pixbuf;
}



/***************************************************************************
 *
 *   th_mkdir_recursive
 *
 *   Creates '$basedir/$addir', creating all directories leading to it
 *     if (starting from basedir!) if necessary
 *
 ***************************************************************************/

gboolean
th_mkdir_recursive (const gchar *basedir, const gchar *adddir, GError **err)
{
	GString  *dir;
	gchar   **subdirs, **pos;
	gint      ret;

	g_return_val_if_fail (basedir != NULL, FALSE);
	g_return_val_if_fail (adddir  != NULL, FALSE);

	dir = g_string_new (basedir);

	subdirs = g_strsplit (adddir, G_DIR_SEPARATOR_S, -1);

	for (pos = subdirs;  subdirs && pos && *pos;  ++pos)
	{
		g_string_append (dir, G_DIR_SEPARATOR_S);
		g_string_append (dir, *pos);

		ret = mkdir(dir->str, 00700);

		if (ret != 0  &&  errno != EEXIST)
		{
			g_set_error (err, TH_UTILS_ERROR, 0,
			             _("mkdir('%s') failed: %s"),
			             dir->str, g_strerror(errno));
			g_strfreev (subdirs);
			g_string_free (dir, TRUE);
			return FALSE;
		}
	}

	g_strfreev (subdirs);
	g_string_free (dir, TRUE);

	return TRUE;
}

/***************************************************************************
 *
 *   utils_find_mount_point
 *
 ***************************************************************************/

static gchar *
utils_find_mount_point (const gchar *fn)
{
#ifdef HAVE_MTAB
	struct mntent *mount_entry = NULL;
#elif defined(HAVE_MNTTAB)
	struct mnttab  mnt;
#endif
	struct stat    s;
	dev_t          mount_device;
	FILE          *mount_table;

	g_return_val_if_fail (fn != NULL, NULL);

	if (stat (fn, &s) != 0)
		return NULL;

	if ((s.st_mode & S_IFMT) == S_IFBLK)
		mount_device = s.st_rdev;
	else
		mount_device = s.st_dev;

#ifdef HAVE_MTAB
	mount_table = setmntent ("/etc/mtab", "r");
	if (mount_table == NULL)
	{
		g_warning (_("Could not open /etc/mtab file: %s"), g_strerror (errno));
		return NULL;
	}

	while ((mount_entry = getmntent (mount_table)))
	{
		if (g_str_equal (fn, mount_entry->mnt_dir)
		 || g_str_equal (fn, mount_entry->mnt_fsname))
			break;

		/* Match the directory's mount point. */
		if (stat (mount_entry->mnt_dir, &s) == 0 && s.st_dev == mount_device)
			break;
	}
	
	endmntent (mount_table);

	if (mount_entry)
		return g_strdup (mount_entry->mnt_dir);

#elif defined(HAVE_MNTTAB)

	mount_table  = fopen ("/etc/mnttab", "r");
	if (mount_table == NULL)
	{
		g_warning (_("Could not open /etc/mnttab file: %s"), g_strerror (errno));
		return NULL;
	}
	
	while (getmntent (mount_table, &mnt) == 0)
	{
		if (is (object, mnt.mnt_special))
		{
			fclose (mount_table);
			return g_strdup (mnt.mnt_mountp);
		}
	}
	
	fclose (mount_table);

#endif

	return NULL;
}

/***************************************************************************
 *
 *   th_utils_get_free_space_from_path
 *
 *   Returns free space on the partition where the given path is, in MB,
 *    or a value < 0.0 on error.
 *
 ***************************************************************************/

gdouble
th_utils_get_free_space_from_path (const gchar *path)
{
#ifdef HAVE_STATFS
	struct statfs filesysinfo;

	if (statfs (path, &filesysinfo) == 0)
		return (gdouble) filesysinfo.f_bavail * filesysinfo.f_bsize / (1024.0 * 1024.0);
#else
# ifdef HAVE_STATVFS
	struct statvfs filesysinfo;

	if (statvfs (path, &filesysinfo) == 0)
		return (gdouble) filesysinfo.f_bavail * filesysinfo.f_bsize / (1024.0 * 1024.0);
# endif
#endif

	return (-1.0);
}

/***************************************************************************
 *
 *   th_utils_free_space_check_init
 *
 *   Returns FALSE if not supported for some reason
 *
 ***************************************************************************/

gboolean
th_utils_free_space_check_init (GList **p_mount_point_list)
{
	gchar *mnt_point;

	g_return_val_if_fail (p_mount_point_list != NULL, FALSE);

	*p_mount_point_list = NULL;

	if (th_utils_get_free_space_from_path ("/") < 0.0)
		return FALSE;

	mnt_point = utils_find_mount_point (g_get_home_dir());
	if (mnt_point == NULL)
		return FALSE;
		
	g_free (mnt_point);

	return TRUE;
}

/***************************************************************************
 *
 *   th_utils_free_space_check_deinit
 *
 ***************************************************************************/

void
th_utils_free_space_check_deinit (GList **p_mount_point_list)
{
	GList *l;

	g_return_if_fail (p_mount_point_list != NULL);

	for (l = *p_mount_point_list; l; l = l->next)
	{
		ThMountPoint *mp = (ThMountPoint*) l->data;
		g_free (mp->mnt_point);
		g_list_foreach (mp->filenames, (GFunc) g_free, NULL);
		g_list_free (mp->filenames);
		memset (mp, 0x88, sizeof (ThMountPoint));
		g_free (mp);
	}
	
	g_list_free (*p_mount_point_list);
	*p_mount_point_list = NULL;
}

/***************************************************************************
 *
 *   utils_free_space_mp_cmp_func
 *
 ***************************************************************************/

static gint
utils_free_space_mp_cmp_func (ThMountPoint *mp, const gchar *mnt_point)
{
	return strcmp (mp->mnt_point, mnt_point);
}

/***************************************************************************
 *
 *   th_utils_free_space_check_add_filename
 *
 ***************************************************************************/

gboolean
th_utils_free_space_check_add_filename (GList        **p_mount_point_list, 
                                        const gchar   *fn, 
                                        gdouble        space_needed)
{
	GList *l;
	gchar *mnt_point, *dir;

	g_return_val_if_fail (p_mount_point_list != NULL, FALSE);
	g_return_val_if_fail (fn != NULL, FALSE);

	dir = g_path_get_dirname (fn);
	mnt_point = utils_find_mount_point (dir);
	g_free (dir);

	if (mnt_point == NULL)
		return FALSE;
	
	/* already got this mount point? */
	if ((l = g_list_find_custom (*p_mount_point_list, mnt_point, (GCompareFunc) utils_free_space_mp_cmp_func)))
	{
		ThMountPoint *mp = (ThMountPoint*) l->data;
		
		if (!g_list_find_custom (mp->filenames, fn, (GCompareFunc) strcmp))
		{
			mp->space_needed += space_needed;
			mp->filenames = g_list_append (mp->filenames, g_strdup (fn));
		}
	}
	else
	{
		ThMountPoint *mp = g_new0 (ThMountPoint, 1);
		
		mp->mnt_point = g_strdup (mnt_point);
		mp->space_needed = space_needed;
		mp->filenames = g_list_append (NULL, g_strdup (fn));
		
		*p_mount_point_list = g_list_append (*p_mount_point_list, mp);
	}
	
	g_free (mnt_point);
	
	return TRUE;
}

/***************************************************************************
 *
 *   th_utils_free_space_check
 *
 *   Returns NULL if there is enough free space for all added files, 
 *    otherwise it returns a GList of ThMountPoint structs for all mount 
 *    points where there is not enough space. The caller has to free the
 *    returned GList, but MUST NOT free the contents of that list.
 *
 ***************************************************************************/

GList *
th_utils_free_space_check (GList **p_mount_point_list)
{
	GList *l, *points = NULL;

	g_return_val_if_fail (p_mount_point_list != NULL, FALSE);

	for (l = *p_mount_point_list;  l;  l = l->next)
	{
		ThMountPoint *mp = (ThMountPoint*) l->data;
		
		mp->space_avail = th_utils_get_free_space_from_path (mp->mnt_point);
		
		if (mp->space_avail >= 0.0  &&  mp->space_avail < mp->space_needed)
			points = g_list_append (points, mp);
	}
	
	return points;
}

/***************************************************************************
 *
 *   th_utils_running_in_valgrind
 *
 *   Returns TRUE if we are running in valgrind, otherwise FALSE
 *
 ***************************************************************************/

gboolean
th_utils_running_in_valgrind (void)
{
#ifdef HAVE_VALGRIND
	return (RUNNING_ON_VALGRIND);
#else
	return FALSE;
#endif
}

/***************************************************************************
 *
 *   th_utils_vg_style_print
 *
 *   Silly function to print a comment valgrind-style 
 *    with the PID in front of each line
 *
 ***************************************************************************/

void
th_utils_vg_style_print (const gchar *txt)
{
	guint64   pid;
	gchar   **lines, **l;

	g_return_if_fail (txt != NULL);
	
	lines = g_strsplit (txt, "\n", -1);
	if (lines == NULL)
		return;
	
	pid = (guint64) getpid();

	for (l = lines; l && *l;  ++l)
	{
		g_printerr ("--%" G_GUINT64_FORMAT "-- %s\n", pid, *l);
	}
	
	g_strfreev (lines);
}

/******************************************************************************
 *
 *   th_utils_get_shaded_pixbuf
 *
 *   Returns a copy of the given pixbuf with the rectangle (x,y,w,h) 
 *    within the pixbuf being made darker than the rest
 *
 ******************************************************************************/

GdkPixbuf *
th_utils_get_shaded_pixbuf (GdkPixbuf *src, guint x, guint y, guint w, guint h, gint shift)
{
	GdkPixbuf *dest;
	guchar    *target_pixels, *original_pixels;
	guchar    *pixsrc, *pixdest;
	gint       width, height, has_alpha;
	gint       src_stride, dest_stride;
	gint       i, j;

	if (src == NULL)
		return NULL;

	width = gdk_pixbuf_get_width (src);
	height = gdk_pixbuf_get_height (src);
	has_alpha = gdk_pixbuf_get_has_alpha (src);
	src_stride = gdk_pixbuf_get_rowstride (src);

	g_return_val_if_fail (x + w <= width, NULL);
	g_return_val_if_fail (y + h <= height, NULL);
	
	dest = gdk_pixbuf_new (gdk_pixbuf_get_colorspace (src), 
	                       gdk_pixbuf_get_has_alpha (src),
	                       gdk_pixbuf_get_bits_per_sample (src), 
	                       width, height);
	
	dest_stride = gdk_pixbuf_get_rowstride (dest);
	target_pixels = gdk_pixbuf_get_pixels (dest);
	original_pixels = gdk_pixbuf_get_pixels (src);

	for (i = 0; i < height; i++)
	{
		pixdest = target_pixels + i * dest_stride;
		pixsrc = original_pixels + i * src_stride;

		for (j = 0; j < width; j++)
		{
			if (i >= y && i <= y+h && j >= x && j <= x+w)
			{
				guchar r, g, b;
				r = *(pixsrc++);
				g = *(pixsrc++);
				b = *(pixsrc++);
				*(pixdest++) = (guchar) ((gint) CLAMP ((gint) r + shift, 0, 255));
				*(pixdest++) = (guchar) ((gint) CLAMP ((gint) g + shift, 0, 255));
				*(pixdest++) = (guchar) ((gint) CLAMP ((gint) b + shift, 0, 255));
			}
			else
			{
				*(pixdest++) = *(pixsrc++);
				*(pixdest++) = *(pixsrc++);
				*(pixdest++) = *(pixsrc++);
			}
			
			if (has_alpha)
				*(pixdest++) = *(pixsrc++);
		}
	}
	
	return dest;
}

/******************************************************************************
 *
 *   th_utils_device_might_be_dvd_drive
 *
 ******************************************************************************/

gboolean
th_utils_device_might_be_dvd_drive (const gchar *device, GError **err)
{
	struct stat  s;
	gint         fd;
	
	g_return_val_if_fail (device != NULL, FALSE);

	if ((fd = open (device, O_RDONLY | O_NONBLOCK)) < 0)
	{
		g_set_error (err, TH_UTILS_ERROR, 0,
		             _("Failed to open dvd device '%s': %s"),
		             device, g_strerror (errno));
		return FALSE;
	}
	
	if (fstat (fd, &s) != 0)
	{
		g_set_error (err, TH_UTILS_ERROR, 0,
		             _("Failed to fstat device '%s': %s"),
		             device, g_strerror (errno));
		close (fd);
		return FALSE;
	}
	
	close (fd);

	if (!S_ISBLK (s.st_mode))
	{
		g_set_error (err, TH_UTILS_ERROR, 0,
		             _("Device '%s' is not a block device."), 
		             device);
		return FALSE;
	}
	
	return TRUE;
}


/******************************************************************************
 *
 *   th_utils_invoke_browser_with_url
 *
 ******************************************************************************/

#if !defined(G_OS_WIN32)

gboolean
th_utils_invoke_browser_with_url (const gchar *url)
{
	const gchar  *browsers[] =
	{
#ifdef MACOSX
		"safari", "chimera",
#endif
		"epiphany", "konqueror", "firefox", "mozilla-bin", 
		"galeon", "opera", "netscape", "phoenix"
	};
	
	const gchar  *envbrowser, *argv[3];
	gchar        *browserpath = NULL;
	GError       *error = NULL;
	guint         i;

	g_return_val_if_fail (url != NULL, FALSE);

	envbrowser = g_getenv ("BROWSER");
	
	/* check whether $BROWSER environment variable is set (thanks to Yorzik for this) */
	if (envbrowser)
		th_log ("Invoking browser. Environment variable $BROWSER is set.\n");
	else
		th_log ("Invoking browser. Environment variable $BROWSER is not set.\n");

	if (envbrowser)
	{
		browserpath = g_find_program_in_path (envbrowser);

		if (browserpath == NULL)
			g_warning ("Environment variable $BROWSER contains invalid path to browser binary '%s'.\n", envbrowser);
	}

	for (i = 0; browserpath == NULL && i < G_N_ELEMENTS (browsers); ++i)
	{
		browserpath = g_find_program_in_path (browsers[i]);
	}
	
	if (browserpath == NULL)
		return FALSE;

	argv[0] = browserpath;
	argv[1] = url;
	argv[2] = NULL;

	if (g_spawn_async (NULL, (gchar**) argv, NULL,
	      G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL, 
	      NULL, NULL, NULL, &error) == FALSE )
	{
		g_warning ("Starting browser failed: %s\n", error->message);
		g_error_free (error);
		g_free (browserpath);
		return FALSE;
	}

	g_free (browserpath);
	return TRUE;
}

#else /* !G_OS_WIN32 */

/******************************************************************************
 *
 *   th_utils_invoke_browser_with_url
 *
 ******************************************************************************/

gboolean
th_utils_invoke_browser_with_url (const gchar *url)
{
	gint ret;

	g_return_val_if_fail (url != NULL, FALSE);
		
	ret = (gint) ShellExecute (NULL, "open", url, NULL, NULL, 0);
	
	if (ret <= 32) 
	{
		g_warning ("Couldn't start browser with URL '%s'.\n", url);
		return FALSE;
	}
	
	return TRUE;
}

#endif /* !G_OS_WIN32 */

/******************************************************************************
 *
 *   th_log
 *
 ******************************************************************************/

#define THOGGEN_LOG_DOMAIN "thoggen"

void
th_log (const gchar *format, ...)
{
	va_list args;
	gchar   *s;

	g_return_if_fail (format != NULL);

	va_start (args, format);
	s = g_strdup_vprintf (format, args);
	va_end (args);
	
	g_log (THOGGEN_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s", s);
	g_free (s);
}



/******************************************************************************
 *
 *   th_utils_get_plugin_rank
 *
 *   Queries the rank of a plugin feature, or -1 if not found.
 *
 ******************************************************************************/

gint
th_utils_get_plugin_rank (const gchar *plugin_name,  /* e.g. 'gstelements' */
                          const gchar *feature_name) /* e.g. 'filesrc'     */
{
	GstPluginFeature *plugin_feature;
	GstRegistry      *registry;
	gint              rank = -1;

	g_return_val_if_fail (plugin_name != NULL, -1);
	g_return_val_if_fail (feature_name != NULL, -1);

	registry = gst_registry_get_default ();
	plugin_feature = gst_registry_lookup_feature (registry, feature_name);
	if (plugin_feature)
	{
		rank = (gint) gst_plugin_feature_get_rank (plugin_feature);
		gst_object_unref (plugin_feature);
	}
	
	return rank;
}

/******************************************************************************
 *
 *   th_utils_get_human_size_str
 *
 ******************************************************************************/

gchar *
th_utils_get_human_size_str (guint64 size_in_bytes)
{
	if (size_in_bytes < 1024)
		return g_strdup_printf (_("%u bytes"), (guint) size_in_bytes);
	
	if (size_in_bytes < (1024*1024))
		return g_strdup_printf (_("%u kB"), (guint) (size_in_bytes / 1024));

	return g_strdup_printf (_("%.1f MB"), size_in_bytes / (1024.0 * 1024.0));
}


/***************************************************************************
 *
 *   cleanup_log_text_buffer
 *
 ***************************************************************************/

static void
cleanup_log_text_buffer (void)
{
	g_object_unref (log_text_buffer);
}

/***************************************************************************
 *
 *   create_log_text_buffer
 *
 ***************************************************************************/

static void
create_log_text_buffer (void)
{
	GtkTextTagTable *ttt;
	GtkTextTag      *tag;

	gui_thread = g_thread_self ();

	log_text_buffer = gtk_text_buffer_new (NULL);
	ttt = gtk_text_buffer_get_tag_table (log_text_buffer);
	tag = gtk_text_tag_new ("log-tag");

	g_object_set (tag, 
	              "family", "Monospace", 
	              "family-set", TRUE, 
	              NULL);

	gtk_text_tag_table_add (ttt, tag);
	g_object_unref (tag);

	g_atexit (cleanup_log_text_buffer);

	g_object_add_weak_pointer (G_OBJECT (log_text_buffer), &log_text_buffer);

	pending_log_lines = g_async_queue_new ();
}

static gboolean
process_pending_log_lines (gpointer data)
{
  gchar *t;

  if ((t = g_async_queue_try_pop (pending_log_lines))) {
    th_log_add_line (t);
    g_free (t);
  }

  return (g_async_queue_length (pending_log_lines) > 0);
}

/******************************************************************************
 *
 *   th_log_add_line
 *
 ******************************************************************************/

void
th_log_add_line (const gchar *txt)
{
	if (txt && *txt)
	{
		GtkTextIter end;
		gsize len;

		/* if a GStreamer element throws a g_warning or something,
		 * we might get called from one of the streaming threads,
		 * ie. in a thread context we really shouldn't be doing any
		 * Gtk/GUI things whatsoever; in this case, just queue the
		 * message for later adding to the buffer */
		if (log_text_buffer != NULL && gui_thread != g_thread_self ()) {
		  g_async_queue_push (pending_log_lines, g_strdup (txt));
		  g_idle_add (process_pending_log_lines, NULL);
		  return;
		}

		if (log_text_buffer == NULL)
			create_log_text_buffer ();

		gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (log_text_buffer), &end);

		len = strlen (txt);
		if (g_utf8_validate (txt, len, NULL))
		{
			gtk_text_buffer_insert_with_tags_by_name (GTK_TEXT_BUFFER (log_text_buffer), &end,
			                                          txt, len, "log-tag", NULL);
		}
		else
		{
			gtk_text_buffer_insert_with_tags_by_name (GTK_TEXT_BUFFER (log_text_buffer), &end,
			                                          "[skipped invalid UTF-8]\n", -1, "log-tag", NULL);
		}
	}
}

/******************************************************************************
 *
 *   log_window_scroll_to_end
 *
 ******************************************************************************/

static gboolean
log_window_scroll_to_end (GtkTextView *view)
{
	GtkTextIter end;

	g_return_val_if_fail (GTK_IS_TEXT_VIEW (view), FALSE);

	gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (log_text_buffer), &end);

	gtk_text_view_scroll_to_iter (view, &end, 0.0, TRUE, 0.0, 1.0);

	return FALSE; /* do not call again */
}

/******************************************************************************
 *
 *   copy_log_to_clipboard
 *
 ******************************************************************************/

static void
copy_log_to_clipboard (GtkButton *copy_button, GtkTextView *view)
{
	GtkClipboard *clipboard;
	GtkTextIter   start, end;
	gchar        *txt;

	gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (log_text_buffer), &start, &end);

	txt = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (log_text_buffer), &start, &end, FALSE);
	
	clipboard = gtk_clipboard_get (gdk_atom_intern ("CLIPBOARD", TRUE));
	g_return_if_fail (clipboard != NULL);

	gtk_clipboard_set_text (clipboard, txt, strlen (txt));

	th_log ("Saved log to clipboard.\n");
	log_window_scroll_to_end (view);

	g_free (txt);
}

/******************************************************************************
 *
 *   save_log_to_file
 *
 ******************************************************************************/

static void
save_log_to_file (GtkButton *saveas_button, GtkWindow *logwin)
{
	GtkTextView *textview;
	GtkWidget   *fc;
	GDate       *date;
	gchar        datestr[128];

	date = g_date_new ();
	g_date_set_time (date, time (NULL));
	g_date_strftime (datestr, sizeof (datestr), "thoggen-debug-log-%Y-%m-%d.txt", date);
	g_date_free (date);

	fc = gtk_file_chooser_dialog_new (NULL, /* "Save Log File As ..." */
	                                  GTK_WINDOW (logwin),
	                                  GTK_FILE_CHOOSER_ACTION_SAVE, 
	                                  GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
	                                  NULL);

	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (fc), g_get_home_dir());
	gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (fc), datestr);

	if (gtk_dialog_run (GTK_DIALOG (fc)) == GTK_RESPONSE_ACCEPT)
	{
		GtkTextIter  start, end;
		gchar       *fn, *txt;
		FILE        *f;

		fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (fc));

		gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (log_text_buffer), &start, &end);

		txt = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (log_text_buffer), &start, &end, FALSE);

		f = fopen (fn, "w");
		if (f != NULL)
		{
			if (fwrite (txt, strlen (txt), 1, f) != 1)
				th_log ("Could not write log to '%s': %s\n", fn, g_strerror (errno));
			else
				th_log ("Wrote log to '%s'.\n", fn);

			fclose (f);
		}
		else th_log ("Could not open '%s' for writing: %s\n", fn, g_strerror (errno));

		g_free (txt);
		g_free (fn);
	}
	
	gtk_widget_destroy (fc);

	textview = GTK_TEXT_VIEW (g_object_get_data (G_OBJECT (logwin), "log-text-view"));
	log_window_scroll_to_end (textview);
}


/******************************************************************************
 *
 *   th_log_show
 *
 ******************************************************************************/

void
th_log_show (void)
{
	GtkWidget *win, *scrollwin, *textview, *align, *hsep;
	GtkWidget *vbox, *bbox, *bbhbox;
	GtkWidget *close_button, *copy_button, *saveas_button;

	if (log_text_buffer == NULL)
		create_log_text_buffer ();

	win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (win, 480, 320);

	align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_container_add (GTK_CONTAINER (win), align);
	gtk_alignment_set_padding (GTK_ALIGNMENT (align), 12, 12, 12, 12);

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (align), vbox);

	scrollwin = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (vbox), scrollwin, TRUE, TRUE, 0);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrollwin),
	                                     GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollwin),
	                                GTK_POLICY_NEVER,
	                                GTK_POLICY_AUTOMATIC);

	textview = gtk_text_view_new_with_buffer (GTK_TEXT_BUFFER (log_text_buffer));
	gtk_container_add (GTK_CONTAINER (scrollwin), textview);
	g_object_set_data (G_OBJECT (win), "log-text-view", textview);

	hsep = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (vbox), hsep, FALSE, FALSE, 0);

	bbhbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), bbhbox, FALSE, FALSE, 0);

	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_end (GTK_BOX (bbhbox), bbox, TRUE, TRUE, 0);
	gtk_box_set_spacing (GTK_BOX (bbox), 12);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_END);

	copy_button = gtk_button_new_from_stock (GTK_STOCK_COPY);
	gtk_container_add (GTK_CONTAINER (bbox), copy_button);
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (bbox), copy_button, TRUE);

	g_signal_connect (copy_button, "clicked", 
	                  G_CALLBACK (copy_log_to_clipboard),
	                  textview);

	saveas_button = gtk_button_new_from_stock (GTK_STOCK_SAVE_AS);
	gtk_container_add (GTK_CONTAINER (bbox), saveas_button);
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (bbox), saveas_button, TRUE);

	g_signal_connect (saveas_button, "clicked", 
	                  G_CALLBACK (save_log_to_file),
	                  win);

	close_button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
	gtk_container_add (GTK_CONTAINER (bbox), close_button);

	g_signal_connect_swapped (close_button, "clicked", 
	                          G_CALLBACK (gtk_widget_destroy),
	                          win);

	/* scroll to end */
	g_idle_add ((GSourceFunc) log_window_scroll_to_end, textview);

	gtk_widget_show_all (win);
}

/***************************************************************************
 *
 *   get_user_foo_fn
 *
 ***************************************************************************/

typedef const gchar * (*ThGetUserFooDirFunc) (void);

static gchar *
get_user_foo_fn (const gchar *filename, ThGetUserFooDirFunc g_get_user_foo_dir)
{
	GError *err = NULL;
	gchar  *fn, *dir;

	g_return_val_if_fail (filename != NULL, NULL);
	g_return_val_if_fail (strchr (filename, '/') == NULL, NULL);

	fn = g_strdup_printf ("%s/thoggen/%s", g_get_user_foo_dir(), filename);
	dir = g_strdup_printf ("%s/thoggen", g_get_user_foo_dir());

	if (!th_mkdir_recursive ("/", dir, &err))
	{
		th_log ("th_mkdir_recursive (\"/\", %s) failed: %s", dir, err->message);
		g_error_free (err);
	}

	g_free (dir);

	return fn;
}

/***************************************************************************
 *
 *   th_get_user_data_fn
 *
 ***************************************************************************/

gchar *
th_get_user_data_fn (const gchar *filename)
{
	return get_user_foo_fn (filename, g_get_user_data_dir);
}

/***************************************************************************
 *
 *   th_get_user_cache_fn
 *
 ***************************************************************************/

gchar *
th_get_user_cache_fn (const gchar *filename)
{
	return get_user_foo_fn (filename, g_get_user_cache_dir);
}

/***************************************************************************
 *
 *   th_get_user_config_fn
 *
 ***************************************************************************/

gchar *
th_get_user_config_fn (const gchar *filename)
{
	return get_user_foo_fn (filename, g_get_user_config_dir);
}

/***************************************************************************
 *
 *   th_bin_set_child_properties
 *
 ***************************************************************************/

void
th_bin_set_child_properties (GstBin *bin, const gchar *child_name,
                             const gchar *prop1_name, ...)
{
	GstElement *child;
	va_list     args;

	g_return_if_fail (GST_IS_BIN (bin));
	g_return_if_fail (child_name != NULL);
	g_return_if_fail (prop1_name != NULL);

	child = gst_bin_get_by_name (bin, child_name);
	if (child == NULL) {
		g_warning ("Pipeline contains no child named '%s'", child_name);
		g_return_if_fail (child != NULL);
	}

	va_start (args, prop1_name);
	g_object_set_valist (G_OBJECT (child), prop1_name, args);
	va_end (args);

	gst_object_unref (child);
}

/***************************************************************************
 *
 *   th_simplify_fraction
 *
 ***************************************************************************/

void
th_simplify_fraction (guint * num, guint * denom)
{
  guint a, b, tmp;

  a = *num;
  b = *denom;
  while (a) {    /* find greatest common divisor */
    tmp = a;
    a = b % tmp;
    b = tmp;
  }
  *num = *num / b;
  *denom = *denom / b;
}

