/***************************************************************************
                            th-device-pool.c
                            ----------------
    begin                : Mon May 17 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   The device pool connects to the HAL daemon via the d-bus system       *
 *    message bus (using libhal), and receives notifications when devices  *
 *    are added to the system (hot-pluggable CD drives, IPods etc.) or     *
 *    have been removed, or a device's status has changed. These           *
 *    notifications are then passed on to all registered device trackers.  *
 *                                                                         *
 *   We do things this way so we don't have to use multiple connections    *
 *    when multiple objects watch for different types of devices (and also *
 *    you never know what this kind of modularity is good for later)       *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <libhal.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "th-device-pool.h"

#include <string.h>

struct _ThDevicePoolPrivate 
{
	DBusConnection *dbus_conn;
	LibHalContext  *hal_ctx;
	GList          *trackers;
	
	gulong          hal_reconnect_id;
};

static void           device_pool_class_init      (ThDevicePoolClass *klass);

static void           device_pool_instance_init   (ThDevicePool *dp);

static void           device_pool_finalize        (GObject *object);

static void           device_pool_send_initial_state (ThDevicePool *pool, ThDeviceTracker *tracker);

static gboolean       dp_reconnect_to_hal_cb         (ThDevicePool *pool);

static void           dp_reconnect_set_hal_ctx       (ThDevicePool *pool, LibHalContext *ctx);

/* HAL callbacks */

static void           dp_hal_main_loop_hook_cb   (LibHalContext *ctx, 
                                                  DBusConnection *dbus_conn);

static void           dp_hal_device_added_cb     (LibHalContext *ctx, 
                                                  const char    *udi);

static void           dp_hal_device_removed_cb   (LibHalContext *ctx, 
                                                  const char    *udi);

static void           dp_hal_device_new_cap_cb   (LibHalContext *ctx, 
                                                  const char    *udi, 
                                                  const char    *cap);

static void           dp_hal_device_lost_cap_cb  (LibHalContext *ctx, 
                                                  const char    *udi, 
                                                  const char    *cap);

static void           dp_hal_device_prop_mod_cb  (LibHalContext *ctx, 
                                                  const char    *udi, 
                                                  const char    *key, 
                                                  dbus_bool_t    is_removed, 
                                                  dbus_bool_t    is_added);

#ifndef USE_NEW_DBUS_HAL_API

static void           dp_hal_device_condition_cb (LibHalContext *ctx, 
                                                  const char    *udi, 
                                                  const char    *cond_name, 
                                                  DBusMessage   *msg);
#else

static void           dp_hal_device_condition_cb (LibHalContext *ctx, 
                                                  const char    *udi, 
                                                  const char    *cond_name, 
                                                  const char    *cond_detail);

#endif

/* variables */

static GObjectClass  *dp_parent_class;          /* NULL */


/***************************************************************************
 *
 *   disc_drive_hal_cb_funcs
 *
 ***************************************************************************/

#ifndef USE_NEW_DBUS_HAL_API
static const LibHalFunctions disc_drive_hal_cb_funcs = 
{
	dp_hal_main_loop_hook_cb,  /* main loop integration    */
	dp_hal_device_added_cb,    /* device added             */
	dp_hal_device_removed_cb,  /* device removed           */
	dp_hal_device_new_cap_cb,  /* device new capability    */
	dp_hal_device_lost_cap_cb, /* device lost capability   */
	dp_hal_device_prop_mod_cb, /* device property modified */
	dp_hal_device_condition_cb /* device condiiton         */
};
#endif /* USE_NEW_DBUS_HAL_API */

/***************************************************************************
 *
 *   hal_ctx_get_device_pool
 *
 ***************************************************************************/

static ThDevicePool *
hal_ctx_get_device_pool (LibHalContext *ctx)
{
	return TH_DEVICE_POOL (hal_ctx_get_user_data (ctx));
}

/***************************************************************************
 *
 *   device_pool_class_init
 *
 ***************************************************************************/

static void
device_pool_class_init (ThDevicePoolClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS(klass);
	
	dp_parent_class = g_type_class_peek_parent(klass);

	object_class->finalize = device_pool_finalize;
}


/***************************************************************************
 *
 *   dp_reconnect_set_hal_ctx
 *
 ***************************************************************************/

static void
dp_reconnect_set_hal_ctx (ThDevicePool *pool, LibHalContext *ctx)
{
	GList *l;

	if (pool->priv->hal_ctx == ctx)
		return;

#if 0
	/* causes crash on dbus/hald restart; dbus 0.22-3, hal 0.4.2-1 */
		if (pool->priv->hal_ctx)
			hal_shutdown (pool->priv->hal_ctx); 
#endif
	
	pool->priv->hal_ctx = ctx;
	
	if (ctx)
	{
		hal_ctx_set_user_data (pool->priv->hal_ctx, pool);

		for (l = pool->priv->trackers;  l;  l = l->next)
		{
			th_device_tracker_set_context (TH_DEVICE_TRACKER (l->data), ctx);
			device_pool_send_initial_state (pool, TH_DEVICE_TRACKER (l->data));
		}
	}
}

/***************************************************************************
 *
 *   device_pool_instance_init
 *
 ***************************************************************************/

static void
device_pool_instance_init (ThDevicePool *pool)
{
	LibHalContext *new_hal_ctx;

	pool->priv = g_new0 (ThDevicePoolPrivate, 1);

#ifdef USE_NEW_DBUS_HAL_API
	new_hal_ctx = libhal_ctx_new();
	libhal_ctx_set_cache (new_hal_ctx, TRUE);
	libhal_ctx_set_device_added (new_hal_ctx, dp_hal_device_added_cb);
	libhal_ctx_set_device_removed (new_hal_ctx, dp_hal_device_removed_cb);
	libhal_ctx_set_device_new_capability (new_hal_ctx, dp_hal_device_new_cap_cb);
	libhal_ctx_set_device_lost_capability (new_hal_ctx, dp_hal_device_lost_cap_cb);
	libhal_ctx_set_device_property_modified (new_hal_ctx, dp_hal_device_prop_mod_cb);
	libhal_ctx_set_device_condition (new_hal_ctx, dp_hal_device_condition_cb);

	{
		DBusConnection *conn = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
		if (conn == NULL)
		{
			g_warning ("dbus_bus_get(DBUS_BUS_SYSTEM) failed at %s!\n", G_STRLOC);
		}
		else
		{
			libhal_ctx_set_dbus_connection (new_hal_ctx, conn);
			dp_hal_main_loop_hook_cb (new_hal_ctx, conn);
		}
	}

	if (!libhal_ctx_init (new_hal_ctx, NULL))
		g_warning ("libhal_ctx_init() failed at %s!\n", G_STRLOC);

#else

	new_hal_ctx = hal_initialize (&disc_drive_hal_cb_funcs, TRUE);

#endif /* USE_NEW_DBUS_HAL_API */


	/* we will get a NULL context if libhal cannot connect
	 *  to the dbus message bus. we will get a valid context
	 *  though if libhal can connect to the message bus, but
	 *  the hal daemon isn't running at the moment. */

	if (new_hal_ctx)
	{
		dp_reconnect_set_hal_ctx (pool, new_hal_ctx);
	}
	else
	{
		g_warning ("Could not connect to DBus system bus (for HAL).\n");
		pool->priv->hal_reconnect_id = 
			g_timeout_add (1000, (GSourceFunc) dp_reconnect_to_hal_cb, pool);

	}
}

/***************************************************************************
 *
 *   device_pool_finalize
 *
 ***************************************************************************/

static void
device_pool_finalize (GObject *object)
{
	ThDevicePool *pool;
	GList        *l;
	
	pool = (ThDevicePool*) object;

	g_print("ThDevicePool: finalize\n");

	if (pool->priv->hal_reconnect_id > 0)
		g_source_remove (pool->priv->hal_reconnect_id);

	if (pool->priv->hal_ctx)
	{
#ifndef USE_NEW_DBUS_HAL_API
		if (hal_shutdown (pool->priv->hal_ctx) != 0)
			g_printerr ("ThDevicePool: hal_shutdown() problem!\n");
#else
		if (!libhal_ctx_shutdown (pool->priv->hal_ctx, NULL))
			g_printerr ("ThDevicePool: libhal_ctx_shutdown() problem!\n");
		libhal_ctx_free (pool->priv->hal_ctx);
#endif
	}
	
	for (l = pool->priv->trackers;  l;  l = l->next)
		g_object_unref (TH_DEVICE_TRACKER (l->data));

	g_list_free (pool->priv->trackers);

	memset (pool->priv, 0xff, sizeof (ThDevicePoolPrivate));
	g_free (pool->priv);

	/* chain up */
	dp_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_device_pool_get_type
 *
 ***************************************************************************/

GType
th_device_pool_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThDevicePoolClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) device_pool_class_init,
			NULL, NULL,
			sizeof (ThDevicePool),
			0,
			(GInstanceInitFunc) device_pool_instance_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "ThDevicePool", &info, 0);
	}

	return type;
}



/***************************************************************************
 *
 *   th_device_pool_new
 *
 ***************************************************************************/

ThDevicePool *
th_device_pool_new (void)
{
	return (ThDevicePool *) g_object_new (TH_TYPE_DEVICE_POOL, NULL);
}

/***************************************************************************
 *
 *   dp_reconnect_to_hal_cb
 *
 ***************************************************************************/

static gboolean
dp_reconnect_to_hal_cb (ThDevicePool *pool)
{
	static guint   retries; /* 0 */
	LibHalContext *new_hal_ctx;

	g_message ("Trying to reconnect to HAL...");
	
#ifdef USE_NEW_DBUS_HAL_API
	new_hal_ctx = libhal_ctx_new();
	libhal_ctx_set_cache (new_hal_ctx, TRUE);
	libhal_ctx_set_device_added (new_hal_ctx, dp_hal_device_added_cb);
	libhal_ctx_set_device_removed (new_hal_ctx, dp_hal_device_removed_cb);
	libhal_ctx_set_device_new_capability (new_hal_ctx, dp_hal_device_new_cap_cb);
	libhal_ctx_set_device_lost_capability (new_hal_ctx, dp_hal_device_lost_cap_cb);
	libhal_ctx_set_device_property_modified (new_hal_ctx, dp_hal_device_prop_mod_cb);
	libhal_ctx_set_device_condition (new_hal_ctx, dp_hal_device_condition_cb);

	{
		DBusConnection *conn = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
		if (conn == NULL)
		{
			g_warning ("dbus_bus_get(DBUS_BUS_SYSTEM) failed at %s!\n", G_STRLOC);
		}
		else
		{
			libhal_ctx_set_dbus_connection (new_hal_ctx, conn);
			dp_hal_main_loop_hook_cb (new_hal_ctx, conn);
		}
	}

	if (!libhal_ctx_init (new_hal_ctx, NULL))
		g_warning ("libhal_ctx_init() failed at %s!\n", G_STRLOC);
#else
	new_hal_ctx = hal_initialize (&disc_drive_hal_cb_funcs, TRUE);
#endif

	if (new_hal_ctx) 
	{
		g_message ("Reconnected to HAL OK.");
		dp_reconnect_set_hal_ctx (pool, new_hal_ctx);
		retries = 0;
		return FALSE;
	}

	/* Retry later if it failed. */
	if (retries++ < 100)
		return TRUE;

	/* Too many retries; clean up and bail. */
	dp_reconnect_set_hal_ctx (pool, NULL);

	return FALSE;
}

/***************************************************************************
 *
 *   dp_do_filter_dbus_msg
 *
 ***************************************************************************/

static DBusHandlerResult
dp_do_filter_dbus_msg (DBusConnection *connection, DBusMessage *message, void *data)
{
	LibHalContext *ctx = (LibHalContext*) data;

	if (dbus_message_is_signal (message,
	                            DBUS_INTERFACE_ORG_FREEDESKTOP_LOCAL,
	                            "Disconnected"))
	{
		ThDevicePool  *pool;
		
		g_message ("Disconnected from HAL.");
		
		pool = hal_ctx_get_device_pool (ctx);
		g_return_val_if_fail (TH_IS_DEVICE_POOL (pool), DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

		if (pool->priv->hal_reconnect_id > 0)
			g_source_remove (pool->priv->hal_reconnect_id);

		pool->priv->hal_reconnect_id = 
			g_timeout_add (1000, (GSourceFunc) dp_reconnect_to_hal_cb, pool);

		dbus_connection_remove_filter (connection, 
		                               dp_do_filter_dbus_msg,
		                               ctx);
		dbus_connection_unref (connection);
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	else
	{
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}

/***************************************************************************
 *
 *   dp_hal_main_loop_hook_cb
 *
 ***************************************************************************/

static void
dp_hal_main_loop_hook_cb (LibHalContext *ctx, DBusConnection *dbus_conn)
{
	dbus_connection_set_exit_on_disconnect (dbus_conn, FALSE);
	dbus_connection_setup_with_g_main (dbus_conn, NULL);
	
	dbus_connection_add_filter (dbus_conn, 
	                            dp_do_filter_dbus_msg,
	                            ctx, NULL);
}

/***************************************************************************
 *
 *   dp_hal_device_added_cb
 *
 *   We are watching for two things here: DVD drives, and DVD discs
 *
 ***************************************************************************/

static void
dp_hal_device_added_cb (LibHalContext *ctx, const char *udi)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_device_added (tracker, udi);
	}
}

/***************************************************************************
 *
 *   dp_hal_device_removed_cb
 *
 ***************************************************************************/

static void
dp_hal_device_removed_cb (LibHalContext *ctx, const char *udi)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_device_removed (tracker, udi);
	}
}


/***************************************************************************
 *
 *   dp_hal_device_new_cap_cb
 *
 ***************************************************************************/

static void
dp_hal_device_new_cap_cb (LibHalContext *ctx, const char *udi, const char *cap)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_new_cap (tracker, udi, cap);
	}
}

/***************************************************************************
 *
 *   dp_hal_device_lost_cap_cb
 *
 ***************************************************************************/

static void
dp_hal_device_lost_cap_cb (LibHalContext *ctx, const char *udi, const char *cap)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_lost_cap (tracker, udi, cap);
	}
}

/***************************************************************************
 *
 *   dp_hal_device_prop_mod_cb
 *
 ***************************************************************************/

static void
dp_hal_device_prop_mod_cb (LibHalContext *ctx, 
                            const char    *udi, 
                            const char    *key, 
                            dbus_bool_t    is_removed, 
                            dbus_bool_t    is_added)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_prop_mod (tracker, udi, key, is_removed, is_added);
	}
}

/***************************************************************************
 *
 *   dp_hal_device_condition_cb
 *
 ***************************************************************************/

#ifndef USE_NEW_DBUS_HAL_API

static void
dp_hal_device_condition_cb (LibHalContext *ctx, const char *udi, const char *cond_name, DBusMessage *msg)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_dev_condition (tracker, udi, cond_name, msg);
	}
}

#else /* USE_NEW_DBUS_HAL_API */

static void
dp_hal_device_condition_cb (LibHalContext *ctx, const char *udi, const char *cond_name, const char *msg_detail)
{
	ThDevicePool *pool; 
	GList        *l;
	
	pool = hal_ctx_get_device_pool (ctx);
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	
	for (l = pool->priv->trackers;  l;  l = l->next)
	{
		ThDeviceTracker *tracker = TH_DEVICE_TRACKER (l->data);
	
		th_device_tracker_dev_condition (tracker, udi, cond_name, msg_detail);
	}
}

#endif /* USE_NEW_DBUS_HAL_API */

/***************************************************************************
 *
 *   device_pool_device_is_volume
 *
 ***************************************************************************/

static gboolean
device_pool_device_is_volume (ThDevicePool *pool, const gchar *udi)
{
	LibHalContext *ctx = pool->priv->hal_ctx;

	if (!hal_device_property_exists (ctx, udi, "info.capabilities")
	 || !hal_device_query_capability (ctx, udi, "volume"))
		return FALSE;
	
	return TRUE;
}

/***************************************************************************
 *
 *   device_pool_send_existing_devices_to_tracker
 *
 ***************************************************************************/

static void       
device_pool_send_initial_state (ThDevicePool *pool, ThDeviceTracker *tracker)
{
	char **devs;
	int    i, num_devs;
	
	if (!hal_device_exists (pool->priv->hal_ctx, "/org/freedesktop/Hal/devices/computer"))
	{
		/* libhal will automatically notice when hald is 
		 *   started and will send device_added messages */
		g_message ("Can't query devices. The HAL daemon (hald) does not seem to be running.");
		return;
	}
	
	devs = hal_get_all_devices (pool->priv->hal_ctx, &num_devs);

	/* We have to assume the list of devices is unordered,
	 *  ie. children could be before their parents. For now,
	 *  we'll simply send 'volumes' after all other devices,
	 *  that should make sure that CD drives get sent before
	 *  the discs they contain. */

	for (i = 0;  devs && i < num_devs;  ++i)
	{
		if (!device_pool_device_is_volume (pool, devs[i]))
			dp_hal_device_added_cb (pool->priv->hal_ctx, devs[i]);
	}

	/* Now send the volumes */
	for (i = 0;  devs && i < num_devs;  ++i)
	{
		if (device_pool_device_is_volume (pool, devs[i]))
			dp_hal_device_added_cb (pool->priv->hal_ctx, devs[i]);
	}
	
	hal_free_string_array (devs);
}

/***************************************************************************
 *
 *   th_device_pool_add_tracker
 *
 *   Adds its own reference to the tracker
 *
 ***************************************************************************/

void       
th_device_pool_add_tracker (ThDevicePool *pool, ThDeviceTracker *tracker)
{
	g_return_if_fail (TH_IS_DEVICE_POOL (pool));
	g_return_if_fail (TH_IS_DEVICE_TRACKER (tracker));
	g_return_if_fail (TH_DEVICE_TRACKER_GET_IFACE (tracker)->set_context != NULL);
	g_return_if_fail (TH_DEVICE_TRACKER_GET_IFACE (tracker)->device_added != NULL);
	g_return_if_fail (TH_DEVICE_TRACKER_GET_IFACE (tracker)->device_removed != NULL);

	/* already have it? */
	if (g_list_find (pool->priv->trackers, tracker))
		return;
	
	pool->priv->trackers = g_list_prepend (pool->priv->trackers, tracker);
	g_object_ref (tracker);

	/* skip this if we're not connected to the system bus */
	if (pool->priv->hal_ctx)
	{
		th_device_tracker_set_context (tracker, pool->priv->hal_ctx);
	
		device_pool_send_initial_state (pool, tracker);
	}
}

