/***************************************************************************
                            th-disc-drive.c
                            ---------------
    begin                : Tue Oct 12 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "th-disc-drive.h"
#include "th-utils.h"

#include <gst/gstregistry.h>

#include <glib/gi18n.h>

#include "_stdint.h"

#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#if defined(__OpenBSD__)
# include <sys/dvdio.h>
#elif defined(__linux__)  /* FIXME */
# include <linux/cdrom.h>
#else
# error "FIX include for dvd ioctls"
#endif

/* Minimum length a title needs to be
 *  to be taken into consideration
 *  (= be shown in the list) in seconds */
#define MINIMUM_JOB_DURATION   15

/* 8 audio streams is the maximum */
#define MAX_AUDIO_STREAMS 8

/* 32 subpicture streams is the maximum */
#define MAX_SUBPICTURE_STREAMS 32

/* how often we poll for a media 
 * change, if we have to do it ourselves */
#define MEDIA_POLL_FREQ (1*1000)

/* when we do the polling ourselves, we won't 
 *  know the disc title until we open the dvd
 *  to get the title infos */
#define PRELIMINARY_TITLE  (_("Retrieving disc title ..."))

enum
{
	PROP_DISC_TITLE = 1,
	PROP_NUM_TITLES,
	PROP_DEVICE,
	PROP_UDI,
	PROP_VENDOR,
	PROP_PRODUCT,
	PROP_POLL_MEDIA_REQUIRED,
	PROP_POLL_MEDIA           
};

enum
{
	AUDIO_FORMAT_AC3 = 0,
	AUDIO_FORMAT_UNKNOWN,
	AUDIO_FORMAT_MPEG1,
	AUDIO_FORMAT_MPEG2EXT,
	AUDIO_FORMAT_LPCM,
	AUDIO_FORMAT_SDDS,
	AUDIO_FORMAT_DTS
};

enum
{
	AUDIO_TYPE_UNDEFINED = 0,
	AUDIO_TYPE_NORMAL,
	AUDIO_TYPE_IMPAIRED,
	AUDIO_TYPE_COMMENTS1,
	AUDIO_TYPE_COMMENTS2  /* music score? */
};

/* FIXME: are these accurate? (and for real?) */
enum
{
	SUBPICTURE_TYPE_UNDEFINED = 0,
	SUBPICTURE_TYPE_NORMAL = 1,
	SUBPICTURE_TYPE_LARGE = 2,
	SUBPICTURE_TYPE_CHILDREN = 3,
	SUBPICTURE_TYPE_NORMAL_CAPTIONS = 5,
	SUBPICTURE_TYPE_LARGE_CAPTIONS = 6,
	SUBPICTURE_TYPE_CHILDRENS_CAPTIONS = 7,
	SUBPICTURE_TYPE_FORCED = 9,
	SUBPICTURE_TYPE_DIRECTORS_COMMENTS = 13,
	SUBPICTURE_TYPE_DIRECTORS_COMMENTS_LARGE = 14,
	SUBPICTURE_TYPE_DIRECTORS_COMMENTS_FOR_CHILDREN = 15
};


struct _ThDiscDrivePrivate
{
	gchar         *udi;         /* HAL device ID string         */
	gchar         *device;      /* OS device                    */

	gchar         *disc_title;  /* disc title if DVD in drive   */
	gchar         *vendor;      /* manufacturer/vendor          */
	gchar         *product;     /* model number                 */

	gboolean       got_dvd;     /* only used if we have to poll ourselves */
	gulong         media_poll_id;
	gboolean       media_poll_required;

	GIOChannel    *ioc;
	GString       *ioc_buf;
	guint          ioc_watch;
	guint          child_watch;

	GList         *titles;      /* ThJob objects */
};

/* Note: ThTitleInfo will be written
 *       flat into the pipe, so don't use
 *       deep allocation here */
typedef struct _ThTitleInfo ThTitleInfo;
struct _ThTitleInfo
{
	guint num;  /* title number */

	guint hours;
	guint mins;
	guint secs;

	guint num_chapters;
	
	guint frame_rate;
	guint video_aspect;      /* 0 = 4:3, otherwise 16:9 */ 

	guint size_x, size_y;

	gchar title_id[64];      /* disc ID + title number as string      */
	gchar volume_id[128];    /* title of disc                         */
	guint num_audio_streams; /* number of (interesting) audio streams */
	struct {
		guint         aid;      /* this is the _logical_ audio stream number */
		guint         phys_aid; /* this is the physical stream number on which
		                         * the mpeg demuxer will base the pad name   */
		audio_attr_t  attr;
	} audio_streams[MAX_AUDIO_STREAMS];

	guint num_subpicture_streams; /* number of interesting subpicture streams */
	struct {
		guint         sid;
		guint         phys_sid;
		subp_attr_t   attr;
	} subpicture_streams[MAX_SUBPICTURE_STREAMS];
};

static void             disc_drive_class_init       (ThDiscDriveClass *klass);

static void             disc_drive_instance_init    (ThDiscDrive *cp);

static void             disc_drive_finalize         (GObject *object);

static void             disc_drive_get_title_info   (ThDiscDrive *drive);

static gboolean         disc_drive_media_poll_cb    (ThDiscDrive *drive);

#ifdef __linux
static gboolean         disc_drive_has_dvd_medium_linux (ThDiscDrive *drive);
#endif


/* variables */

static GObjectClass    *drive_parent_class;          /* NULL */


/***************************************************************************
 *
 *   disc_drive_set_property
 *
 ***************************************************************************/

static void
disc_drive_set_property (GObject      *object,
                         guint         param_id,
                         const GValue *value,
                         GParamSpec   *pspec)
{
	ThDiscDrive *drive = (ThDiscDrive*) object;

	g_return_if_fail (TH_IS_DISC_DRIVE (drive));
	
	switch (param_id)
	{
		case PROP_DISC_TITLE:
		{
			gpointer old_title_ptr = drive->priv->disc_title;
			g_free (drive->priv->disc_title);
			drive->priv->disc_title = g_value_dup_string (value);
			if (drive->priv->disc_title && old_title_ptr == NULL)
				disc_drive_get_title_info (drive);
		}
		break;

		case PROP_DEVICE:
			drive->priv->device = g_value_dup_string (value);
			break;

		case PROP_UDI:
			drive->priv->udi = g_value_dup_string (value);
			break;
		
		case PROP_VENDOR:
			drive->priv->vendor = g_value_dup_string (value);
			break;
		
		case PROP_PRODUCT:
			drive->priv->product = g_value_dup_string (value);
			break;

		case PROP_POLL_MEDIA_REQUIRED:
			drive->priv->media_poll_required = g_value_get_boolean (value);
			break;

		case PROP_POLL_MEDIA:
		{
			gboolean do_poll = g_value_get_boolean (value);

			g_return_if_fail (!do_poll || drive->priv->media_poll_required);

			if (!do_poll && drive->priv->media_poll_id)
			{
				g_source_remove (drive->priv->media_poll_id);
				drive->priv->media_poll_id = 0;
			}
			else if (do_poll && drive->priv->media_poll_id == 0)
			{
				drive->priv->media_poll_id =
					g_timeout_add (MEDIA_POLL_FREQ, 
				                       (GSourceFunc) disc_drive_media_poll_cb,
					               drive);
			}
		}
		break;
		
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   disc_drive_get_property
 *
 ***************************************************************************/

static void
disc_drive_get_property (GObject      *object,
                         guint         param_id,
                         GValue       *value,
                         GParamSpec   *pspec)
{
	ThDiscDrive *drive = (ThDiscDrive*) object;

	g_return_if_fail (TH_IS_DISC_DRIVE (drive));
	
	switch (param_id)
	{
		case PROP_DISC_TITLE:
			g_value_set_string (value, drive->priv->disc_title);
			break;

		case PROP_UDI:
			g_value_set_string (value, drive->priv->udi);
			break;
		
		case PROP_DEVICE:
			g_value_set_string (value, drive->priv->device);
			break;

		case PROP_VENDOR:
			g_value_set_string (value, drive->priv->vendor);
			break;

		case PROP_PRODUCT:
			g_value_set_string (value, drive->priv->product);
			break;

		case PROP_NUM_TITLES:
			if (drive->priv->disc_title == NULL) 
				g_value_set_int (value, -2); /* no disc in drive */
			else if (drive->priv->ioc != NULL)
				g_value_set_int (value, -1); /* still checking */
			else
				g_value_set_int (value, g_list_length (drive->priv->titles));
			break;
	
		case PROP_POLL_MEDIA_REQUIRED:
			g_value_set_boolean (value, drive->priv->media_poll_required);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
			break;
	}
}

/***************************************************************************
 *
 *   disc_drive_class_init
 *
 ***************************************************************************/

static void
disc_drive_class_init (ThDiscDriveClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	drive_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize     = disc_drive_finalize;
	object_class->set_property = disc_drive_set_property;
	object_class->get_property = disc_drive_get_property;

	g_object_class_install_property (object_class, PROP_DISC_TITLE,
	                                 g_param_spec_string ("disc-title", 
	                                                      "disc-title", 
	                                                      "disc-title", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class, PROP_NUM_TITLES,
	                                 g_param_spec_int ("num-titles", 
	                                                   "num-titles", 
	                                                   "num-titles", 
	                                                   -2, 999, 0, 
	                                                   G_PARAM_READABLE));

	g_object_class_install_property (object_class, PROP_UDI,
	                                 g_param_spec_string ("udi", 
	                                                      "udi", 
	                                                      "udi", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_DEVICE,
	                                 g_param_spec_string ("device", 
	                                                      "device", 
	                                                      "device", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_PRODUCT,
	                                 g_param_spec_string ("product", 
	                                                      "product", 
	                                                      "product", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_VENDOR,
	                                 g_param_spec_string ("vendor", 
	                                                      "vendor", 
	                                                      "vendor", 
	                                                      NULL, 
	                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class, PROP_POLL_MEDIA_REQUIRED,
	                                 g_param_spec_boolean ("poll-media-required", 
	                                                       "we have to check for media changes ourselves", 
	                                                       "poll-media-required", 
	                                                       FALSE,
	                                                       G_PARAM_READWRITE));

	g_object_class_install_property (object_class, PROP_POLL_MEDIA,
	                                 g_param_spec_boolean ("poll-media", 
	                                                       "check for media changes regularly", 
	                                                       "poll-media", 
	                                                       FALSE,
	                                                       G_PARAM_WRITABLE));
}

/***************************************************************************
 *
 *   disc_drive_instance_init
 *
 ***************************************************************************/

static void
disc_drive_instance_init (ThDiscDrive *drive)
{
	drive->priv = g_new0 (ThDiscDrivePrivate, 1);
}

/***************************************************************************
 *
 *   disc_drive_finalize
 *
 ***************************************************************************/

static void
disc_drive_finalize (GObject *object)
{
	ThDiscDrive *drive;

	drive = (ThDiscDrive*) object;

	th_log ("ThDiscDrive: finalize\n");

	/* will unref ioc and close fd */
	if (drive->priv->ioc_watch)
		g_source_remove (drive->priv->ioc_watch);

	if (drive->priv->ioc_buf)
		g_string_free (drive->priv->ioc_buf, TRUE);

	g_list_foreach (drive->priv->titles, (GFunc) g_object_unref, NULL);
	g_list_free (drive->priv->titles);

	g_free (drive->priv->device);
	g_free (drive->priv->udi);

	if (drive->priv->media_poll_id)
		g_source_remove (drive->priv->media_poll_id);

	if (drive->priv->child_watch)
		g_source_remove (drive->priv->child_watch);

	memset (drive->priv, 0xab, sizeof (ThDiscDrivePrivate));
	g_free (drive->priv);

	/* chain up */
	drive_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_disc_drive_get_type
 *
 ***************************************************************************/

GType
th_disc_drive_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThDiscDriveClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) disc_drive_class_init,
			NULL, NULL,
			sizeof (ThDiscDrive),
			0,
			(GInstanceInitFunc) disc_drive_instance_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "ThDiscDrive", &info, 0);
	}

	return type;
}


/***************************************************************************
 *
 *   th_disc_drive_new
 *
 ***************************************************************************/

ThDiscDrive *
th_disc_drive_new (const gchar  *udi, 
                   const gchar  *device, 
                   const gchar  *vendor, 
                   const gchar  *product)
{
	ThDiscDrive *drive;
	
	drive = (ThDiscDrive *) g_object_new (TH_TYPE_DISC_DRIVE,
	                                      "device", device, 
	                                      "vendor", vendor, 
	                                      "product", product, 
	                                      "udi", udi, 
	                                      NULL);
	
	return drive;
}


/***************************************************************************
 *
 *   dd_audio_stream_get_format_str
 *
 ***************************************************************************/

static const gchar *
dd_audio_stream_get_format_str (audio_attr_t *attr)
{
	const gchar *formats[] = { N_("AC3"), N_("Unknown"), N_("MPEG1"), N_("MPEG2EXT"), 
	                           N_("Linear PCM"), N_("SDDS"), N_("DTS") };

	if (attr->audio_format >= G_N_ELEMENTS (formats))
		return _("Unknown");
	
	return _(formats[attr->audio_format]);
}

static const gchar *
dd_audio_stream_get_decoder (audio_attr_t *attr)
{
  switch (attr->audio_format) {
    case AUDIO_FORMAT_AC3:
      return g_intern_static_string ("a52dec");
    case AUDIO_FORMAT_MPEG1:
      return g_intern_static_string ("mad");
    case AUDIO_FORMAT_LPCM:
      return g_intern_static_string ("dvdlpcmdec");
    case AUDIO_FORMAT_DTS:
      return g_intern_static_string ("dtsdec");
    case AUDIO_FORMAT_MPEG2EXT:
    case AUDIO_FORMAT_UNKNOWN:
    case AUDIO_FORMAT_SDDS:
      break;
  }
  return NULL;
}

/***************************************************************************
 *
 *   dd_audio_stream_get_channel_str
 *
 ***************************************************************************/

static const gchar *
dd_audio_stream_get_channel_str (audio_attr_t *attr)
{
	const gchar *channels[] = { N_("Mono"), N_("Stereo"), N_("Unknown"), 
	                            N_("Surround"), N_("Unknown"), N_("5.1") };

	if (attr->channels >= G_N_ELEMENTS (channels))
		return _("Unknown");
	
	return _(channels[attr->channels]);
}

/***************************************************************************
 *
 *   dd_audio_stream_get_language_str_short
 *
 ***************************************************************************/

static const gchar *
dd_audio_stream_get_language_str_short (audio_attr_t *attr)
{
	static gchar lang[3];
	
	lang[0] = (attr->lang_code >> 8);
	lang[1] = (attr->lang_code & 0xFF);
	lang[2] = 0x00;

	if (!g_ascii_isalnum (lang[0])
	 || !g_ascii_isalnum (lang[1]))
		lang[0] = '\0';

	return lang;
}

/***************************************************************************
 *
 *   dd_subp_stream_get_language_str_short
 *
 ***************************************************************************/

static const gchar *
dd_subp_stream_get_language_str_short (subp_attr_t *attr)
{
	static gchar lang[3];
	
	lang[0] = (attr->lang_code >> 8);
	lang[1] = (attr->lang_code & 0xFF);
	lang[2] = 0x00;

	if (!g_ascii_isalnum (lang[0])
	 || !g_ascii_isalnum (lang[1]))
		lang[0] = '\0';

	return lang;
}

/***************************************************************************
 *
 *   dd_dump_title_info
 *
 ***************************************************************************/

static void
dd_dump_title_info (ThTitleInfo *info)
{
	const gchar *rates[5] = { "? (0)", "25.00", "? (2)", "29.97", "???" };
	gint j;

	th_log (" [%02u] [%02u:%02u:%02u] [Video 00] [%2u %s]\t[%4s] [%ux%u] [%s fps]\n",
	         info->num+1, info->hours, info->mins, info->secs,
	         info->num_chapters, (info->num_chapters == 1) ? _("chapter") : _("chapters"),
	         (info->video_aspect == 0) ? "4:3" : "16:9",
	         info->size_x, info->size_y,
	         rates[CLAMP (info->frame_rate, 0, 4)]);

	for (j = 0;  j < info->num_audio_streams;  ++j)
	{
		th_log ("      [Audio %02u] [%s] %s, %s\n",info->audio_streams[j].phys_aid,
		         dd_audio_stream_get_language_str_short (&info->audio_streams[j].attr),
		         dd_audio_stream_get_format_str (&info->audio_streams[j].attr),
		         dd_audio_stream_get_channel_str (&info->audio_streams[j].attr));
	}

	for (j = 0;  j < info->num_subpicture_streams;  ++j)
	{
		th_log ("      [Sub   %02u] [%s]\n", info->subpicture_streams[j].phys_sid,
		         dd_subp_stream_get_language_str_short (&info->subpicture_streams[j].attr));
	}
}


/***************************************************************************
 *
 *   dd_job_add_audio_streams
 *
 ***************************************************************************/

static void
dd_job_add_audio_streams (ThJob *job, ThTitleInfo *info)
{
	gint i;

	for (i = 0;  i < info->num_audio_streams;  ++i)
	{
		audio_attr_t *attr;
		const gchar  *lang_str, *chan_str, *fmt_str, *decoder;
		gchar        *desc, *pad_name;
		
		attr = &info->audio_streams[i].attr;
		
		fmt_str  = dd_audio_stream_get_format_str (attr); 
		chan_str = dd_audio_stream_get_channel_str (attr);
		lang_str = dd_audio_stream_get_language_str_short (attr);
		decoder = dd_audio_stream_get_decoder (attr);

		desc = g_strdup_printf ("(%s, %s)", fmt_str, chan_str);
		pad_name = g_strdup_printf ("audio_%02u", info->audio_streams[i].phys_aid);
		th_job_add_audio_stream (job, info->audio_streams[i].aid,
		                         pad_name, lang_str, desc, decoder);
		g_free (pad_name);
		g_free (desc);
	}
}

/***************************************************************************
 *
 *   dd_job_add_subpicture_streams
 *
 ***************************************************************************/

static void
dd_job_add_subpicture_streams (ThJob *job, ThTitleInfo *info)
{
	gint i;

	for (i = 0;  i < info->num_subpicture_streams;  ++i)
	{
		subp_attr_t *attr;
		const gchar *lang_str;
		gchar       *pad_name;
		
		attr = &info->subpicture_streams[i].attr;
		
		lang_str = dd_subp_stream_get_language_str_short (attr);
		
		pad_name = g_strdup_printf ("subpicture_%02u",
		    info->subpicture_streams[i].phys_sid);
		th_job_add_subpicture_stream (job, info->subpicture_streams[i].sid,
		                              pad_name, lang_str);
		g_free (pad_name);
	}
}


/***************************************************************************
 *
 *   disk_drive_import_titles_from_pipe_data
 *
 ***************************************************************************/

static void
disk_drive_import_titles_from_pipe_data (ThDiscDrive *drive, ThTitleInfo *titles, guint num_titles)
{
	GList *l;
	guint  i, longest_title_len;

	longest_title_len = 0;

	for (i = 0;  i < num_titles;  ++i)
		dd_dump_title_info (&titles[i]);

	if (drive->priv->disc_title
	 && g_str_equal (drive->priv->disc_title, PRELIMINARY_TITLE)
	 && num_titles > 0 && titles[0].volume_id[0] != 0x00)
	{
		g_object_set (drive, "disc-title", titles[0].volume_id, NULL);
	}

	for (i = 0;  i < num_titles;  ++i)
	{
		guint secs;
	
		secs = (titles[i].hours * 3600) + (titles[i].mins * 60) + titles[i].secs;
	
		longest_title_len = MAX (longest_title_len, secs);

		/* skip titles that are just too
		 *  short to be of interest */
		if (secs >= MINIMUM_JOB_DURATION 
		 && titles[i].num_audio_streams > 0)
		{
			ThJob  *job;

			job = th_job_new (drive->priv->device, titles[i].title_id, 
			                  i, secs, titles[i].num_chapters,
			                  titles[i].size_x, titles[i].size_y);

			if (titles[i].frame_rate == 3) {
			  job->video_info.fps_num = 30000; /* 29.97 fps */
			  job->video_info.fps_denom = 1001;
			} else {
			  job->video_info.fps_num = 25;    /* 25.00 fps */
			  job->video_info.fps_denom = 1;
			}

			if (titles[i].video_aspect == 0) {
			  job->video_info.aspect_num = 4;
			  job->video_info.aspect_denom = 3;
			} else {
			  job->video_info.aspect_num = 16;
			  job->video_info.aspect_denom = 9;
			}
			job->video_info.aspect_num *= titles[i].size_y;
			job->video_info.aspect_denom *= titles[i].size_x;

			th_simplify_fraction (&job->video_info.aspect_num,
			    &job->video_info.aspect_denom);

			dd_job_add_audio_streams (job, &titles[i]);
			dd_job_add_subpicture_streams (job, &titles[i]);

			drive->priv->titles = g_list_append (drive->priv->titles, job);
		}
	}

	/* set up titles */
	for (l = drive->priv->titles;  l;  l = l->next)
	{
		gchar  *title_tag;
		guint   secs, title_num;
		ThJob  *job;
		
		job = TH_JOB (l->data);
		
		g_object_get (job, 
		              "title-length", &secs, 
		              "title-tag", &title_tag, 
		              "title-num", &title_num,
		              NULL);

		/* flag the longest title(s) */
		if ((gdouble) (secs * 1.0/ longest_title_len) >= 0.75)
			g_object_set (job, "is-main-title", TRUE, NULL);
		
		/* Set default title if one hasn't been loaded from file */
		if (title_tag == NULL || *title_tag == 0x00)
		{
			g_free (title_tag);

			title_tag = th_job_get_default_title (job);
			
			g_object_set (job, "title-tag", title_tag, NULL);
		}
		
		g_free (title_tag);
	}

	/* notify interested parties that 
	 * we've received the title data */
	g_object_notify (G_OBJECT (drive), "num-titles");
}

/***************************************************************************
 *
 *   dd_ioc_read_cb
 *
 ***************************************************************************/

static gboolean
dd_ioc_read_cb (GIOChannel *ioc, GIOCondition cond, ThDiscDrive *drive)
{
	if (cond & (G_IO_IN | G_IO_PRI))
	{
		GIOStatus  ret;
		GError    *err = NULL;
		gchar      buf[8192];
		gsize      num_read;

		do
		{
			num_read = 0;

			ret = g_io_channel_read_chars (drive->priv->ioc, buf,
			                               sizeof (buf), &num_read, &err);
		
			if (err)
			{
				th_log ("error in dd_ioc_read_cb(): %s\n", err->message);
				g_error_free (err);
			}
		
			if (ret != G_IO_STATUS_NORMAL)
				cond |= G_IO_ERR;
			 
			if (num_read <= 0)
				cond |= G_IO_HUP;
		
			if (num_read > 0)
			{
				th_log ("pipe read %u bytes\n", (guint) num_read);
				g_string_append_len (drive->priv->ioc_buf, buf, num_read);
			}
		}
		while (ret == G_IO_STATUS_NORMAL && num_read == sizeof (buf));
	}
	
	if ((cond & (G_IO_ERR | G_IO_HUP | G_IO_NVAL)))
	{
		th_log ("pipe hangup. %" G_GSIZE_FORMAT " bytes in total\n",
                        drive->priv->ioc_buf->len);
		
		g_assert (drive->priv->titles == NULL);
		
		if (drive->priv->ioc_buf->len % sizeof (ThTitleInfo) == 0)
		{
			ThTitleInfo *titles;
			guint        num_titles;
			
			num_titles = drive->priv->ioc_buf->len / sizeof (ThTitleInfo);
			
			titles = g_new0 (ThTitleInfo, num_titles);
			memcpy (titles, drive->priv->ioc_buf->str, drive->priv->ioc_buf->len);

			disk_drive_import_titles_from_pipe_data (drive, titles, num_titles);
			
			g_free (titles);
		}
		else
		{
			g_warning ("%s: unexpected number of bytes received "
                                   "(%" G_GSIZE_FORMAT "u is not a multiple "
                                   "of %" G_GSIZE_FORMAT ")\n", G_STRLOC, 
			           drive->priv->ioc_buf->len,
			           sizeof (ThTitleInfo));
		}
		
		drive->priv->ioc = NULL;
		drive->priv->ioc_watch = 0;

		/* FALSE = remove source, which will unref the 
		 *  io channel, which in turn will close the fd */
		return FALSE; 
	}
	
	return TRUE; /* call again */
}

/***************************************************************************
 *
 *   disc_drive_setup_read_io_channel
 *
 ***************************************************************************/

static void
disc_drive_setup_read_io_channel (ThDiscDrive *drive, gint fd_write, gint fd_read)
{
	drive->priv->ioc = g_io_channel_unix_new (fd_read); /* use reading end */

	if (drive->priv->ioc_buf == NULL)
		drive->priv->ioc_buf = g_string_new (NULL);

	if (drive->priv->titles != NULL)
		g_object_notify (G_OBJECT (drive), "num-titles");
	
	g_list_foreach (drive->priv->titles, (GFunc) g_object_unref, NULL);
	g_list_free (drive->priv->titles);
	drive->priv->titles = NULL;
	
	g_string_truncate (drive->priv->ioc_buf, 0);

	g_io_channel_set_encoding (drive->priv->ioc, NULL, NULL); /* read binary data */
		
	g_io_channel_set_close_on_unref (drive->priv->ioc, TRUE);

	drive->priv->ioc_watch = g_io_add_watch (drive->priv->ioc,
	                                         G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL | G_IO_PRI,
	                                         (GIOFunc) dd_ioc_read_cb,
	                                         drive);

	g_io_channel_unref (drive->priv->ioc); /* io watch added own reference */
}

/***************************************************************************
 *
 *   disc_drive_child_exit_cb
 *
 ***************************************************************************/

static void
disc_drive_child_exit_cb (GPid pid, gint status, ThDiscDrive *drive)
{
	th_log ("PID %u exited with status %d.\n", pid, status);
	drive->priv->child_watch = 0;
}

/***************************************************************************
 *
 *   dd_audio_stream_compare
 *
 *   Returns a value > 0 if attr2 is 'better' than attr1
 *
 ***************************************************************************/

static gint
dd_audio_stream_compare (audio_attr_t *attr1, audio_attr_t *attr2)
{
	/* the more channels, the better (even though we're
	 *   converting them back to stereo for now anyway) */
	if (attr2->channels != attr1->channels)
		return ((gint) attr2->channels - (gint) attr1->channels);

	if (attr2->audio_format == attr1->audio_format)
		return 0;

	/* if the same number of channels, prefer LPCM to any other format */
	if (attr2->audio_format == AUDIO_FORMAT_LPCM && attr1->audio_format != AUDIO_FORMAT_LPCM)
		return 1;

	return -1;
}

/***************************************************************************
 *
 *   dd_audio_format_supported
 *
 ***************************************************************************/

static gboolean
dd_audio_format_supported (guint audio_format)
{
	static gboolean  checked; /* FALSE */
	static gboolean  lpcm_ok; /* FALSE */
	static gboolean  dts_ok;  /* FALSE */
	static gboolean  mp3_ok;  /* FALSE */

	if (audio_format == AUDIO_FORMAT_AC3)
		return TRUE;
	
	if (checked == FALSE)
	{
		if (gst_default_registry_check_feature_version ("dvdlpcmdec", 0, 10, 0))
		{
			th_log ("dvdlpcmdec plugin available");
			lpcm_ok = TRUE;
		} else th_log ("dvdlpcmdec plugin not available");

		if (gst_default_registry_check_feature_version ("dtsdec", 0, 10, 0))
		{
			th_log ("dtsdec plugin available");
			dts_ok = TRUE;
		} else th_log ("dtsdec plugin not available");

		/* ffdec_mp3 isn't tested well enough and mad is hardcoded in th-job.c */
		if (gst_default_registry_check_feature_version ("mad", 0, 10, 0))
		{
			th_log ("mad plugin available");
			mp3_ok = TRUE;
		} else th_log ("mad plugin not available");

		checked = TRUE;
	}

	if (audio_format == AUDIO_FORMAT_LPCM)
		return lpcm_ok;

	if (audio_format == AUDIO_FORMAT_DTS)
		return dts_ok;

	if (audio_format == AUDIO_FORMAT_MPEG1)
		return mp3_ok;

	return FALSE;
}

/***************************************************************************
 *
 *   disc_drive_get_title_set_audio_info
 *
 *   Extracts the audio streams which interest us
 *
 ***************************************************************************/

static void
disc_drive_get_title_set_audio_info (ThDiscDrive  *drive, 
                                     ifo_handle_t *vts_file, 
                                     pgc_t        *pgc,
                                     ThTitleInfo  *tinfo)
{
	guint i;

	tinfo->num_audio_streams = 0;
	
	g_return_if_fail (pgc != NULL);

	/* do checks and print debug info before the other stuff */
	(void) dd_audio_format_supported (AUDIO_FORMAT_LPCM);
	(void) dd_audio_format_supported (AUDIO_FORMAT_DTS);

	th_log ("\n");

	for (i = 0;  i < vts_file->vtsi_mat->nr_of_vts_audio_streams;  ++i)
	{
		audio_attr_t  attr;
		guint phys_streamid;

		/* audio stream not present? */ 
		if ((pgc->audio_control[i] & 0x8000) == 0)
			continue;

		phys_streamid = (pgc->audio_control[i] & 0x700) >> 8;

		attr = vts_file->vtsi_mat->vts_audio_attr[i];

		th_log ("%3u  audio_%02u      (%02u)  %s  %s, %-8s  "
		        "(type = %u, ext = %u)\n", 
		         tinfo->num + 1, phys_streamid, i, 
			       dd_audio_stream_get_language_str_short (&attr),
			       dd_audio_stream_get_format_str (&attr),
			       dd_audio_stream_get_channel_str (&attr),
			       attr.lang_type, attr.lang_extension);
		
		if (attr.lang_code == 0xffff)
			attr.lang_code = 0;

		if (dd_audio_format_supported (attr.audio_format)
		 && ((attr.lang_type == 1 && attr.lang_extension <= AUDIO_TYPE_NORMAL)
		  || attr.lang_type == 0))
		{
			/* Check if we already have an audio
			 *  stream in the same language */
			gint j;
			for (j = 0;  j < tinfo->num_audio_streams;  ++j)
			{
				if (tinfo->audio_streams[j].attr.lang_code == attr.lang_code)
				{
					/* Is this audio track better than the one we got? */
					if (dd_audio_stream_compare (&tinfo->audio_streams[j].attr, &attr) > 0)
					{
						tinfo->audio_streams[j].aid  = i;
						tinfo->audio_streams[j].phys_aid  = phys_streamid;
						tinfo->audio_streams[j].attr = attr;
					}
					break; /* => this stream will not be added below */
				}
			}
			
			/* Don't have a stream in this language yet? add it */
			if (j == tinfo->num_audio_streams)
			{
				tinfo->audio_streams[tinfo->num_audio_streams].aid  = i;
				tinfo->audio_streams[tinfo->num_audio_streams].phys_aid  = phys_streamid;
				tinfo->audio_streams[tinfo->num_audio_streams].attr = attr;
				
				++tinfo->num_audio_streams;
				
				if (tinfo->num_audio_streams == MAX_AUDIO_STREAMS)
					return; /* bail out if more streams than expected/allowed */
			}
		}
	}
}

/***************************************************************************
 *
 *   disc_drive_get_title_set_subp_info
 *
 *   Extracts the subtitle streams which interest us
 *
 ***************************************************************************/

static void
disc_drive_get_title_set_subp_info (ThDiscDrive  *drive, 
                                    ifo_handle_t *vts_file, 
                                    pgc_t        *pgc,
                                    ThTitleInfo  *tinfo)
{
	guint i;

	tinfo->num_subpicture_streams = 0;
	
	g_return_if_fail (pgc != NULL);

	for (i = 0;  i < vts_file->vtsi_mat->nr_of_vts_subp_streams;  ++i)
	{
		subp_attr_t  attr;
		guint phys_streamid;
		gint j;

		/* subpicture stream not present? */ 
		if ((pgc->subp_control[i] & 0x80000000) == 0)
			continue;

		/* get stream ID (depending on aspect ratio) (FIXME?) */
		if (tinfo->video_aspect == 0) {
			/* 4:3 */
			phys_streamid = (pgc->subp_control[i] >> 16) & 0x3;
		} else {
			/* 16:9 */
			phys_streamid = (pgc->subp_control[i] >> 16) & 0x3;
		}

		attr = vts_file->vtsi_mat->vts_subp_attr[i];

		th_log ("%3u  subpicture_%02u (%02u)  %s                 "
		        "(type = %u, ext = %u, coding = %u)\n", 
		         tinfo->num + 1, phys_streamid, i, 
		         dd_subp_stream_get_language_str_short (&attr),
		         attr.type, attr.lang_extension, attr.code_mode);

		if (attr.lang_code == 0xffff)
			attr.lang_code = 0;

		/* ignore subpicture streams without specified language */
		if (attr.type != 1 || attr.lang_code == 0)
			continue;

		/* ignore subpicture streams that are not run-length encoded */
		if (attr.code_mode != 0)
			continue;

		/* FIXME: allow these other types too? */
		if (attr.lang_extension >= SUBPICTURE_TYPE_LARGE)
			continue;

		/* Check if we already have a stream in the same language */
		for (j = 0;  j < tinfo->num_subpicture_streams;  ++j) {
			if (tinfo->subpicture_streams[j].attr.lang_code == attr.lang_code)
				break; /* => this stream will not be added below */
		}
			
		/* Don't have a stream in this language yet? add it */
		if (j == tinfo->num_subpicture_streams) {
			tinfo->subpicture_streams[tinfo->num_subpicture_streams].sid  = i;
			tinfo->subpicture_streams[tinfo->num_subpicture_streams].phys_sid  = phys_streamid;
			tinfo->subpicture_streams[tinfo->num_subpicture_streams].attr = attr;
				
			++tinfo->num_subpicture_streams;
				
			if (tinfo->num_subpicture_streams == MAX_AUDIO_STREAMS)
				return; /* bail out if more streams than expected/allowed */
		}
	}
}

/***************************************************************************
 *
 *   disc_drive_get_title_set_info
 *
 ***************************************************************************/

static gboolean
disc_drive_get_title_set_info (ThDiscDrive  *drive, 
                               dvd_reader_t *dvd, 
                               tt_srpt_t    *tt_srpt, 
                               ThTitleInfo  *tinfo)
{
	ifo_handle_t  *vts_file;
	dvd_time_t    *time;
	pgc_t         *cur_pgc;    
	guint          title_set_nr;
	gint           ttn, pgc_id;
	
	title_set_nr = tt_srpt->title[tinfo->num].title_set_nr;
	
	/* g_warning ("Opening title #%u\n", title_num); */
	
	vts_file = ifoOpen (dvd, title_set_nr);
	
	if (vts_file == NULL) 
	{
		th_log ("Can't open the title %d info file.\n", tinfo->num);
		return FALSE;
	}

	ttn = tt_srpt->title[tinfo->num].vts_ttn;
	pgc_id = vts_file->vts_ptt_srpt->title[ttn-1].ptt[0].pgcn;
	cur_pgc = vts_file->vts_pgcit->pgci_srp[pgc_id-1].pgc;

	/* FIXME: this does not always seem to be right, 
	 * e.g. Kurz+Schmerzlos DVD, Title #1 is 20 secs instead of 31 secs */
	time = &cur_pgc->playback_time;
	
	g_assert ((time->hour>>4) < 0xa && (time->hour&0xf) < 0xa);
	g_assert ((time->minute>>4) < 0x7 && (time->minute&0xf) < 0xa);
	g_assert ((time->second>>4) < 0x7 && (time->second&0xf) < 0xa);

	tinfo->hours = (time->hour >> 4) * 10 + (time->hour & 0xf);
	tinfo->mins  = (time->minute >> 4) * 10 + (time->minute & 0xf);
	tinfo->secs  = (time->second >> 4) * 10 + (time->second & 0xf);
	
	tinfo->num_chapters = tt_srpt->title[tinfo->num].nr_of_ptts;
	
	/* video stream info */
	tinfo->frame_rate = (time->frame_u >> 6) & 0x03;
	tinfo->video_aspect = vts_file->vtsi_mat->vts_video_attr.display_aspect_ratio;
	
	if (vts_file->vtsi_mat->vts_video_attr.picture_size == 0)
		tinfo->size_x = 720;
	else if (vts_file->vtsi_mat->vts_video_attr.picture_size == 1)
		tinfo->size_x = 704;
	else if (vts_file->vtsi_mat->vts_video_attr.picture_size == 2)
		tinfo->size_x = 352;
	else if (vts_file->vtsi_mat->vts_video_attr.picture_size == 3)
		tinfo->size_x = 176;
	else
		g_assert_not_reached ();

	if (vts_file->vtsi_mat->vts_video_attr.video_format == 0)
		tinfo->size_y = 480;
	else
		tinfo->size_y = 576;
	
	disc_drive_get_title_set_audio_info (drive, vts_file, cur_pgc, tinfo);
	disc_drive_get_title_set_subp_info (drive, vts_file, cur_pgc, tinfo);

	ifoClose (vts_file);
	
	return TRUE;
}

/***************************************************************************
 *
 *   disc_drive_query_rpc_status
 *
 *   Query the region code settings of the drive (for RPC-II drives), so
 *   we can show an appropriate error message if there's a problem
 *
 ***************************************************************************/

static void
disc_drive_query_rpc_status (ThDiscDrive  *drive)
{
#if defined (DVD_AUTH) && defined (DVD_LU_SEND_RPC_STATE)
  dvd_authinfo auth_info;
  int fd, ret, i;

  fd = open (drive->priv->device, O_RDONLY | O_NONBLOCK);
  if (fd < 0) {
    th_log ("Can't open dvd device: %s\n", g_strerror (errno));
    return;
  }

  memset (&auth_info, 0, sizeof (auth_info));
  auth_info.type = DVD_LU_SEND_RPC_STATE;

  ret = ioctl (fd, DVD_AUTH, &auth_info);

  if (ret < 0) {
    th_log ("ioctl DVD_AUTH failed: %s\n", g_strerror (errno));
    goto out;
  }
   
  th_log ("Current Region Code settings:\n");
  th_log ("  RPC Scheme                           : %s\n",
      (auth_info.lrpcs.rpc_scheme == 1) ? "RPC-II" : "Unknown");
  switch (auth_info.lrpcs.type) {
    case 0:
      th_log ("  Type                                 : NO REGION CODE SET YET\n");
      break;
    case 1:
      th_log ("  Type                                 : REGION CODE SET\n");
      break;
    case 2:
      th_log ("  Type                                 : LAST CHANCE\n");
      break;
    case 3:
      th_log ("  Type                                 : REGION CODE PERMANENT\n");
      break;
    default:
      break;
  }
  th_log ("  Vendor resets left                   : %d\n", auth_info.lrpcs.vra);
  th_log ("  User controlled changes left         : %d\n", auth_info.lrpcs.ucca);

  th_log ("  Region_mask=0x%02X\n", auth_info.lrpcs.region_mask);
  if (auth_info.lrpcs.region_mask == 0xff) {
    th_log ("  Drive will play discs from region(s) : %s\n",
        "NONE, you must set a region first");
  } else {
    gchar regions[256] = { 0, };

    for (i = 0; i < 8; ++i) {
      if ((auth_info.lrpcs.region_mask & (1 << i)) == 0) {
        regions[i * 2] = (i + 1) + '0';
        regions[(i * 2)+1] = ' ';
      }
    }
    th_log ("  Drive will play discs from region(s) : %s\n", regions);
  }

out:

  close (fd);

#else

  th_log ("Can't query DVD RPC status on this operating system.\n");

#endif
}

/***************************************************************************
 *
 *   disc_drive_get_title_info_real
 *
 *   This code is executed in the child process after the fork. fd is the
 *     writing end of the pipe through which we pass the results to the
 *     parent.
 *
 ***************************************************************************/

static void
disc_drive_get_title_info_real (ThDiscDrive *drive, gint fd)
{
	ThTitleInfo    *titles;
	dvd_reader_t   *dvd;
	ifo_handle_t   *vmg_file;
	tt_srpt_t      *tt_srpt;
	gboolean        err = FALSE;
	guint8          discid[16], volsetid[128];
	gchar           discid_str[33], volid[128];
	guint           i;
	
	/* Open the disc */
	dvd = DVDOpen (drive->priv->device);
	if (dvd == NULL)
	{
		g_warning ("DVDOpen (%s) failed.\n", drive->priv->device);
		return;
	}

	/* check region code settings, if possible */
	disc_drive_query_rpc_status (drive);

	if (DVDDiscID (dvd, discid) != 0)
	{
		g_warning ("DVDDiscID (%s) failed.\n", drive->priv->device);
		return;
	}

	memset (volid, 0x00, sizeof (volid));
	memset (volsetid, 0x00, sizeof (volsetid));
	if (DVDUDFVolumeInfo (dvd, volid, sizeof (volid), volsetid, sizeof (volsetid) - 1) == 0)
	{
		/* th_log ("Volume ID:\t%s\nVolume Set ID:\t%s\n", volid, volsetid); */
	}
	
	for (i = 0;  i < 16;  ++i)
	{
		const gchar *hexdigits = "0123456789abcdef";
		discid_str [(i << 1) + 0] = hexdigits [((discid[i] & 0xf0) >> 4)];
		discid_str [(i << 1) + 1] = hexdigits [((discid[i] & 0x0f) >> 0)];
	}
	discid_str[32] = 0x00;
	
	th_log ("DiscID = '%s'\n", discid_str);

	/* Load video manager to find out info
	 *  about the titles on this disc */
	vmg_file = ifoOpen (dvd, 0);
	if (vmg_file == NULL)
	{
		g_warning ("Could not open VMG info\n");
		DVDClose (dvd);
		return;
	}

	tt_srpt = vmg_file->tt_srpt;

	th_log ("Number of titles: %u\n", tt_srpt->nr_of_srpts);

	titles = g_new0 (ThTitleInfo, tt_srpt->nr_of_srpts);

	for (i = 0;  i < tt_srpt->nr_of_srpts && !err;  ++i)
	{
		titles[i].num = i;
		
		g_snprintf (titles[i].title_id, 
		            sizeof (titles[i].title_id), 
		            "%s-%03u", discid_str, i);
		
		g_snprintf (titles[i].volume_id, 
		            sizeof (titles[i].volume_id), 
		            "%s", volid);

		if (!disc_drive_get_title_set_info (drive, dvd, tt_srpt, &titles[i]))
			err = TRUE;
	}

	if (err == FALSE)
	{
		ssize_t  ret, written = 0;
		ssize_t  left = tt_srpt->nr_of_srpts * sizeof (ThTitleInfo);
		void    *buf = (void*) titles;
		
		do
		{
			ret = write (fd, buf, left);
	
			th_log ("%d = write (%d, %p, %u)\n",
			        (int) ret, fd, buf, (guint) left);

			if (ret < 0)
			{
				/* if the parent is gone, just exit silently */
				if (errno == EPIPE)
					_exit (1);
				g_warning ("%s: write() to pipe failed: %s\n", G_STRLOC, g_strerror (errno));
				break;
			}
			
			written += ret;
			buf += ret;
			left -= ret;
		
			th_log ("\t=>written = %u, buf = %p, left = %u\n",
			        (guint) written, buf, (guint) left);
		}
		while (ret > 0 && left > 0);
	}
	
	g_free (titles);

	ifoClose (vmg_file);
	DVDClose (dvd);
}

/***************************************************************************
 *
 *   disc_drive_get_title_info
 *
 ***************************************************************************/

static void
disc_drive_get_title_info (ThDiscDrive *drive)
{
	struct rlimit   rl;
	pid_t           child_pid;
	gint            i, filedes[2];

	g_return_if_fail (drive->priv->ioc == NULL);

	if (pipe (filedes) != 0)
	{
		g_warning ("%s: pipe() failed: %s\n", G_STRLOC, g_strerror (errno));
		return;
	}
	
	child_pid = fork ();
	
	/* Has an error occured? */
	if (child_pid == (pid_t) -1)
	{
		g_warning ("%s: fork() failed: %s\n", G_STRLOC, g_strerror (errno));
		close (filedes[0]);
		close (filedes[1]);
		return;
	}
	
	/* Are we the parent? */
	if (child_pid > 0)
	{
		disc_drive_setup_read_io_channel (drive, filedes[1], filedes[0]);
		
		/* so our child doesn't stay around as zombie after it exits */
		drive->priv->child_watch = g_child_watch_add ((GPid) child_pid, 
		                                              (GChildWatchFunc) disc_drive_child_exit_cb,
		                                              drive);
		
		close (filedes[1]); /* close our copy of the writing end of the pipe */
		return;
	}
	
	/* We are the child. Close all inherited file descriptors,
	 *  including our copy of the reading end of the pipe,
	 *  but not the writing end of the pipe. */
	if (getrlimit (RLIMIT_NOFILE, &rl) == 0)
	{
		for (i = rl.rlim_max - 1;  i >= 3;  --i)
		{
			if (i != filedes[1])
				(void) close (i);
		}
	}

	disc_drive_get_title_info_real (drive, filedes[1]);

	close (filedes[1]); /* close our writing end */
	
	_exit (0);
}

/***************************************************************************
 *
 *   th_disc_drive_get_titles
 *
 ***************************************************************************/

const GList *
th_disc_drive_get_titles (ThDiscDrive *drive)
{
	g_return_val_if_fail (TH_IS_DISC_DRIVE (drive), NULL);
	g_return_val_if_fail (drive->priv->ioc == NULL, NULL);
	
	return drive->priv->titles;
}

/***************************************************************************
 *
 *   disc_drive_media_poll_cb
 *
 ***************************************************************************/

static gboolean
disc_drive_media_poll_cb (ThDiscDrive *drive)
{
	gboolean got_dvd;

	got_dvd = disc_drive_has_dvd_medium_linux (drive);

	if (got_dvd == drive->priv->got_dvd)
		return TRUE; /* no status change, check again later */

	if (got_dvd)
	{
		g_object_set (drive, "disc-title", PRELIMINARY_TITLE, NULL);
	}
	else
	{
		g_object_set (drive, "disc-title", NULL, NULL);
	}

	drive->priv->got_dvd = got_dvd;

	// g_signal_emit (ddpool, ddp_signals[DISC_CHANGED], 0, drive);

	return TRUE; /* call again until removed */ 
}

#ifdef __linux

/*
 * This is part of dvd+rw-tools by Andy Polyakov <appro@fy.chalmers.se>
 *
 * Use-it-on-your-own-risk, GPL bless...
 *
 * For further details see http://fy.chalmers.se/~appro/linux/DVD+RW/
 */

#define CREAM_ON_ERRNO(s)	do {				\
    switch ((s)[2]&0x0F)					\
    {	case 2:	if ((s)[12]==4) errno=EAGAIN;	break;		\
	case 5:	errno=EINVAL;					\
		if ((s)[13]==0)					\
		{   if ((s)[12]==0x21)		errno=ENOSPC;	\
		    else if ((s)[12]==0x20)	errno=ENODEV;	\
		}						\
		break;						\
    }								\
} while(0)
#define ERRCODE(s)	((((s)[2]&0x0F)<<16)|((s)[12]<<8)|((s)[13]))
#define	SK(errcode)	(((errcode)>>16)&0xF)
#define	ASC(errcode)	(((errcode)>>8)&0xFF)
#define ASCQ(errcode)	((errcode)&0xFF)

#ifndef _LARGEFILE_SOURCE
#define _LARGEFILE_SOURCE
#endif
#ifndef _LARGEFILE64_SOURCE
#define _LARGEFILE64_SOURCE
#endif
#ifndef _FILE_OFFSET_BITS
#define _FILE_OFFSET_BITS 64
#endif
#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <errno.h>
#include <string.h>
#include <mntent.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <scsi/scsi.h>
#include <scsi/sg.h>
#include <poll.h>
#include <sys/time.h>

typedef enum {
	NONE = CGC_DATA_NONE,	// 3
	READ = CGC_DATA_READ,	// 2
	WRITE = CGC_DATA_WRITE	// 1
} Direction;

typedef struct ScsiCommand ScsiCommand;

struct ScsiCommand {
	int fd;
	int autoclose;
	char *filename;
	struct cdrom_generic_command cgc;
	union {
		struct request_sense s;
		unsigned char u[18];
	} _sense;
	struct sg_io_hdr sg_io;
};

#define DIRECTION(i) (Dir_xlate[i]);

/* 1,CGC_DATA_WRITE
 * 2,CGC_DATA_READ
 * 3,CGC_DATA_NONE
 */
const int Dir_xlate[4] = {
	0,			// implementation-dependent...
	SG_DXFER_TO_DEV,	// 1,CGC_DATA_WRITE
	SG_DXFER_FROM_DEV,	// 2,CGC_DATA_READ
	SG_DXFER_NONE		// 3,CGC_DATA_NONE
};

static ScsiCommand *
scsi_command_new (void)
{
	ScsiCommand *cmd;

	cmd = (ScsiCommand *) malloc (sizeof (ScsiCommand));
	memset (cmd, 0, sizeof (ScsiCommand));
	cmd->fd = -1;
	cmd->filename = NULL;
	cmd->autoclose = 1;

	return cmd;
}

static ScsiCommand *
scsi_command_new_from_fd (int f)
{
	ScsiCommand *cmd;

	cmd = scsi_command_new ();
	cmd->fd = f;
	cmd->autoclose = 0;

	return cmd;
}

static void
scsi_command_free (ScsiCommand * cmd)
{
	if (cmd->fd >= 0 && cmd->autoclose) {
		close (cmd->fd);
		cmd->fd = -1;
	}
	if (cmd->filename) {
		free (cmd->filename);
		cmd->filename = NULL;
	}

	free (cmd);
}

static int
scsi_command_transport (ScsiCommand * cmd, Direction dir, void *buf,
			size_t sz)
{
	int ret = 0;

	cmd->sg_io.dxferp = buf;
	cmd->sg_io.dxfer_len = sz;
	cmd->sg_io.dxfer_direction = DIRECTION (dir);

	if (ioctl (cmd->fd, SG_IO, &cmd->sg_io))
		return -1;

	if ((cmd->sg_io.info & SG_INFO_OK_MASK) != SG_INFO_OK) {
		errno = EIO;
		ret = -1;
		if (cmd->sg_io.masked_status & CHECK_CONDITION) {
			CREAM_ON_ERRNO (cmd->sg_io.sbp);
			ret = ERRCODE (cmd->sg_io.sbp);
			if (ret == 0)
				ret = -1;
		}
	}

	return ret;
}

static void
scsi_command_init (ScsiCommand * cmd, size_t i, int arg)
{
	if (i == 0) {
		memset (&cmd->cgc, 0, sizeof (cmd->cgc));
		memset (&cmd->_sense, 0, sizeof (cmd->_sense));
		cmd->cgc.quiet = 1;
		cmd->cgc.sense = &cmd->_sense.s;
		memset (&cmd->sg_io, 0, sizeof (cmd->sg_io));
		cmd->sg_io.interface_id = 'S';
		cmd->sg_io.mx_sb_len = sizeof (cmd->_sense);
		cmd->sg_io.cmdp = cmd->cgc.cmd;
		cmd->sg_io.sbp = cmd->_sense.u;
		cmd->sg_io.flags = SG_FLAG_LUN_INHIBIT | SG_FLAG_DIRECT_IO;
	}
	cmd->sg_io.cmd_len = i + 1;
	cmd->cgc.cmd[i] = arg;
}

int
get_disc_type (int fd)
{
	ScsiCommand *cmd;
	int retval = -1;
	unsigned char header[8];

	cmd = scsi_command_new_from_fd (fd);

	scsi_command_init (cmd, 0, 0x46);
	scsi_command_init (cmd, 1, 1);
	scsi_command_init (cmd, 8, 8);
	scsi_command_init (cmd, 9, 0);
	if (scsi_command_transport (cmd, READ, header, 8)) {
		/* GET CONFIGURATION failed */
		scsi_command_free (cmd);
		return -1;
	}
	
	retval = (header[6]<<8)|(header[7]);


	scsi_command_free (cmd);
	return retval;
}

/***************************************************************************
 *
 *   The CD media check code is nicked from hald and
 *   Copyright (C) 2003 David Zeuthen, <david@fubar.dk>
 *
 *   This kind of code should never have made it in here *sigh*
 * 
 ***************************************************************************/

static gboolean
disc_drive_is_mounted_linux (ThDiscDrive *drive)
{
	struct mntent  *mnte, mnt;
	FILE           *f;
	gchar           buf[512];

	if ((f = setmntent ("/etc/mtab", "r")) == NULL)
		return FALSE;

	while ((mnte = getmntent_r (f, &mnt, buf, sizeof(buf))) != NULL)
	{
		if (strcmp (drive->priv->device, mnt.mnt_fsname) == 0) 
		{
			endmntent (f);
			return TRUE;
		}
	}

	endmntent (f);
	return FALSE;
}

static gboolean
disc_drive_has_dvd_medium_linux (ThDiscDrive *drive)
{
	gboolean  got_media;
	gint      fd, drive_status, disc_type;

	got_media = FALSE;

	fd = open (drive->priv->device, O_RDONLY | O_NONBLOCK | O_EXCL);

	if (fd < 0 && errno == EBUSY) {
		/* this means the disc is mounted or some other app,
		 * like a cd burner, has already opened O_EXCL */
			
		/* HOWEVER, when starting hald, a disc may be
		 * mounted; so check /etc/mtab to see if it
		 * actually is mounted. If it is we retry to open
		 * without O_EXCL
		 */
		if (!disc_drive_is_mounted_linux (drive))
		{
			g_message ("Disc not mounted\n");
			return FALSE;
		}
		
		fd = open (drive->priv->device, O_RDONLY | O_NONBLOCK);
	}

	if (fd < 0)
	{
		g_message ("fd < 0 : %s\n", g_strerror (errno));
		return FALSE;
	}

	/* Check if a disc is in the drive
	 *
	 * @todo Use MMC-2 API if applicable
	 */
	drive_status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
	switch (drive_status) 
	{
		/* explicit fallthrough */
		case CDS_NO_INFO:
		case CDS_NO_DISC:
		case CDS_TRAY_OPEN:
		case CDS_DRIVE_NOT_READY:
		break;
			
		case CDS_DISC_OK:
			/* some CD-ROMs report CDS_DISK_OK even with an open
			 * tray; if media check has the same value two times in
			 * a row then this seems to be the case and we must not
			 * report that there is a media in it. */
			if (ioctl (fd, CDROM_MEDIA_CHANGED, CDSL_CURRENT) && 
			    ioctl (fd, CDROM_MEDIA_CHANGED, CDSL_CURRENT))
			{
				g_message ("CD-ROM drive %s: media checking is broken, assuming no CD is inside.", drive->priv->device);
			} else {
				/* g_message ("CD-ROM drive %s: got medium.", drive->priv->device); */
				got_media = TRUE;
			}
			break;
			
		case -1:
			g_warning ("CDROM_DRIVE_STATUS failed: %s\n", strerror(errno));
			break;

		default:
			break;
	}

	if (!got_media)
	{
		close (fd);
		return FALSE;
	}

	/* so, we have a medium in the drive - but is it a DVD? */

	/* check for audio/data/blank */
	disc_type = ioctl (fd, CDROM_DISC_STATUS, CDSL_CURRENT);
	switch (disc_type) {
	case CDS_MIXED:		/* mixed mode CD */
	case CDS_DATA_1:	/* data CD */
	case CDS_DATA_2:
	case CDS_XA_2_1:
	case CDS_XA_2_2:
		/* might be a DVD, dunno */
		break;

	case CDS_AUDIO:		/* audio CD */
	case CDS_NO_INFO:	/* blank or invalid CD */
	default:		/* should never see this */
		close (fd);
		return FALSE;
	}
	
	/* see table 373 in MMC-3 for details on disc type
	 * http://www.t10.org/drafts.htm#mmc3 */
	disc_type = get_disc_type (fd);	
	switch (disc_type) 
	{
		case 0x08: /* CD-ROM */
		case 0x09: /* CD-R */
		case 0x0a: /* CD-RW */
		default: 
			close (fd);
			return FALSE;

		case -1:
			g_message ("get_disc_type() failed. Assuming it is a DVD.\n");
			/* fallthrough */
		case 0x10: /* DVD-ROM */
		case 0x11: /* DVD-R Sequential */
		case 0x12: /* DVD-RAM */
		case 0x13: /* DVD-RW Restricted Overwrite */
		case 0x14: /* DVD-RW Sequential */
		case 0x1A: /* DVD+RW */
		case 0x1B: /* DVD+R */
			break;
	}

	close (fd);

	return TRUE;
}

#endif /* linux */

