/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

static TickerEnv	*env;
static Resource		*resrc;
static Params		*prm;
static int		shift_counter = 0;
#if USE_GUI
static GtkWidget	*popup_menu;
static GtkAccelGroup	*popup_menu_accel_group;
static int		popup_menu_accel_group_attached;
static int		position_x_in_drwa;
#endif
static GtkWidget	*main_hbox, *vbox_ticker, *vbox_clock;
static int		speed_up_flag = FALSE;
static int		speed_down_flag = FALSE;
static int		instance_id = 0;

#ifdef G_OS_WIN32
FILE			*stdout_fp, *stderr_fp;
#endif

/* prototypes for some static funcs */
static gint shift2left_callback();

#if USE_GUI
void toggle_speed_up_flag()
{
	speed_up_flag = TRUE;
}

void toggle_speed_down_flag()
{
	speed_down_flag = TRUE;
}
#endif

void check_main_win_always_on_top()
{
	if (prm->always_on_top == 'y')
		gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), TRUE);
	else
		gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);
}

#if USE_GUI
/*
 * funct_name2(***NO args***) only used to call funct_name(***args***)
 */
static void get_new_url2()
{
	get_new_url(resrc);
}

static void open_new_txt_file2()
{
	open_new_txt_file(resrc);
}

static void modify_params2()
{
	if (modify_params(prm) == 1)	/* 1 means we want to reopen the dialog */
		modify_params2(prm);
}

static void connection_settings2()
{
	int	suspend_rq_bak;
	char	disable_popups_bak;

	suspend_rq_bak = get_ticker_env()->suspend_rq;
	get_ticker_env()->suspend_rq = TRUE;
	disable_popups_bak = get_params()->disable_popups;	/* IS THIS ANY */
	get_params()->disable_popups = 'n';			/* BETTER? */
	if (connection_settings(AUTH_PAGE) == GTK_RESPONSE_OK) {
		current_feed();
		get_ticker_env()->reload_rq = TRUE;
	}
	get_params()->disable_popups = disable_popups_bak;
	get_ticker_env()->suspend_rq = suspend_rq_bak;
}

static void first_feed2()
{
	if (env->selection_mode == MULTIPLE)
		first_feed();
	else
		info_win_wait("Single selection mode", INFO_WIN_WAIT_TIMEOUT);
}

static void last_feed2()
{
	if (env->selection_mode == MULTIPLE)
		last_feed();
	else
		info_win_wait("Single selection mode", INFO_WIN_WAIT_TIMEOUT);
}

static void previous_feed2()
{
	if (env->selection_mode == MULTIPLE)
		previous_feed();
	else
		info_win_wait("Single selection mode", INFO_WIN_WAIT_TIMEOUT);
}

static void next_feed2()
{
	if (env->selection_mode == MULTIPLE)
		next_feed();
	else
		info_win_wait("Single selection mode", INFO_WIN_WAIT_TIMEOUT);
}

static void show_resource_info()
{
#define INFO_OPEN_TAG	""	/*"<b>"*/
#define INFO_CLOSE_TAG	""	/*"</b>"*/
	char	tmp[512];
	int	suspend_rq_bak;
	char	disable_popups_bak;

	/* TODO: improve layout - use a table? */
	suspend_rq_bak = get_ticker_env()->suspend_rq;
	get_ticker_env()->suspend_rq = TRUE;
	disable_popups_bak = get_params()->disable_popups;
	get_params()->disable_popups = 'n';
	if (resrc->type == RESRC_URL) {
		snprintf(tmp, 511, "\n"
			"Resource type:        "INFO_OPEN_TAG"%s Feed"INFO_CLOSE_TAG"\n\n"
			"Feed title:                "INFO_OPEN_TAG"%s"INFO_CLOSE_TAG"\n\n"
			"Feed URL:                "INFO_OPEN_TAG"%s"INFO_CLOSE_TAG"\n",
			(resrc->format == RSS_2_0 ? "RSS 2.0" : (resrc->format == RSS_ATOM ? "Atom" : "RSS 1.0")),
			resrc->feed_title, resrc->id);
		info_win("Resource Properties", tmp, INFO, TRUE);
	} else if (resrc->type == RESRC_FILE) {
		snprintf(tmp, 511, "\n"
			"Resource type:        "INFO_OPEN_TAG"File"INFO_CLOSE_TAG"\n\n"
			"File name:               "INFO_OPEN_TAG"%s"INFO_CLOSE_TAG"\n",
			resrc->id);
		info_win("Resource Properties", tmp, INFO, TRUE);
	} else
		info_win("Resource Properties", "\nNo information available\n", INFO_ERROR, FALSE);
	get_params()->disable_popups = disable_popups_bak;
	get_ticker_env()->suspend_rq = suspend_rq_bak;
}

#ifdef G_OS_WIN32
static int easily_link_with_browser2()
{
	char		*browser_cmd_p, browser_cmd[512], tmp[1024];
	unsigned int	i, count, start = 0, end = 0;
	int		exit_status = NO;

	if ((browser_cmd_p = (char *)get_default_browser_from_win32registry()) != NULL) {
		/* find 1st str in browser_cmd */
		str_n_cpy(tmp, browser_cmd_p, 511);
		for (i = 0, count = 0; i < strlen(tmp); i++) {
			if (tmp[i] == '"') {
				count++;
				if (count == 1)
					start = i + 1;
				else if (count == 2)
					end = i - 1;
			}
		}
		str_n_cpy(browser_cmd, tmp + start, MIN((end - start + 1), 511));
		snprintf(tmp, 1024,
			"\nThis is the Shell command that opens your default Browser:\n\n%s\n\n"
			"Do you want to use it (recommended) ?\n", browser_cmd);
		if (question_win(tmp) == YES) {
			exit_status = YES;
			str_n_cpy(prm->open_link_cmd, browser_cmd, FILE_NAME_MAXLEN);
			str_n_cpy(prm->open_link_args, "", FILE_NAME_MAXLEN);
			if (question_win("Save this parameter ?") == YES)
				save_to_config_file(prm);
		} else
			info_win(APP_NAME, "\nCancelled.\n"
				"You may too set this parameter manually in the Preferences window.\n",
				INFO, FALSE);
	} else
		info_win(APP_NAME" - Error", "\nCan't find Browser shell command."
			"You will have to set this parameter manually in the Preferences window.\n",
			INFO_ERROR, FALSE);
	return exit_status;
}

static void easily_link_with_browser()
{
	gtk_window_set_keep_above(GTK_WINDOW(env->win), FALSE);
	easily_link_with_browser2();
	check_main_win_always_on_top();
}
#endif

static void ticker_play()
{
	env->suspend_rq = FALSE;
}

static void ticker_pause()
{
	env->suspend_rq = TRUE;
}

static void ticker_reload()
{
	current_feed();
	env->reload_rq = TRUE;
	env->suspend_rq = FALSE;
}

static int get_visible_link()
{
	int	i, location_on_surface;

	location_on_surface = shift_counter * prm->shift_size + position_x_in_drwa;
	for (i = 0; i < NFEEDLINKANDOFFSETMAX; i++) {
		if (resrc->link_and_offset[i].offset_in_surface > location_on_surface) {
			str_n_cpy(env->active_link, resrc->link_and_offset[i].url, FILE_NAME_MAXLEN);
			return TRUE;
		}
	}
	env->active_link[0] = '\0';
	return FALSE;
}

static void open_link()
{
	char		tmp1[FILE_NAME_MAXLEN + 1], tmp2[FILE_NAME_MAXLEN + 1];
#ifdef G_OS_WIN32
	char		*tmp;
#endif
	char		*argv[32];	/* up to 32 - 3 (prog name, url, NULL) args */
	GPid		pid;
	GError		*error = NULL;
	int		i, j;

	if (prm->open_link_cmd[0] == '\0') {
#ifndef G_OS_WIN32
		/* TODO: should look for default browser */
		warning("Can't launch Browser: no command is defined.\n"
			"Please set the 'Open in Browser' option in the Preferences window.",
			"", "", "", FALSE);
		return;
#else
		easily_link_with_browser();
		if (prm->open_link_cmd[0] == '\0') {
			warning("Can't launch Browser: no command is defined.\n"
				"Please set the 'Open in Browser' option in the Preferences window.",
				"", "", "", FALSE);
			return;
		}
#endif
	}

	if (env->active_link [0] == '\0') {
		warning("No link found", "", "", "", FALSE);
		return;
	}

	fprintf(STD_OUT, "Spawning: %s %s %s\n", prm->open_link_cmd, prm->open_link_args, env->active_link);
#ifndef G_OS_WIN32
	str_n_cpy(tmp1, prm->open_link_cmd, FILE_NAME_MAXLEN);
#else
	tmp = g_win32_locale_filename_from_utf8(prm->open_link_cmd);
	str_n_cpy(tmp1, tmp, FILE_NAME_MAXLEN);
	g_free(tmp);
#endif
	str_n_cpy(tmp2, prm->open_link_args, FILE_NAME_MAXLEN);
	argv[0] = tmp1;
	for (i = 0, j = 1; tmp2[i] != '\0' && j < 32 + 1 - 3; j++) {
		argv[j] = &tmp2[i];
		while (tmp2[++i] != ' ')
			if (tmp2[i] == '\0')
				break;
		tmp2[i] = '\0';
		while (tmp2[++i] == ' ');
	}
	argv[j] = env->active_link;
	argv[j + 1] = NULL;
	if (!g_spawn_async(NULL, argv, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, &pid, &error)) {
		warning(APP_NAME": Can't create process ", argv[0], " - ", error->message, FALSE);
#ifndef G_OS_WIN32
		info_win(APP_NAME" - Warning", "Please check the 'Open in Browser' option in the Preferences window.",
			INFO_WARNING, FALSE);
#else
		if (easily_link_with_browser2() == YES)
			open_link();
#endif
	} else
		g_spawn_close_pid(pid);
}

static void open_link2()
{
	position_x_in_drwa = env->drwa_width / 2;
	get_visible_link();
	open_link();
}

static gint left_click_on_drawing_area(GtkWidget *widget, GdkEventButton *event, gpointer unused)
{
	widget = widget;
	unused = unused;
	if (event->type == GDK_BUTTON_PRESS && event->button == 1) {	/* 1 = mouse left button */
		/* get pointer x location on drawing_area */
		position_x_in_drwa = event->x;
		get_visible_link();
		open_link();
		return TRUE;
	} else
		return FALSE;
}

static gint right_click_on_drawing_area(GtkWidget *widget, GdkEventButton *event, gpointer unused)
{
	widget = widget;
	unused = unused;
	if (event->type == GDK_BUTTON_PRESS && event->button == 3) {	/* 3 = mouse right button */
		/* popup main menu */
		gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL, NULL, NULL, event->button, event->time);
		return TRUE;
	} else
		return FALSE;
}

/* pause ticker on mouseover */
static gint mouse_over_drawing_area(GtkWidget *widget, GdkEvent *event, gpointer unused)
{
	widget = widget;
	unused = unused;
	if (prm->pause_on_mouseover == 'y') {
		if (event->type == GDK_ENTER_NOTIFY)
			env->suspend_rq = TRUE;
		else if (event->type == GDK_LEAVE_NOTIFY)
			env->suspend_rq = FALSE;
	}
	if (resrc->type == RESRC_URL)
		gtk_widget_set_tooltip_text(widget, resrc->feed_title);
	else if (resrc->type == RESRC_FILE)
		gtk_widget_set_tooltip_text(widget, resrc->id);
	return TRUE;
}

static gint mouse_wheel_scroll_on_drawing_area(GtkWidget *widget, GdkEvent *event, gpointer unused)
{
	widget = widget;
	unused = unused;
	if (event->type == GDK_SCROLL) {
		env->suspend_rq = FALSE;
		if (event->scroll.direction == GDK_SCROLL_UP) {
			if (prm->mouse_wheel_scroll == 's')
				toggle_speed_up_flag();
			else if (prm->mouse_wheel_scroll == 'f')
				next_feed2();
		} else if (event->scroll.direction == GDK_SCROLL_DOWN) {
			if (prm->mouse_wheel_scroll == 's')
				toggle_speed_down_flag();
			else if (prm->mouse_wheel_scroll == 'f')
				previous_feed2();
		}
	}
	return TRUE;
}

/*
 * popup menu stuff
 */
static GtkItemFactoryEntry popup_menu_item[] = {
	{"/_File",		NULL, NULL, 0, "<Branch>", NULL},

	{"/File/Open Feed (_RSS|Atom)", "<control>R", get_new_url2, 0, "<StockItem>",
				(gconstpointer)RSS_ICON},

	{"/File/Open _Text File", "<control>T", open_new_txt_file2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_OPEN},

	{"/File/sep",		NULL, NULL, 0, "<Separator>", NULL},

	{"/File/_Import Feed List (OPML)", "<control>I", import_opml_file, 0, NULL, NULL},

	{"/File/_Export Feed List (OPML)", "<control>E", export_opml_file, 0, NULL, NULL},

	{"/File/sep",		NULL, NULL, 0, "<Separator>", NULL},

	{"/File/Import Preferences", "<control>", import_params, 0, NULL, NULL},

	{"/File/Export Preferences", "<control>", export_params, 0, NULL, NULL},

	{"/File/sep",		NULL, NULL, 0, "<Separator>", NULL},

	{"/File/Resource _Properties", "<control>P", show_resource_info, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_PROPERTIES},

	{"/File/sep",		NULL, NULL, 0, "<Separator>", NULL},

	{"/File/_Quit",		"<control>Q", gtk_main_quit, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_QUIT},

	{"/_Edit",		NULL, NULL, 0, "<Branch>", NULL},

	{"/Edit/Preference_s",	"<control>S", modify_params2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_PREFERENCES},

	{"/Edit/Connection Settings", "<control>", connection_settings2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_CONNECT},

	{"/_Control",		NULL, NULL, 0, "<Branch>", NULL},

	{"/Control/Open Link in _Browser", "<control>B", open_link2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_JUMP_TO/*GTK_STOCK_REDO*/},

	{"/Control/sep",	NULL, NULL, 0, "<Separator>", NULL},

	{"/Control/Play",	"<control>J", ticker_play, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_MEDIA_PLAY},

	{"/Control/Pause",	"<control>K", ticker_pause, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_MEDIA_PAUSE},

	{"/Control/Reload",	"<control>L", ticker_reload, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_REFRESH},

	{"/Control/sep",	NULL, NULL, 0, "<Separator>", NULL},

	{"/Control/First Feed",	"<control>", first_feed2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_GOTO_FIRST/*GTK_STOCK_MEDIA_PREVIOUS*/},

	{"/Control/Previous Feed", "<control>", previous_feed2, 0,  "<StockItem>",
				(gconstpointer)GTK_STOCK_GO_BACK/*GTK_STOCK_MEDIA_REWIND*/},

	{"/Control/Next Feed",	"<control>", next_feed2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_GO_FORWARD/*GTK_STOCK_MEDIA_FORWARD*/},

	{"/Control/Last Feed",	"<control>", last_feed2, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_GOTO_LAST/*GTK_STOCK_MEDIA_NEXT*/},

	{"/Control/sep",	NULL, NULL, 0, "<Separator>", NULL},

	{"/Control/Speed Up",	"<control>U", toggle_speed_up_flag, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_GO_UP},

	{"/Control/Speed Down",	"<control>D", toggle_speed_down_flag, 0,  "<StockItem>",
				(gconstpointer)GTK_STOCK_GO_DOWN},

	{"/_Help",		NULL, NULL, 0, "<Branch>", NULL},

	{"/Help/Quick Help",	"F1", help_win, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_HELP},

	{"/Help/Online Help",	"<control>H", online_help, 0, NULL, NULL},

	{"/Help/sep",		NULL, NULL, 0, "<Separator>", NULL},

	{"/Help/_About",	"<control>A", about_win, 0, "<StockItem>",
				(gconstpointer)GTK_STOCK_ABOUT},
};

static gint n_popup_menu_items = sizeof(popup_menu_item) / sizeof(popup_menu_item[0]);

static GtkWidget *get_popup_menu_menu(GtkWidget *win)
{
	GtkItemFactory	*item_factory;

	win = win;
	popup_menu_accel_group = gtk_accel_group_new();
	item_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<main>", popup_menu_accel_group);
	gtk_item_factory_create_items(item_factory, n_popup_menu_items, popup_menu_item, NULL);
	return gtk_item_factory_get_widget(item_factory, "<main>");
}
#endif

static gint update_win_dims()
{
	gtk_widget_set_size_request(env->drw_a, env->drwa_width, env->drwa_height);
	gtk_widget_set_size_request(env->drwa_clock, env->drwa_clock_width, env->drwa_height);

	gtk_widget_set_size_request(env->win, 0, 0);
	gtk_window_resize(GTK_WINDOW(env->win), env->drwa_width + env->drwa_clock_width, env->drwa_height);

	gtk_window_unmaximize(GTK_WINDOW(env->win));

	/* this is to prevent win from becoming 'unshowable' */
	if (prm->icon_in_taskbar == 'n')
		gtk_window_deiconify(GTK_WINDOW(env->win));
	/*
	 * not about win dims but this must be checked too
	 * (is there a better place?)
	 */
	check_main_win_always_on_top();
	return TRUE;
}

gboolean params_have_been_changed(Params *new_prm, int n_required_runs)
{
	static Params	prm_bak;
	static int	initial_runs = 0;

	if (initial_runs < n_required_runs) {
		/* 'fake params changes' are required at program startup */
		initial_runs++;
		memcpy((void *)&prm_bak, (const void *)new_prm, sizeof(Params));
		return TRUE;
	} else if (memcmp((const void *)new_prm, (const void *)&prm_bak, sizeof(Params)) != 0) {
		memcpy((void *)&prm_bak, (const void *)new_prm, sizeof(Params));
		return TRUE;
	} else
		return FALSE;
}

/*
 * create new or next cairo image surface & compute stuff to redraw window
 */
static int compute_surface_and_win()
{
	char 		fname[FONT_NAME_MAXLEN + 1], fsize[FONT_SIZE_MAXLEN + 1];
	char		cfname[FONT_NAME_MAXLEN + 1], cfsize[FONT_SIZE_MAXLEN + 1];
	gint		height1, height2;
	int		size, render_exit_status, i;
	char		tmp[80];

	gtk_window_set_keep_above(GTK_WINDOW(env->win), FALSE);
	gtk_window_set_skip_taskbar_hint(GTK_WINDOW(env->win), FALSE);
	gtk_window_unstick(GTK_WINDOW(env->win));
	/*
	 * compute font size from requested height if > 0
	 */
	split_font(prm->font_n_s, fname, fsize);
	if (prm->win_h > 0)
		snprintf(fsize, FONT_SIZE_MAXLEN + 1, "%3d",
			get_fsize_from_layout_height((gint)prm->win_h, fname));
	/* in all cases, font size can't be > FONT_MAXSIZE */
	if (atoi(fsize) > FONT_MAXSIZE)
		snprintf(fsize, FONT_SIZE_MAXLEN + 1, "%3d", FONT_MAXSIZE);
	compact_font(prm->font_n_s, fname, fsize);
	/*
	 * compute clock font size
	 * clock height = ticker height unless clock font size is set
	 * in which case clock height is always <= ticker height
	 */
	split_font(prm->clock_font_n_s, cfname, cfsize);
	if (strcmp(fname, cfname) == 0) {
		if (atoi(cfsize) > atoi(fsize))
			str_n_cpy(cfsize, fsize, FONT_SIZE_MAXLEN);
	} else {
		height1 = get_layout_height_from_fnamesize(prm->clock_font_n_s);
		height2 = get_layout_height_from_fnamesize(prm->font_n_s);
		if (height1 > height2) {
			size = get_fsize_from_layout_height(height2, cfname);
			snprintf(cfsize, FONT_SIZE_MAXLEN + 1, "%3d", size);
		}
	}
	compact_font(prm->clock_font_n_s, cfname, cfsize);
	/*
	 * compute ticker width
	 */
	if (prm->win_w >= DRWA_WIDTH_MIN && prm->win_w <= env->screen_w)
		env->drwa_width = prm->win_w;
	else if (prm->win_w < DRWA_WIDTH_MIN)
		env->drwa_width = DRWA_WIDTH_MIN;
	else if (prm->win_w > env->screen_w)
		env->drwa_width = env->screen_w;
	env->drwa_width -= (env->drwa_clock_width = get_clock_width(prm));
	/*
	 * reset link_and_offset stuff
	 */
	if(resrc->id[0] == '\0' && resrc->fp != NULL) {
		fclose(resrc->fp);
		resrc->fp = NULL;
		for (i = 0; i < NFEEDLINKANDOFFSETMAX; i++) {
			resrc->link_and_offset[i].url[0] = '\0';
			resrc->link_and_offset[i].offset_in_surface = 0;
		}
	}
	/*
	 * create cairo image surface of rendered text (one long single line)
	 */
	env->c_surf = render_stream_to_surface(resrc->fp, resrc->link_and_offset, prm, &render_exit_status);
	if (render_exit_status != OK && render_exit_status != RENDER_NO_RESOURCE) {
		switch(render_exit_status) {
			case RENDER_NULL_ENDINGSTR:
				warning("render_stream_to_surface():", "ending string = NULL", "", "", FALSE);
				break;
			case RENDER_GETLINE_ERROR:
				warning("render_stream_to_surface():", "getline() error", "", "", FALSE);
				break;
			case RENDER_LINE_TOO_LONG:
				warning("render_stream_to_surface():", "line too long\n", "", "", FALSE);
				break;
			case RENDER_CAIRO_IMAGE_SURFACE_TOO_WIDE:
				big_error(RENDER_CAIRO_IMAGE_SURFACE_TOO_WIDE, "render_stream_to_surface():",
					"cairo image surface too wide (> 32 K pixels)", "", "");
				break;
			case RENDER_CREATE_CAIRO_IMAGE_SURFACE_ERROR:
				big_error(RENDER_CREATE_CAIRO_IMAGE_SURFACE_ERROR, "render_stream_to_surface():",
					"can't create cairo image surface", "", "");
				break;
			default: warning("render_stream_to_surface():", "unknown error", "", "", FALSE);
				break;
		}
	}
	if (env->c_surf == NULL)
		big_error(render_exit_status,
			"render_stream_to_surface():", "cairo image surface = NULL", "", "");
	env->surf_width = cairo_image_surface_get_width(env->c_surf);
	env->surf_height = cairo_image_surface_get_height(env->c_surf);
	env->drwa_height = (MIN(env->surf_height, env->screen_h));
	/*
	 * window layout stuff
	 */
	/* win title */
	if (resrc->type == RESRC_URL)
		snprintf(tmp, 80, "%s-%s  |  %s", APP_NAME, APP_VERSION_NUMBER, resrc->feed_title);
	else if (resrc->type == RESRC_FILE)
		snprintf(tmp, 80, "%s-%s  |  %s", APP_NAME, APP_VERSION_NUMBER, resrc->id);
	else
		snprintf(tmp, 80, "%s-%s  |  %s", APP_NAME, APP_VERSION_NUMBER, "No resource");
	gtk_window_set_title(GTK_WINDOW(env->win), tmp);
	/*
	 * when window-always-on-top is disabled, the following code needs to be run
	 * only twice at program startup (ie once after gtk_widget_show_all() has been
	 * called), then whenever params are changed, but not when a new feed is loaded
	 * (this is to fix an issue with Unity on Ubuntu)
	 */
	if (prm->always_on_top == 'n' && !params_have_been_changed(prm, 2))
		return render_exit_status;
	/* move win */
	gtk_window_move(GTK_WINDOW(env->win), prm->win_x, prm->win_y);
	/* win decoration */
	if (prm->windec == 'y') {
		gtk_window_set_decorated(GTK_WINDOW(env->win), TRUE);
#if USE_GUI
		if (!popup_menu_accel_group_attached) {
			gtk_window_add_accel_group(GTK_WINDOW(env->win), popup_menu_accel_group);
			popup_menu_accel_group_attached = TRUE;
		}
#endif
	} else {
		gtk_window_set_decorated(GTK_WINDOW(env->win), FALSE);
#if USE_GUI
		if (popup_menu_accel_group_attached) {
			gtk_window_remove_accel_group(GTK_WINDOW(env->win), popup_menu_accel_group);
			popup_menu_accel_group_attached = FALSE;
		}
#endif
	}
	/* clock */
	if (prm->clock == 'l')
		gtk_box_reorder_child(GTK_BOX(main_hbox), vbox_clock, 0);
	else if (prm->clock == 'r')
		gtk_box_reorder_child(GTK_BOX(main_hbox), vbox_clock, 1);
	/* whole window transparency */
	gtk_window_set_opacity(GTK_WINDOW(env->win), prm->win_transparency);
	/* window icon in taskbar? */
	if (prm->icon_in_taskbar == 'y')
		gtk_window_set_skip_taskbar_hint(GTK_WINDOW(env->win), FALSE);
	else if (prm->icon_in_taskbar == 'n')
		gtk_window_set_skip_taskbar_hint(GTK_WINDOW(env->win), TRUE);
	/* window on all desktops? */
	if (prm->win_sticky == 'y') {
		gtk_window_stick(GTK_WINDOW(env->win));
		gtk_window_present(GTK_WINDOW(env->win));	/* seems necessary but not sure why */
	} else if (prm->win_sticky == 'n')
		gtk_window_unstick(GTK_WINDOW(env->win));
	check_main_win_always_on_top();
	return render_exit_status;
}

/*
 * timeout handler to render one part of cairo image surface onto drawing
 * area - image is shifted to left by <shift_size> pixels
 *
 * handler 'listens' to flags: suspend_rq, reload_rq, compute_rq and
 * stream_fully_read
 *
 * notes:
 * - reload_rq may be a misleading name as, in multiple selection mode,
 * we will load next stream, not reload the same one
 * - stream_fully_read is only relevant in multiple selection mode because
 * there is no need to reload each time in single selection mode
 */
static gint shift2left_callback()
{
	cairo_t		*cr;
	GdkRectangle	r;
	char		tmp[256];
	int		i;

	if (env->suspend_rq)
		return TRUE;	/* does nothing - just return */
	else if ((shift_counter * prm->shift_size < env->surf_width - env->drwa_width - 8) &&\
			/*!(env->selection_mode == MULTIPLE && env->stream_fully_read) &&\*/
			!env->reload_rq && !env->compute_rq) {
		env->suspend_rq = TRUE;
		/*
		 * draw onto ticker area
		 * (we now use cairo instead of deprecated gdk_draw_ stuff)
		 */
		/* double buffering disabled for drawing area so we need this */
		gdk_window_get_geometry(env->drw_a->window, &(r.x), &(r.y), NULL, NULL, NULL);
		if (prm->clock == 'l')			/* we must do that but noy quite sure why */
			r.x -= env->drwa_clock_width;
		r.width = env->drwa_width;
		r.height = env->drwa_height;
		gdk_window_begin_paint_rect(env->drw_a->window, &r);
		/* cairo stuff */
		cr = gdk_cairo_create(GDK_DRAWABLE(env->drw_a->window));
		/* dest_x - src_x, dest_y - src_y */
		cairo_set_source_surface(cr, env->c_surf, - (shift_counter++ * prm->shift_size), 0);
		/* dest_x, dest_y */
		cairo_rectangle(cr, 0, 0, env->drwa_width, env->drwa_height);
		cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
		cairo_fill(cr);
		cairo_destroy(cr);
		/* we're done */
		gdk_window_end_paint(env->drw_a->window);
		env->suspend_rq = FALSE;
		/*
		 * speeding up/down on the fly
		 */
		if (speed_up_flag == TRUE || speed_down_flag == TRUE) {
			if (speed_up_flag == TRUE) {
				speed_up_flag = FALSE;
				if (prm->delay > 1)
					prm->delay--;
				else
					return TRUE;
			} else if (speed_down_flag == TRUE) {
				speed_down_flag = FALSE;
				if (prm->delay < 50)
					prm->delay++;
				else
					return TRUE;
			}
			g_timeout_add_full(G_PRIORITY_DEFAULT, prm->delay, shift2left_callback, NULL, NULL);
			return FALSE;
		} else
			return TRUE;
	} else {
		/*
		 * when cairo_image surface scrolling completed or if request to
		 * reload or recompute everything, will create new surface or next one
		 */
		env->suspend_rq = TRUE;
		if (env->c_surf != NULL) {
			cairo_surface_destroy(env->c_surf);
			env->c_surf = NULL;
		}
		i = OK;
		if (env->stream_fully_read || env->reload_rq) {
    			if (env->selection_mode == SINGLE && env->reload_rq)
				load_resource(resrc, NULL);
			else {
				while ((i = load_resource(resrc, get_selected_url_list())) != OK) {
					if (i == CONNECT_TOO_MANY_ERRORS && env->selection_mode == MULTIPLE) {
						if (get_params()->disable_popups == 'n') {
							snprintf(tmp, 256,
							"\nFailed to connect %d times in a row.\n\n"
							"Please check your internet connection and/or "
							"your connection settings.\n\n"
							"Switch to single selection mode ?\n",
							CONNECT_FAIL_MAX);
							if (question_win(tmp) == YES) {
								env->selection_mode = SINGLE;
								break;
							}
						} else {
							fprintf(STD_ERR,
							"Failed to connect %d times in a row\n"
							"Please check your internet connection and/or "
							"your connection settings\n",
							CONNECT_FAIL_MAX);
							/* TODO: improve that */
							if (gtk_events_pending()) {
								if (gtk_main_iteration_do(FALSE))
									break;
							} else
#ifndef G_OS_WIN32
								sleep(CONNECT_FAIL_TIMEOUT);
#else
								Sleep(CONNECT_FAIL_TIMEOUT * 1000);
#endif
						}
					}
				}
			}
			check_time_load_resource(TIME_RESET);
		}
		compute_surface_and_win();
		shift_counter = prm->shift_size;	/* to avoid displaying twice the same thing */
		g_timeout_add_full(G_PRIORITY_DEFAULT, prm->delay, shift2left_callback, NULL, NULL);
		env->reload_rq = FALSE;
		env->compute_rq = FALSE;
		env->suspend_rq = FALSE;
		return FALSE;
	}
}

static gint display_time2()
{
	display_time(prm);
	return TRUE;
}

static gint get_win_position()	/* only if win is 'draggable' */
{
	if (prm->windec == 'y')
		gtk_window_get_position(GTK_WINDOW(env->win), &prm->win_x, &prm->win_y);
	return TRUE;
}

/*
 * reload stuff every <resrc->rss_ttl * 60> seconds
 */
void check_time_load_resource(check_time_mode mode)
{
	static unsigned long	elapsed_time_in_sec;

	if (mode == TIME_RESET)
		elapsed_time_in_sec = 0;
	else if (mode == TIME_CHECK) {
		if (resrc->type == RESRC_FILE)
			resrc->rss_ttl = prm->rss_refresh;
		if ((++elapsed_time_in_sec) / 60 >= (unsigned long)resrc->rss_ttl) {
			elapsed_time_in_sec = 0;
			env->reload_rq = TRUE;
		}
	}
}

gint check_time_load_resource2()
{
	check_time_load_resource(TIME_CHECK);
	return TRUE;
}

TickerEnv *get_ticker_env()
{
	return env;
}

Resource *get_resource()
{
	return resrc;
}

Params *get_params()
{
	return prm;
}

static void quit_with_cli_info(tickr_error_code error_code)
{
	if (resrc->fp != NULL)
		fclose(resrc->fp);
	xmlCleanupParser();
	free2(env);
	free2(resrc);
	free2(prm);
	fprintf(STD_ERR, "Try '"APP_CMD" --help' for more information\n");
	exit(error_code);
}

static void print_help()
{
	int	i;

	for (i = 0; get_help_str1()[i] != NULL; i++)
		fprintf(STD_OUT, "%s", get_help_str1()[i]);
	fprintf(STD_OUT, "\n");
}

static void print_version()
{
	fprintf(STD_OUT, APP_NAME" version "APP_VERSION_NUMBER"\n");
}

static void print_license()
{
	int	i;

	for (i = 0; get_license_str1()[i] != NULL; i++)
		fprintf(STD_OUT, "%s", get_license_str1()[i]);
	fprintf(STD_OUT, "%s\n", get_license_str2());
}

int get_instance_id()
{
	return instance_id;
}

/* reload resource as soon as scrolling has started to update all display params
 * TODO: reset open stream to begining and recompute instead of reload (which
 * is useless) -> must 1st make sure this doesn't break anything
 */
gint update_everything()
{
	if (shift_counter > 4) {
		current_feed();
		return FALSE;
	} else
		return TRUE;
}

#ifdef G_OS_WIN32
/* this is meant to be used with the new gtk runtime */
static int init_win32_mmtimer()
{
	TIMECAPS	tc;
	MMRESULT	mmr;
	unsigned	int highest_res;

	mmr = timeGetDevCaps(&tc, sizeof(TIMECAPS));
	highest_res = tc.wPeriodMin;
	if (timeBeginPeriod(highest_res) == TIMERR_NOERROR)
		return 0;
	else {
		fprintf(STD_ERR, "init_win32_mmtimer() error");
		return -1;
	}
}
#endif

int main(int argc, char *argv[])
{
	GdkPixbuf	*pixb = NULL;
	GtkIconFactory	*icon_factory = NULL;
	GError		*error = NULL;
#ifndef G_OS_WIN32
	GdkColormap	*colormap = NULL;
#endif
	int		n_options, n_resources, i, j;
#ifdef G_OS_WIN32
	WSADATA		wsadata;
#endif

	gtk_init(&argc, &argv);
	LIBXML_TEST_VERSION
#ifndef LIBXML_TREE_ENABLED
# error Libxml2: tree support not compiled in
#endif
	xmlInitParser();
	env = malloc2(sizeof(TickerEnv));
	resrc = malloc2(sizeof(Resource));
	prm = malloc2(sizeof(Params));
	env->win = env->drw_a = env->drwa_clock = NULL;
	env->c_surf = NULL,
	env->screen = NULL;
	env->visual = NULL;
	env->screen_w = env->screen_h = env->depth = 0;
	env->drwa_width = env->drwa_height = env->surf_width = env->surf_height = 0;
	env->active_link[0] = '\0';
	env->suspend_rq = env->compute_rq = env->reload_rq = env->stream_fully_read = FALSE;
	/* env->selection_mode: default = MULTIPLE, set to SINGLE internally if
	 * selection is unavalaible/empty or after picking a single feed */
	env->selection_mode = MULTIPLE;
	/*
	 * toplevel window
	 */
	env->win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(env->win), APP_NAME"-"APP_VERSION_NUMBER);
	gtk_container_set_border_width(GTK_CONTAINER(env->win), 0);
	g_signal_connect(G_OBJECT(env->win), "delete_event", G_CALLBACK(gtk_main_quit), NULL);
	g_signal_connect(G_OBJECT(env->win), "destroy", G_CALLBACK(gtk_widget_destroy), &(env->win));

#ifdef G_OS_WIN32
	if ((i = WSAStartup(MAKEWORD(2, 2), &wsadata)) != 0)
		big_error(WIN32V_ERROR, "WSAStartup() error ", itoa2(i), "", "");
	else if (LOBYTE(wsadata.wVersion) != 2 || HIBYTE(wsadata.wVersion) != 2) {
		WSACleanup();
		big_error(WIN32V_ERROR, "Couldn't find a usable version of Winsock.dll", "", "", "");
	} else if (get_progfiles_dir() == NULL)
		big_error(WIN32V_ERROR, "Can't find Program Files directory", "", "", "");
	else if (get_appdata_dir() == NULL)
		big_error(WIN32V_ERROR, "Can't find Application Data directory", "", "", "");
	else if (init_win32_mmtimer() != 0)
		if (question_win("Can't initialize win32 multimedia timer. "
				"Scrolling may be very slow. Continue ?") == NO)
			exit(WIN32V_ERROR);
#endif
	/*
	 * create news dir in user home dir if it doesn't exist
	 */
	g_mkdir(get_datadir_full_path(), S_IRWXU);
#ifdef G_OS_WIN32
	STD_OUT = open_new_datafile_with_name(STDOUT_FILENAME, "wb+");
	STD_ERR = open_new_datafile_with_name(STDERR_FILENAME, "wb+");
#endif
	/*
	 * get help, show license, set instance id or check available system fonts
	 */
	if (argc >= 2 && (strcmp(argv[1], "-help") == 0 || strcmp(argv[1], "--help") == 0 ||\
			strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "-?") == 0)) {
		print_help();
		return 0;
	} else if (argc >= 2 && (strcmp(argv[1], "-version") == 0 || strcmp(argv[1], "--version") == 0 ||\
			strcmp(argv[1], "-v") == 0)) {
		print_version();
		return 0;
	} else if (argc >= 2 && (strcmp(argv[1], "-license") == 0 || strcmp(argv[1], "--license") == 0 ||\
			strcmp(argv[1], "-l") == 0)) {
		print_license();
		return 0;
	} else if (argc >= 2 && strncmp(argv[1], "-instance-id=", strlen("-instance-id=")) == 0) {
		if ((instance_id = atoi(argv[1] + strlen("-instance-id="))) < 1 || instance_id > 99) {
			fprintf(STD_ERR, "Instance ID invalid value: %d\n", instance_id);
			instance_id = 0;
		}
	} else if (argc >= 2 && (strcmp(argv[1], "-dumpfontlist") == 0 || strcmp(argv[1], "--dumpfontlist") == 0)) {
		dump_font_list();
		return 0;
	}
	set_news_icon_to_dialog(GTK_WINDOW(env->win));
	/*
	 * create rss icon
	 */
	 pixb = gdk_pixbuf_new_from_file(get_imagefile_full_name_from_name(RSS_ICON), &error);
	icon_factory = gtk_icon_factory_new();
	gtk_icon_factory_add(icon_factory, RSS_ICON, gtk_icon_set_new_from_pixbuf(pixb));
	gtk_icon_factory_add_default(icon_factory);
	if (pixb != NULL)
		g_object_unref(pixb);
	/*
	 * get some system graphical params
	 */
	env->screen = gtk_widget_get_screen(env->win);
#ifndef G_OS_WIN32
	if ((colormap = gdk_screen_get_rgba_colormap(env->screen)) != NULL) {
#ifdef VERBOSE_OUTPUT
		fprintf(STD_OUT, "Composited GDK screen found\n");
#endif
	} else
		colormap = gdk_screen_get_rgb_colormap(env->screen);
	gtk_widget_set_colormap(env->win, colormap);
	env->visual = gdk_colormap_get_visual(colormap);
#else
	env->visual = gdk_visual_get_best();
#endif
	env->depth = env->visual->depth;
	env->screen_w = gdk_screen_get_width(gtk_window_get_screen(GTK_WINDOW(env->win)));
	env->screen_h = gdk_screen_get_height(gtk_window_get_screen(GTK_WINDOW(env->win)));
#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "GDK visual: screen width = %d, screen height = %d, depth = %d\n",
		env->screen_w, env->screen_h, env->depth);
#endif
	resrc->type = RESRC_UNSPECIFIED;
	resrc->format = RSS_FORMAT_UNDETERMINED;
	resrc->id[0] = '\0';
	resrc->xml_dump[0] = '\0';
	resrc->fp = NULL;
	for (i = 0; i < NFEEDLINKANDOFFSETMAX; i++) {
		resrc->link_and_offset[i].url[0] = '\0';
		resrc->link_and_offset[i].offset_in_surface = 0;
	}
	init_authentication();
	init_proxy();
	/*
	 * options and resource stuff
	 */
	set_default_options(prm);	/* init all params */
	get_config_file_options(prm);

	n_options = 0;
	n_resources = 0;
	if (argc > 1) {
		/* parse args: first -options... then resource - the rest is ignored */
		for (i = 1; i < argc; i++)
			if (argv[i][0] == '-')
				n_options ++;
			else
				break;
		if (i < argc)
			n_resources++;
		if (n_resources == 0) {
			fprintf(STD_ERR, "No resource specified\n");
			if (RESOURCE_NONE_QUIT)
				quit_with_cli_info(RESOURCE_NONE);
		} else {
			env->selection_mode = SINGLE;
			/* will open text file or get rss feed */
			str_n_cpy(resrc->id, argv[n_options + 1], FILE_NAME_MAXLEN);
		}
		if (n_options > 0)
			if (parse_options_array(prm, n_options, (const char **)(argv + 1)) != OK)
				if (OPTION_ERROR_QUIT)
					quit_with_cli_info(OPTION_ERROR);
	} else {
		fprintf(STD_ERR, "No resource specified\n");
		if (RESOURCE_NONE_QUIT)
			quit_with_cli_info(RESOURCE_NONE);
	}
	compute_auth_and_proxy_str();
	init_url_list();
	if ((i = load_url_list(get_url_list())) == OK)
		save_url_list(get_url_list());
	init_selected_url_list();
	j = load_selected_url_list(get_selected_url_list());
	if (n_resources == 0 && env->selection_mode == MULTIPLE && i == OK && j == OK)
		fprintf(STD_OUT, "Will use feed selection\n");
	else {
		if (env->selection_mode == MULTIPLE && (i != OK || j != OK)) {
			if (j == SELECTION_EMPTY)
				warning("No feed selected", "-",
					"Switching to single selection mode", "", TRUE);
			else if (j == SELECTION_ERROR)
				warning("No feed selection available", "-",
					"Switching to single selection mode", "", TRUE);
		}
		env->selection_mode = SINGLE;
		if (n_resources == 0) {
			if (prm->homefeed[0] != '\0') {
				str_n_cpy(resrc->id, prm->homefeed, FILE_NAME_MAXLEN);
				fprintf(STD_OUT, "Will use default resource (Homefeed): %s\n", resrc->id);
			} else {
				fprintf(STD_ERR, "No default resource available\n");
				if (RESOURCE_NONE_QUIT)
					quit_with_cli_info(RESOURCE_NONE);
			}
		}
	}
	first_feed();
	if (load_resource(resrc, get_selected_url_list()) != OK)
		if (RESOURCE_NONE_QUIT)
			quit_with_cli_info(RESOURCE_NONE);
	resrc->rss_ttl = prm->rss_refresh;
	first_feed();
	/*
	 * set window layout
	 */
	main_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(env->win), main_hbox);
#if USE_GUI
	popup_menu = get_popup_menu_menu(NULL);
	popup_menu_accel_group_attached = FALSE;
#endif
	/*
	 * create 2 drawing areas
	 */
	env->drw_a = gtk_drawing_area_new();
	/* we disable double buffering for drawing area and will use
	 * gdk_window_begin_paint_rect() and gdk_window_end_paint()
	 */
	gtk_widget_set_double_buffered(env->drw_a, FALSE);
	env->drwa_clock = gtk_drawing_area_new();
	gtk_widget_add_events(env->drw_a, GDK_ALL_EVENTS_MASK);	/* change mask? */
#if USE_GUI
	g_signal_connect(G_OBJECT(env->drw_a), "button-press-event", G_CALLBACK(left_click_on_drawing_area), NULL);
	g_signal_connect(G_OBJECT(env->drw_a), "button-press-event", G_CALLBACK(right_click_on_drawing_area), NULL);
	g_signal_connect(G_OBJECT(env->drw_a), "enter-notify-event", G_CALLBACK(mouse_over_drawing_area), NULL);
	g_signal_connect(G_OBJECT(env->drw_a), "leave-notify-event", G_CALLBACK(mouse_over_drawing_area), NULL);
	g_signal_connect(G_OBJECT(env->drw_a), "scroll-event", G_CALLBACK(mouse_wheel_scroll_on_drawing_area), NULL);
#endif
	vbox_ticker = gtk_vbox_new(FALSE, 0);	/* vbox_ticker is only used to have a v-centered ticker */
	gtk_box_pack_start(GTK_BOX(vbox_ticker), GTK_WIDGET(env->drw_a), TRUE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_hbox), vbox_ticker, FALSE, FALSE, 0);
	vbox_clock = gtk_vbox_new(FALSE, 0);	/* vbox_clock is only used to have a v-centered clock */
	gtk_box_pack_start(GTK_BOX(vbox_clock), GTK_WIDGET(env->drwa_clock), TRUE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_hbox), vbox_clock, FALSE, FALSE, 0);
	/*
	 * create cairo image surface from text file or rss feed & compute stuff to draw window
	 */
	compute_surface_and_win();

	/* === timeout callbacks === */
	/*
	 * refresh the image every <delay> milliseconds
	 */
	g_timeout_add_full(G_PRIORITY_DEFAULT, prm->delay, shift2left_callback, NULL, NULL);
	/*
	 * display clock every 0.5 seconds
	 */
	g_timeout_add_full(G_PRIORITY_DEFAULT, 500, display_time2, NULL, NULL);
	/*
	 * get current win position (if 'dragged') every 0.5 seconds
	 */
	g_timeout_add_full(G_PRIORITY_DEFAULT, 500, get_win_position, NULL, NULL);
	/*
	 * check every sec until elapsed time in mn > <resrc->rss_ttl> mn then reload rss feed (or text file)
	 */
	check_time_load_resource(TIME_RESET);
	g_timeout_add_full(G_PRIORITY_DEFAULT, 1000, check_time_load_resource2, NULL, NULL);

	g_timeout_add_full(G_PRIORITY_DEFAULT, 500, update_win_dims, NULL, NULL);

	g_timeout_add_full(G_PRIORITY_DEFAULT, 100, update_everything, NULL, NULL);

	env->suspend_rq = FALSE;
	env->compute_rq = FALSE;
	env->reload_rq = TRUE;

	gtk_widget_show_all(env->win);
	update_win_dims();
	gtk_main();
	/*
	 * cleanup stuff
	 */
	if (resrc->fp != NULL)
		fclose(resrc->fp);
	xmlCleanupParser();
#ifdef G_OS_WIN32
	WSACleanup();
#endif
	free2(env);
	free2(resrc);
	free2(prm);
	return 0;
}
