/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2011
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif

static int	depth;
static xmlNode	*item_element, *entry_element;
static int	n;
static int	counter;

/*
 * here, 'rss' is sometimes used as a synonym of 'feed' and sometimes
 * used by opposition to 'atom' (this can be confusing)
 */

/*
 * look for url and, if valid, parse it then dump result into
 * '<user_home_dir>/'NEWS_DIR_NAME/XMLDUMP'
 * if url not valid, only set error code and return
 */
int get_feed(Resource *resrc, const Params *prm)
{
	char		feed_title[FEED_TITLE_MAXLEN + 1];
	char		feed_link[FILE_NAME_MAXLEN + 1];
	char		feed_ttl[32];
	char		file_name[FILE_NAME_MAXLEN + 1];
	char		url[FILE_NAME_MAXLEN + 1];
	int		exit_status, i;

	get_ticker_env()->suspend_rq = TRUE;

	resrc->rss_ttl = prm->rss_refresh;
	str_n_cpy(resrc->xml_dump, get_datafile_full_name_from_name(XMLDUMP), FILE_NAME_MAXLEN);
	if (resrc->fp != NULL)
		fclose(resrc->fp);
	resrc->fp = open_new_datafile_with_name(XMLDUMP, "wb");
	str_n_cpy(file_name, get_datafile_full_name_from_name(RESOURCEDUMP), FILE_NAME_MAXLEN);

	/* we replace resrc->id with modified url, file_name = downloaded resource */
	if ((exit_status = fetch_resource((const char *)resrc->id, (const char *)file_name, url)) == OK)
#ifdef VERBOSE_OUTPUT
		fprintf(STD_OUT, "Resource fetched: %s\n", (const char *)resrc->id);
#endif
	else if (exit_status == FEED_FORMAT_ERROR) {
		warning("Feed format error (RSS2.0/Atom) in:", (const char *)resrc->id, "", "",
			get_ticker_env()->selection_mode == MULTIPLE);
		return FEED_FORMAT_ERROR;
	} else if (exit_status == CONNECT_TOO_MANY_ERRORS) {
		return CONNECT_TOO_MANY_ERRORS;
	} else {
		warning("Can't fetch resource:", (const char *)resrc->id, "", "",
			get_ticker_env()->selection_mode == MULTIPLE);
		return FEED_DOWNLOAD_ERROR;
	}
	for (i = 0; i < NFEEDLINKANDOFFSETMAX; i++) {
		resrc->link_and_offset[i].offset_in_surface = 0;
		(resrc->link_and_offset[i].url)[0] = '\0';
	}
	resrc->format = RSS_2_0;
	/* we use file_name instead of resrc->id */
	if ((exit_status = parse_rss20_xml_file(resrc->fp, file_name,
			resrc->link_and_offset, prm)) == RSS_NO_ITEM_ELEMENT) {
		resrc->format = RSS_ATOM;
		for (i = 0; i < NFEEDLINKANDOFFSETMAX; i++) {
			resrc->link_and_offset[i].offset_in_surface = 0;
			(resrc->link_and_offset[i].url)[0] = '\0';
		}
		/* we use file_name instead of resrc->id */
		if ((exit_status = parse_atom_xml_file(resrc->fp, file_name,
				resrc->link_and_offset, prm)) == ATOM_NO_ENTRY_ELEMENT)
			warning("No 'Item' or 'Entry' element found in:", resrc->id,
				"\nIs feed format RSS 2.0 or Atom?",
				"\n(RSS 1.0 format is not currently supported)",
				get_ticker_env()->selection_mode == MULTIPLE);
	}
	if (exit_status != OK) {
		resrc->format = RSS_FORMAT_UNDETERMINED;
		return exit_status;
	}
	fclose(resrc->fp);
	resrc->fp = open_new_datafile_with_name(XMLDUMP, "rb");

	/* we use file_name instead of resrc->id */
	get_feed_info(file_name, feed_title, feed_link, feed_ttl);
	str_n_cpy(resrc->feed_title, feed_title, FEED_TITLE_MAXLEN);
	if (feed_ttl[0] != '\0')
		resrc->rss_ttl = atoi(feed_ttl);

	get_ticker_env()->suspend_rq = FALSE;
	return OK;
}

/*
 * must be utf-8 encoded
 */
int parse_rss20_xml_file(FILE *fp, const char* file_name, FeedLinkAndOffset *link_and_offset, const Params *prm)
{
	xmlDoc	*doc;
	xmlNode	*root_element;

#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "Parsing XML file ... ");
#endif
	if ((doc = xmlParseFile(file_name)) == NULL) {
		warning("XML parser error:", xmlGetLastError()->message, "", "",
			get_ticker_env()->selection_mode == MULTIPLE);
		return FEED_UNPARSABLE;
	}
	if ((root_element = xmlDocGetRootElement(doc)) == NULL) {
		xmlFreeDoc(doc);
		warning("Empty XML document:", file_name, "", "",
			get_ticker_env()->selection_mode == MULTIPLE);
		return FEED_EMPTY;
	}

	depth = 0;
	item_element = NULL;
	n = 1;
	counter = 0;

	get_rss20_selected_elements1(root_element, doc);
	if (item_element != NULL)
		get_rss20_selected_elements2(item_element, doc, fp, link_and_offset, prm);
	else {
		xmlFreeDoc(doc);
		return RSS_NO_ITEM_ELEMENT;
	}
	xmlFreeDoc(doc);
#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "Done\n");
#endif
	return OK;
}

/*
 * must be utf-8 encoded
 */
int parse_atom_xml_file(FILE *fp, const char* file_name, FeedLinkAndOffset *link_and_offset, const Params *prm)
{
	xmlDoc	*doc;
	xmlNode	*root_element;

#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "Parsing XML file ... ");
#endif
	if ((doc = xmlParseFile(file_name)) == NULL) {
		warning("XML parser error:", xmlGetLastError()->message, "", "",
			get_ticker_env()->selection_mode == MULTIPLE);
		return FEED_UNPARSABLE;
	}
	if ((root_element = xmlDocGetRootElement(doc)) == NULL) {
		xmlFreeDoc(doc);
		warning("Empty XML document:", file_name, "", "",
			get_ticker_env()->selection_mode == MULTIPLE);
		return FEED_EMPTY;
	}

	depth = 0;
	entry_element = NULL;
	n = 1;
	counter = 0;

	get_atom_selected_elements1(root_element, doc);
	if (entry_element != NULL)
		get_atom_selected_elements2(entry_element, doc, fp, link_and_offset, prm);
	else {
		xmlFreeDoc(doc);
		return ATOM_NO_ENTRY_ELEMENT;
	}
	xmlFreeDoc(doc);
#ifdef VERBOSE_OUTPUT
	fprintf(STD_OUT, "Done\n");
#endif
	return OK;
}

/*
 * for every link found, we insert in text LINKTAG_CHAR"00n"
 * with n = link rank and we fill link_and_offset with url
 * this is used later in render_stream_to_surface()
 */
void get_rss20_selected_elements1(xmlNode *some_element, xmlDoc *doc)
{
	xmlNode	*cur_node;

	for (cur_node = some_element; cur_node != NULL; cur_node = cur_node->next) {
		if (item_element != NULL)
			return;

		if (xmlStrcmp(cur_node->name, (const xmlChar *)"rss") == 0 && depth == 0)
			depth = 1;
		else if (xmlStrcmp(cur_node->name, (const xmlChar *)"channel") == 0 && depth == 1)
			depth = 2;
		else if (xmlStrcmp(cur_node->name, (const xmlChar *)"item") == 0 && depth == 2)
			depth = 3;

		if (depth == 3)
			item_element = cur_node;
		else
			get_rss20_selected_elements1(cur_node->children, doc);
	}
}

void get_rss20_selected_elements2(xmlNode *some_element, xmlDoc *doc,
		FILE *fp, FeedLinkAndOffset *link_and_offset, const Params *prm)
{
	xmlNode	*cur_node, *cur_node_bak;
	xmlChar	*str;

	for (cur_node = some_element; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)"item") == 0) {
			cur_node_bak = cur_node;
			cur_node = cur_node->children;
			for (; cur_node != NULL; cur_node = cur_node->next) {
				if (prm->rss_title == 'y' && xmlStrcmp(cur_node->name, (const xmlChar *)"title") == 0) {
					if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {
						/* we remove any LINKTAG_CHAR from str because it will be used in "link tag" */
						remove_char_from_str((char *)str, LINKTAG_CHAR);
						fprintf(fp, "%s%s\n", str, prm->rss_title_delimiter);
						xmlFree(str);
					}
				} else if (prm->rss_description == 'y' && xmlStrcmp(cur_node->name, (const xmlChar *)"description") == 0) {
					if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {
						/* we remove any LINKTAG_CHAR from str because it will be used in "link tag" */
						remove_char_from_str((char *)str, LINKTAG_CHAR);
						fprintf(fp, "%s%s\n", str, prm->rss_description_delimiter);
						xmlFree(str);
					}
				}
			}
			cur_node = cur_node_bak;
			cur_node = cur_node->children;
			for (; cur_node != NULL; cur_node = cur_node->next) {
				if (xmlStrcmp(cur_node->name, (const xmlChar *)"link") == 0) {
					if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {
						if (n < NFEEDLINKANDOFFSETMAX + 1) {
							str_n_cpy((link_and_offset + n)->url, (const char *)str,
								FILE_NAME_MAXLEN);
							fprintf(fp, "%c%03d\n", LINKTAG_CHAR, n++);
						}
						xmlFree(str);
					}
				}
			}
			cur_node = cur_node_bak;
			if (prm->n_items_per_feed != 0)
				if (++counter >= prm->n_items_per_feed)
					break;
		}
	}
}

/*
 * for every link found, we insert in text LINKTAG_CHAR"00n"
 * with n = link rank and we fill link_and_offset with url
 * this is used later in render_stream_to_surface()
 */
void get_atom_selected_elements1(xmlNode *some_element, xmlDoc *doc)
{
	xmlNode	*cur_node;

	for (cur_node = some_element; cur_node != NULL; cur_node = cur_node->next) {
		if (entry_element != NULL)
			return;

		if (xmlStrcmp(cur_node->name, (const xmlChar *)"feed") == 0)/* &&\
				xmlStrcmp(xmlGetProp(cur_node, (const xmlChar *)"xmlns"),\
				(const xmlChar *)"http://www.w3.org/2005/Atom") == 0 && depth == 0)*/
			depth = 1;
		else if (xmlStrcmp(cur_node->name, (const xmlChar *)"entry") == 0 && depth == 1)
			depth = 2;

		if (depth == 2)
			entry_element = cur_node;
		else
			get_atom_selected_elements1(cur_node->children, doc);
	}
}

void get_atom_selected_elements2(xmlNode *some_element, xmlDoc *doc,
		FILE *fp, FeedLinkAndOffset *link_and_offset, const Params *prm)
{
	xmlNode	*cur_node, *cur_node_bak;
	xmlChar	*str;

	for (cur_node = some_element; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)"entry") == 0) {
			cur_node_bak = cur_node;
			cur_node = cur_node->children;
			for (; cur_node != NULL; cur_node = cur_node->next) {
				if (prm->rss_title == 'y' && xmlStrcmp(cur_node->name, (const xmlChar *)"title") == 0) {
					if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {
						/* we remove any LINKTAG_CHAR from str because it will be used in "link tag" */
						remove_char_from_str((char *)str, LINKTAG_CHAR);
						fprintf(fp, "%s%s\n", str, prm->rss_title_delimiter);
						xmlFree(str);
					}
				} else if (prm->rss_description == 'y' && xmlStrcmp(cur_node->name, (const xmlChar *)"summary") == 0) {
					if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {
						/* we remove any LINKTAG_CHAR from str because it will be used in "link tag" */
						remove_char_from_str((char *)str, LINKTAG_CHAR);
						fprintf(fp, "%s%s\n", str, prm->rss_description_delimiter);
						xmlFree(str);
					}
				}
			}
			cur_node = cur_node_bak;
			cur_node = cur_node->children;
			for (; cur_node != NULL; cur_node = cur_node->next) {
				if (xmlStrcmp(cur_node->name, (const xmlChar *)"link") == 0) {
					/* node attribute instead of node content (atom/rss) */
					if ((str = xmlGetProp(cur_node, (const xmlChar *)"href")) != NULL) {
					/*if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {*/
						if (n < NFEEDLINKANDOFFSETMAX + 1) {
							str_n_cpy((link_and_offset + n)->url, (const char *)str,\
								FILE_NAME_MAXLEN);
							fprintf(fp, "%c%03d\n", LINKTAG_CHAR, n++);
						}
						xmlFree(str);
					}
				}
			}
			cur_node = cur_node_bak;
			if (prm->n_items_per_feed != 0)
				if (++counter >= prm->n_items_per_feed)
					break;
		}
	}
}

/*
 * info is 4 strings 255 chars long each
 * feed_* can be NULL
 */
int get_feed_info(const char *file_name, char *feed_title, char *feed_link, char *feed_ttl)
{
	xmlDoc	*doc;
	xmlNode	*root_element;

	if (feed_title != NULL)
		feed_title[0] = '\0';
	if (feed_link != NULL)
		feed_link[0] = '\0';
	if (feed_ttl != NULL)
		feed_ttl[0] = '\0';

	if ((doc = xmlParseFile(file_name)) == NULL) {
		return FEED_UNPARSABLE;
	} else {
		if ((root_element = xmlDocGetRootElement(doc)) == NULL) {
			xmlFreeDoc(doc);
			return FEED_EMPTY;
		} else {
			if (feed_title != NULL)
				get_xml_first_element(root_element->children, doc,
					"title", feed_title, FEED_TITLE_MAXLEN);
			if (feed_link != NULL)
				get_xml_first_element(root_element->children, doc,
					"link", feed_link, FILE_NAME_MAXLEN);
			if (feed_ttl != NULL)
				get_xml_first_element(root_element->children, doc,
					"ttl", feed_ttl, 31);
			xmlFreeDoc(doc);
			return OK;
		}
	}
}

void get_xml_first_element(xmlNode *some_element, xmlDoc *doc, char *name, char *string, int length)
{
	xmlNode	*cur_node;
	xmlChar	*str;

	for (cur_node = some_element; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)name) == 0) {
			if ((str = xmlNodeListGetString(doc, cur_node->children, 1)) != NULL) {
				str_n_cpy(string, (const char *)str, length);
				xmlFree(str);
			} else
				string[0] = '\0';
			break;
		}
		get_xml_first_element(cur_node->children, doc, name, string, length);
	}
}
