/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#if USE_GUI
#define URL_ENTRY_LENGTH	100
#define SC_WIN_HEIGHT		350

/* predefined GTK_RESPONSE_ are < 0, those ones are app level defined and >= 0 */
enum {
	GTK_RESPONSE_SELECT_ALL, GTK_RESPONSE_UNSELECT_ALL, GTK_RESPONSE_TOP, GTK_RESPONSE_HOME,
	GTK_RESPONSE_REMOVE, GTK_RESPONSE_ADD, GTK_RESPONSE_SINGLE, GTK_RESPONSE_SELECTION
};

enum {COLUMN_INT, COLUMN_BOOLEAN_CHECKED, COLUMN_STRING_TITLE, COLUMN_STRING_URL, N_COLUMNS};

static GtkWidget	*dialog, *entry_url;
static FList		*flist;
static int		f_index;	/* starting at 0 (row starts at 1) */
static gchar		home_feed[FILE_NAME_MAXLEN + 1];

static gint esc_key_pressed(GtkWidget *dialog2, GdkEventKey *event)
{
	if (event->keyval == GDK_Escape) {
		gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CLOSE);
		return TRUE;
	} else
		return FALSE;
}

static void force_quit_dialog(GtkWidget *dialog2)
{
	gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CLOSE);
}

/* tree selection callback - update f_index and copy selected url to entry_url (override) */
static gint tree_selection_changed(GtkTreeSelection *selection)
{
	GtkTreeModel	*tree_model = NULL;
	GtkTreeIter	iter;
	gchar		*str_url;

	if (gtk_tree_selection_get_selected(selection, &tree_model, &iter)) {
		gtk_tree_model_get(tree_model, &iter,
			COLUMN_INT, &f_index,
			COLUMN_STRING_URL, &str_url,
			-1);
		gtk_entry_set_text(GTK_ENTRY(entry_url), str_url);
		if (IS_FLIST(flist))
			if ((f_index = f_list_search(flist, str_url)) > -1)
				flist = f_list_nth(flist, f_index + 1);
		g_free(str_url);
	}
	return TRUE;
}

/* catch double-click on tree view */
static gint double_click_on_tree_view(GtkTreeView *tree_view, GtkTreePath *tree_path)
{
	GtkTreeModel	*tree_model = NULL;
	GtkTreeIter	iter;
	gchar		*str_url;

	tree_model = gtk_tree_view_get_model(tree_view);
	if (gtk_tree_model_get_iter(tree_model, &iter, tree_path)) {
		gtk_tree_model_get(tree_model, &iter,
			COLUMN_INT, &f_index,
			COLUMN_STRING_URL, &str_url,
			-1);
		gtk_entry_set_text(GTK_ENTRY(entry_url), str_url);
		if (IS_FLIST(flist))
			if ((f_index = f_list_search(flist, str_url)) > -1)
				flist = f_list_nth(flist, f_index + 1);
		g_free(str_url);
	}
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_SINGLE);
	return TRUE;
}

static gint enter_key_pressed_in_entry(GtkWidget *widget)
{
	widget = widget;
	gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_SINGLE);
	return TRUE;
}

static gint clear_entry(GtkWidget *widget)
{
	widget = widget;
	gtk_entry_set_text(GTK_ENTRY(entry_url), "");
	gtk_widget_grab_focus(GTK_WIDGET(entry_url));
	return TRUE;
}

static gint checkbox_toggled(GtkCellRendererToggle *renderer, gchar *path_str, gpointer tree_view)
{
	GtkTreeModel	*tree_model;
	GtkTreeIter	iter;
	FList		*flist2;
	gboolean	checked;

	renderer = renderer;
	tree_model = gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
	if (gtk_tree_model_get_iter_from_string(tree_model, &iter, path_str)) {
		gtk_tree_model_get(tree_model, &iter,
			COLUMN_INT, &f_index,
			COLUMN_BOOLEAN_CHECKED, &checked,
			-1);
			checked = !checked;
			gtk_list_store_set(GTK_LIST_STORE(tree_model), &iter,
				COLUMN_BOOLEAN_CHECKED, checked,
				-1);
			if (IS_FLIST(flist)) {
				flist2 = f_list_nth(flist, f_index + 1);
				if (IS_FLIST(flist2))
					flist2->selected = checked;
			}
	}
	return TRUE;
}

static void select_all(GtkTreeView *tree_view)
{
	GtkTreeModel	*tree_model;
	GtkTreeIter	iter;
	FList		*flist2;

	tree_model = gtk_tree_view_get_model(tree_view);
	if (gtk_tree_model_get_iter_first(tree_model, &iter)) {
		do
			gtk_list_store_set(GTK_LIST_STORE(tree_model), &iter,
				COLUMN_BOOLEAN_CHECKED, TRUE,
				-1);
		while (gtk_tree_model_iter_next(tree_model, &iter));
		if (IS_FLIST(flist))
			for (flist2 = f_list_first(flist); IS_FLIST(flist2); flist2 = flist2->next)
				flist2->selected = TRUE;
	}
}

static void unselect_all(GtkTreeView *tree_view)
{
	GtkTreeModel	*tree_model;
	GtkTreeIter	iter;
	FList		*flist2;

	tree_model = gtk_tree_view_get_model(tree_view);
	if (gtk_tree_model_get_iter_first(tree_model, &iter)) {
		do
			gtk_list_store_set(GTK_LIST_STORE(tree_model), &iter,
				COLUMN_BOOLEAN_CHECKED, FALSE,
				-1);
		while (gtk_tree_model_iter_next(tree_model, &iter));
		if (IS_FLIST(flist))
			for (flist2 = f_list_first(flist); IS_FLIST(flist2); flist2 = flist2->next)
				flist2->selected = FALSE;
	}
}

/* select and scroll to row */
void highlight_and_go_to_row(GtkTreeView *tree_view, GtkTreeSelection *selection, int row)
{
	GtkTreeModel	*tree_model;
	GtkTreePath	*tree_path;
	GtkTreeIter	iter;
	int		i = 0;

	if (row > 0) {
		tree_model = gtk_tree_view_get_model(tree_view);
		tree_path = gtk_tree_path_new_first();
		if (gtk_tree_model_get_iter_first(tree_model, &iter)) {
			while (gtk_tree_model_iter_next(tree_model, &iter) && i++ < row - 1)
				gtk_tree_path_next(tree_path);
			gtk_tree_selection_select_path(selection, tree_path);
			gtk_tree_view_set_cursor_on_cell(tree_view, tree_path, NULL, NULL, FALSE);
			gtk_tree_view_scroll_to_cell(tree_view, tree_path, NULL, FALSE, 0.5, 0);
		}
	}
}

/* clear the list before filling it */
static void fill_list_store_from_flnode(GtkListStore *list_store)
{
	FList		*flist2;
	GtkTreeIter	iter;
	int		i = 0;

	gtk_list_store_clear(list_store);
	if (IS_FLIST(flist))
		for (flist2 = f_list_first(flist); IS_FLIST(flist2); flist2 = flist2->next) {
			gtk_list_store_append(list_store, &iter);
			gtk_list_store_set(list_store, &iter,
				COLUMN_INT, i,
				COLUMN_BOOLEAN_CHECKED, flist2->selected,
				COLUMN_STRING_TITLE, flist2->title,
				COLUMN_STRING_URL, flist2->url,
				-1);
			i++;
		}
}

static int add_feed_to_flnode_and_list_store(Resource *resrc, GtkListStore *list_store)
{
	int	rss_status;
	char	rss_feed_title[FEED_TITLE_MAXLEN + 1];
	char	rss_feed_link[FILE_NAME_MAXLEN + 1];
	char	rss_feed_ttl[32];

	str_n_cpy(resrc->id, (char *)gtk_entry_get_text(GTK_ENTRY(entry_url)), FILE_NAME_MAXLEN);
	if ((rss_status = get_feed(resrc, get_params())) == OK) {
		if (IS_FLIST(flist))
			flist = f_list_last(flist);
		if (get_feed_info(resrc->id, rss_feed_title, rss_feed_link, rss_feed_ttl) == OK)
			/* added url is set unselected */
			flist = f_list_add_at_end(flist, resrc->id, rss_feed_title, FALSE);
		else
			flist = f_list_add_at_end(flist, resrc->id, "", FALSE);
		flist = f_list_sort(flist);
		fill_list_store_from_flnode(list_store);
	}
	return rss_status;
}

/*
 * open a dialog with a list of urls to choose from and an url entry
 *
 * entry format in url list file:
 *	["*" (selected) or "-" (unselected) + url [+ ">" + title] + '\n']
 *
 * entry max length = FILE_NAME_MAXLEN
 * see also: (UN)SELECTED_URL_CHAR in tickr.h
 */
void get_new_url(Resource *resrc)
{
	TickerEnv		*env;
	GtkWidget		*vbox, *sc_win, *hbox, *label[3];
	GtkWidget		*cancel_but, *selectall_but, *unselectall_but;
	GtkWidget		*top_but, *home_but, *remove_but, *add_but;
	GtkWidget		*single_but, *selection_but, *clear_but;
	GtkTreeView		*tree_view = NULL;
	GtkListStore		*list_store = NULL;
	GtkTreeModel		*tree_model = NULL;
	GtkTreeIter		iter;
	GtkCellRenderer		*renderer1 = NULL, *renderer2 = NULL, *renderer3 = NULL;
	GtkTreeViewColumn	*column1 = NULL, *column2 = NULL, *column3 = NULL;
	GtkTreeSelection	*selection = NULL;
	FList			*flist2 = NULL;
	Resource		resrc_bak;
	gint			response;
	gint			cursor_position;

	env = get_ticker_env();
	env->suspend_rq = TRUE;
	gtk_window_set_keep_above(GTK_WINDOW(env->win), FALSE);

	dialog = gtk_dialog_new_with_buttons(
			"Feed Picker (RSS/Atom)", GTK_WINDOW(env->win),
			GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
			NULL);

	cancel_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
	selectall_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "Select All", GTK_RESPONSE_SELECT_ALL);
	unselectall_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "Unselect All", GTK_RESPONSE_UNSELECT_ALL);
	top_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "Top", GTK_RESPONSE_TOP);
	home_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_HOME, GTK_RESPONSE_HOME);
	remove_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_REMOVE, GTK_RESPONSE_REMOVE);
	add_but = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_ADD, GTK_RESPONSE_ADD);
	single_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "OK (Single)", GTK_RESPONSE_SINGLE);
	selection_but = gtk_dialog_add_button(GTK_DIALOG(dialog), "OK (Selec)", GTK_RESPONSE_SELECTION);
	clear_but = gtk_button_new_with_label("Clear");

	cancel_but = cancel_but;	/* to get rid of one compiler warning */
	selectall_but = selectall_but;
	unselectall_but = unselectall_but;
	top_but = top_but;
	home_but = home_but;
	remove_but = remove_but;
	add_but = add_but;
	single_but = single_but;
	selection_but = selection_but;
	clear_but = clear_but;		/* until here */

	set_tickr_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER_ALWAYS);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), 15);

	str_n_cpy(home_feed, get_params()->homefeed, FILE_NAME_MAXLEN);
	list_store = gtk_list_store_new(N_COLUMNS, G_TYPE_INT, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING);

	if (IS_FLIST(get_feed_list()))
		f_list_free_all(get_feed_list());
	if (f_list_load_from_file(&flist, NULL) == OK) {
		flist = f_list_sort(flist);
		fill_list_store_from_flnode(list_store);
	} else
		flist = NULL;
	set_feed_list(flist);

	tree_view = GTK_TREE_VIEW(gtk_tree_view_new_with_model(GTK_TREE_MODEL(list_store)));

	renderer1 = gtk_cell_renderer_toggle_new();
	gtk_cell_renderer_toggle_set_radio(GTK_CELL_RENDERER_TOGGLE(renderer1), FALSE);
	column1 = gtk_tree_view_column_new_with_attributes(NULL, renderer1,
		"active", COLUMN_BOOLEAN_CHECKED, NULL);
	gtk_tree_view_append_column(tree_view, column1);

	renderer2 = gtk_cell_renderer_text_new();
	column2 = gtk_tree_view_column_new_with_attributes("Feed Title", renderer2,
		"text", COLUMN_STRING_TITLE, NULL);
	gtk_tree_view_append_column(tree_view, column2);

	renderer3 = gtk_cell_renderer_text_new();
	column3 = gtk_tree_view_column_new_with_attributes("Feed URL", renderer3,
		"text", COLUMN_STRING_URL, NULL);
	gtk_tree_view_append_column(tree_view, column3);

	sc_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sc_win),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_size_request(sc_win, -1, SC_WIN_HEIGHT);

	gtk_container_add(GTK_CONTAINER(vbox), sc_win);
	gtk_container_add(GTK_CONTAINER(sc_win), GTK_WIDGET(tree_view));

	label[0] = gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(vbox), label[0]);

	selection = gtk_tree_view_get_selection(tree_view);
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

	g_signal_connect(G_OBJECT(selection), "changed", G_CALLBACK(tree_selection_changed), NULL);
	g_signal_connect(G_OBJECT(tree_view), "row_activated", G_CALLBACK(double_click_on_tree_view), NULL);
	g_signal_connect(G_OBJECT(renderer1), "toggled", G_CALLBACK(checkbox_toggled), tree_view);

	hbox = gtk_hbox_new(FALSE, 0);
	label[1] = gtk_label_new("Enter new URL:");
	gtk_box_pack_start(GTK_BOX(hbox), label[1], FALSE, FALSE, 0);
	label[2] = gtk_label_new("  ");
	gtk_box_pack_start(GTK_BOX(hbox), label[2], TRUE, FALSE, 0);
	entry_url = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry_url), FILE_NAME_MAXLEN);
	gtk_entry_set_width_chars(GTK_ENTRY(entry_url), URL_ENTRY_LENGTH);
	gtk_box_pack_start(GTK_BOX(hbox), entry_url, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), clear_but, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	g_signal_connect(G_OBJECT(entry_url), "activate", G_CALLBACK(enter_key_pressed_in_entry), NULL);
	g_signal_connect(G_OBJECT(clear_but), "clicked", G_CALLBACK(clear_entry), NULL);
	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);
	gtk_widget_set_size_request(sc_win, -1, SC_WIN_HEIGHT);

	/* backup last valid opened resource (if any) */
	str_n_cpy(resrc_bak.id, resrc->id, FILE_NAME_MAXLEN);
	/* is it better like that ? */
	if (IS_FLIST(flist) && resrc->id[0] != '\0')
		if ((f_index = f_list_search(flist, resrc->id)) > -1)
			highlight_and_go_to_row(tree_view, selection, f_index + 1);
	gtk_widget_grab_focus(GTK_WIDGET(tree_view));

	while ((response = gtk_dialog_run(GTK_DIALOG(dialog))) != GTK_RESPONSE_CLOSE) {
		if (response == GTK_RESPONSE_SELECT_ALL) {
			if (question_win("Select all URLs ?") == YES)
				select_all(tree_view);
		} else if (response == GTK_RESPONSE_UNSELECT_ALL) {
			if (question_win("Unselect all URLs ?") == YES)
				unselect_all(tree_view);
		} else if (response == GTK_RESPONSE_TOP) {
			highlight_and_go_to_row(tree_view, selection, 1);
		} else if (response == GTK_RESPONSE_HOME) {
			gtk_entry_set_text(GTK_ENTRY(entry_url), (const gchar *)home_feed);
			if (IS_FLIST(flist)) {
				if ((f_index = f_list_search(flist,
						gtk_entry_get_text(GTK_ENTRY(entry_url)))) > -1)
					highlight_and_go_to_row(tree_view, selection, f_index + 1);
			}
		} else if (response == GTK_RESPONSE_REMOVE) {
			if (IS_FLIST(flist) && f_index > -1) {
				if (gtk_tree_selection_get_selected(selection, &tree_model, &iter) &&
						question_win("Remove selected URL from list ?") == YES) {
					gtk_list_store_remove(list_store, &iter);
					flist = f_list_remove(flist);
					if (f_list_count(flist) > 0)
						set_feed_list(f_list_first(flist));
					else
						set_feed_list(NULL);
					f_index = -1;
					gtk_entry_set_text(GTK_ENTRY(entry_url), "");
				}
			} else
				warning(FALSE, 1, "You must select an URL first");
		} else if (response == GTK_RESPONSE_ADD) {
			if (gtk_entry_get_text(GTK_ENTRY(entry_url))[0] != '\0') {
				cursor_position = gtk_editable_get_position(GTK_EDITABLE(entry_url));
				str_n_cpy(resrc->id, gtk_entry_get_text(GTK_ENTRY(entry_url)), FILE_NAME_MAXLEN);
				if (add_feed_to_flnode_and_list_store(resrc, list_store) == OK) {
					if ((f_index = f_list_search(flist,
							gtk_entry_get_text(GTK_ENTRY(entry_url)))) > -1)
						highlight_and_go_to_row(tree_view, selection, f_index + 1);
					/* backup last valid opened resource (if any) */
					str_n_cpy(resrc_bak.id, resrc->id, FILE_NAME_MAXLEN);
				} else {
					/* restore last valid opened resource (if any) */
					str_n_cpy(resrc->id, resrc_bak.id, FILE_NAME_MAXLEN);
					gtk_widget_grab_focus(GTK_WIDGET(entry_url));
					gtk_editable_set_position(GTK_EDITABLE(entry_url), cursor_position);
					continue;
				}
			} else {
				warning(FALSE, 1, "You must enter an URL first");
				gtk_widget_grab_focus(GTK_WIDGET(entry_url));
				continue;
			}
		} else if (response == GTK_RESPONSE_SINGLE) {
			if (f_index > -1 && gtk_entry_get_text(GTK_ENTRY(entry_url))[0] == '\0')
				tree_selection_changed(selection);
			if (gtk_entry_get_text(GTK_ENTRY(entry_url))[0] != '\0') {
				if (add_feed_to_flnode_and_list_store(resrc, list_store) == OK) {
					env->selection_mode = SINGLE;
					env->reload_rq = TRUE;
					break;
				} else
					/* restore last valid opened resource (if any) */
					str_n_cpy(resrc->id, resrc_bak.id, FILE_NAME_MAXLEN);
			} else
				warning(FALSE, 1, "You must enter or select an URL first");
		} else if (response == GTK_RESPONSE_SELECTION) {
			if (IS_FLIST(flist)) {
				for (flist2 = f_list_first(flist); IS_FLIST(flist2); flist2 = flist2->next)
					if (flist2->selected)
						break;
				if (IS_FLIST(flist2)) {
					if (flist2->selected) {
						env->selection_mode = MULTIPLE;
						env->reload_rq = TRUE;
						break;
					}
				}
				warning(FALSE, 1, "Selection is empty");
			} else
				warning(FALSE, 1, "URL list is empty");
		}
		gtk_widget_grab_focus(GTK_WIDGET(tree_view));
	}
	if (list_store != NULL)
		g_object_unref(list_store);
	if (IS_FLIST(flist)) {
		flist = f_list_first(flist);
		f_list_save_to_file(flist, NULL);
		set_feed_list(flist);
		build_feed_selection_from_feed_list();
		if (IS_FLIST(get_feed_selection()) && response == GTK_RESPONSE_SELECTION)
			first_feed();
	}
	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
	env->suspend_rq = FALSE;
}
#endif
