/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#if USE_GUI
#ifdef G_OS_WIN32
extern FILE	*stdout_fp, *stderr_fp;
#endif
static char	*opml_str_global;
static char	tmp_url[FILE_NAME_MAXLEN + 1], tmp_str[256];

static gint esc_key_pressed(GtkWidget *dialog2, GdkEventKey *event)
{
	if (event->keyval == GDK_Escape) {
		gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CANCEL);
		return TRUE;
	} else
		return FALSE;
}

static void force_quit_dialog(GtkWidget *dialog2)
{
	gtk_dialog_response(GTK_DIALOG(dialog2), GTK_RESPONSE_CANCEL);
}

/*
 * importing
 */

static int *get_feed_counter()
{
	static int	feed_counter;

	return &feed_counter;
}

static void zero_feed_counter()
{
	*get_feed_counter() = 0;
}

static void inc_feed_counter()
{
	*get_feed_counter() += 1;
}

void import_opml_file()
{
	char	*opml_file_name, *url_list_file_name;
	FList	*node;
	FILE	*fp;
	int	status =  OPML_ERROR;

	if ((opml_file_name = pick_opml_file())[0] != '\0') {
		url_list_file_name = l_str_new(get_datafile_full_name_from_name(URL_LIST_FILE));
		if ((fp = g_fopen(url_list_file_name, "rb")) != NULL) {
			fclose(fp);
			if (question_win("Imported URLs will be merged with currently saved ones. "
					"Continue ?") == NO) {
				l_str_free(url_list_file_name);
				return;
			}
		} else if ((fp = g_fopen(url_list_file_name, "wb")) != NULL)
			fclose(fp);
		else {
			warning(FALSE, 4, "Can't create URL list ", url_list_file_name, ": ", strerror(errno));
			return;
		}
		win_with_progress_bar(WIN_WITH_PROGRESS_BAR_OPEN,
			"    Importing (and updating) data from OPML file, please wait...    ");
		zero_feed_counter();
		get_ticker_env()->selection_mode = MULTIPLE;
		opml_str_global = l_str_new(NULL);
		if (parse_opml_xml_file(opml_file_name) == OK)
			if ((fp = g_fopen(url_list_file_name, "ab")) != NULL) {
				fprintf(fp, "\n%s\n", opml_str_global);
				fclose(fp);
				if (f_list_load_from_file(&node, NULL) == OK) {
					node = f_list_sort(node);
					if (f_list_save_to_file(node, NULL) == OK) {
						if (IS_FLIST(get_feed_list()))
							f_list_free_all(get_feed_list());
						set_feed_list(node);
						status = OK;
					}
				}
			}
		l_str_free(url_list_file_name);
		l_str_free(opml_str_global);
		win_with_progress_bar(WIN_WITH_PROGRESS_BAR_CLOSE, NULL);
		if (status == OK) {
			/*snprintf(tmp_str, 256, "\nFeed list (%d URLs) has been imported\n",
				*get_feed_counter());
			info_win(APP_NAME, tmp_str, INFO, FALSE);*/
			/* ok like that ? */
			snprintf(tmp_str, 256, "\nFeed list (%d URLs) has been imported. "
				"Open the Feed Picker window ?\n", *get_feed_counter());
			if (question_win(tmp_str) == YES)
				get_new_url(get_resource());
		}
	}
}

char *pick_opml_file()
{
	GtkWidget	*dialog;
	GtkFileFilter	*filter;
	char		*file_name;
	static char	file_name2[FILE_NAME_MAXLEN + 1];

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_file_chooser_dialog_new("Import Feed List (OPML)", GTK_WINDOW(get_ticker_env()->win),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);

	set_tickr_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);

	filter = gtk_file_filter_new();
	gtk_file_filter_add_pattern(filter, "*.opml");
	gtk_file_filter_add_pattern(filter, "*.xml");
	gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(dialog), filter);

	file_name2[0] = '\0';
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		if ((file_name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog))) != NULL) {
			str_n_cpy(file_name2, file_name, FILE_NAME_MAXLEN);
			g_free(file_name);
		}
	}
	gtk_widget_destroy(dialog);
	check_main_win_always_on_top();
	return file_name2;
}

int parse_opml_xml_file(const char *file_name)
{
	xmlDoc		*doc = NULL;
	xmlNode		*cur_node = NULL, *cur_node_start = NULL;
	int		body_element = NO, outline_element = NO;

	fprintf(STD_OUT, "Parsing OPML file... ");
	if ((doc = xmlParseFile(file_name)) == NULL) {
		warning(FALSE, 2, "Can't parse XML file: ", xmlGetLastError()->message);
		return XML_UNPARSABLE;
	} else if ((cur_node = xmlDocGetRootElement(doc)) == NULL) {
		warning(FALSE, 2, "Empty XML document: ", file_name);
		xmlFreeDoc(doc);
		return XML_EMPTY;
	} else if (xmlStrcmp(cur_node->name, (const xmlChar *)"opml") != 0) {
		warning(FALSE, 2, "Not an OPML document: ", file_name);
		xmlFreeDoc(doc);
		return OPML_ERROR;
	}
	cur_node_start = cur_node;
	for (cur_node = cur_node->children; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)"body") == 0) {
			body_element = YES;
			break;
		}
	}
	if (body_element == YES) {
		for (cur_node = cur_node->children; cur_node != NULL; cur_node = cur_node->next) {
			if (xmlStrcmp(cur_node->name, (const xmlChar *)"outline") == 0) {
				outline_element = YES;
				break;
			}
		}
	}
	if (body_element == YES && outline_element == YES) {
		get_opml_selected_element(cur_node_start, "outline", "type");
		fprintf(STD_OUT, "Done\n");
		xmlFreeDoc(doc);
		return OK;
	} else {
		warning(FALSE, 2, "Couldn't find all required OPML elements in: ",
			file_name);
		xmlFreeDoc(doc);
		return OPML_ERROR;
	}
}

void get_opml_selected_element(xmlNode *some_node, const char *selected_element, const char *selected_attribute)
{
	xmlNode	*cur_node;
	xmlChar	*str1, *str2;
	gchar	*unesc_str;
	char	feed_title[FEED_TITLE_MAXLEN + 1];

	for (cur_node = some_node; cur_node != NULL; cur_node = cur_node->next) {
		if (xmlStrcmp(cur_node->name, (const xmlChar *)selected_element) == 0) {
			/* need to get attribute name and value
			 * then use only those whose value = "rss" for name = 'type'
			 * then get value for name = 'xmlUrl' -> rss feed url
			 */
			if ((str1 = xmlGetProp(cur_node, (xmlChar *)selected_attribute)) != NULL) {
				if (xmlStrcmp(str1, (xmlChar *)"rss") == 0) {
					if ((str2 = xmlGetProp(cur_node, (xmlChar *)"xmlUrl")) != NULL) {
						/* we add 1st one UNSELECTED_URL_CHAR */
						opml_str_global = l_str_cat(opml_str_global, UNSELECTED_URL_STR);
						unesc_str = g_uri_unescape_string((const char *)str2, NULL);
						opml_str_global = l_str_cat(opml_str_global, (const char*)unesc_str);
						/* get feed title from feed url - ok like that? */
						str_n_cpy(tmp_url, unesc_str, FILE_NAME_MAXLEN);
/* TODO: double-check following 2 lines hack */
						if (fetch_resource(tmp_url, get_datafile_full_name_from_name(TMP1),
								tmp_url) == OK)
							if (get_feed_info(get_datafile_full_name_from_name(TMP1),
									feed_title, NULL, NULL) == OK)
								if (feed_title[0] != '\0') {
									opml_str_global = l_str_cat(opml_str_global,
										TITLE_TAG_STR);
									opml_str_global = l_str_cat(opml_str_global,
										feed_title);
								}
						opml_str_global = l_str_cat(opml_str_global, "\n");
						inc_feed_counter();
						snprintf(tmp_str, 256, "%d feeds imported so far, please wait...",
							*get_feed_counter());
						win_with_progress_bar(WIN_WITH_PROGRESS_BAR_PULSE, tmp_str);
						g_free(unesc_str);
						xmlFree(str2);
					}
				}
				xmlFree(str1);
			}
		}
		get_opml_selected_element(cur_node->children, selected_element, selected_attribute);
	}
}

/*
 * exporting
 */
void export_opml_file()
{
	FList	*node;
	char	*str;

	if (f_list_load_from_file(&node, NULL) == OK) {
		str = create_opml_str_from_feed_list_node(node);
		save_str_as_opml_file(str);
		l_str_free(str);
		f_list_free_all(node);
	}
}

/* must l_str_free() opml_str afterwards */
char *create_opml_str_from_feed_list_node(FList *node)
{
#define OPML_TITLE	APP_NAME"-"APP_VERSION_NUMBER" Feed List"
#define STR_SIZE	(4 * 1024)

	char 	*opml_header =
		"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<opml version=\"1.0\">\n"
		"  <head>\n"
		"    <title>%s</title>\n"
		"  </head>\n"
		"  <body>\n";
	char 	*opml_item =
		"    <outline title=\"%s\" type=\"rss\" xmlUrl=\"%s\"/>\n";
	char 	*opml_footer =
		"  </body>\n"
		"</opml>\n";
	char	*opml_str, tmp[STR_SIZE];
	gchar	*esc_str1, *esc_str2;

	snprintf(tmp, STR_SIZE, opml_header, OPML_TITLE);
	opml_str = l_str_new(tmp);
	for (; node != NULL; node = node->next) {
		if (!g_utf8_validate(node->url, -1, NULL)) {
			warning(FALSE, 3, "Non UTF-8 encoded string: ", node->url, " - Skipped");
			continue;
		}
		esc_str1 = g_uri_escape_string(node->title, " AÄääàçéèêÖöôÜüùà'()[]-_|{}/.,;:?!%€£$*+=", TRUE);
		esc_str2 = g_uri_escape_string(node->url, G_URI_RESERVED_CHARS_GENERIC_DELIMITERS, TRUE);
		snprintf(tmp, STR_SIZE, opml_item, esc_str1, esc_str2);
		g_free(esc_str1);
		g_free(esc_str2);
		opml_str = l_str_cat(opml_str, tmp);
	}
	return l_str_cat(opml_str, opml_footer);
}

int save_str_as_opml_file(const char *str)
{
	GtkWidget	*dialog;
	char		*file_name = NULL;
	FILE		*fp;
	int		exit_status = CREATE_FILE_ERROR;
	char		tmp[TMPSTR_SIZE + 1];

	gtk_window_set_keep_above(GTK_WINDOW(get_ticker_env()->win), FALSE);

	dialog = gtk_file_chooser_dialog_new("Export Feed List (OPML)", GTK_WINDOW(get_ticker_env()->win),
			GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK, GTK_RESPONSE_OK,
			NULL);

	set_tickr_icon_to_dialog(GTK_WINDOW(dialog));
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	g_signal_connect(G_OBJECT(dialog), "key-press-event", G_CALLBACK(esc_key_pressed), NULL);
	g_signal_connect(G_OBJECT(dialog), "delete_event", G_CALLBACK(force_quit_dialog), NULL);

	gtk_widget_show_all(dialog);

	/*gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), "~");*/
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), APP_CMD"-feed-list.opml");
	gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK) {
		if ((file_name = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog))) != NULL) {
			if ((fp = g_fopen(file_name, "wb")) != NULL) {
				fprintf(fp, "%s",  str);
				fclose(fp);
				exit_status = OK;
			} else
				warning(FALSE, 3, "Can't save OPML file ", file_name, ": ",
					strerror(errno));
		}
	}
	gtk_widget_destroy(dialog);
	if (exit_status == OK) {
		snprintf(tmp, TMPSTR_SIZE + 1,
			"\nFeed list has been exported to OPML file: %s\n", file_name);
		info_win(APP_NAME, tmp, INFO, FALSE);
	}
	if (file_name != NULL)
			g_free(file_name);
	check_main_win_always_on_top();
	return exit_status;
}
#endif
