######################################################################
#
# $Header: /home/cvs/tkabber/tkabber/jabberlib-tclxml/jabberlib.tcl,v 1.52 2003/11/17 20:55:45 aleksey Exp $
#
# This is JabberLib (abbreviated jlib), the Tcl library for 
# use in making Jabber clients.
#
#
# Variables used in JabberLib :
#	roster(users)                : Users currently in roster
#
#	roster(group,$username)      : Groups $username is in.
#
#	roster(name,$username)       : Name of $username.
#
#	roster(subsc,$username)      : Subscription of $username 
#                                  ("to" | "from" | "both" | "")
#
#	roster(ask,$username)        : "Ask" of $username 
#                                  ("subscribe" | "unsubscribe" | "")
#
#	lib(wrap)                    : Wrap ID
#
#	lib(sck)                     : SocketName
#
#	lib(sckstats)                : Socket status, "on" or "off"
#
#	lib(disconnect)              : disconnect procedure
#
#	iq(num)                      : Next iq id-number. Sent in 
#                                  "id" attributes of <iq> packets.
#
#	iq($id)                      : Callback to run when result packet 
#                                  of $id is received.
#
#
######################################################################
#
# Procedures defined in this library
#
if {0} {
proc jlib::connect {sck server}
proc jlib::disconnect {}
proc jlib::got_stream {vars}
proc jlib::end_of_parse {}
proc jlib::outmsg {msg}
proc jlib::inmsg {}
proc jlib::clear_vars {}
proc jlib::clear_iqs {}
proc jlib::parse {xmldata}
proc jlib::parse_send_auth {cmd type data}
proc jlib::parse_send_create {cmd type data}
proc jlib::parse_roster_get {connid ispush cmd type data}
proc jlib::parse_roster_set {item cmd groups name type data}
proc jlib::parse_roster_del {item cmd type data}
proc jlib::send_iq {type xmldata args}
proc jlib::send_auth {user pass res cmd}
proc jlib::send_create {user pass name mail cmd}
proc jlib::send_msg {to args}
proc jlib::send_presence {args}
proc jlib::roster_get {args}
proc jlib::roster_set {item args}
proc jlib::roster_del {item args}
proc jlib::error {type condition args}
proc ::LOG text
proc jlib::noop args
}

lappend auto_path [file dirname [info script]]
if {![info exists use_external_tclxml] || $use_external_tclxml == 0} {
    package require -exact xml 2.0
} else {
    package require xml 2.0
}
package require sha1


namespace eval jlib {
    # Load XML:Wrapper
    source [file join [file dirname [info script]] "wrapper.tcl"]

    #set lib(wrap) [wrapper:new \
    #    	       [namespace current]::got_stream \
    #    	       [namespace current]::end_of_parse \
    #    	       [namespace current]::parse]

    if {[catch {package require sasl 1.0}]} {
	set lib(have_sasl) 0
    } else {
	set lib(have_sasl) 1
	sasl::client_init -callbacks [list [list log ::LOG]]
    }

    set lib(connections) {}
    set lib(connid) 0
    set iq(num) 0
    set lib(handshake) ""

    # Export procedures.
    #
    namespace export \
	"wrapper:splitxml" "wrapper:createtag" \
	"wrapper:createxml" "wrapper:xmlcrypt" \
	"wrapper:isattr" "wrapper:getattr"

    variable sasl_ns "urn:ietf:params:xml:ns:xmpp-sasl"
}


if {![info exists keep_alive]} {
    set keep_alive 0
}
if {![info exists keep_alive_interval]} {
    set keep_alive_interval 10
}

######################################################################

proc jlib::connect {sck server args} {
    variable lib
    variable connhist

    set xmlns jabber:client
    set newconn 0
    set user ""
    set httppoll 0
    set httppollint 3000
    set httppollmin 3000
    set httppollmax 30000
    set httpproxy 0
    set httpproxyauth ""
    set httppollurl ""
    set use_sasl 0

    foreach {attr val} $args {
	switch -- $attr {
	    -xmlns         {set xmlns $val}
	    -newconnection {set newconn $val}
	    -user          {set user $val}
	    -httppoll      {set httppoll $val}
	    -pollint       {set httppollint $val}
	    -pollmin       {set httppollmin $val}
	    -pollmax       {set httppollmax $val}
	    -pollurl       {set httppollurl $val}
	    -proxy         {set httpproxy $val}
	    -proxyauth     {set httpproxyauth $val}
	    -usesasl       {set use_sasl $val}
	}
    }

    if {[info exists connhist(${user}@$server)]} {
	set connid $connhist(${user}@$server)
	if {[lsearch -exact $lib(connections) $connid] >= 0} {
	    set connid [incr lib(connid)]
	}
    } else {
	set connid [incr lib(connid)]
	set connhist(${user}@$server) $connid
    }

    ::LOG "(jlib::connect) Socket:'$sck' Server:'$server' ConnectionID:'$connid'"

    if {!$newconn && $lib(connections) != {}} {
	::LOG "error (jlib::connect) Already connected"
	return -1
	# Already connected
    }

    if {!$httppoll && [catch {fconfigure $sck}] != 0} {
	::LOG "error (jlib::connect) Socket doesn't exist"
	return -2
	# Socket doesn't exist
    }

    lappend lib(connections) $connid
    set lib($connid,sck)        $sck
    set lib($connid,mechanisms) {}
    set lib($connid,server)     $server
    add_connection_route $connid $server
    set lib($connid,disconnect) "client:reconnect"
    set lib($connid,parse_end)  0
    set lib($connid,httppoll) $httppoll
    set lib($connid,httpwait) disconnected
    set lib($connid,httpoutdata) ""
    set lib($connid,httpseskey) 0
    set lib($connid,httppollint) $httppollint
    set lib($connid,httppollmin) $httppollmin
    set lib($connid,httppollmax) $httppollmax
    set lib($connid,httppollid) ""
    set lib($connid,httppollurl) $httppollurl
    set lib($connid,httpproxy) $httpproxy
    set lib($connid,httpproxyauth) $httpproxyauth
    set lib($connid,httptimeout) 15000
    set lib($connid,use_sasl) $use_sasl

    if {$lib($connid,use_sasl)} {
	set version " version='1.0'"
    } else {
	set version ""
    }	

    set lib($connid,wrap) \
	[wrapper:new \
	     [namespace current]::got_stream \
	     [namespace current]::end_of_parse \
	     [namespace current]::parse]

    if {!$httppoll} {
	fconfigure $sck -blocking 0 -buffering none \
	    -translation auto -encoding utf-8
    } else {
	set_httpwait $connid connected
    }
    outmsg [cconcat "<stream:stream xmlns:stream='" \
		[wrapper:xmlcrypt {http://etherx.jabber.org/streams}] \
		"' xmlns='[wrapper:xmlcrypt $xmlns]' " \
		"to='[wrapper:xmlcrypt $server]'$version>"] \
	-connection $connid
    if {!$httppoll} {
	fileevent $sck readable "[namespace current]::inmsg $connid"
    }

    return $connid
}

######################################################################
proc jlib::disconnect {{connections {}}} {
    variable lib

    ::LOG "(jlib::disconnect) $connections"

    if {$connections == {}} {
	set connections $lib(connections)
    }

    foreach connid $connections {
	catch { if {!$lib($connid,httppoll)} {
		flush $lib($connid,sck)

		outmsg "</stream:stream>" -connection $connid
		close $lib($connid,sck)
	    } else {
		outmsg "</stream:stream>" -connection $connid
		hook::run http_hook $connid disconnected
		set_httpwait $connid wait
	    }
	}

	clear_vars $connid
	set idx [lsearch -exact $lib(connections) $connid]
	set lib(connections) [lreplace $lib(connections) $idx $idx]
    }

    if {$lib(connections) == {}} {
	clear_iqs
    }
}

######################################################################
proc jlib::got_stream {vars} {
    variable lib

    set connid [get_connid]
    set lib($connid,sessionid) [jlib::wrapper:getattr $vars id]
    set lib($connid,use_sasl) \
	[expr {([jlib::wrapper:getattr $vars version] == "1.0") && \
		   $lib(have_sasl)}]

    ::LOG "(jlib::got_stream)\
Session ID = $lib($connid,sessionid), Use SASL = $lib($connid,use_sasl)"
}

######################################################################
proc jlib::end_of_parse {} {
    variable lib

    ::LOG "(jlib::end_of_parse)"

    set connid [get_connid]
    set lib($connid,parse_end) 1
    if { $lib(connections) == {} } {
	::LOG "error (jlib::end_of_parse) No connection"
	return -1
	# Already disconnected
    }

    catch {close $lib($connid,sck)}

    after idle [list [namespace current]::emergency_disconnect $connid]
}

######################################################################
proc jlib::outmsg {msg args} {
    global keep_alive keep_alive_interval
    variable keep_alive_id
    variable lib

    foreach {attr val} $args {
	switch -- $attr {
	    -connection {set connid $val}
	}
    }
    if {![info exists connid] || ![lcontain $lib(connections) $connid]} {
	set connid [lindex $lib(connections) 0]
    }

    ::LOG "(jlib::outmsg) ($connid) '$msg'"
    ::LOG_OUTPUT $connid $msg

    #catch { set msg [encoding convertto utf-8 $msg] }

    if { $lib(connections) == {} } {
	::LOG "error (jlib::outmsg) No connection"
	return -1
    }

    if {$lib($connid,httppoll)} {
	http_poll $connid $msg
	return
    }

    if {$keep_alive} {
	if {[info exists keep_alive_id]} {
	    after cancel $keep_alive_id
	}
	set keep_alive_id [after [expr $keep_alive_interval * 60 * 1000] \
			       [namespace current]::out_keepalive $connid]
    }

    if { [catch {puts $lib($connid,sck) $msg}] != 0 } {
	::LOG "error (jlib::outmsg) Cannot write to socket: $lib($connid,sck)"
	return -2
    }
}

######################################################################
proc jlib::inmsg {connid} {
    global keep_alive keep_alive_interval
    variable keep_alive_id
    variable lib
    variable current_connid

    #if { $lib(connections) == {} } {return}

    # TODO
    if {$keep_alive} {
	if {[info exists keep_alive_id]} {
	    after cancel $keep_alive_id
	}
	set keep_alive_id [after [expr $keep_alive_interval * 60 * 1000] \
			       [namespace current]::out_keepalive $connid]
    }

    set temp ""
    catch { set temp [read $lib($connid,sck)] }

    set_connid $connid
    ::LOG "(jlib::inmsg) ($connid) '$temp'"
    ::LOG_INPUT $connid $temp
    wrapper:parser $lib($connid,wrap) parse $temp

    if { !$lib($connid,parse_end) && [eof $lib($connid,sck)] } {
	::LOG "error (jlib::inmsg) Socket is closed by server. Disconnecting..."
	catch { fileevent $lib($connid,sck) readable {} }
	catch { close $lib($connid,sck) }
	after idle [list [namespace current]::emergency_disconnect $connid]
    }
}

######################################################################
proc jlib::emergency_disconnect {connid} {
    variable lib

    set idx [lsearch -exact $lib(connections) $connid]
    set lib(connections) [lreplace $lib(connections) $idx $idx]

    set disconnect $lib($connid,disconnect)
    clear_vars $connid
    if {$lib(connections) == {}} {
	clear_iqs
    }
    uplevel #0 "$disconnect $connid"
}

######################################################################
proc jlib::out_keepalive {sockid} {
    outmsg " " -connection $sockid
}

######################################################################
proc jlib::clear_vars {connid} {
    #
    # unset all the variables
    #
    variable roster
    variable pres
    variable lib
    variable iq

    if {[info exists lib($connid,sasl_token)]} {
	rename $lib($connid,sasl_token) ""
    }

    array unset lib $connid,*

    set lib($connid,disconnect) "client:reconnect"
    set lib(handshake) ""

    #set iq(num) 0

    # TODO: delete parser
    #wrapper:reset $lib(wrap)
}

######################################################################
proc jlib::clear_iqs {} {
  variable iq

  foreach id [array names iq] {
	if {$id != "num"} {
		uplevel #0 "$iq($id) DISCONNECT {}"
		unset iq($id)
	}
  }
}

######################################################################
proc jlib::set_connid {connid} {
    variable current_connid
    set current_connid $connid
}

proc jlib::get_connid {} {
    variable current_connid
    return $current_connid
}

proc jlib::connections {} {
    variable lib
    return $lib(connections)
}

proc jlib::connection_jid {connid} {
    variable lib
    return $lib($connid,user)@$lib($connid,server)/$lib($connid,res)
}

proc jlib::connection_bare_jid {connid} {
    variable lib
    return $lib($connid,user)@$lib($connid,server)
}

######################################################################
proc jlib::parse {xmldata} {
    after idle [list [namespace current]::parse1 [get_connid] $xmldata]
}

proc jlib::parse1 {connid xmldata} {
    variable global
    variable roster
    variable pres
    variable lib
    variable iq

    ::LOG "(jlib::parse) xmldata:'$xmldata'"
    ::LOG_INPUT_XML $connid $xmldata

    if { $lib(connections) == {} } {
        ::LOG "error (jlib::parse) No connection"
        return -1
    }

    set usefrom 0
    set from ""

    wrapper:splitxml $xmldata tag vars isempty chdata children
    if {[wrapper:isattr $vars from] == 1} {
	set usefrom 1
	set from [wrapper:getattr $vars from]
    }

    switch -- $tag {
	iq {
	    set useid   0
	    set id ""
	    set type [wrapper:getattr $vars type]

	    if {[wrapper:isattr $vars id] == 1} {
		set useid 1
		set id [wrapper:getattr $vars id]
	    }

	    if {$type != "result" && $type != "error" && $type != "get" && $type != "set"} {
		::LOG "(error) iq: unknown type:'$type' id ($useid):'$id'"
		return
	    }

	    if {$type == "result"} {
		if {$useid == 0} {
		    ::LOG "(error) iq:result: no id reference"
		    return
		}
		if {[info exists iq($id)] == 0} {
		    ::LOG "(error) iq:result: id doesn't exists in memory. Probably a re-replied iq"
		    return
		}

		set cmd $iq($id)
		unset iq($id)

		uplevel \#0 "$cmd OK [list [lindex $children 0]]"
	    } elseif {$type == "error"} {
		if {$useid == 0} {
		    ::LOG "(error) iq:result: no id reference"
		    return
		}
		if {[info exists iq($id)] == 0} {
		    ::LOG "(error) iq:result: id doesn't exists in memory. Probably a re-replied iq."
		    return
		}

		set cmd $iq($id)
		unset iq($id)

		set child ""
		foreach child $children {
		    if {[lindex $child 0] == "error"} {break}
		    set child ""
		}
		if {$child == ""} {
		    set errcode ""
		    set errtype ""
		    set errmsg ""
		} else {
		    set errcode [wrapper:getattr [lindex $child 1] code]
		    set errtype [wrapper:getattr [lindex $child 1] type]
		    set errmsg [lindex $child 3]
		}
		if {$errtype == ""} {
		    uplevel #0 "$cmd ERR [list [list $errcode $errmsg]]"
		} else {
		    uplevel #0 "$cmd ERR [list [list $errtype $child]]"
		}
	    } elseif {$type == "get" || $type == "set"} {
		set child [lindex $children 0]

		if {$child == ""} {
		    ::LOG "(error) iq:$type: Cannot find 'query' tag"
		    return
		}

		#
		# Before calling the 'client:iqreply' procedure, we should check
		# the 'xmlns' attribute, to understand if this is some 'iq' that
		# should be handled inside jlib, such as a roster-push.
		#
		if {$type == "set" && [wrapper:getattr [lindex $child 1] xmlns] == "jabber:iq:roster"} {
		    if {$from != "" && \
			    !([string equal -nocase $from $lib($connid,server)] || \
			    [string equal -nocase $from [connection_bare_jid $connid]] || \
			    [string equal -nocase $from [connection_jid $connid]])} {
			send_iq "error" \
			    [jlib::error cancel not-allowed \
				 -xml $child] \
			    -id [wrapper:getattr $vars "id"] \
			    -to $from \
			    -connection $connid
			return
		    }

		    # Found a roster-push
		    ::LOG "(info) iq packet is roster-push. Handling internally"

		    # First, we reply to the server, saying that, we 
		    # got the data, and accepted it.
		    #
		    if [wrapper:isattr $vars "id"] {
			send_iq "result" \
			    [wrapper:createtag query \
				 -vars [list "xmlns" "jabber:iq:roster"]] \
			    -id [wrapper:getattr $vars "id"] \
			    -connection $connid
		    } else {
			send_iq "result" \
			    [wrapper:createtag query \
				 -vars [list "xmlns" "jabber:iq:roster"]] \
			    -connection $connid
		    }

		    # And then, we call the jlib::parse_roster_get, because this
		    # data is the same as the one we get from a roster-get.
		    parse_roster_get \
			$connid 1 "[namespace current]::noop" "OK" $child
		    return
		}

		uplevel \#0 [list client:iqreply $connid $from $useid $id $type $child]}
	}
	message {set type [wrapper:getattr $vars type]
	    set id [wrapper:getattr $vars id]

	    set body     ""
	    set err      [list "" ""]
	    set subject  ""
	    set priority ""
	    set thread   ""
	    set x        ""

	    foreach child $children {
		wrapper:splitxml $child ctag cvars cisempty cchdata cchildren

		switch -- $ctag {
		    body {set body $cchdata}
		    error {
			set errmsg $cchdata
			set errcode [wrapper:getattr $cvars code]
			set errtype [wrapper:getattr $cvars type]
			if {$errtype == ""} {
			    set err [list $errcode $errmsg]
			} else {
			    set err [list $errtype $child]
			}
		    }
		    subject {set subject $cchdata}
		    priority {set priority $cchdata}
		    thread {set thread $cchdata}
		    default {
			if {[wrapper:getattr $cvars xmlns] != ""} {
			    lappend x $child
			}
		    }
		}
	    }

	    uplevel \#0 [list client:message $connid $from $id $type $subject $body $err $thread $priority $x]
	}
	presence {
	    set type [wrapper:getattr $vars type]

	    set status   ""
	    set priority ""
	    set meta     ""
	    set icon     ""
	    set show     ""
	    set loc      ""
	    set x        ""

	    set param    ""

	    foreach child $children {
		wrapper:splitxml $child ctag cvars cisempty cchdata cchildren

		switch -- $ctag {
		    status   {lappend param -status   $cchdata}
		    priority {lappend param -priority $cchdata}
		    meta     {lappend param -meta     $cchdata}
		    icon     {lappend param -icon     $cchdata}
		    show     {lappend param -show     $cchdata}
		    loc      {lappend param -loc      $cchdata}
		    x        {lappend x $child}
		    error {
			set errcode [wrapper:getattr $cvars code]
			set errtype [wrapper:getattr $cvars type]
			if {$errtype == ""} {
			    set err [list $errcode $cchdata]
			} else {
			    set err [list $errtype $child]
			}
			lappend param -status [lindex [error_to_list $err] 2]
		    }
		}
	    }

	    uplevel #0 "client:presence [list $connid $from $type $x] $param"
	}
        handshake {
	    set cmd $lib(handshake)
	    set lib(handshake) ""

	    if {[string compare $cmd ""]} {
                uplevel #0 $cmd OK
	    }
	}
	stream:error {
	    if {[string compare [set cmd $lib(handshake)] ""]} {
		set lib(handshake) ""

		uplevel #0 $cmd ERR $chdata
		return
	    }

	    switch -- $chdata {
		Disconnected {
		    set lib($connid,disconnect) "client:disconnect"
		}
		default {
		    set lib($connid,disconnect) "client:reconnect"
		}
	    }
	}
	stream:features {
	    parse_stream_features $connid $children
	}
	challenge {
	    variable sasl_ns
	    if {[wrapper:getattr $vars xmlns] == $sasl_ns} {
		sasl_step $connid $chdata
	    }
	}
	failure -
	success {
	    variable sasl_ns
	    if {[wrapper:getattr $vars xmlns] == $sasl_ns} {
		sasl_result $connid $tag
	    }
	}
    }
}

######################################################################
proc jlib::parse_send_auth {cmd type data} {
    variable lib

    ::LOG "(jlib::parse_send_auth) type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_send_auth) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_send_auth) errdesc:'[lindex $data 1]'"
	client:status [::msgcat::mc "Authentication failed"]
	uplevel #0 "$cmd ERR [list $data]"
	return
    }
    client:status [::msgcat::mc "Authentication successful"]
    uplevel #0 "$cmd OK {}"
}

######################################################################
proc jlib::parse_send_create {cmd type data} {
  variable lib

  ::LOG "(jlib::parse_send_create) type:'$type'"

  if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_send_create) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_send_create) errdesc:'[lindex $data 1]'"
	uplevel #0 "$cmd ERR [list [lindex $data 1]]"
	return
  }
  uplevel #0 "$cmd OK {}"
}

######################################################################
proc jlib::parse_roster_get {connid ispush cmd type data} {
    variable lib
    variable roster
    
    ::LOG "(jlib::parse_roster_get) ispush:'$ispush' type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_roster_get) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_roster_get) errdesc:'[lindex $data 1]'"
	uplevel #0 "$cmd $connid ERR"
	return
    }
    if {!$ispush} {
	client:status [::msgcat::mc "Got roster"]
	uplevel #0 "$cmd $connid BEGIN_ROSTER"
    }

    wrapper:splitxml $data tag vars isempty chdata children

    if {![cequal [wrapper:getattr $vars xmlns] jabber:iq:roster]} {::LOG "warning (jlib::parse_roster_get) 'xmlns' attribute of query tag doesn't match 'jabber:iq:roster': '[wrapper:getattr $vars xmlns]"}

    foreach child $children {
	wrapper:splitxml $child ctag cvars cisempty cchdata cchildren
	
	switch -- $ctag {
	    default {
		set groups ""
		set jid   [wrapper:getattr $cvars jid]
		set name  [wrapper:getattr $cvars name]
		set subsc [wrapper:getattr $cvars subscription]
		set ask   [wrapper:getattr $cvars ask]
		if {$ctag != {item}} {
		    set category $ctag
		} else {
		    set category [wrapper:getattr $cvars category]
		}
		set subtype [wrapper:getattr $cvars type]
		
		foreach subchild $cchildren {
		    wrapper:splitxml $subchild subtag tmp tmp subchdata tmp
		    
		    switch -- $subtag {
			group {lappend groups $subchdata}
		    }
		}
		
		# Ok, collected information about item.
		# Now we can set our variables...
		#
		if {[lsearch $roster(users) $jid] == -1} {
		    lappend roster(users) $jid
		}
		
		set roster(group,$jid) $groups
		set roster(name,$jid)  $name
		set roster(subsc,$jid) $subsc
		set roster(ask,$jid)   $ask

		add_connection_route $connid $jid

		# ...and call client procedures
		if $ispush {
		    uplevel \#0 [list client:roster_push $connid $jid \
				     $name $groups $subsc $ask \
				     $category $subtype]
		} else {
		    uplevel \#0 [list client:roster_item $connid $jid \
				     $name $groups $subsc $ask \
				     $category $subtype]
		}
	    }
	}
    }
    if {!$ispush} {
	uplevel #0 "$cmd $connid END_ROSTER"
    }
}

######################################################################
proc jlib::parse_roster_set {item cmd groups name type data} {
  variable lib
  variable roster

  ::LOG "(jlib::parse_roster_set) item:'$item' type:'$type'"

  if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_roster_set) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_roster_set) errdesc:'[lindex $data 1]'"
	uplevel #0 "$cmd ERR"
	return
  }

  if { [lsearch $roster(users) $item] == -1}   {
	lappend roster(users) $item
	set roster(subsc,$item) "none"
	set roster(ask,$item)   ""
  }

  set roster(group,$item) $groups
  set roster(name,$item)  $name

  uplevel #0 "$cmd OK"
}

######################################################################
proc jlib::parse_roster_del {item cmd type data} {
  variable lib
  variable roster

  ::LOG "(jlib::parse_roster_del) item:'$item' type:'$type'"

  if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_roster_set) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_roster_set) errdesc:'[lindex $data 1]'"
	uplevel #0 "$cmd ERR"
	return
  }

  if {[set num [lsearch $roster(users) $item]] != -1} {
	set roster(users) [lreplace $roster(users) $num $num]

	catch {unset roster(group,$item) }
	catch {unset roster(name,$item)  }
	catch {unset roster(subsc,$item) }
	catch {unset roster(ask,$item)   }
  } else {
	::LOG "warning (jlib::parse_roster_del) Item '$item' doesn't exist in roster for deletion."
  }
  uplevel #0 "$cmd OK"
}

######################################################################
proc jlib::parse_stream_features {connid childrens} {
    variable lib

    foreach child $childrens {
	wrapper:splitxml $child tag vars isempty chdata children

	switch -- $tag {
	    mechanisms {
		set mechanisms {}
		foreach mechelem $children {
		    wrapper:splitxml $mechelem \
			tag1 vars1 isempty1 chdata1 children1
		    if {$tag1 == "mechanism"} {
			lappend mechanisms $chdata1
		    }
		}
		set lib($connid,mechanisms) $mechanisms
	    }
	}
    }
    set lib($connid,features) ""
}

######################################################################
proc jlib::send_iq {type xmldata args} {
    variable lib
    variable iq

    ::LOG "(jlib::send_iq) type:'$type'"
    if { $lib(connections) == {} } {
          ::LOG "error (jlib::send_iq) No connection"
          return -1
    }

    set useto  0
    set useid  0
    set to     ""
    set id     ""
    set cmd    "[namespace current]::noop"
    set vars   ""

    foreach {attr val} $args {
	switch -- $attr {
	    -command    {set cmd $val}
	    -to         {set useto 1; set to $val}
	    -id         {set useid 1; set id $val}
	    -connection {set connid $val}
	}
    }
    if {![info exists connid]} {
	set connid [jlib::route $to]
    }

    if {$type != "set" && $type != "result" && $type != "error"} {
	set type "get"
    }

  ::LOG "(jlib::send_iq) type:'$type' to ($useto):'$to' cmd:'$cmd' xmldata:'$xmldata'"

    # Temporary hack that allows to insert more than 1 subtag in error iqs
    if {($type != "error") && ($xmldata != "")} {
	set xmldata [list $xmldata]
    }

    if {$type == "get" || $type == "set"} {
	lappend vars "id" $iq(num)
	set iq($iq(num)) $cmd
	incr iq(num)
    } elseif { $useid == 1 } {
	lappend vars "id" $id
    }

    if { $useto == 1 } {
	lappend vars "to" $to
    }
    lappend vars "type" $type

    if {$xmldata != ""} {
	set data [wrapper:createtag iq -vars $vars -subtags $xmldata]
    } else {
	set data [wrapper:createtag iq -vars $vars]
    }
    ::LOG_OUTPUT_XML $connid $data
    outmsg [wrapper:createxml $data] -connection $connid
}

######################################################################

proc jlib::parse_get_authtypes {connid type data} {
    variable lib

    ::LOG "(jlib::parse_get_authtypes) type:'$type'"

    if {$type == "DISCONNECT"} {
	return
    }
    if {$type == "ERR"} {
	::LOG "error (jlib::parse_get_authtypes) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_get_authtypes) errdesc:'[lindex $data 1]'"
	set lib($connid,authtypes) [list ERR $data]
	return
    }
    set authtypes {}
    wrapper:splitxml $data tag vars isempty chdata children
    foreach child $children {
	wrapper:splitxml $child tag1 vars1 isempty1 chdata1 children1
	switch -- $tag1 {
	    password {
		lappend authtypes plain
	    }
	    digest {
		lappend authtypes $tag1
	    }
	}
    }
    set lib($connid,authtypes) [list OK $authtypes]
}

proc jlib::get_authtypes {user {connid ""}} {
    variable lib

    if {$lib($connid,use_sasl)} {
	return SASL
    }

    if {$connid == ""} {
	set connid [lindex $lib(connections) 0]
    }

    ::LOG "(jlib::get_authtypes) username:'$user'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::get_authtypes) No connection"
	return -1
    }

    set data [wrapper:createtag query \
		  -vars    [list xmlns "jabber:iq:auth"] \
		  -subtags [list [wrapper:createtag username \
				     -chdata $user]]]

    send_iq get $data \
	-command "[namespace current]::parse_get_authtypes $connid" \
	-connection $connid
    
    client:status [::msgcat::mc "Waiting for authentication mechanisms"]
    
    vwait [namespace current]::lib($connid,authtypes)

    client:status [::msgcat::mc "Got authentication mechanisms"]
    return $lib($connid,authtypes)
}


proc jlib::send_auth {user pass res cmd {authtype plain} {connid ""}} {
    variable lib

    set lib($connid,user) $user
    set lib($connid,pass) $pass
    set lib($connid,res)  $res

    if {$connid == ""} {
	set connid [lindex $lib(connections) 0]
    }

    ::LOG "(jlib::send_auth) username:'$user' password:'$pass' resource:'$res'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::send_auth) No connection"
	return -1
    }

    if {$lib($connid,use_sasl)} {
	sasl_auth $connid $user $pass $res $cmd
	return
    }

    switch -- $authtype {
	plain {
	    set data [wrapper:createtag query \
			  -vars    [list xmlns "jabber:iq:auth"] \
			  -subtags [list \
					[wrapper:createtag username \
					     -chdata $user] \
					[wrapper:createtag password \
					     -chdata $pass] \
					[wrapper:createtag resource \
					     -chdata $res]]]
	}
	digest {
	    set digest [sha1::sha1 [cconcat $lib($connid,sessionid) \
					[encoding convertto utf-8 $pass]]]
	    set data [wrapper:createtag query \
			  -vars    [list xmlns "jabber:iq:auth"] \
			  -subtags [list \
					[wrapper:createtag username \
					     -chdata $user] \
					[wrapper:createtag digest \
					     -chdata $digest] \
					[wrapper:createtag resource \
					     -chdata $res]]]
	}

        handshake {
            set lib(handshake) $cmd
	    set digest [sha1::sha1 $lib($connid,sessionid)$pass]
            return [outmsg "<handshake>$digest</handshake>"]
	}
    }

    send_iq set $data \
	-command "[namespace current]::parse_send_auth [list $cmd]" \
	-connection $connid
    client:status [::msgcat::mc "Waiting for authentication results"]
}

######################################################################
proc jlib::sasl_auth {connid user pass res cmd} {
    variable lib
    variable sasl_ns

    set lib($connid,auth_cmd) $cmd

    if {![info exists lib($connid,mechanisms)] || \
	    $lib($connid,mechanisms) == {}} {
	vwait [namespace current]::lib($connid,mechanisms)
    }

    ::LOG "(jlib::sasl_auth) start"

    foreach id {authname pass getrealm cnonce} {
	lappend callbacks \
	    [list $id [list [namespace current]::sasl_callback $connid]]
    }

    set lib($connid,sasl_token) \
	[sasl::client_new \
	     -service     xmpp \
	     -serverFQDN  $lib($connid,server) \
	     -callbacks   $callbacks \
	     -flags       success_data]

    #$lib($connid,sasl_token) -operation setprop \
    #    -property auth_external \
    #    -value server@example.com

    $lib($connid,sasl_token) -operation setprop \
	-property sec_props \
	-value [list \
		    min_ssf 0 \
		    max_ssf 0 \
		    flags {noplaintext}]

    #$lib($connid,sasl_token) -operation setprop \
    #    -property ssf_external \
    #    -value 0

    ::LOG "(jlib::sasl_auth) mechs: $lib($connid,mechanisms)"

    set code [catch {
	$lib($connid,sasl_token) \
	    -operation  start \
	    -mechanisms $lib($connid,mechanisms) \
	    -interact   [namespace current]::sasl_interact
    } result]

    switch -- $code {
	0 {
	    ::LOG "(jlib::sasl_auth) SASL OK: $result"
	    array set resarray $result
	    ::LOG "(jlib::sasl_auth) $resarray(output)"

	    set data [wrapper:createtag auth \
			  -vars   [list \
				       xmlns $sasl_ns \
				       mechanism $resarray(mechanism)] \
			  -chdata [sasl::encode64 $resarray(output)]]

	    outmsg [wrapper:createxml $data] -connection $connid
	}
	4 {
	    ::LOG "(jlib::sasl_auth) SASL CONTINUE: $result"
	    array set resarray $result
	    ::LOG "(jlib::sasl_auth) $resarray(output)"

	    set data [wrapper:createtag auth \
			  -vars   [list \
				       xmlns $sasl_ns \
				       mechanism $resarray(mechanism)] \
			  -chdata [sasl::encode64 $resarray(output)]]

	    outmsg [wrapper:createxml $data] -connection $connid
	}
	default {
	    ::LOG "(jlib::sasl_auth) SASL code $code: $result"
	    return
	}
    }
}

proc jlib::sasl_step {connid serverin64} {
    variable lib
    variable sasl_ns

    set serverin [sasl::decode64 $serverin64]

    set code [catch {
	$lib($connid,sasl_token) \
	    -operation  step \
	    -input      $serverin \
	    -interact   [namespace current]::sasl_interact
    } result]

    switch -- $code {
	0 {
	    ::LOG "(jlib::sasl_step) SASL OK: $result"
	    array set resarray $result
	    if {![info exists resarray(output)]} {
		set resarray(output) ""
	    }
	    ::LOG "(jlib::sasl_step) $resarray(output)"

	    set data [wrapper:createtag response \
			  -vars   [list xmlns $sasl_ns] \
			  -chdata [sasl::encode64 $resarray(output)]]

	    outmsg [wrapper:createxml $data] -connection $connid
	}
	4 {
	    ::LOG "(jlib::sasl_step) SASL CONTINUE: $result"

	    set data [wrapper:createtag response \
			  -vars   [list \
				       xmlns $sasl_ns] \
			  -chdata [sasl::encode64 $result]]

	    outmsg [wrapper:createxml $data] -connection $connid
	}
	default {
	    ::LOG "(jlib::sasl_step) SASL code $code: $result"
	    return
	}
    }
}

proc jlib::sasl_result {connid result} {
    variable lib

    switch -- $result {
	success {
	    wrapper:reset $lib($connid,wrap)

	    outmsg [cconcat "<stream:stream xmlns:stream='" \
			"http://etherx.jabber.org/streams" \
			"' xmlns='jabber:client' " \
			"to='[wrapper:xmlcrypt $lib($connid,server)]' "\
			"version='1.0'>"] \
		-connection $connid
	
	    vwait [namespace current]::lib($connid,features)

	    set ns_bind urn:ietf:params:xml:ns:xmpp-bind
	    set data [wrapper:createtag bind \
			  -vars [list xmlns $ns_bind] \
			  -subtags [list [wrapper:createtag resource \
					      -chdata $lib($connid,res)]]]

	    send_iq set $data \
		-command [list [namespace current]::parse_resource_bind \
			      $lib($connid,auth_cmd) $connid] \
		-connection $connid
	}
	failure {
	    uplevel \#0 [list eval $lib($connid,auth_cmd) [list ERR [list 401 Unauthorized]]]
	}
    }
}



proc jlib::sasl_callback {connid data} {
    variable lib

    ::LOG "(jlib::sasl_callback) $data"

    array set params $data

    switch -- $params(id) {
        authname {
            set value [encoding convertto utf-8 $lib($connid,user)]
        }
        user {
            set value \
		[encoding convertto utf-8 \
		     $lib($connid,user)@$lib($connid,server)]
        }
        pass {
            set value [encoding convertto utf-8 $lib($connid,pass)]
        }
        getrealm {
	    set value [encoding convertto utf-8 $lib($connid,server)]
        }
    }

    ::LOG "(jlib::sasl_callback) return: $value"
    return $value
}


proc jlib::sasl_interact {data} {
    variable lib

    array set params $data

    ::LOG "(jlib::sasl_interact) $data"
}


proc jlib::parse_resource_bind {cmd connid type data} {
    variable lib

    ::LOG "(jlib::parse_resource_bind) type:'$type'"

    if {$type == "ERR"} {           ;# Got an error reply
	::LOG "error (jlib::parse_send_auth) errtype:'[lindex $data 0]'"
	::LOG "error (jlib::parse_send_auth) errdesc:'[lindex $data 1]'"
	client:status [::msgcat::mc "Authentication failed"]
	uplevel #0 "$cmd ERR [list $data]"
	return
    }

    set ns_session urn:ietf:params:xml:ns:xmpp-session
    set data [wrapper:createtag session \
		  -vars [list xmlns $ns_session]]

    send_iq set $data \
	-command [list [namespace current]::parse_send_auth \
		      $lib($connid,auth_cmd)] \
	-connection $connid
}


######################################################################
proc jlib::route {jid} {
    variable lib

    if { $lib(connections) == {} } {
	::LOG "error (jlib::send_create) No connection"
	return -1
    }

    set user $jid
    regexp {([^/]*)/.*} $jid temp user
    set serv $user
    regexp {[^@]*@(.*)} $user temp serv

    set connid [lindex $lib(connections) 0]
    foreach dest [list $user $serv] {
	foreach c $lib(connections) {
	    if {[info exists lib($c,route,$dest)]} {
		return $c
	    }
	}
    }

    return $connid
}


proc jlib::add_connection_route {connid jid} {
    variable lib

    set lib($connid,route,$jid) 1
}


######################################################################
# TODO
proc jlib::send_create {user pass name mail cmd} {
    variable lib

    ::LOG "(jlib::send_create) username:'$user' password:'$pass' name:'$name' email:'$mail'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::send_create) No connection"
	return -1
    }

    set data [wrapper:createtag query \
		  -vars    [list xmlns "jabber:iq:register"] \
		  -subtags [list \
				[wrapper:createtag name     -chdata $name] \
				[wrapper:createtag email    -chdata $mail] \
				[wrapper:createtag username -chdata $user] \
				[wrapper:createtag password -chdata $pass]]]

    send_iq set $data \
	-command "[namespace current]::parse_send_create [list $cmd]"
}

######################################################################
proc jlib::send_msg {to args} {
    variable lib

    ::LOG "(jlib::send_msg) to:'$to'"
    if { $lib(connections) == {} } {::LOG "error (jlib::send_msg) No connection"; return -1}

    set connid [route $to]
    if [wrapper:isattr $args -connection] {set connid [wrapper:getattr $args -connection]   }

    set children ""

    if {[wrapper:isattr $args -subject] == 1}  {lappend children [wrapper:createtag subject  -chdata [wrapper:getattr $args -subject]]}
    if {[wrapper:isattr $args -thread] == 1}   {lappend children [wrapper:createtag thread   -chdata [wrapper:getattr $args -thread]]}
    if {[wrapper:isattr $args -body] == 1}     {lappend children [wrapper:createtag body     -chdata [wrapper:getattr $args -body]]}
    if {[wrapper:isattr $args -xlist] == 1}    {foreach a [wrapper:getattr $args -xlist] {lappend children $a}}

    set vars [list "to" $to]
    if {[wrapper:isattr $args -type] == 1} {lappend vars "type" [wrapper:getattr $args -type]}

    set data [wrapper:createtag message -vars $vars -subtags $children]
    ::LOG_OUTPUT_XML $connid $data
    outmsg [wrapper:createxml $data] -connection $connid
}

######################################################################
proc jlib::send_presence {args} {
    variable lib

    ::LOG "(jlib::send_presence)"
    if { $lib(connections) == {} } {::LOG "error (jlib::send_presence) No connection"; return -1}

    set children ""
    set vars     ""

    # TODO
    if [wrapper:isattr $args -to]   {
	set to [wrapper:getattr $args -to]
	lappend vars to $to
    }
    if [wrapper:isattr $args -type] {lappend vars type [wrapper:getattr $args -type] }

    if {[info exists to]} {
	set connid [route $to]
    } else {
	set connid [lindex $lib(connections) 0]
    }
    if [wrapper:isattr $args -connection] {set connid [wrapper:getattr $args -connection]   }

    if [wrapper:isattr $args -stat] {lappend children [wrapper:createtag status   -chdata [wrapper:getattr $args -stat]] }
    if [wrapper:isattr $args -pri]  {lappend children [wrapper:createtag priority -chdata [wrapper:getattr $args -pri]]  }
    if [wrapper:isattr $args -meta] {lappend children [wrapper:createtag meta     -chdata [wrapper:getattr $args -meta]] }
    if [wrapper:isattr $args -icon] {lappend children [wrapper:createtag icon     -chdata [wrapper:getattr $args -icon]] }
    if [wrapper:isattr $args -show] {lappend children [wrapper:createtag show     -chdata [wrapper:getattr $args -show]] }
    if [wrapper:isattr $args -loc]  {lappend children [wrapper:createtag loc      -chdata [wrapper:getattr $args -loc]]  }

    if [wrapper:isattr $args -xlist] {foreach a [wrapper:getattr $args -xlist] {lappend children $a}}

    set data [wrapper:createtag presence -vars $vars -subtags $children]
    ::LOG_OUTPUT_XML $connid $data
    outmsg [wrapper:createxml $data] -connection $connid
}

######################################################################
proc jlib::roster_get {args} {
    variable lib
    variable roster

    ::LOG "(jlib::roster_get)"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::roster_get) No connection"
	return -1
    }

    set cmd "[namespace current]::noop"
    set connid [lindex $lib(connections) 0]
    foreach {attr val} $args {
	switch -- $attr {
	    -command    {set cmd $val}
	    -connection {set connid $val}
	}
    }

    foreach array [array names roster] {
	unset roster($array)
    }
    set roster(users) ""

    set vars [list xmlns "jabber:iq:roster"]
    set data [wrapper:createtag query -empty 1 -vars $vars]
    send_iq get $data \
	-command [list [namespace current]::parse_roster_get $connid 0 $cmd] \
	-connection $connid
    client:status [::msgcat::mc "Waiting for roster"]
}

######################################################################
proc jlib::roster_set {item args} {
  variable lib
  variable roster

  ::LOG "(jlib::roster_set) item:'$item'"
  if { $lib(connections) == {} } {
	::LOG "error (jlib::roster_set) No connection"
	return -1
  }

  set usename 0; set name ""
  if { [lsearch $roster(users) $item] == -1 } {
	set groups ""
  } else {
	set groups $roster(group,$item)
  }

  if [wrapper:isattr $args "-name"]    {set usename 1; set name [wrapper:getattr $args "-name"] }
  if [wrapper:isattr $args "-groups"]  {set groups [wrapper:getattr $args "-groups"] }
  if [wrapper:isattr $args "-command"] {set cmd    [wrapper:getattr $args "-command"] } else {set cmd "[namespace current]::noop"}

  set vars [list jid $item]
  if $usename  {lappend vars name $name }

  set subdata ""
  foreach group $groups {
	lappend subdata [wrapper:createtag group -chdata $group]
  }

  set xmldata [wrapper:createtag query \
		-vars    [list xmlns "jabber:iq:roster"] \
		-subtags [list [wrapper:createtag item -vars $vars -subtags $subdata]]]

  send_iq set $xmldata -command "[namespace current]::parse_roster_set [list $item $cmd $groups $name]"
}

######################################################################
proc jlib::roster_del {item args} {
    variable lib
    variable roster

    ::LOG "(jlib::roster_del) item:'$item'"
    if { $lib(connections) == {} } {
	::LOG "error (jlib::roster_del) No connection"
	return -1
    }

    # TODO

    if [wrapper:isattr $args -command] {
	set cmd [wrapper:getattr $args -command]
    } else {
	set cmd "[namespace current]::noop"
    }

    set xmldata [wrapper:createtag query \
		     -vars    [list xmlns "jabber:iq:roster"] \
		     -subtags [list [wrapper:createtag item -vars [list jid $item subscription "remove"]]]]

    send_iq set $xmldata \
	-command "[namespace current]::parse_roster_del [list $item $cmd]"
}

######################################################################
proc jlib::wait_for_stream {{connid ""}} {
    variable lib

    client:status [::msgcat::mc "Waiting for stream"]
    if {$connid == ""} {
	set connid [lindex $lib(connections) 0]
    }
    vwait [namespace current]::lib($connid,sessionid)
    client:status [::msgcat::mc "Got stream"]
}

######################################################################

set ::NS(stanzas) "urn:ietf:params:xml:ns:xmpp-stanzas"

array set ::error_type [list \
	auth	    "Authentication Error" \
	cancel	    "Unrecoverable Error" \
	continue    "Warning" \
	modify	    "Request Error" \
	wait	    "Temporary Error"]

set ::defined_error_conditions {}
# Code is zero iff the condition isn't mentioned in JEP-0086
foreach {clist lcode type cond description} [list \
      {400 406}     400 modify	bad-request		"Bad Request" \
      {409}	    409 cancel	conflict		"Conflict" \
      {501}	    501 cancel	feature-not-implemented "Feature Not Implemented" \
      {403}	    403 auth	forbidden		"Forbidden" \
      {500}	    500 wait	internal-server-error   "Internal Server Error" \
      {404}	    404 cancel	item-not-found		"Item Not Found" \
      {}	    400 modify	jid-malformed		"JID Malformed" \
      {401 405}     405 cancel	not-allowed		"Not Allowed" \
      {402}	    402 auth	payment-required	"Payment Required" \
      {}	    404 wait	recipient-unavailable   "Recipient Unavailable" \
      {407}	    407 auth	registration-required   "Registration Required" \
      {}	    404 cancel	remote-server-not-found "Remote Server Not Found" \
      {408 504}     504 wait	remote-server-timeout   "Remote Server Timeout" \
      {}	    500 wait	resource-constraint	"Resource Constraint" \
      {502 503 510} 503 cancel	service-unavailable	"Service Unavailable" \
      {}	    407 auth	subscription-required   "Subscription Required" \
      {}	    500 any	undefined-condition	"Undefined Condition" \
      {}	    400 wait	unexpected-request	"Unexpected Request"] \
    {
	lappend ::defined_error_conditions $cond
	set ::error_description($type,$::NS(stanzas),$cond) $description
	# JEP-0086
	foreach code $clist {
	    set ::error_type_descelem($code) [list $type $cond]
	}
	set ::legacy_error_codes($cond) $lcode
    }

proc jlib::error_to_list {errmsg} {
    global error_type defined_error_conditions error_description error_type_descelem

    lassign $errmsg code desc
    if {[string is integer $code]} {
	if {[info exists error_type_descelem($code)]} {
	    lassign $error_type_descelem($code) type descelem
	} else {
	    lassign {none none} type descelem
	}
	return [list $type $descelem "$code ([::msgcat::mc $desc])"]
    } else {
	set type $code
	set errelem $desc
	set condition "undefined-condition"
	set description ""
	wrapper:splitxml $errelem tag vars isempty chdata children
	foreach child $children {
	    jlib::wrapper:splitxml $child tag1 vars1 isempty1 chdata1 children1
	    set xmlns [jlib::wrapper:getattr $vars1 xmlns]
	    set cond $tag1
	    switch -- $cond {
		text {
		    if {$xmlns == $::NS(stanzas)} {
			set description $chdata1
		    }
		}
		undefined-condition {
		    # TODO
		}
		default {
		    if {[lsearch -exact $defined_error_conditions $cond] >= 0} {
			set condition $cond
			if {[info exists error_description($type,$xmlns,$cond)] && \
				($description == "")} {
			    set description $error_description($type,$xmlns,$cond)
			}
		    } else {
			# TODO
		    }
		}
	    }
	}
	if {[info exists error_type($type)]} {
	    set typedesc $error_type($type)
	}
	set res ""
	if {$description != ""} {
	    set res "[::msgcat::mc $description]"
	}
	if {[info exists typedesc] && $typedesc != ""} {
	    if {$res == ""} {
		set res "[::msgcat::mc $typedesc]"
	    } else {
		set res "[::msgcat::mc $typedesc] ($res)"
	    }
	}
	return [list $type $condition $res]
    }
}

proc jlib::error {type condition args} {
    return [eval xmpp_error $type $condition $args]
}

proc jlib::legacy_error {type condition args} {
    global legacy_error_codes error_description

    if {[info exists legacy_error_codes($condition)] && \
	    $legacy_error_codes($condition)} {
	set code $legacy_error_codes($condition)
    } else {
	set code 501
    }
    if {[info exists error_description($type,$::NS(stanzas),$condition)]} {
	set description $error_description($type,$::NS(stanzas),$condition)
    } else {
	set description ""
    }
    set xml ""
    foreach {opt val} $args {
	switch -- $opt {
	    -xml {
		set xml $val
	    }
	    -text {
		set description $val
	    }
	}
    }
    set err [wrapper:createtag error \
		-vars [list code $code] \
		-chdata $description]
    if {$xml == ""} {
	return [list $err]
    } else {
	return [list $xml $err]
    }
}

proc jlib::xmpp_error {type condition args} {
    global legacy_error_codes

    set subtags [list [wrapper:createtag $condition \
			   -vars [list xmlns $::NS(stanzas)]]]
    set xml ""
    foreach {opt val} $args {
	switch -- $opt {
	    -xml {
		set xml $val
	    }
	    -text {
		lappend subtags [wrapper:createtag text \
				     -vars [list xmlns $::NS(stanzas)] \
				     -chdata $val]
	    }
	    -application-specific {
		lappend subtags $val
	    }
	}
    }
    set vars [list type $type]
    if {[info exists legacy_error_codes($condition)] && \
	    $legacy_error_codes($condition)} {
	lappend vars code $legacy_error_codes($condition)
    }
    set err [wrapper:createtag error \
		-vars $vars \
		-subtags $subtags]
    if {$xml == ""} {
	return [list $err]
    } else {
	return [list $xml $err]
    }
}

######################################################################
#
proc ::LOG text {
#
# For debugging purposes.
#

    puts "LOG: $text\n"

}

proc ::LOG_OUTPUT     {connid t} {}
proc ::LOG_OUTPUT_XML {connid x} {}
proc ::LOG_INPUT      {connid t} {}
proc ::LOG_INPUT_XML  {connid x} {}

######################################################################
proc jlib::noop args {}


######################################################################
#
# HTTP Polling
#
######################################################################

package require http

proc httpdebug {msg} {
    puts $msg
}

proc jlib::set_httpwait {connid opt} {
    variable lib

    if {$lib($connid,httpwait) != $opt} {
	hook::run http_hook $connid $opt
    }
    set lib($connid,httpwait) $opt
    if {$opt == "disconnected"} {
	if {$lib($connid,httppollid) != ""} {
	    after cancel $lib($connid,httppollid)
	}
    }
}

proc jlib::process_httpreply {connid token} {
    variable lib

    upvar #0 $token state

    if {[set temp [::http::ncode $token]] != 200} {
	::LOG "error (jlib::process_httpreply) Http returned $temp"
	set_httpwait $connid disconnected
	jlib::emergency_disconnect $connid
	::http::cleanup $token
	return
    }

    foreach {name value} $state(meta) {
	if {[string equal -nocase "Set-Cookie" $name]} {
	    set start 0
	    set end [string first ";" $value]
	    if {$end < 1} {
		set end [string length $value]
	    }
	    if {[string equal -nocase -length 3 "ID=" $value]} {
		set start 3
	    }
	    set lib($connid,httpseskey) [string range $value $start [expr {$end - 1}]]
	}
    }
    set inmsg [encoding convertfrom utf-8 $state(body)]
    ::http::cleanup $token

    if {[regexp {:0$} $lib($connid,httpseskey)] || \
	    [regexp {%3A0$} $lib($connid,httpseskey)]} {
	::LOG "error (jlib::process_httpreply) Cookie Error"
	set_httpwait $connid disconnected
	jlib::emergency_disconnect $connid
	return
    }

    if {[string length $inmsg] > 5 } {
	set lib($connid,httppollint) [expr $lib($connid,httppollint) / 2]
	if {$lib($connid,httppollint) < $lib($connid,httppollmin)} {
	    set lib($connid,httppollint) $lib($connid,httppollmin)
	}
    } else {
	set lib($connid,httppollint) [expr $lib($connid,httppollint) * 11 / 10]
	if {$lib($connid,httppollint) > $lib($connid,httppollmax)} {
	    set lib($connid,httppollint) $lib($connid,httppollmax)
	}
    }

    http_inmsg $connid $inmsg


    switch -- $lib($connid,httpwait) {
	disconnected -
	wait {
	    set_httpwait $connid disconnected
	}
	default {
	    set_httpwait $connid connected
	}
    }
}

proc jlib::http_keep {connid} {
    variable lib

    if {[info exists lib($connid,httpwait)] && \
	    $lib($connid,httpwait) == "connected"} {
	http_poll $connid " "
    }
}

proc jlib::http_poll {connid what} {
    variable lib

    append lib($connid,httpoutdata) [encoding convertto utf-8 $what]
    if {$lib($connid,httpwait) != "connected"} {
        httpdebug "poll ($connid): Waiting for godot!"
        return
    }
    set query "$lib($connid,httpseskey),$lib($connid,httpoutdata)"
    set_httpwait $connid polling
    if {$lib($connid,httpproxy)} {
        ::http::geturl $lib($connid,httppollurl) -query $query \
	    -command [list jlib::process_httpreply $connid] \
	    -timeout $lib($connid,httptimeout) \
	    -headers $lib($connid,httpproxyauth)
    } else {
        ::http::geturl $lib($connid,httppollurl) -query $query \
	    -command [list jlib::process_httpreply $connid]\
	    -timeout $lib($connid,httptimeout)
    }
    set lib($connid,httpoutdata) ""

    if {[info exists lib($connid,httppollid)]} {
        after cancel $lib($connid,httppollid)
    }
    httpdebug "poll ($connid): $lib($connid,httppollint)"
    set lib($connid,httppollid) \
	[after $lib($connid,httppollint) [list jlib::http_keep $connid]]
}

proc jlib::http_inmsg {connid body} {
    variable lib

    ::LOG_INPUT $connid $body
    if {[string length $body] > 2} {
	set_connid $connid
        wrapper:parser $lib($connid,wrap) parse $body
    }
}

######################################################################
#
# Now that we're done...
#
#jlib::clear_vars
package provide jabberlib 0.8.4

