# $Id: gpgme.tcl,v 1.23 2004/07/15 21:56:10 aleksey Exp $

namespace eval ssj {
    variable options

    if {![info exists options(one-passphrase)]} {
        set options(one-passphrase) 0
    }

    if {[llength [array names options encrypt,*]] > 0} {
        set needP 1
    } else {
        set needP 0
    }
    if {![info exists options(sign-traffic)]} {
        if {[info exists options(encrypt-traffic)]} {
            set options(sign-traffic) 1
        } else {
            set options(sign-traffic) $needP
        }
    }

    if {![info exists options(encrypt-traffic)]} {
        set options(encrypt-traffic) $options(sign-traffic)
    }

    if {[catch { package require gpgme }]} {
        if {($options(sign-traffic)) || ($options(encrypt-traffic))} {
            set needP 1
        }

        if {$needP} {
            debugmsg ssj \
                     "unable to load the PGPME package, so no crypto!

See contrib/pgpme for details"
        }

        return
    }
}


package require base64


namespace eval ssj {
    variable ctx
    variable e4me
    variable j2k
    variable options
    variable passphrase
    variable s2e
    variable signers
    variable warnings
    variable gpg_error_id 0

    if {![info exists ctx]} {
        set ctx ""
    }

    array set j2k {}

    array set options {}

    array set passphrase {}

    array set s2e \
          [list none       "No information available"                         \
                bad        "Invalid signature"                                \
                nokey      "Signature not processed due to missing key"       \
                nosig      "Malformed signature block"                        \
                error      "Error in signature processing"                    \
                diff       "Multiple signatures having different authenticity"\
                expired    "The signature is good but has expired"            \
                expiredkey "The signature is good but the key has expired"]

    catch { unset warnings }
    array set warnings {}
}


proc ssj::once_only {{armorP 0}} {
    global env loginconf

    variable ctx
    variable selectkey

    if {![cequal $ctx ""]} {
        $ctx -operation set   \
             -property  armor \
             -value     $armorP

        return
    }

    set ctx [gpgme::context]
    $ctx -operation set   \
         -property  armor \
         -value     $armorP


    if {![info exists env(GPG_AGENT_INFO)]} {
        $ctx -operation set                 \
             -property  passphrase-callback \
             -value     [namespace current]::passphrase
    }

    set pattern [jlib::connection_bare_jid [jlib::route ""]]

    set firstP 1
    foreach p [list $pattern ""] {
        set command [list $ctx -operation start-key -secretonly true]
        if {![cequal $p ""]} {
            lappend command -patterns [list $p]
        }
        eval $command

        for {set keys {}} \
            {![cequal [set key [$ctx -operation next-key]] ""]} \
            {lappend keys $key} {}
        $ctx -operation done-key

        if {[llength $keys] > 0} {
            break
        }
        if {[cequal $p ""]} {
            return
        }
        set firstP 0
    }

    switch -- [llength $keys] {
        0 {
            return
        }

        1 {
            if {$firstP} {
                e4meP $keys
                return
            }
        }

        default {
        }
    }

    set dw .selectkey
    catch { destroy $dw }

    set titles {}
    set balloons {}
    foreach key $keys {
        foreach {k v} [$ctx -operation info-key -key $key] {
            if {![cequal $k subkeys]} {
                continue
            }
            foreach subkey $v {
                catch { unset params }
                array set params $subkey
                if {![info exists params(email)]} {
                    continue
                }
		lappend titles $key $params(email)
                if {![catch { format "%d%s/%s %s %s" $params(length)         \
                                     [string range $params(algorithm) 0 0]   \
                                     [string range $params(keyid) end-7 end] \
                                     [clock format $params(created)          \
                                            -format "%Y-%m-%d"]              \
                                     $params(userid) } text]} {
		    lappend balloons $key $text
		}
	    }
	}
    }

    CbDialog $dw [::msgcat::mc "Select Key for Signing Traffic"] \
        [list [::msgcat::mc "Select"] "[namespace current]::once_only_aux $dw" \
	      [::msgcat::mc "Cancel"] "destroy $dw"] \
	[namespace current]::selectkey $titles $balloons \
	-modal local
}

proc ssj::once_only_aux {dw} {
    variable ctx
    variable selectkey

    set keys {}
    foreach key [array names selectkey] {
        if {$selectkey($key)} {
            lappend keys $key
        }
    }

    destroy $dw

    if {[llength $keys] > 0} {
        e4meP $keys
    }
}


proc ssj::passphrase {data} {
    variable ctx
    variable passphrase
    variable options

    array set params $data
    set lines [split [string trimright $params(description)] "\n"]
    set text [lindex $lines 0]

    if {[set x [string first " " [set keyid [lindex $lines 1]]]] > 0} {
        set userid [string range $keyid [expr $x+1] end]
        if {!$options(one-passphrase)} {
            set keyid [string range $keyid 0 [expr $x-1]]
        } else {
            regexp { +([^ ]+)} [lindex $lines 2] ignore keyid
        }
    } else {
        set userid unknown!
    }

    if {([cequal $text ENTER]) \
            &&([info exists passphrase($keyid)]) \
            && (![cequal $passphrase($keyid) ""])} {
        return $passphrase($keyid)
    }

    set pw .passphrase
    if {[winfo exists $pw]} {
        destroy $pw
    }

    set title [::msgcat::mc "Please enter passphrase"]
    switch -- $text {
        ENTER {
        }

        TRY_AGAIN {
            set title [::msgcat::mc "Please try again"]
        }

        default {
            append title ": " $text
        }
    }
    Dialog $pw -title $title -separator 1 -anchor e -default 0 -cancel 1

    set pf [$pw getframe]
    grid columnconfigure $pf 1 -weight 1

    foreach {k v} [list keyid [::msgcat::mc "Key ID"] userid [::msgcat::mc "User ID"]] {
        label $pf.l$k -text ${v}:
        entry $pf.$k
        $pf.$k insert 0 [set $k]
        if {[string length [set $k]] <= 72} {
            $pf.$k configure -width 0
        }
        if {[info tclversion] >= 8.4} {
            set bgcolor [lindex [$pf.$k configure -background] 4]
            $pf.$k configure -state readonly -readonlybackground $bgcolor
        } else {
            $pf.$k configure -state disabled
        }
    }

    label $pf.lpassword -text [::msgcat::mc "Passphrase:"]
    entry $pf.password  -textvariable [namespace current]::passphrase($keyid) \
                        -show *
    set passphrase($keyid) ""

    grid $pf.lkeyid    -row 0 -column 0 -sticky e
    grid $pf.keyid     -row 0 -column 1 -sticky ew
    grid $pf.luserid   -row 1 -column 0 -sticky e
    grid $pf.userid    -row 1 -column 1 -sticky ew
    grid $pf.lpassword -row 2 -column 0 -sticky e
    grid $pf.password  -row 2 -column 1 -sticky ew

    $pw add -text [::msgcat::mc "OK"] -command "$pw enddialog 0"
    $pw add -text [::msgcat::mc "Cancel"] -command "$pw enddialog 1"

    if {[set abort [$pw draw $pf.password]]} {
        $params(token) -operation cancel
    }

    destroy $pw

    if {!$abort} {
        return $passphrase($keyid)
    }
}


proc ssj::armor:encode {text} {
    if {[set x [string first "\n\n" $text]] >= 0} {
        set text [string range $text [expr $x+2] end]
    }
    if {[set x [string first "\n-----" $text]] > 0} {
        set text [string range $text 0 [expr $x-1]]
    }

    return $text
}

proc ssj::armor:decode {text} {
    return "-----BEGIN PGP MESSAGE-----\n\n$text\n-----END PGP MESSAGE-----"
}


proc ssj::signed:input {from signature data what} {
    variable ctx
    variable j2k
    variable s2e
    variable warnings
    variable gpg_error_id

    once_only

    if {[catch { $ctx -operation verify \
                      -input     [binary format a* [encoding convertto utf-8 $data]]  \
                      -signature [armor:decode $signature] } result]} {
        debugmsg ssj "verify processing error: $result ($from)"

        if {![info exists warnings(verify-traffic)]} {

            set warnings(verify-traffic) 1
            after idle [list NonmodalMessageDlg .verify_error -aspect 50000 -icon error \
                -message [format [::msgcat::mc "Error in signature verification software: %s."] $result]]
        }

        set params(reason) $result

        return [array get params]
    }

    debugmsg ssj "VERIFY: $from ($data); $result"

    array set params $result
    set result $params(status)

    set signatures {}
    foreach signature $params(signatures) {
        catch { unset sparams }
        array set sparams $signature

        if {[info exists sparams(key)]} {
            set sparams(key) [$ctx -operation info-key -key $sparams(key)]
            foreach {k v} $sparams(key) {
                if {![cequal $k subkeys]} {
                    continue
                }
                foreach subkey $v {
                    catch { unset kparams }
                    array set kparams $subkey
                    if {[info exists kparams(keyid)]} {
                        set j2k($from) $kparams(keyid)
                        break
                    }
                }
            }
        }

        lappend signatures [array get sparams]
    }
    catch { unset params }
    array set params [list signatures $signatures]

    if {![cequal $result good]} {
        if {[info exists s2e($result)]} {
            set result $s2e($result)
        }
        set params(reason) $result

        if {![info exists warnings(verify,$from)]} {
            set warnings(verify,$from) 1
            incr gpg_error_id
            after idle [list NonmodalMessageDlg .verify_error$gpg_error_id -aspect 50000 -icon error \
                -message [format [::msgcat::mc "%s purportedly signed by %s can't be verified.\n\n%s."] $what $from $result]]
        }
    }

    return [array get params]
}


proc ssj::signed:output {data args} {
    variable ctx
    variable options
    variable warnings
    variable gpg_error_id

    if {(!$options(sign-traffic)) || ([cequal $data ""])} {
        return
    }

    once_only 1

    if {[catch { $ctx -operation sign  \
                      -input     [binary format a* [encoding convertto utf-8 $data]] \
                      -mode      detach } result]} {
        set options(sign-traffic) 0

        debugmsg ssj "signature processing error: $result ($data)"

        if {[llength $args] == 0} {
            set buttons ok
            set cancel 0
            set message [format [::msgcat::mc "Unable to sign presence information: %s.\n\nPresence will be sent, but signing traffic is now disabled."] $result]
        } else {
            set buttons {ok cancel}
            set cancel 1
            set message [format [::msgcat::mc "Unable to sign message body: %s.\n\nSigning traffic is now disabled.\n\nSend it WITHOUT a signature?"] $result]
        }

        incr gpg_error_id
        if {[MessageDlg .sign_error$gpg_error_id -aspect 50000 -icon error -type user \
                        -buttons $buttons -default 0 -cancel $cancel \
                        -message $message]} {
            error ""
        }           

        return
    }
    set result [armor:encode $result]

    debugmsg ssj "SIGN: $data; $result"
    whichkeys sign

    return $result
}

proc ssj::signed:info {pinfo} {

    set text ""
    array set params $pinfo

    foreach {k v} $pinfo {
	if {![cequal $k signatures]} {
	    if {![cequal $v ""]} {
		append text [format "%s: %s\n" $k $v]
	    }
	}
    }

    foreach signature $params(signatures) {
	set info ""
	set addrs ""
	set s ""
	foreach {k v} $signature {
	    switch -- $k {
		key {
		    foreach {k v} $v {
			if {![cequal $k subkeys]} {
			    continue
			}
			foreach subkey $v {
			    catch { unset sparams }
			    array set sparams $subkey
			    if {[info exists sparams(email)]} {
				append addrs $s $sparams(email)
                                set s "\n     "
			    }
			}
		    }
		}

		created {
		    append info "created: [clock format $v]\n"
		}

		default {
		    if {![cequal $v ""]} {
			append info [format "%s: %s\n" $k $v]
		    }
		}
	    }
	}

	if {![cequal $addrs ""]} {
	    set info "email: $addrs\n$info"
	}
	if {![cequal $info ""]} {
	    append text "\n" [string trimright $info]
	}
    }

    return [string trimleft $text]
}

proc ssj::signed:Label {lb pinfo} {
    global messageicon

    array set params $pinfo
    if {[info exists params(reason)]} {
	set args [list -image $messageicon(badsigned)]
    } else {
	set args [list -image $messageicon(signed)]
    }

    if {![cequal [set info [signed:info $pinfo]] ""]} {
	lappend args -helptext $info -helptype balloon
    }

    return [eval [list Label $lb] $args -cursor arrow \
		-padx 0 -pady 0 -borderwidth 0 -highlightthickness 0]
}


proc ssj::encrypted:input {from data} {
    variable ctx
    variable warnings
    variable gpg_error_id

    once_only

    if {[catch { $ctx -operation decrypt \
                      -input     [armor:decode $data] } result]} {
        debugmsg ssj "decryption processing error: $result ($from)"

        if {![info exists warnings(decrypt,$from)]} {
            set warnings(decrypt,$from) 1
            incr gpg_error_id
            after idle [list NonmodalMessageDlg .decrypt_error$gpg_error_id -aspect 50000 -icon error \
                -message [format [::msgcat::mc "Data purported sent by %s can't be deciphered.\n\n%s."] $from $result]]
        }

        error $result
    }

    debugmsg ssj "DECRYPT: $from; $result"

    array set params $result
    binary scan $params(plaintext) a* temp_utf8
    return [encoding convertfrom utf-8 $temp_utf8]
}


proc ssj::encrypted:output {data to} {
    variable ctx
    variable e4me
    variable j2k
    variable options
    variable gpg_error_id

    if {[cequal $data ""]} {
        return
    }

    #if {[cequal [set jid [roster::find_jid $to]] ""]} {
    #    set jid $to
    #}
    set jid [node_and_server_from_jid $to]
    if {![encryptP $jid]} {
        return
    }

    if {[info exists j2k($to)]} {
        set name $j2k($to)
    } elseif {[llength [set k [array names j2k $to/*]]] > 0} {
        set name $j2k([lindex $k 0])
    } else {
        set name $jid
    }
    [set recipient [gpgme::recipient]] \
            -operation add   \
            -name      $name \
            -validity  full
    foreach signer $e4me {
        $recipient -operation add \
                   -name      $signer \
                   -validity  full
    }

    once_only 1

    set code [catch { $ctx -operation   encrypt \
                           -input       [binary format a* [encoding convertto utf-8 $data]]   \
                           -recipients $recipient } result]

    rename $recipient {}

    if {$code} {
        debugmsg ssj "encryption processing error: $result ($data)"

        set options(encrypt,$jid) 0
        incr gpg_error_id
        if {[MessageDlg .encrypt_error$gpg_error_id -aspect 50000 -icon error -type user \
                -buttons {ok cancel} -default 0 -cancel 1 \
                -message [format [::msgcat::mc "Unable to encipher data for %s: %s.\n\nEncrypting traffic to this user is now disabled.\n\nSend it as PLAINTEXT?"] $to $result]]} {
            error ""
        }

        return
    }
    set result [armor:encode $result]

    debugmsg ssj "ENCRYPT: $data; $result"

    return $result
}

proc ssj::whichkeys {what} {
    variable ctx
    variable warnings

    if {[cequal [set s [$ctx -operation get -property  last-op-info]] ""]} {
        return
    }

    set keys {}
    while {([set x [string first <fpr> $s]] > 0) \
                && ([set y [string first </fpr> $s]] > $x) \
                && ($x+45 == $y)} {
        lappend keys [string range $s [expr $x+20] [expr $y-1]]
        set s [string range $s $y end]
    }

    if {![info exists warnings($what)]} {
        set warnings($what) ""
    } elseif {[cequal $warnings($what) $keys]} {
        return
    }

    set warnings($what) $keys
    debugmsg ssj "${what}ing with $keys"
}


proc ssj::prefs {jid} {
    variable ctx
    variable options
    variable optionsX

    set w [win_id security_preferences $jid]

    if {[winfo exists $w]} {
        focus -force $w
        return
    }

    Dialog $w -title [format [::msgcat::mc "Change security preferences for %s"] $jid] -separator 1 \
        -anchor e -default 0 -cancel 1

    set f [$w getframe]

    if {![info exists options(encrypt,$jid)]} {
        set options(encrypt,$jid) [encryptP $jid]
    }

    set optionsX(encrypt,$jid) $options(encrypt,$jid)
    checkbutton $f.encrypt \
        -text     [::msgcat::mc "Encrypt traffic"] \
        -variable [namespace current]::optionsX(encrypt,$jid)

    pack $f.encrypt -side left -expand yes -fill x

    $w add -text [::msgcat::mc "OK"] -command [list [namespace current]::prefs_ok $w $jid]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]

    $w draw $f.name
}

proc ssj::prefs_ok {w jid} {
    variable options
    variable optionsX

    set options(encrypt,$jid) $optionsX(encrypt,$jid)

    destroy $w
}


proc ssj::signP {} {
    variable options

    return $options(sign-traffic)
}

proc ssj::encryptP {{jid ""}} {
    variable ctx
    variable j2k
    variable options

    if {[cequal $jid ""]} {
        return $options(encrypt-traffic)
    }

    lassign [roster::get_category_and_subtype \
                 [jlib::route $jid] $jid] \
            category subtype
    switch -- $category {
	conference
	    -
	service {
	    set resP 0
	}

	default {
	    set resP 1
	}
    }

    if {[info exists options(encrypt,$jid)]} {
        return $options(encrypt,$jid)
    }

    if {!$options(encrypt-traffic)} {
        return 0
    }

    if {[info exists options(encrypt-tried,$jid)]} {
        return $options(encrypt-tried,$jid)
    }

    once_only

    if {[info exists j2k($jid)]} {
        set name $j2k($jid)
    } elseif {($resP) && ([llength [set k [array names j2k $jid/*]]] > 0)} {
        set name $j2k([lindex $k 0])
    } else {
        set name $jid
    }

    [set recipient [gpgme::recipient]] \
            -operation add   \
            -name      $name \
            -validity  full

    if {[catch { $ctx -operation  encrypt        \
                      -input      "Hello world." \
                      -recipients $recipient }]} {
        set options(encrypt-tried,$jid) 0
    } else {
        set options(encrypt-tried,$jid) 1
    }

    rename $recipient {}

    return $options(encrypt-tried,$jid)
}


proc ssj::e4meP {keys} {
    variable ctx
    variable e4me
    variable signers

    $ctx -operation set     \
         -property  signers \
         -value     [set signers $keys]

    set e4me {}
    foreach signer $signers {
        [set recipient [gpgme::recipient]] \
                -operation add     \
                -name      $signer \
                -validity  full

        if {![catch { $ctx -operation  encrypt       \
                          -input      "Hello world." \
                          -recipients $recipient } result]} {
            lappend e4me $signer
        }

        rename $recipient {}
    }
}

proc ssj::sign:toggleP {} {
    variable options

    set options(sign-traffic) [expr ($options(sign-traffic)+1)%2]
}

proc ssj::encrypt:toggleP {{jid ""}} {
    variable options

    if {[cequal $jid ""]} {
        set options(encrypt-traffic) [expr ($options(encrypt-traffic)+1)%2]
        return
    }

    if {![info exists options(encrypt,$jid)]} {
        set options(encrypt,$jid) [encryptP $jid]
    }
    set options(encrypt,$jid) [expr ($options(encrypt,$jid)+1)%2]
}


proc ssj::signed:trace {script} {
    variable options
    variable trace

    if {![info exists trace(sign-traffic)]} {
        set trace(sign-traffic) {}

        ::trace variable options(sign-traffic) w \
                [namespace current]::trace
    }

    lappend trace(sign-traffic) $script
}

proc ssj::encrypted:trace {script {jid ""}} {
    variable options
    variable trace

    if {[cequal $jid ""]} {
        set k encrypt-traffic
    } else {
        set k encrypt,$jid 
    }
    if {![info exists trace($k)]} {
        set trace($k) {}

        ::trace variable [namespace current]::options($k) w \
                [namespace current]::trace
    }

    lappend trace($k) $script
}

proc ssj::trace {name1 name2 op} {
    variable trace

    set new {}
    foreach script $trace($name2) {
        if {[catch { eval $script } result]} {
            debugmsg ssj "$result -- $script"
        } else {
            lappend new $script
        }
    }
    set trace($name2) $new
}
