#  jlibsasl.tcl --
#  
#      This file is part of the jabberlib. It provides support for the
#      SASL authentication layer via the tclsasl or tcllib SASL package.
#      
#  Copyright (c) 2005 Sergei Golovan <sgolovan@nes.ru>
#  Based on jlibsasl by Mats Bengtson
#  
# $Id: jlibsasl.tcl

if {![catch { package require sasl 1.0 }]} {
    set [namespace current]::jlib::saslpack tclsasl
} elseif {![catch { package require SASL 1.0 }]} {
    set [namespace current]::jlib::saslpack tcllib
} else {
    return -code error "No SASL package found"
}

package require base64

package provide jlibsasl 1.0


proc jlib::sasl_init {} {
    variable saslpack

    switch -- $saslpack {
	tclsasl {
	    sasl::client_init -callbacks [list [list log ::LOG]]
	}
	default {
	    # empty
	}
    }

    # SASL error messages
    set ::error_type(sasl) [::msgcat::mc "Authentication Error"]

    foreach {lcode type cond description} [list \
	401 sasl aborted		[::msgcat::mc "Aborted"] \
	401 sasl incorrect-encoding	[::msgcat::mc "Incorrect encoding"] \
	401 sasl invalid-authzid	[::msgcat::mc "Invalid authzid"] \
	401 sasl invalid-mechanism	[::msgcat::mc "Invalid mechanism"] \
	401 sasl mechanism-too-weak	[::msgcat::mc "Mechanism too weak"] \
	401 sasl not-authorized		[::msgcat::mc "Not Authorized"] \
	401 sasl temporary-auth-failure	[::msgcat::mc "Temporary auth failure"]] \
    {
	lappend ::defined_error_conditions $cond
	set ::error_description($type,$::NS(stanzas),$cond) $description
	set ::legacy_error_codes($cond) $lcode
    }
}

proc jlib::sasl_new {connid args} {
    variable NS
    variable sasl

    ::LOG "(jlib::sasl_new $connid)"

    if {![info exists sasl(init)]} {
	sasl_init
	set sasl(init) 1
    }

    catch {unset sasl($connid,mechanisms)}

    jlib::register_xmlns $connid $NS(sasl) \
	[list [namespace current]::sasl_parse $connid]
}

proc jlib::sasl_free {connid} {
    variable NS
    variable saslpack
    variable sasl

    jlib::unregister_xmlns $connid $NS(sasl)

    if {[info exists sasl($connid,token)]} {
	switch -- $saslpack {
	    tclsasl {
		rename $sasl($connid,token) ""
	    }
	    tcllib {
		SASL::cleanup $sasl($connid,token)
	    }
	}
    }

    catch {array unset sasl $connid,*}
}

proc jlib::sasl_parse {connid xmldata} {
    variable sasl

    jlib::wrapper:splitxml $xmldata tag vars isempty cdata children

    switch -- $tag {
	mechanisms {
	    set mechanisms {}
	    foreach child $children {
		wrapper:splitxml $child tag1 vars1 isempty1 cdata1 children1
		if {$tag1 == "mechanism"} {
		    lappend mechanisms $cdata1
		}
	    }
	    set sasl($connid,mechanisms) $mechanisms
	}
	challenge {
	    sasl_step $connid $cdata
	}
	success {
	    sasl_success $connid
	}
	failure {
	    sasl_failure $connid $children
	}
    }
}


proc jlib::sasl_auth {connid username resource password allow_plain cmd} {
    variable saslpack
    variable sasl
    variable connjid

    ::LOG "(jlib::sasl_auth $connid) start"

    set sasl($connid,username)    $username
    set sasl($connid,server)      $connjid($connid,server)
    set sasl($connid,resource)    $resource
    set sasl($connid,password)    $password
    set sasl($connid,allow_plain) $allow_plain
    set sasl($connid,cmd)         $cmd

    switch -- $saslpack {
	tclsasl {
	    foreach id {authname pass getrealm cnonce} {
		lappend callbacks \
		    [list $id [list [namespace current]::tclsasl_callback $connid]]
	    }

	    set sasl($connid,token) \
		[sasl::client_new -service     xmpp \
				  -serverFQDN  $sasl($connid,server) \
				  -callbacks   $callbacks \
				  -flags       success_data]

	    if {$sasl($connid,allow_plain)} {
		set flags {}
	    } else {
		set flags {noplaintext}
	    }

	    $sasl($connid,token) -operation setprop \
				 -property sec_props \
				 -value [list min_ssf 0 \
					      max_ssf 0 \
					      flags $flags]
	}
	tcllib {
	    set sasl($connid,token) \
		[SASL::new -service xmpp \
			   -type client \
			   -server $sasl($connid,server) \
			   -callback [list [namespace current]::tcllib_callback $connid]]
	}
    }

    if {[info exists sasl($connid,mechanisms)]} {
	sasl_auth_continue $connid
    } else {
	# Must be careful so this is not triggered by a reset or something...
	trace variable [namespace current]::sasl($connid,mechanisms) w \
	      [list [namespace current]::sasl_auth_mechanisms_write $connid]
    }
}

proc jlib::sasl_auth_mechanisms_write {connid name1 name2 op} {
    
    ::LOG "(jlib::sasl_auth_mechanisms_write $connid)"
    
    trace vdelete [namespace current]::sasl($connid,mechanisms) w \
	  [list [namespace current]::sasl_auth_mechanisms_write $connid]
    sasl_auth_continue $connid
}

proc jlib::sasl_auth_continue {connid} {
    variable NS
    variable saslpack
    variable sasl

    ::LOG "(jlib::sasl_auth $connid) mechs: $sasl($connid,mechanisms)"

    switch -- $saslpack {
	tclsasl {
	    set code [catch {
		$sasl($connid,token) \
		    -operation start \
		    -mechanisms $sasl($connid,mechanisms) \
		    -interact [list [namespace current]::sasl_interact $connid]
	    } result]
	}
	tcllib {
	    set code [catch { sasl_choose_mech $connid } result]

	    if {!$code} {
		SASL::configure $sasl($connid,token) -mech $result
		set result [list mechanism $result output ""]
	    }
	}
    }

    ::LOG "(jlib::sasl_auth $connid) SASL code $code: $result"
    switch -- $code {
	0 - 
	4 {
	    array set resarray $result
	    set data [wrapper:createtag auth \
			  -vars   [list xmlns $NS(sasl) \
					mechanism $resarray(mechanism)] \
			  -chdata [base64::encode $resarray(output)]]

	    outmsg [wrapper:createxml $data] -connection $connid
	}
	default {
	    sasl_finish $connid ERR \
			[concat sasl \
				[error sasl undefined-condition \
				       -text [format "SASL auth error: %s" \
						     $result]]]
	}
    }
}

proc jlib::sasl_choose_mech {connid} {
    variable sasl

    if {$sasl($connid,allow_plain)} {
	set forbidden_mechs {}
    } else {
	set forbidden_mechs {PLAIN LOGIN}
    }

    foreach m [SASL::mechanisms] {
	if {[lsearch -exact $sasl($connid,mechanisms) $m] >= 0 && \
		[lsearch -exact $forbidden_mechs $m] < 0} {
	    return $m
	}
    }
    return -code error "no mechanism available"
}

proc jlib::sasl_step {connid serverin64} {
    variable NS
    variable saslpack
    variable sasl

    set serverin [base64::decode $serverin64]

    switch -- $saslpack {
	tclsasl {
	    set code [catch {
		$sasl($connid,token) \
		    -operation  step \
		    -input      $serverin \
		    -interact   [list [namespace current]::sasl_interact $connid]
	    } result]
	}
	tcllib {
	    set code [catch {SASL::step $sasl($connid,token) $serverin} result]

	    if {!$code} {
		set result [SASL::response $sasl($connid,token)]
	    }
	}
    }

    ::LOG "(jlib::sasl_step) SASL code $code: $result"
    switch -- $code {
	0 -
	4 {
	    set data [jlib::wrapper:createtag response \
			  -vars   [list xmlns $NS(sasl)] \
			  -chdata [base64::encode $result]]

	    outmsg [jlib::wrapper:createxml $data] -connection $connid
	}
	default {
	    sasl_finish $connid ERR \
			[concat sasl \
				[error sasl undefined-condition \
				       -text [format "SASL step error: %s" \
						     $result]]]
	}
    }
}

# jlib::tclsasl_callback --
# 
#       TclSASL's callback id's seem to be a bit mixed up.

proc jlib::tclsasl_callback {connid data} {
    variable sasl

    ::LOG "(jlib::tclsasl_callback $connid) $data"

    array set params $data

    switch -- $params(id) {
        authname {
	    #username
            return [encoding convertto utf-8 $sasl($connid,username)]
        }
        user {
	    # authzid
            return [encoding convertto utf-8 \
			$sasl($connid,username)@$sasl($connid,server)]
        }
        pass {
            return [encoding convertto utf-8 $sasl($connid,password)]
        }
        getrealm {
	    return [encoding convertto utf-8 $sasl($connid,server)]
        }
	default {
	    return -code error \
		"SASL callback error: client needs to write $params(id)"
	}
    }
}

proc jlib::tcllib_callback {connid token command args} {
    variable sasl

    ::LOG "(jlib::tcllib_callback $connid) $token $command"

    switch -exact -- $command {
	login {
	    # authzid
	    return [encoding convertto utf-8 \
			$sasl($connid,username)@$sasl($connid,server)]
	}
	username {
	    return [encoding convertto utf-8 $sasl($connid,username)]
	}
	password {
	    return [encoding convertto utf-8 $sasl($connid,password)]
	}
	realm {
	    return [encoding convertto utf-8 $sasl($connid,server)]
	}
	hostname {
	    return [info host]
	}
	default {
	    return -code error \
		"SASL callback error: client needs to write $command"
	}
    }
}

proc jlib::sasl_interact {connid data} {
    # empty
}


proc jlib::sasl_failure {connid children} {
    variable sasl

    ::LOG "(jlib::sasl_failure $connid)"

    set error [lindex $children 0]
    if {$error == ""} {
	set err not-authorized
    } else {
	wrapper:splitxml $error tag vars empty cdata children
	set err $tag
    }
    sasl_finish $connid ERR [concat sasl [error sasl $err]]
}

proc jlib::sasl_success {connid} {
    variable lib

    ::LOG "(jlib::sasl_success $connid)"
    
    # xmpp-core sect 6.2:
    # Upon receiving the <success/> element,
    # the initiating entity MUST initiate a new stream by sending an
    # opening XML stream header to the receiving entity (it is not
    # necessary to send a closing </stream> tag first...
    
    reset $connid
    
    outmsg [wrapper:streamheader [connection_server $connid] \
		-xml:lang [get_lang] -version "1.0"] \
	   -connection $connid
	
    # Must be careful so this is not triggered by a reset or something...
    trace variable [namespace current]::lib($connid,features) w \
	  [list [namespace current]::sasl_auth_features_write $connid]
    
    return {}
}

proc jlib::sasl_auth_features_write {connid name1 name2 op} {
    variable NS
    variable sasl

    trace vdelete [namespace current]::lib($connid,features) w \
	  [list [namespace current]::sasl_auth_features_write $connid]

    set data [wrapper:createtag bind \
		  -vars [list xmlns $NS(bind)] \
		  -subtags [list [wrapper:createtag resource \
				      -chdata $sasl($connid,resource)]]]

    send_iq set $data \
	    -command [list [namespace current]::resource_bind_cb $connid] \
	    -connection $connid
    
}

proc jlib::resource_bind_cb {connid res child} {
    variable NS
    variable sasl

    switch -- $res {
	ERR {
	    ::LOG "error (jlib::resource_bind_cb) $data"
	    sasl_finish $connid $res $child
	}
	default {
	    # Establish the session.
	    set data [wrapper:createtag session \
			  -vars [list xmlns $NS(session)]]

	    send_iq set $data \
		-command [list [namespace current]::send_session_cb $connid] \
		-connection $connid
	}
    }
}

proc jlib::send_session_cb {connid res child} {
    sasl_finish $connid $res $child
}

proc jlib::sasl_finish {connid res child} {
    variable sasl

    ::LOG "(jlib::sasl_finish $connid)"

    if {$res != "OK"} {
	client:status [::msgcat::mc "Authentication failed"]
    } else {
	client:status [::msgcat::mc "Authentication successful"]
    }
    uplevel #0 $sasl($connid,cmd) [list $res $child]
}

