# $Id: muc.tcl,v 1.67 2005/08/06 22:13:32 aleksey Exp $

# Multi-User Chat support (JEP-0045)

namespace eval muc {
    set winid 0
    custom::defvar options(gen_events) 1 \
	[::msgcat::mc "Generate event messages in MUC compatible conference rooms."] \
	-type boolean -group Chat
    custom::defvar options(history_maxchars) 10000 \
	[::msgcat::mc "Maximum number of characters in the history in MUC compatible conference rooms."] \
	-type integer -group Chat
    custom::defvar options(history_maxstanzas) 20 \
	[::msgcat::mc "Maximum number of stanzas in the history in MUC compatible conference rooms."] \
	-type integer -group Chat
    custom::defvar options(request_only_unseen_history) 0 \
	[::msgcat::mc "Request only unseen (which aren't displayed in the chat window) messages in the history in MUC compatible conference rooms."] \
	-type boolean -group Chat
    custom::defvar options(report_muc_rooms) 0 \
	[::msgcat::mc "Report the list of current MUC rooms on disco#items query."] \
	-type boolean -group IQ
}


set ::NS(muc)       http://jabber.org/protocol/muc
set ::NS(muc#admin) http://jabber.org/protocol/muc#admin
set ::NS(muc#owner) http://jabber.org/protocol/muc#owner
set ::NS(muc#user)  http://jabber.org/protocol/muc#user


proc muc::add_groupchat_user_menu_items {m connid jid} {
    set group [node_and_server_from_jid $jid]

    if {![is_compatible $group]} return

    set mm [menu $m.muc -tearoff 0]
    $mm add command -label [::msgcat::mc "Whois"] \
	-command [list muc::whois $connid $jid]
    $mm add command -label [::msgcat::mc "Kick"] \
	-command [list muc::change_item_param admin {role none} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Ban"] \
	-command [list muc::change_item_param admin {affiliation outcast} \
		      $connid $jid ""]
    $mm add command -label [::msgcat::mc "Grant Voice"] \
	-command [list muc::change_item_param admin \
		      {role participant} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Revoke Voice"] \
	-command [list muc::change_item_param admin \
		      {role visitor} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Grant Membership"] \
	-command [list muc::change_item_param admin \
		      {affiliation member} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Revoke Membership"] \
	-command [list muc::change_item_param admin \
		      {affiliation none} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Grant Moderator Privileges"] \
	-command [list muc::change_item_param admin \
		      {role moderator} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Revoke Moderator Privileges"] \
	-command [list muc::change_item_param admin \
		      {role participant} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Grant Administrative Privileges"] \
	-command [list muc::change_item_param owner \
		      {affiliation admin} $connid $jid ""]
    $mm add command -label [::msgcat::mc "Revoke Administrative Privileges"] \
	-command [list muc::change_item_param admin \
		      {affiliation member} $connid $jid ""]
    #$mm add command -label [::msgcat::mc "Grant Ownership Privileges"] \
    #    -command [list muc::change_item_param owner \
    #    	      {affiliation owner} $connid $jid ""]
    #$mm add command -label [::msgcat::mc "Revoke Ownership Privileges"] \
    #    -command [list muc::change_item_param admin \
    #		      {affiliation admin} $connid $jid ""]

    $m add cascade -label [::msgcat::mc "MUC"] -menu $mm
}

hook::add roster_create_groupchat_user_menu_hook \
    muc::add_groupchat_user_menu_items


proc muc::create_conference_menu_items {m chatid} {
    set group [chat::get_jid $chatid]

    trace variable ::muc::muc_compatible($group) w \
	[list muc::add_conference_menu_items $m $chatid]
}
    
hook::add chat_create_conference_menu_hook muc::create_conference_menu_items


proc muc::add_conference_menu_items {m chatid args} {
    set group [chat::get_jid $chatid]

    if {![is_compatible $group] || ![winfo exists $m]} return

    trace vdelete ::muc::muc_compatible($group) w \
	[list muc::add_conference_menu_items $m $chatid]

    $m add command -label [::msgcat::mc "Configure room..."] \
	-command [list muc::request_config query $chatid]
    $m add command -label [::msgcat::mc "Edit voice list..."] \
	-command [list muc::request_list admin role participant $chatid]
    $m add command -label [::msgcat::mc "Edit ban list..."] \
	-command [list muc::request_list admin affiliation outcast $chatid]
    $m add command -label [::msgcat::mc "Edit member list..."] \
	-command [list muc::request_list admin affiliation member $chatid]
    $m add command -label [::msgcat::mc "Edit moderator list..."] \
	-command [list muc::request_list admin role moderator $chatid]
    $m add command -label [::msgcat::mc "Edit admin list..."] \
	-command [list muc::request_list owner affiliation admin $chatid]
    $m add command -label [::msgcat::mc "Edit owner list..."] \
	-command [list muc::request_list owner affiliation owner $chatid]
    $m add command -label [::msgcat::mc "Destroy room"] \
	-command [list muc::request_destruction_dialog $chatid "" ""]
}


proc muc::handle_commands {chatid user body type} {
    if {$type != "groupchat"} return

    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]
    if {[cequal [crange $body 0 5] "/kick "]} {
	set level admin
	set params {role none}
	set nick_reason [string trim [crange $body 6 end]]
	#set we [string wordend $nick_reason 0]
	set we [string first " " $nick_reason]
	if {$we < 0} {
	    set we end
	}
	set nick   [string trim [crange $nick_reason 0 $we]]
	if {[cequal $we end]} {
	    set reason ""
	} else {
	    set reason [string trim [crange $nick_reason $we end]]
	}
    } elseif {[cequal [crange $body 0 4] "/ban "]} {
	set level admin
	set params {affiliation outcast}
	set nick_reason [string trim [crange $body 5 end]]
	#set we [string wordend $nick_reason 0]
	set we [string first " " $nick_reason]
	if {$we < 0} {
	    set we end
	}
	set nick   [string trim [crange $nick_reason 0 $we]]
	if {[cequal $we end]} {
	    set reason ""
	} else {
	    set reason [string trim [crange $nick_reason $we end]]
	}
    } elseif {[cequal [crange $body 0 6] "/whois "]} {
	set nick [string trim [crange $body 7 end]]
	whois $connid $group/$nick
	return stop
    } elseif {[cequal [crange $body 0 6] "/voice "]} {
	set level admin
	set params {role participant}
	set nick [string trim [crange $body 7 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 8] "/devoice "]} {
	set level admin
	set params {role visitor}
	set nick [string trim [crange $body 9 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 7] "/member "]} {
	set level admin
	set params {affiliation member}
	set nick [string trim [crange $body 8 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 9] "/demember "]} {
	set level admin
	set params {affiliation none}
	set nick [string trim [crange $body 10 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 10] "/moderator "]} {
	set level admin
	set params {role moderator}
	set nick [string trim [crange $body 11 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 12] "/demoderator "]} {
	set level admin
	set params {role participant}
	set nick [string trim [crange $body 13 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 6] "/admin "]} {
	set level admin
	set params {affiliation admin}
	set nick [string trim [crange $body 7 end]]
	set reason ""
    } elseif {[cequal [crange $body 0 8] "/deadmin "]} {
	set level admin
	set params {affiliation member}
	set nick [string trim [crange $body 9 end]]
	set reason ""
    } else {
	return
    }
    
    change_item_param $level $params $connid $group/$nick $reason

    return stop
}

hook::add chat_send_message_hook muc::handle_commands 50


proc muc::commands_comps {chatid compsvar wordstart line} {
    set group [chat::get_jid $chatid]
    if {![is_compatible $group]} return

    upvar 0 $compsvar comps

    if {!$wordstart} {
	lappend comps {/whois } {/kick } {/ban } \
	    {/voice } {/devoice } \
	    {/member } {/demember } \
	    {/moderator } {/demoderator } \
	    {/admin } {/deadmin }
    }
}

hook::add generate_completions_hook muc::commands_comps


proc muc::get_real_jid {connid jid} {
    variable users

    if {[info exists users(jid,$connid,$jid)] && \
	    $users(jid,$connid,$jid) != ""} {
	return $users(jid,$connid,$jid)
    } else {
	return ""
    } 
}

proc muc::whois {connid user} {
    set group [node_and_server_from_jid $user]
    set chatid [chat::chatid $connid $group]
    set nick [chat::get_nick $connid $user groupchat]

    set real_jid [get_real_jid $connid $user]

    if {$real_jid != ""} {
	chat::add_message $chatid $group info "whois $nick: $real_jid" {}
    } else {
	chat::add_message $chatid $group error "whois $nick: no info" {}
    }
}


proc muc::change_item_param {level params connid user reason} {
    set group [node_and_server_from_jid $user]
    set nick  [chat::get_nick $connid $user groupchat]

    set itemsubtags {}
    if {$reason != ""} {
	lappend itemsubtags [jlib::wrapper:createtag reason \
				 -chdata $reason]
    }
    set item [jlib::wrapper:createtag item \
		  -vars [concat [list nick $nick] $params] \
		  -subtags $itemsubtags]

    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars [list xmlns $::NS(muc#$level)] \
	     -subtags [list $item]] \
	-to $group \
	-connection $connid \
	-command [list muc::test_error_res "$params $nick" $connid $group]
}


proc muc::request_destruction_dialog {chatid alt reason} {
    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]

    set warning \
	[format [::msgcat::mc "Conference room %s will be destroyed permanently.\n\nProceed?"] \
		$group]

    set res [MessageDlg .muc_request_destruction -aspect 50000 -icon warning \
			-type user -buttons {yes no} -default 1 \
			-cancel 1 \
			-message $warning]
    if {!$res} {
	request_destruction $chatid $alt $reason
    }
}

proc muc::request_destruction {chatid alt reason} {
    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]

    jlib::send_iq set \
	[jlib::wrapper:createtag query \
	     -vars [list xmlns $::NS(muc#owner)] \
	     -subtags [list \
			   [jlib::wrapper:createtag destroy \
				-subtags [list \
					      [jlib::wrapper:createtag alt \
						   -chdata $alt] \
					      [jlib::wrapper:createtag reason \
						   -chdata $reason]]]]] \
	-to $group \
	-connection $connid \
	-command [list muc::test_error_res "destroy" $connid $group]
}


proc muc::request_list {level attr val chatid} {
    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]
    jlib::send_iq get \
	[jlib::wrapper:createtag query \
	     -vars [list xmlns $::NS(muc#$level)] \
	     -subtags [list [jlib::wrapper:createtag item \
				 -vars [list $attr $val]]]] \
	-to $group \
	-connection $connid \
	-command [list muc::receive_list $level $attr $val $chatid]
}


proc muc::receive_list {level attr val chatid res child} {
    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]
    if {![cequal $res OK]} {
	chat::add_message $chatid $group error "$attr $val list: [error_to_string $child]" {}
	return
    }

    jlib::wrapper:splitxml $child tag vars isempty chdata children

    #data::draw_window $children [list muc::send_list $level $role $group]

    variable winid

    set w .muc_list$winid
    incr winid

    if {[winfo exists $w]} {
	destroy $w
    }

    Dialog $w -title [::msgcat::mc [format "Edit %s list" $val]] \
        -modal none -separator 1 -anchor e -default 0 -cancel 1 \
        -parent .

    set wf [$w getframe]

    set sw [ScrolledWindow $wf.sw -scrollbar vertical]
    set sf [ScrollableFrame $w.fields -constrainedwidth yes]
    set f [$sf getframe]
    $sw setwidget $sf
    fill_list $sf $f $children $attr $val
    list_add_item $sf $f $attr $val
 
    $w add -text [::msgcat::mc "Send"] \
	-command [list muc::send_list $chatid $level $attr $val $w $f]
    $w add -text [::msgcat::mc "Cancel"] -command [list destroy $w]
    bind $w <Destroy> [list muc::list_cleanup $w $f]

    button $w.add -text [::msgcat::mc "Add"] \
	-command [list muc::list_add_item $sf $f $attr $val]
    pack $w.add -side bottom -anchor e -in $wf -padx 1m -pady 1m
    pack $sw -side top -expand yes -fill both

    bindscroll $f $sf

    set hf [frame $w.hf]
    pack $hf -side top
    set vf [frame $w.vf]
    pack $vf -side left

    update idletasks
    $hf configure -width [expr {[winfo reqwidth $f] + [winfo pixels $f 1c]}]

    set h [winfo reqheight $f]
    set sh [winfo screenheight $w]
    if {$h > $sh - 200} {
	set h [expr {$sh - 200}]
    }
    $vf configure -height $h

    $w draw
}


proc muc::fill_list {sf f items attr val} {
    global font
    variable listdata
    variable origlistdata

    grid columnconfigure $f 0 -weight 1
    grid columnconfigure $f 1 -weight 1
    grid columnconfigure $f 2 -weight 1
    grid columnconfigure $f 3 -weight 2

    label $f.lnick -text [::msgcat::mc "Nick"]
    grid $f.lnick -row 0 -column 0 -sticky we -padx 1m
    bindscroll $f.lnick $sf
    label $f.ljid -text JID
    grid $f.ljid -row 0 -column 1 -sticky we -padx 1m
    bindscroll $f.ljid $sf
    switch -- $attr {
	role {
	    label $f.lrole -text [::msgcat::mc "Role"]
	    grid $f.lrole -row 0 -column 2 -sticky we -padx 1m
	    bindscroll $f.lrole $sf
	}
	affiliation {
	    label $f.laffiliation -text [::msgcat::mc "Affiliation"]
	    grid $f.laffiliation -row 0 -column 2 -sticky we -padx 1m
	    bindscroll $f.laffiliation $sf
	}
    }
    label $f.lreason -text [::msgcat::mc "Reason"]
    grid $f.lreason -row 0 -column 3 -sticky we -padx 1m
    bindscroll $f.lreason $sf

    set row 1

    set items2 {}
    foreach item $items {
	jlib::wrapper:splitxml $item tag vars isempty chdata children
	switch -- $tag {
	    item {
		set nick [jlib::wrapper:getattr $vars nick]
		set jid [jlib::wrapper:getattr $vars jid]
		set role [jlib::wrapper:getattr $vars role]
		set affiliation [jlib::wrapper:getattr $vars affiliation]
		lappend items2 [list $nick $jid $role $affiliation]
	    }
	}
    }

    foreach item [lsort -dictionary -index 1 $items2] {
	lassign $item nick jid role affiliation

	label $f.nick$row -text $nick -font $font \
	    -textvariable muc::listdata($f,nick,$row)
	grid $f.nick$row -row $row -column 0 -sticky w -padx 1m
	bindscroll $f.nick$row $sf

	label $f.jid$row -text $jid -font $font \
	    -textvariable muc::listdata($f,jid,$row)
	grid $f.jid$row -row $row -column 1 -sticky w -padx 1m
	bindscroll $f.jid$row $sf

	switch -- $attr {
	    role {
		ComboBox $f.role$row -text $role \
		    -values {moderator participant visitor none} \
		    -editable no \
		    -width 11 \
		    -textvariable muc::listdata($f,role,$row)
		grid $f.role$row -row $row -column 2 -sticky we -padx 1m
		bindscroll $f.role$row $sf
	    }
	    affiliation {
		ComboBox $f.affiliation$row -text $affiliation \
		    -values {owner admin member none outcast} \
		    -editable no \
		    -width 7 \
		    -textvariable muc::listdata($f,affiliation,$row)
		grid $f.affiliation$row -row $row -column 2 -sticky we -padx 1m
		bindscroll $f.affiliation$row $sf
	    }
	}

	entry $f.reason$row -font $font \
	    -textvariable muc::listdata($f,reason,$row)
	grid $f.reason$row -row $row -column 3 -sticky we -padx 1m
	bindscroll $f.reason$row $sf
	
	incr row
    }

    set listdata($f,rows) [incr row -1]
    array set origlistdata [array get listdata ${f}*]
}


proc muc::list_add_item {sf f attr val} {
    global font
    variable listdata

    set row [incr listdata($f,rows)]

    entry $f.nick$row -font $font -textvariable muc::listdata($f,nick,$row)
    grid $f.nick$row -row $row -column 0 -sticky we -padx 1m
    bindscroll $f.nick$row $sf
    
    entry $f.jid$row -font $font -textvariable muc::listdata($f,jid,$row)
    grid $f.jid$row -row $row -column 1 -sticky we -padx 1m
    bindscroll $f.jid$row $sf

    switch -- $attr {
	role {
	    ComboBox $f.role$row -text none \
		-values {moderator participant visitor none} \
		-editable no \
		-width 11 \
		-textvariable muc::listdata($f,role,$row)
	    grid $f.role$row -row $row -column 2 -sticky we -padx 1m
	    bindscroll $f.role$row $sf
	}
	affiliation {
	    ComboBox $f.affiliation$row -text none \
		-values {owner admin member none outcast} \
		-editable no \
		-width 7 \
		-textvariable muc::listdata($f,affiliation,$row)
	    grid $f.affiliation$row -row $row -column 2 -sticky we -padx 1m
	    bindscroll $f.affiliation$row $sf
	}
    }

    entry $f.reason$row -font $font -textvariable muc::listdata($f,reason,$row)
    grid $f.reason$row -row $row -column 3 -sticky we -padx 1m
    bindscroll $f.reason$row $sf

    set listdata($f,$attr,$row) $val
}


proc muc::send_list {chatid level attr val w f} {
    variable origlistdata
    variable listdata

    set items {}

    for {set i 1} {$i <= $origlistdata($f,rows)} {incr i} {
	set vars {}
	if {$listdata($f,$attr,$i) != $origlistdata($f,$attr,$i)} {
	    lappend vars $attr $listdata($f,$attr,$i)
	}

	if {$vars != {}} {
	    if {$origlistdata($f,nick,$i) != ""} {
		lappend vars nick $origlistdata($f,nick,$i)
	    }
	    if {$origlistdata($f,jid,$i) != ""} {
		lappend vars jid $origlistdata($f,jid,$i)
	    }
	    set itemsubtags {}
	    set reason $listdata($f,reason,$i)
	    if {$reason != ""} {
	        lappend itemsubtags [jlib::wrapper:createtag reason \
	        			 -chdata $reason]
	    }
	    lappend items [jlib::wrapper:createtag item \
			       -vars $vars \
			       -subtags $itemsubtags]
	}
    }

    for {} {$i <= $listdata($f,rows)} {incr i} {
	set vars1 {}
	set vars2 {}
	if {$listdata($f,$attr,$i) != ""} {
	    lappend vars1 $attr $listdata($f,$attr,$i)
	}
	if {$listdata($f,nick,$i) != ""} {
	    lappend vars2 nick $listdata($f,nick,$i)
	}
	if {$listdata($f,jid,$i) != ""} {
	    lappend vars2 jid $listdata($f,jid,$i)
	}

	if {$vars1 != {} && $vars2 != {}} {
	    set vars [concat $vars2 $vars1]
	    set itemsubtags {}
	    set reason $listdata($f,reason,$i)
	    if {$reason != ""} {
	        lappend itemsubtags [jlib::wrapper:createtag reason \
	        			 -chdata $reason]
	    }
	    lappend items [jlib::wrapper:createtag item \
			       -vars $vars \
			       -subtags $itemsubtags]
	}
    }

    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]

    if {$items != {}} {
	jlib::send_iq set [jlib::wrapper:createtag query \
			       -vars [list xmlns $::NS(muc#$level)] \
			       -subtags $items] \
	    -to $group \
	    -connection $connid \
	    -command [list muc::test_error_res "Sending list" $connid $group]
    }
    destroy $w
}


proc muc::list_cleanup {w f} {
    variable listdata
    variable origlistdata

    array unset listdata ${f}*
    array unset origlistdata ${f}*
}


proc muc::request_config {op chatid} {
    jlib::send_iq get \
	[jlib::wrapper:createtag $op \
	     -vars [list xmlns $::NS(muc#owner)]] \
	-to [chat::get_jid $chatid] \
	-connection [chat::get_connid $chatid] \
	-command [list muc::receive_config $op $chatid]
}


proc muc::receive_config {op chatid res child} {
    set group [chat::get_jid $chatid]
    if {![cequal $res OK]} {
	chat::add_message $chatid $group error "$op list: [error_to_string $child]" {}
	return
    }

    jlib::wrapper:splitxml $child tag vars isempty chdata children

    #if {$tag != $op} {
    #    chat::add_message $group $group error \
    #        "Ban list: receiving wrong reply ('$tag' tag instead of '$op')" {}
    #    return
    #}

    data::draw_window $children [list muc::send_config $op $chatid]
    return
}


proc muc::send_config {op chatid w restags} {
    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]
    jlib::send_iq set [jlib::wrapper:createtag $op \
			   -vars [list xmlns $::NS(muc#owner)] \
			   -subtags $restags] \
	-to $group \
	-connection $connid \
	-command [list muc::test_error_res "Sending $op list" $connid $group]
    destroy $w
}


proc muc::test_error_res {op connid group res child} {
    if {![cequal $res OK]} {
	set chatid [chat::chatid $connid $group]
	chat::add_message $chatid $group error "$op: [error_to_string $child]" {}
	return
    }
}

proc muc::process_presence_x {connid from type xs} {
    foreach x $xs {
	jlib::wrapper:splitxml $x tag vars isempty chdata children
	if {[jlib::wrapper:getattr $vars xmlns] == $::NS(muc#user)} {
	    process_muc_user $connid $from $type $children
	}
    }
}

hook::add presence_process_x_hook muc::process_presence_x

proc muc::process_muc_user {connid user type childrens} {
    variable users
    variable options

    foreach child $childrens {
	jlib::wrapper:splitxml $child tag vars isempty chdata children
	switch -- $tag {
	    item {
		if {$type != "unavailable"} {
		    set users(jid,$connid,$user) [jlib::wrapper:getattr $vars jid]
		    set users(role,$connid,$user) [jlib::wrapper:getattr $vars role]
		    set users(affiliation,$connid,$user) \
			[jlib::wrapper:getattr $vars affiliation]
		} else {
		    set new_nick [jlib::wrapper:getattr $vars nick]
		    foreach ch $children {
			jlib::wrapper:splitxml $ch tag1 vars1 isempty1 \
			    chdata1 children1
			switch -- $tag1 {
			    reason {
				set reason $chdata1
			    }
			    actor {
				set actor [jlib::wrapper:getattr $vars1 jid]
			    }
			}
		    }
		}
	    }
	    status {
		set code [jlib::wrapper:getattr $vars code]
		switch -- $code {
		    301 -
		    307 {
			set group [node_and_server_from_jid $user]
			set chatid [chat::chatid $connid $group]
			set nick [chat::get_nick $connid $user groupchat]
			switch -- $code {
			    301 {set action \
				     [format \
					  [::msgcat::mc \
					       "%s has been banned" \
					       $nick]]}
			    307 {set action \
				     [format \
					  [::msgcat::mc \
					       "%s has been kicked" \
					       $nick]]}
			}

			if {[info exists actor] && $actor != ""} {
			    append action \
				[format [::msgcat::mc " by %s" $actor]]
			}

			if {[info exists reason] && $reason != ""} {
			    append action ": $reason"
			}

			if {$options(gen_events)} {
			    variable ignore_unavailable $nick
			    chat::add_message \
				$chatid $group groupchat $action {}
			}
		    }
		    303 {
			set group [node_and_server_from_jid $user]
			set chatid [chat::chatid $connid $group]
			set nick [chat::get_nick $connid $user groupchat]
			if {[info exists new_nick] && $new_nick != ""} {
			    if {$nick == [get_our_groupchat_nick $chatid]} {
				set_our_groupchat_nick $chatid $new_nick
			    }
			    if {$options(gen_events)} {
				variable ignore_available   $new_nick
				variable ignore_unavailable $nick

				set real_jid [get_real_jid $connid $group/$nick]
				if {$real_jid != ""} {
				    set real_jid " ($real_jid)"
				}
				chat::add_message \
				    $chatid $group groupchat \
				    [format \
					 [::msgcat::mc "%s is now known as %s"] \
					 $nick$real_jid $new_nick] {}
			    }
			}
		    }
		}
	    }
	}
    }
}


proc muc::process_available {chatid nick} {
    variable options
    variable ignore_available
    variable pending_nicks

    set connid [chat::get_connid $chatid]
    set group [chat::get_jid $chatid]

    if {![is_compatible $group] && [is_changing_nick $chatid] && \
	    [string equal $pending_nicks($chatid) $nick]} {
	set_our_groupchat_nick $chatid $nick
	unset pending_nicks($chatid)
    }
    
    if {[is_compatible $group] && $options(gen_events) && \
	    (![info exists ignore_available] || \
		 $ignore_available != $nick)} {
	set real_jid [get_real_jid $connid $group/$nick]
	if {$real_jid != ""} {
	    set real_jid " ($real_jid)"
	}
	chat::add_message $chatid $group groupchat \
	    [format [::msgcat::mc "%s has become available"] \
	    $nick$real_jid] {}
    }
    catch { unset ignore_available }
}


proc muc::process_unavailable {chatid nick} {
    variable options
    variable ignore_unavailable

    set group [chat::get_jid $chatid]

    if {[is_compatible $group] && $options(gen_events) && \
	    (![info exists ignore_unavailable] || \
		 $ignore_unavailable != $nick)} {
	set connid [chat::get_connid $chatid]
	set status [get_jid_presence_info status $connid $group/$nick]
	if {$status != ""} {
	    set end ": $status"
	} else {
	    set end ""
	}
	chat::add_message $chatid $group groupchat \
	    [cconcat [format [::msgcat::mc "%s has left"] $nick] $end] {}
    }
    catch { unset ignore_unavailable }
}

###############################################################################

proc muc::change_nick {chatid nick} {
    set group [chat::get_jid $chatid]

    if {![is_compatible $group]} {
	variable pending_nicks
	set pending_nicks($chatid) $nick
    }

    jlib::send_presence -to $group/$nick \
	-connection [chat::get_connid $chatid] \
	-command [list muc::process_change_nick $chatid]
}


proc muc::process_change_nick {chatid connid from type x args} {
    if {![cequal $type error]} {
	return
    }

    if {![cequal [lindex [jlib::wrapper:getattr $args -error] 1] conflict]} {
	return
    }

    if {![is_compatible [chat::get_jid $chatid]] && \
	    [is_changing_nick $chatid]} {
	variable pending_nicks
	unset pending_nicks($chatid)
    }
    chat::add_message $chatid [node_and_server_from_jid $from] error \
	[format [::msgcat::mc "Error %s"] \
	     [jlib::wrapper:getattr $args -status]] {}
    return break
}


proc muc::is_changing_nick {chatid} {
    variable pending_nicks

    if {[info exists pending_nicks($chatid)]} {
	return 1
    } else {
	return 0
    }
}

###############################################################################

proc muc::request_negotiation {connid group} {
    variable muc_compatible

    set muc_compatible($group) 0

    disco::request_info [server_from_jid $group] "" \
	-connection $connid \
	-cache yes \
	-handler [list muc::recv_negotiation $group]
}


proc muc::recv_negotiation {group res identities features extras} {
    variable muc_compatible

    if {[cequal $res OK]} {
	foreach f $features {
	    set var [jlib::wrapper:getattr $f var]
	    if {$var == $::NS(muc)} {
		set muc_compatible($group) 1
		return
	    }
	}
    }
    set muc_compatible($group) 0
}


proc muc::is_compatible {group} {
    variable muc_compatible

    if {[info exists muc_compatible($group)]} {
	return $muc_compatible($group)
    } else {
	return 0
    }
}

###############################################################################

proc muc::add_user_popup_info {infovar connid user} {
    variable users
    upvar 0 $infovar info

    if {[info exists users(jid,$connid,$user)] && \
	    $users(jid,$connid,$user) != ""} {
	append info [format [::msgcat::mc "\n\tJID: %s"] $users(jid,$connid,$user)]
    }
    if {[info exists users(affiliation,$connid,$user)]} {
	append info [format [::msgcat::mc "\n\tAffiliation: %s"] \
	    $users(affiliation,$connid,$user)]
    }
}

hook::add roster_user_popup_info_hook muc::add_user_popup_info

###############################################################################

proc muc::join_group {connid group nick {password ""}} {
    global userstatus textstatus
    variable options
    variable timestamps
    variable muc_password

    set chatid [chat::chatid $connid $group]
    set_our_groupchat_nick $chatid $nick

    chat::open_window $chatid groupchat
    update idletasks

    request_negotiation $connid $group

    set x_subtags {}
    if {$password != ""} {
	lappend x_subtags [jlib::wrapper:createtag password -chdata $password]
    }
    set muc_password($chatid) $password

    set history_vars {}
    if {$options(history_maxchars) >= 0} {
	lappend history_vars maxchars $options(history_maxchars)
    }
    if {$options(history_maxstanzas) >= 0} {
	lappend history_vars maxstanzas $options(history_maxstanzas)
    }
    if {$options(request_only_unseen_history) && \
	    [info exists timestamps($chatid)]} {
	lappend history_vars seconds [expr {[clock seconds] - $timestamps($chatid) + 2}]
    }
    if {![lempty $history_vars]} {
	lappend x_subtags [jlib::wrapper:createtag history -vars $history_vars]
    }

    set command [list jlib::send_presence \
	-to ${group}/${nick} \
	-connection $connid \
	-xlist [list [jlib::wrapper:createtag x \
			  -vars [list xmlns $::NS(muc)] \
			  -subtags $x_subtags]]]
    switch -- $userstatus {
	available { }
	default {
	    lappend command -show $userstatus
	}
    }
    if {$textstatus != ""} {
	lappend command -stat $textstatus
    }
    eval $command
}

proc muc::set_message_timestamp {chatid from type body x} {
    variable timestamps
    
    set timestamps($chatid) [clock seconds]
}

hook::add draw_message_hook muc::set_message_timestamp 15

proc muc::clear_message_timestamp {chatid} {
    variable timestamps

    catch { unset timestamps($chatid) }
}

hook::add close_chat_post_hook muc::clear_message_timestamp

###############################################################################

proc muc::invitation {chatid jid reason} {
    global invite_muc

    set connid [chat::get_connid $chatid]
    set to [chat::get_jid $chatid]

    message::send_msg $to -type normal \
	-xlist [list [jlib::wrapper:createtag x \
	    -vars [list xmlns $::NS(muc#user)] \
	    -subtags [list [jlib::wrapper:createtag invite \
		-vars [list to $jid] \
		-subtags [list [jlib::wrapper:createtag reason \
		    -chdata $reason]]]]]] \
	-connection $connid
}


proc muc::join {jid args} {
    global gr_nick

    set category conference
    foreach {opt val} $args {
	switch -- $opt {
	    -category { set category $val }
	}
    }

    if {![cequal $category conference]} {
	return
    }

    if {![cequal [node_from_jid $jid] {}]} {
	::join_group $jid -nick [get_group_nick $jid $gr_nick]
    } else {
	::join_group_dialog -server [server_from_jid $jid] -group {}
    }
}

hook::add postload_hook \
    [list browser::register_ns_handler jabber:iq:conference muc::join \
	 -desc [list conference [::msgcat::mc "Join conference"]]]
hook::add postload_hook \
    [list disco::browser::register_feature_handler jabber:iq:conference muc::join \
	 -desc [list conference [::msgcat::mc "Join conference"]]]
hook::add postload_hook \
    [list browser::register_ns_handler $::NS(muc) muc::join \
	 -desc [list conference [::msgcat::mc "Join conference"]]]
hook::add postload_hook \
    [list disco::browser::register_feature_handler $::NS(muc) muc::join \
	 -desc [list conference [::msgcat::mc "Join conference"]]]
hook::add postload_hook \
    [list browser::register_ns_handler "gc-1.0" muc::join \
	 -desc [list conference [::msgcat::mc "Join groupchat"]]]
hook::add postload_hook \
    [list disco::browser::register_feature_handler "gc-1.0" muc::join \
	 -desc [list conference [::msgcat::mc "Join groupchat"]]]


proc muc::iq_reply {connid from child} {
    set res [jlib::wrapper:createtag query \
		     -vars [list xmlns $::NS(muc)]]
    return [list result $res]
}

iq::register_handler get query $::NS(muc) muc::iq_reply


proc muc::disco_reply {type connid from child} {
    variable options

    if {!$options(report_muc_rooms)} {
	return {error cancel not-allowed}
    }

    switch -- $type {
	info {
	    return {}
	}
	items {
	    set res {}
	    foreach chatid [array names chat::opened] {
		set connid1 [chat::get_connid $chatid]
		set group [chat::get_jid $chatid]
		if {[cequal $connid $connid1] &&
			[cequal $chat::chats(type,$chatid) groupchat] && \
			[is_compatible $group]} {
		    lappend res [jlib::wrapper:createtag item \
				     -vars [list jid $group]]
		}
	    }
	    return $res
	}
    }
}

hook::add postload_hook \
    {disco::register_node muc_rooms muc::disco_reply ""}

