namespace eval man {
    variable man

    # Define the man array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set man {
	list.reset ""
	list.clear ""
	entry.fstype ""
	entry.xtype ""
	fstype ""
	dialog ""
	viewer "xterm"
    }
}

# man::create --
#
#   Method to create the dialog box for the man command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::create { } {
    global tkWorld
    variable man

    # Put the focus on the man dialog if it is already open.
    if [winfo exists $man(dialog)] {
	switch -- [wm state $man(dialog)] {
	    normal {
		raise $man(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $man(dialog)
	    }
	}
	focus $man(dialog)
	return
    } else {
	set man(dialog) [dialog::create .man Man]
    }

    # Now define the viewers a user can use for the man page.
    if [info exists man(viewer_list)] {
	# Do Nothing
    } else {
	foreach v [system::which "xterm nxterm rxvt"] {
	    lappend man(viewer_list) [file tail $v]
	}
    }

    # There is only 1 tab.
    set tab1 [tabnotebook::page [dialog::interior \
	    $man(dialog)] "Options"]

    # Use a frame to encapsulate the file selection so that the
    # frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.label_manpage \
	    -text "Man Pages" \
	    -width 10 \
	    -command man::select
    set man(entry.manpage) [entry $f1.entry_manpage \
	    -width 35 \
	    -textvariable man::man(manpage)]
    label $f1.label_viewer \
	    -text "Viewer" \
	    -width 10 \
	    -anchor e
    set man(entry.viewer) [ddlistbox::create $f1 man::man(viewer) \
	    $man(viewer_list)]
    grid $f1.label_manpage $f1.entry_manpage \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f1.label_viewer $man(entry.viewer) \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.all_pages \
	    -text "All Pages" \
	    -variable man::man(all_pages) \
	    -onvalue "a" \
	    -offvalue ""
    checkbutton $f2.reformat \
	    -text "Reformat" \
	    -variable man::man(reformat) \
	    -onvalue "c" \
	    -offvalue ""
    checkbutton $f2.debug \
	    -text "debug" \
	    -variable man::man(debug) \
	    -onvalue "d" \
	    -offvalue ""
    checkbutton $f2.display_debug \
	    -text "Display and Debug" \
	    -variable man::man(display_debug) \
	    -onvalue "D" \
	    -offvalue ""
    checkbutton $f2.whatis \
	    -text "Whatis" \
	    -variable man::man(whatis) \
	    -onvalue "f" \
	    -offvalue ""
    checkbutton $f2.help \
	    -text "Help" \
	    -variable man::man(help) \
	    -onvalue "h" \
	    -offvalue ""
    checkbutton $f2.apropos \
	    -text "Apropos" \
	    -variable man::man(apropos) \
	    -onvalue "k" \
	    -offvalue ""
    checkbutton $f2.long_search \
	    -text "Extended Search" \
	    -variable man::man(long_search) \
	    -onvalue "K" \
	    -offvalue ""
    checkbutton $f2.postscript \
	    -text "Postscript" \
	    -variable man::man(postscript) \
	    -onvalue "t" \
	    -offvalue ""
    checkbutton $f2.path \
	    -text "Path" \
	    -variable man::man(path_flag) \
	    -onvalue "w" \
	    -offvalue ""
    checkbutton $f2.single_path \
	    -text "Single Line Path" \
	    -variable man::man(single_path) \
	    -onvalue "W" \
	    -offvalue ""
    grid $f2.all_pages $f2.reformat $f2.debug \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.display_debug $f2.whatis $f2.help \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.apropos $f2.long_search $f2.postscript \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.path $f2.single_path \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build the only tab.
    pack $f1 $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Manhhhhhh......
    focus $man(entry.manpage)

    # Define the lists for the reset and clear methods
    set man(list.reset) "all_pages reformat debug display_debug \
	    whatis help apropos long_search postscript path_flag \
	    single_path"
    set man(list.clear) "manpage viewer"
    set man(list.nonview) "debug whatis help apropos postscript \
	    path_flag single_path"
    set man(list.view) "all_pages reformat display_debug long_search"
}

# man::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::ok { } {
    global tkWorld
    variable man

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.
    
    # Check to see if any of the nonview checkbuttons are set.
    set nonview 0
    foreach x $man(list.nonview) {
	if [string length $man($x)] {
	    set nonview 1
	    break
	}
    }

    # Insert the right portion of the command based on whether
    # we need a viewer or not.
    if $nonview {
	$tkWorld(cmd_center) insert insert "man "
    } else {
	$tkWorld(cmd_center) insert insert "$man(viewer) -e man "
    }
    unset nonview

    # Build the command line argument.
    set cmd_arg ""
    foreach x $man(list.reset) {
	if [string length $man($x)] {
	    append cmd_arg $man($x)
	}
    }
    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert "-$cmd_arg "
    }

    # Insert the users selected man pages.
    if [string length $man(manpage)] {
	$tkWorld(cmd_center) insert insert "$man(manpage) "
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# man::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::reset { } {
    variable man

    # Reset all of the list elements.
    foreach x $man(list.reset) {
	set man($x) ""
    }
}

# man::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::clear { } {
    variable man

    # Reset the data structure elements and bg/fg.
    foreach x $man(list.clear) {
	set man($x) ""
	if {$x == "viewer"} {
	    set man($x) "xterm"
	} else {
	    $man(entry.$x) configure -bg #ffffff -fg #000000
	}
    }

    focus $man(entry.manpage)
}

# man::help --
#
#   Method to invoke the Man Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::help { } {
    global tkWorld

    help::create "help/man.html" "Man Command Help"
}

# man::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::close { } {
    variable man
    
    balloonhelp::cancel
    destroy $man(dialog)
}

# man::select --
#
#   Method to provide the user with a picklist of manpages to
#   display and view.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc man::select { } {
    variable man

    # Define the path to search for man pages on the system.  If
    # the manpath is already built, then don't build it again.

    if [info exists man(path)] {
	# Skip building the path
    } else {
	set man(path) ""
	if [catch {open "[file join /etc man.config]" r} fhandle] {
	    # Configuration file not found, use the man command to find
	    # the path
	    if [catch {exec man -W man} fhandle] {
		tk_message \
			-type ok \
			-message "Cannot find man pages"
		unset fhandle
		return
	    } else {
		# Got the path from the backup man command.
		set man(path) [file dirname $fhandle]
		unset fhandle
	    }
	} else {
	    # Open the man configuration file and extract the
	    # MANPATH lines only.
	    foreach l [split [read $fhandle] \n] {
		if [regexp -nocase "^MANPATH( |\t|\n)*/" $l] {
		    lappend man(path) [file join [lindex [split $l] 1] man1]
		}
	    }
	    unset l fhandle
	}
    }

    # Now define the page index which include the names of all
    # of the man pages which are available in the path.  If the page
    # index has already been constructed, then don't build it again.

    if [info exists man(page_index)] {
	# Skip the rebuilding of the manpage index
    } else {
	foreach p $man(path) {
	    foreach f [glob -nocomplain $p/*] {
		set f [lindex [split [file tail $f] .] 0]
		lappend tmp_index $f
	    }
	}
	set man(page_index) [lsort -increasing $tmp_index]
	unset tmp_index p f
    }

    # First clear out the existing entries.
    $man(entry.manpage) delete 0 end

    # Now give the user a picklist of man pages to choose from.
    set picklist::picklist(status) 0
    picklist::create $man(page_index)
    picklist::wait

    # Insert the users selection in the entry, move the cursor,
    # and set the focus.
    $man(entry.manpage) insert insert [picklist::cancel]
    $man(entry.manpage) icursor end
    focus $man(entry.manpage)
}