/*
** Jabby (a C library for Jabber protocol)
** Copyright (c) 2002 Hubert Sokoowski <who_ami@tlen.pl>
**
** This code is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License.
**
*/


#include "jabby.h"
#include "events.h"
#include "parser.h"
#include "mem.h"
#include <string.h>
#include "search.h"
#include <stdlib.h>

gint jabby_parse_message(jabby_session *ses, ikspak *pak)
{
  jabby_event *event;
  jabby_event_message *message;
  iks *x;

  event = jabby_event_new(JABBY_EVENT_MESSAGE);
  message = g_malloc(sizeof(*message));
  memset(message, '\0', sizeof(*message));
  event->event.message = message;
  message->from = g_strdup(iks_find_attrib(pak->x,"from"));
  message->id = g_strdup(iks_find_attrib(pak->x,"id"));
  message->body = g_strdup(iks_find_cdata(pak->x,"body"));
  message->subject = g_strdup(iks_find_cdata(pak->x,"subject"));
  message->thread = g_strdup(iks_find_cdata(pak->x,"thread"));
  message->error = g_strdup(iks_find_cdata(pak->x,"error"));
  x = iks_find(pak->x, "x");
  if(x)
    if(!strcmp(iks_find_attrib(x,"xmlns"),IKS_NS_DELAY))
      message->delay = g_strdup(iks_find_attrib(x,"stamp"));
  switch(pak->subtype)
  {
    case IKS_TYPE_ERROR:
      message->type = JABBY_MESSAGE_ERROR;
      break;
    case IKS_TYPE_CHAT:
      message->type = JABBY_MESSAGE_CHAT;
      break;
    default:
      message->type = JABBY_MESSAGE_NORMAL;
  }
  jabby_event_add(event);
  return 0;
}

gint jabby_parse_presence(jabby_session *ses, ikspak *pak)
{
  jabby_event *event;
  jabby_event_presence *presence;
  jabby_event_subscribe *subscribe;
  jabby_event_subscribed *subscribed;
  jabby_event_unsubscribe *unsubscribe;
  jabby_event_unsubscribed *unsubscribed;
  gchar *s, *s2;

  switch(pak->subtype)
  {
    case IKS_TYPE_AVAILABLE:
      presence = jabby_presence_new();
      s = iks_find_attrib(pak->x,"from");
      s2 = strchr(s,'/');
      if(s2)
        *s2 = '\0';
      presence->from = g_strdup(s);
      if(s2)
        *s2 = '/';
      s = iks_find_cdata(pak->x, "show");
      if(!s)
        presence->type = JABBY_PRESENCE_AVAILABLE;
      else
      {
        if(!strcmp(s,"away"))
          presence->type = JABBY_PRESENCE_AWAY;
        else
        if(!strcmp(s,"dnd"))
          presence->type = JABBY_PRESENCE_DND;
        else
        if(!strcmp(s,"xa"))
          presence->type = JABBY_PRESENCE_XA;
        else
        if(!strcmp(s,"chat"))
          presence->type = JABBY_PRESENCE_CHAT;
        else
        if(!strcmp(s,"available"))
          presence->type = JABBY_PRESENCE_AVAILABLE;
        else
          presence->type = JABBY_PRESENCE_UNAVAILABLE;
      }
      presence->message = g_strdup(iks_find_cdata(pak->x,"status"));
      event = jabby_event_new(JABBY_EVENT_PRESENCE);
      event->event.presence = presence;
      break;
    case IKS_TYPE_UNAVAILABLE:
      presence = jabby_presence_new();
      s = iks_find_attrib(pak->x,"from");
      s2 = NULL;
      s2 = strchr(s,'/');
      if(s2)
        *s2 = '\0';
      presence->from = g_strdup(s);
      if(s2)
        *s2 = '/';
      presence->type = JABBY_PRESENCE_UNAVAILABLE;
      presence->message = g_strdup(iks_find_cdata(pak->x,"status"));
      event = jabby_event_new(JABBY_EVENT_PRESENCE);
      event->event.presence = presence;
      break;
    case IKS_TYPE_SUBSCRIBE:
      subscribe = g_malloc(sizeof(*subscribe));
      subscribe->from = g_strdup(iks_find_attrib(pak->x, "from"));
      subscribe->status = g_strdup(iks_find_cdata(pak->x, "status"));
      event = jabby_event_new(JABBY_EVENT_SUBSCRIBE);
      event->event.subscribe = subscribe;
      break;
    case IKS_TYPE_SUBSCRIBED:
      subscribed = g_malloc(sizeof(*subscribed));
      subscribed->from = g_strdup(iks_find_attrib(pak->x, "from"));
      event = jabby_event_new(JABBY_EVENT_SUBSCRIBED);
      event->event.subscribed = subscribed;
      break;
    case IKS_TYPE_UNSUBSCRIBE:
      unsubscribe = g_malloc(sizeof(*unsubscribe));
      unsubscribe->from = g_strdup(iks_find_attrib(pak->x, "from"));
      unsubscribe->status = g_strdup(iks_find_cdata(pak->x, "status"));
      event = jabby_event_new(JABBY_EVENT_UNSUBSCRIBE);
      event->event.unsubscribe = unsubscribe;
      break;
    case IKS_TYPE_UNSUBSCRIBED:
      unsubscribed = g_malloc(sizeof(*unsubscribed));
      unsubscribed->from = g_strdup(iks_find_attrib(pak->x, "from"));
      unsubscribed->status = g_strdup(iks_find_cdata(pak->x, "status"));
      event = jabby_event_new(JABBY_EVENT_UNSUBSCRIBED);
      event->event.unsubscribed = unsubscribed;
      break;
    default:
      g_print("unknown subtype of a presence packet!!!\n");
      return 1;
  }
  jabby_event_add(event);
  return 0;
}

static void
jabby_parse_agent(iks *x)
{
  jabby_event *event;

  event = jabby_event_new(JABBY_EVENT_AGENT);
  event->event.agent = jabby_event_agent_new();
  event->event.agent->jid = g_strdup(iks_find_attrib(x, "jid"));
  event->event.agent->name = g_strdup(iks_find_cdata(x, "name"));
  event->event.agent->service = g_strdup(iks_find_cdata(x, "service"));
  event->event.agent->transport = g_strdup(iks_find_cdata(x, "transport"));
  jabby_event_add(event);
}

static void
jabby_parse_item(iks *x, int year)
{
  jabby_event *event;
  gchar *s, *r;
  iks *y;

  event = jabby_event_new(JABBY_EVENT_SEARCH_ITEM);
  event->event.search = jabby_search_res_new();
  y = iks_child(x);
  while(y)
  {
    s = iks_find_attrib(y, "var");
    r = iks_find_cdata(y, "value");
//    g_print("var: %s cdata: %s\n", s, r);
    if(!iks_strcmp(s, "jid"))
    {
      event->event.search->id = g_strdup(r);
    }
    if(!iks_strcmp(s, "firstname"))
    {
      event->event.search->first = g_strdup(r);
    }
    if(!iks_strcmp(s, "lastname"))
    {
      event->event.search->last = g_strdup(r);
    }
    if(!iks_strcmp(s, "nick"))
    {
      event->event.search->nick = g_strdup(r);
    }
    if(!iks_strcmp(s, "city"))
    {
      event->event.search->city = g_strdup(r);
    }
    if(!iks_strcmp(s, "birthyear"))
    {
      if (r)
        event->event.search->age = year - atoi(r);
    }
    if(!iks_strcmp(s, "gender"))
    {
      if (r)
        event->event.search->sex = atoi(r);
    }
    if(!iks_strcmp(s, "online"))
    {
      event->event.search->online = (!iks_strcmp(r, "no"))?0:1;
    }
    y = iks_next(y);
  }
  jabby_event_add(event);
}

gint jabby_parse_iq(jabby_session *ses, ikspak *pak)
{
  jabby_event *event;
  gchar *s;
  iks *x;

  if (!ses || !pak)
    return 1;
  switch (pak->subtype)
  {
    case IKS_TYPE_GET:
      {
        break;
      }
    case IKS_TYPE_SET:
      {
        if(pak->ns && !strcmp(pak->ns,IKS_NS_ROSTER))
          jabby_parse_roster(ses, iks_find(pak->x,"query"));
        break;
      }
    case IKS_TYPE_RESULT:
      {
		  if(ses->debug)
			  g_print("result state %d\n",ses->state);
        if((ses->state == JABBY_NET_STREAM) &&
           !iks_strcmp(iks_find_attrib(pak->x,"id"), "auth"))
        {
          ses->state = JABBY_NET_AUTH;
	  	  if(ses->debug)
              g_print("getting roster\n");
          jabby_get_roster(ses);
        }
        if((ses->state == JABBY_NET_AUTH) &&
           !iks_strcmp(iks_find_attrib(pak->x,"id"), "roster_1"))
        {
          jabby_parse_roster(ses, iks_child(pak->x));
          event = jabby_event_new(JABBY_EVENT_ROSTER_END);
          jabby_event_add(event);
		  if(ses->debug)
          	g_print("end of roster\n");
//          ses->state = JABBY_NET_ON;
        }
        if(!iks_strcmp(pak->ns, "jabber:iq:search"))
        {
          int year;
          gchar *s2;

          s2 = iks_find_attrib(pak->x, "id");
          year = (s2) ? (atoi(s2)) : (-1);
		  if(ses->debug)
			  g_print("parser: %s %d\n", s2, year);
          x = iks_child(pak->x);//query
          x = iks_child(x);//x
          x = iks_child(x);//title
          x = iks_next(x);//reported
          x = iks_next(x);//item
          while(x)
          {
//            g_print("jid: %s\n", iks_find_attrib(x, "jid"));
            jabby_parse_item(x, year);
            x = iks_next(x);//item
          }
          event = jabby_event_new(JABBY_EVENT_SEARCH_END);
          jabby_event_add(event);
        }
        if(!iks_strcmp(iks_find_attrib(pak->x,"id"), "register_1"))
        {
          event = jabby_event_new(JABBY_EVENT_REGISTERED);
          event->event.registered = jabby_event_registered_new();
          event->event.registered->to = g_strdup(iks_find_attrib(pak->x,
                                                                 "from"));
		  if(ses->debug)
			  g_print("registered to %s\n", event->event.registered->to);
          jabby_event_add(event);
        }
        if(!iks_strcmp(iks_find_attrib(pak->x,"id"), "agents"))
        {
          x = iks_child(pak->x);//query
          x = iks_child(x);//agent
          while(x)
          {
//            g_print("jid: %s\n", iks_find_attrib(x, "jid"));
            if(!iks_strcmp(x->name, "agent"))
              jabby_parse_agent(x);
            x = iks_next(x);//agent
          }
          event = jabby_event_new(JABBY_EVENT_AGENTS_END);
          jabby_event_add(event);
        }
		if(ses->debug)
			g_print("ns %s\n",pak->ns);
        break;
      }
    case IKS_TYPE_ERROR:
      {
		  if(ses->debug)
			  g_print("error packet received!!!\n");
        x = iks_find(pak->x, "error");
        if(x)
        {
          gchar *id;

          id = iks_find_attrib(pak->x,"id");
          s = iks_find_attrib(x,"code");
          if(s)
          {
            if(!strcmp(s,"401"))
            {
              if (id)
              {
                if (!iks_strcmp(id, "auth"))
                {
                  event = jabby_event_new (JABBY_EVENT_UNAUTHORIZED);
                  jabby_event_add(event);
                }
                else
                if(!iks_strcmp(id, "register_1"))
                {
                  event = jabby_event_new (JABBY_EVENT_UNAUTHORIZED_TRANSPORT);
                  jabby_event_add(event);
                }
              }
            }
            else
            if(!strcmp(s,"510"))
            {
              event = jabby_event_new(JABBY_EVENT_CONNECTION_FAILED);
              jabby_event_add(event);
            }
          }
        }
        break;
      }
    default:
      {
      }
  }
  return 0;
}

void jabby_parse_roster(jabby_session *ses, iks *x)
{
  iks *c;
  jabby_event *event;
  jabby_roster *roster;
  gchar *subscription;

  if(ses->debug)
	  g_print("cdata %s\n",x->cdata);
  c = iks_child(x);
  while(c)
  {
    subscription = iks_find_attrib(c,"subscription");
    if(subscription)
      if(strcmp(subscription, "remove"))
      {
        event = jabby_event_new(JABBY_EVENT_ROSTER_ITEM);
        roster = jabby_roster_new();
        roster->jid = g_strdup(iks_find_attrib(c,"jid"));
        roster->name = g_strdup(iks_find_attrib(c,"name"));
        roster->ask = g_strdup(iks_find_attrib(c,"ask"));
        roster->subscription = g_strdup(subscription);
        roster->group = g_strdup(iks_find_cdata(c, "group"));
        event->event.roster = roster;
        jabby_event_add(event);
		if(ses->debug)
			g_print("event added\n");
	  }
	c = iks_next(c);
  }
}
