/*
** TleenX2 (Tlen.pl Client)
** Copyright (c) 2002-2005 Hubert Sokoowski <who_ami@tlen.pl>
**                         Pawe Biliski <rael@fr.pl>
**                         Kamil Strzelecki <esack@browarek.net>
**
** This code is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License.
**
*/

#include "main.h"
#include "support.h"
#include "utils.h"
#include "lista.h"
#include "interface.h"
#include "callbacks.h"
#include "groups.h"
#include "conffile.h"


GSList *_expanded_groups = NULL;

gboolean _reloading_gui_atm = FALSE; // TRUE during list_reload_all()


gboolean list_is_refreshing_gui()
{
	if(_reloading_gui_atm)
		return TRUE;
	
	return FALSE;
}


gboolean list_is_group_expanded(const gchar *name)
{
	GSList *l = NULL;
	
	for(l = _expanded_groups; l; l=l->next)
		if(!strcmp(l->data, name)) 
			return TRUE;
	
	return FALSE;
}


void list_expanded_group_set(const gchar *name)
{
	tleenx_print(DEBUG, "list_expanded_group_set(): tryin to add %s\n", name);
	if(!list_is_group_expanded(name) && !_reloading_gui_atm)
		_expanded_groups = g_slist_prepend(_expanded_groups, g_strdup(name));
}


void _expanded_group_remove(const gchar *name)
{
	GSList *l = _expanded_groups;
	
	while(l) {
		if(!strcmp(l->data, name)) {
			g_free(l->data);
			_expanded_groups = g_slist_delete_link(_expanded_groups, l);
			l = _expanded_groups;
		}
		else
			l=l->next;
	}

}


void list_expanded_group_unset(const gchar *name)
{
	if(_reloading_gui_atm)
		return;
	
	_expanded_group_remove(name);	
}


void _expand_groups(GtkTreeView *treeview)
{
	GList *list;
	int i;
	struct group *g;
	GtkTreePath *treepath;
	gchar *path;

	for(i=0, list=groups_list; list; list=list->next, i++) {
		g = (struct group*)list->data;
		if(list_is_group_expanded(g->name)) {
			path = g_strdup_printf("%d",i);
			treepath = gtk_tree_path_new_from_string(path);
			gtk_tree_view_expand_row(treeview, treepath, TRUE);
			gtk_tree_path_free(treepath);
			g_free(path);
		}
	}
  
}


void list_expanded_groups_clear()
{
	g_slist_foreach(_expanded_groups, (GFunc)g_free, NULL);
	g_slist_free(_expanded_groups);
	_expanded_groups = NULL;
}


void _expanded_groups_remove_old()
{
	gchar *s = NULL;
	GSList *expanded = NULL;
	
	while(expanded) {
		s = (gchar *)expanded->data;
		if(!get_group(s)) 
			_expanded_group_remove(s);
		tleenx_print(DEBUG, "_expanded_groups_remove_old(): %s removed\n", s);
		g_free(s);
		s = NULL;
		expanded = expanded->next;
	}
}


/* return TRUE if user should be displayed in roster */
gboolean _is_user_visible(struct user *user)
{
	if(!user)
		return FALSE;
	
	//TODO: add PREF_SHOW_AGENTS option
	//if( !(preferences & PREF_SHOW_AGENTS) && (user->type == USER_AGENT) )
	if( (user->type == USER_AGENT) )
		return FALSE;
	
	if( (user->type == USER_VIRTUAL) && !(user->pending_event) )
		return FALSE;

	if( (preferences & PREF_HIDE_OFFLINE) && 
			(get_status(user) == STATUS_UNAVAILABLE) )
		return FALSE;
	
	return TRUE;
}


static
gint _compare_func(gconstpointer a, gconstpointer b)
{
  struct user *u1, *u2;
  guint status1, status2;
  gchar *user1, *user2;
  gint retval = 0;

  u1 = (struct user*)a;
  u2 = (struct user*)b;
  
  status1 = get_status(u1);
  status2 = get_status(u2);
  if(status1 > status2)
    return 1;

  if(strlen(u1->name))
	  user1 = g_strdup(u1->name);
  else 
	  user1 = g_strdup(u1->jid);
  
  if(strlen(u2->name))
	  user2 = g_strdup(u2->name);
  else
	  user2 = g_strdup(u2->jid);
  
  if(status1 == status2)
	  retval = strcasecmp(user1, user2);
  
  g_free(user1);
  g_free(user2);
  
  return retval ? retval : -1;
}


GList* sort_users(GList *l)
{
  return g_list_sort(l, _compare_func);
}


static
void create_model_add(GtkTreeStore *store, GList *l,
                      GtkTreeIter *child_iter, GtkTreeIter *iter)
{
	struct user *u = NULL;
	gchar *s = NULL, *s2 = NULL;

	while(l) {
		u=(struct user*) l->data;

		gtk_tree_store_append (store, child_iter, iter);
		s2 = utf(u->name);
		s = g_markup_escape_text(s2, strlen(s2));
		g_free(s2);
		s2 = NULL;
		
		if(!s)
			gtk_tree_store_set (store, child_iter,
					0, u->status,
					1, u->jid,
					2, u->jid,
					-1);
		else {
			gchar *d = user_description(u);
			if((preferences & PREF_INFO_MENU) && d)	{
				gchar *a = g_markup_escape_text(d, strlen(d));
				s2 = g_strdup_printf("%s\n<span size='smaller'>%s</span>",
						(strlen(s))?s:u->jid, a);
				g_free(a);
			}
			else
				s2 = g_strdup(strlen(s)?s:u->jid);
			
			g_free(d);
			
			gtk_tree_store_set (store, child_iter,
					0, u->status,
					1, s2,
					2, u->jid,
					-1);

			g_free(s);
			g_free(s2);
		}
		l = l->next;
		u = NULL;
	}
}


static
GtkTreeModel *create_model(GList *list)
{
  GtkTreeStore *store;
  GtkTreeIter child_iter, iter;
  GList *l;
  struct user *u;
  struct group *g;
  gchar *s, *gr, *path;
  gint i=0;

  store = gtk_tree_store_new(3, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING);

  //najpierw osoby zasubskrybowane
  if(preferences & PREF_GROUPS_LIST)
  {
	for(i=0, l=groups_list; l; l=l->next, i++)
    {
	  GList *l2 = NULL, *can_show = NULL;
		
      g=(struct group*)l->data;
      gr=g_strdup_printf("%s(%d/%d)",g->name,users_list_count_online(g->users),
                         users_list_count(g->users));
      s=utf(gr);
      g_free(gr);
      path=g_strdup_printf("%d",i);
      gtk_tree_store_append (store, &iter,NULL);
      gtk_tree_store_set (store, &iter,
                          0, 1,
                          1, s,
                          2, path,
                          -1);
      g_free(s);
      g_free(path);

      g->users = sort_users(g->users);
	  
	  u = NULL;
	  l2 = g->users;
	  while(l2) {
		  u = (struct user*)l2->data;
		  if( _is_user_visible(u) ) 
			  can_show = g_list_append(can_show, (gpointer)u);
		  l2 = l2->next;
		  u = NULL;
	  }
	  
      create_model_add(store, can_show, &child_iter,&iter);
	  g_list_free(can_show);
	  can_show = NULL;
    }
  }
  else
    create_model_add(store, list, &child_iter,NULL);

  // jabber agents
  l=list;
  GList *agents = NULL;

  while(l) {
	  u = (struct user*)l->data;
	  if( u->type == USER_AGENT ) 
		  agents = g_list_append(agents,(gpointer)u);
	  l = l->next;
  }

  if(agents) {
	  gtk_tree_store_append(store, &iter, NULL);
	  path = g_strdup_printf("%d", i++);
	  gtk_tree_store_set (store, &iter,
			  0, 1,
			  1, "Agenty/transporty",
			  2, path,
			  -1);
	  g_free(path);

	  create_model_add(store, agents, &child_iter, &iter);
	  g_list_free(agents);
  }
  
  return GTK_TREE_MODEL (store);
}


GdkPixbuf* _get_user_icon(gchar *jid)
{
	GdkPixbuf *retval = NULL;
	struct user *user = NULL; 
	
	tleenx_print(DEBUG, "_get_user_icon()\n");
	
	if(!jid) 
		return retval;
	
	user = get_user(jid);

	// we have unread messages from user
	if(user->pending_event) 
		return gtk_image_get_pixbuf(GTK_IMAGE(notify_icons[user->pending_event]));

	// waiting for authorisation
	if (!strcmp(user->ask,"subscribe") && !strcmp(user->subscription,"none") ) {
		retval = gtk_image_get_pixbuf(GTK_IMAGE(icon_wait_auth));
	}
	// no authorisation
	else if( !strcmp(user->ask,"") && !strcmp(user->subscription,"none") ) {
		retval = gtk_image_get_pixbuf(GTK_IMAGE(icon_cant_talk));
	}
	// subscribed users and rest
	//if( !strcmp(user->subscription,"both") ||	!strcmp(user->subscription,"to") ) 
	else {
		switch(get_type(jid)) {
			case USER_TLEN:
				retval = gtk_image_get_pixbuf(GTK_IMAGE(icons[user->status]));
				break;
			case USER_JABBER:
			case USER_AGENT:
				retval = gtk_image_get_pixbuf(GTK_IMAGE(j_icons[user->status-2]));
				break;
			case USER_WP:
				retval = gtk_image_get_pixbuf(GTK_IMAGE(wp_icons[user->status-2]));
				break;
			case USER_GG:
				retval = gtk_image_get_pixbuf(GTK_IMAGE(gg_icons[user->status-2]));
				break;
		}
	}
	
	return retval;
}


void
func_pixbuf (GtkTreeViewColumn *tree_column,
             GtkCellRenderer   *cell,
             GtkTreeModel      *model,
             GtkTreeIter       *iter,
             gpointer           data)
{
	guint status;
	gchar *jid;

	gtk_tree_model_get(model, iter, 0, &status, -1);
	gtk_tree_model_get(model, iter, 2, &jid, -1);

	if ((status==1) || (status==0))
		g_object_set(GTK_CELL_RENDERER(cell), "visible", FALSE, NULL);
	else
	{
		g_object_set(GTK_CELL_RENDERER(cell), "visible", TRUE, NULL);
		g_object_set(GTK_CELL_RENDERER(cell), "pixbuf", _get_user_icon(jid), NULL);
	}
	g_free(jid);
}


void
func_text (GtkTreeViewColumn *tree_column,
             GtkCellRenderer   *cell,
             GtkTreeModel      *model,
             GtkTreeIter       *iter,
             gpointer           data)
{
  guint status;
  gchar *s;
  
#ifdef GTK_2_8
  g_object_set(GTK_CELL_RENDERER(cell), "wrap-width", 160, NULL);
#endif

  gtk_tree_model_get (model, iter,
                      0, &status,
                      -1);
  gtk_tree_model_get (model, iter,
                      1, &s,
                      -1);
  switch(status)
  {
    case 0://brak autoryzacji
      g_object_set (GTK_CELL_RENDERER (cell),
                  "font", "normal",
                  "scale", 0.7,
                  NULL);
      g_object_set (GTK_CELL_RENDERER (cell),
                  "text", s,
                  NULL);
      break;
    case 1://grupa
      g_object_set (GTK_CELL_RENDERER (cell),
                  "font", "bold",
                  "scale", 1.0,
                  NULL);
      g_object_set (GTK_CELL_RENDERER (cell),
                  "markup", NULL,
                  "text", s,
                  NULL);
      break;
    default://osoby
      g_object_set (GTK_CELL_RENDERER (cell),
                    "font", "normal",
                    "scale", 1.0,
                    NULL);
      g_object_set (GTK_CELL_RENDERER (cell),
                    "text", NULL,
                    "markup", s,
                    NULL);
  }
  g_free(s);
}


static
void add_columns(GtkTreeView *treeview)
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  renderer = gtk_cell_renderer_pixbuf_new();
  column = gtk_tree_view_column_new();
  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_set_cell_data_func (column, renderer,
                                           func_pixbuf, NULL, NULL);

  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start(column, renderer, FALSE);
  gtk_tree_view_column_set_cell_data_func (column, renderer,
                                           func_text, NULL, NULL);
  gtk_tree_view_column_set_expand(column, TRUE);
  gtk_tree_view_append_column (treeview, column);

}


void refresh_list(GtkTreeView *treeview, GList **l, gboolean sort)
{
  GtkTreeModel *model;
  GList *selected_rows = NULL;
  GtkTreeRowReference *row_reference = NULL;
  
  /* kinda ugly but necessary to preserve selection */
  selected_rows = gtk_tree_selection_get_selected_rows(
		  gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview)), NULL);
  if(selected_rows) {
	row_reference = gtk_tree_row_reference_new(
			gtk_tree_view_get_model(GTK_TREE_VIEW(treeview)),
			g_list_first(selected_rows)->data);
	g_list_foreach(selected_rows, (GFunc )gtk_tree_path_free, NULL);
	g_list_free (selected_rows);
  }
  
  if(sort)
    *l = sort_users(*l);
  model = create_model(*l);
  gtk_tree_view_set_model(treeview, model);
  g_object_unref(G_OBJECT(model));
  
  if(preferences & PREF_GROUPS_LIST) 
	  _expand_groups(treeview);

  /* reselect row */
  if(gtk_tree_row_reference_valid(row_reference)) {
	  GtkTreePath *path = gtk_tree_row_reference_get_path(row_reference);
	  gtk_tree_selection_select_path(
			  gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview)), path);
	  gtk_tree_path_free(path);
  }
  
}


static
GtkWidget *create_treeview(GtkWidget *cont)
{
  GtkWidget *treeview, *widget;

  widget=gtk_viewport_new (NULL, NULL);
  gtk_widget_show (widget);
  gtk_container_add (GTK_CONTAINER (cont), widget);

  treeview = gtk_tree_view_new ();
  gtk_widget_show (treeview);
  gtk_container_add (GTK_CONTAINER (widget), treeview);
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), FALSE);
  g_signal_connect (G_OBJECT (treeview), "row_activated",
                      G_CALLBACK (on_treeview_row_activated),
                      NULL);
  g_signal_connect (G_OBJECT (treeview), "button_release_event",
                      G_CALLBACK (on_treeview_button_release_event),
                      NULL);
  g_signal_connect (G_OBJECT (gtk_tree_view_get_selection(GTK_TREE_VIEW(
                                                                   treeview))),
                    "changed",
                      G_CALLBACK (on_treeview_select_row),
                      NULL);
  g_signal_connect(G_OBJECT(treeview), "row-expanded", 
		  G_CALLBACK(on_treeview_row_expanded), NULL);
  g_signal_connect(G_OBJECT(treeview), "row-collapsed", 
		  G_CALLBACK(on_treeview_row_collapsed), NULL);
  g_signal_connect(G_OBJECT(treeview), "motion-notify-event", 
		  G_CALLBACK(on_treeview_motion), NULL);
  g_signal_connect(G_OBJECT(treeview), "leave-notify-event", 
		  G_CALLBACK(on_treeview_leave), NULL);
  
  
  return treeview;
}


/* sort of lightweight variant of list_reload_all */
void list_refresh()
{
  if(preferences & PREF_GROUPS_NONE) 
	  refresh_list(GTK_TREE_VIEW(treeview1), &users_list, TRUE);
  else
	  refresh_list(GTK_TREE_VIEW(treeview1), &users_list, FALSE);
  
  gtk_widget_realize(treeview1);
  gtk_tree_view_columns_autosize(GTK_TREE_VIEW(treeview1));
  
  return;
}


void list_reload_all()
{
  GList *l;
  struct user *u;
  GtkWidget *scroll;

  _reloading_gui_atm = TRUE;
  
  list_destroy_tooltip();
  
  if(!treeview1) {
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show(scroll);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scroll),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(lookup_widget(window1, "vbox")),
			scroll, TRUE, TRUE, 0);
	treeview1 = create_treeview(scroll);
    scroll1 = scroll;
    add_columns(GTK_TREE_VIEW(treeview1));
  }

  if(groups_list) 
	  groups_list_clear();
  
  if(preferences & PREF_GROUPS_NONE) {
    refresh_list(GTK_TREE_VIEW(treeview1), &users_list, TRUE);
    gtk_widget_realize(treeview1);
    gtk_tree_view_columns_autosize(GTK_TREE_VIEW(treeview1));
    return;
  }
  
  users_list = sort_users(users_list);
  
  l = users_list;
  while(l) {
    u = (struct user*) l->data;
    if(!get_group(u->group)) add_group(u->group);
    if(strlen(u->group) > 0) add_togroup(u->group,u);
    l = l->next;
  }
  
  prepend_group("Kontakty");
  for(l=users_list;l;l=l->next) {
    u=(struct user*) l->data;
    if(!strlen(u->group)) add_togroup("Kontakty",u);
  }
  
  _expanded_groups_remove_old();
  
  refresh_list(GTK_TREE_VIEW(treeview1), &users_list, FALSE);
  gtk_widget_realize(treeview1);
  gtk_tree_view_columns_autosize (GTK_TREE_VIEW(treeview1));

  _reloading_gui_atm = FALSE;

}


/**************************** Tooltips ********************************/


/* Create markup text */
gchar *_get_text(gchar *jid)
{
	struct user *u = get_user(jid);
	GString *str = NULL;
	gchar *s = NULL, *s2 = NULL;
	
	tleenx_print(DEBUG, "_get_text(): creating markup for %s\n", jid);
	
	str = g_string_new("");
	
	if(strlen(u->name))
		s2 = utf(u->name);
	else 
		s2 = utf(u->jid);
	
	s = g_markup_escape_text(s2, strlen(s2));
	g_free(s2);
	
	g_string_append_printf(str, "<b>%s</b> (%s)\n<small>Autoryzacja: %s</small>", 
			s, u->jid, u->subscription);
	g_free(s);
	s = NULL;
	s2 = NULL;

	if((s2 = user_description(u))) {
		s = g_markup_escape_text(s2, strlen(s2));
		g_string_append_printf(str,"\n<small><i>%s</i></small>", s);
		g_free(s2);
		g_free(s);
	}

	return g_string_free(str, FALSE);
}


/* paint tooltip layout (callback for expose event)*/
static void _paint_tip(GtkWidget *widget, GdkEventExpose *event, gchar *jid)
{
	
	tleenx_print(DEBUG, "_paint_tip()\n");
	
	if(!list_tooltip.text || !list_tooltip.window)
		return;

	gtk_paint_flat_box(list_tooltip.window->style, list_tooltip.window->window,
			GTK_STATE_NORMAL, GTK_SHADOW_OUT, NULL, 
			list_tooltip.window, "tooltip", 0, 0, -1, -1);
	
	gdk_draw_pixbuf(GDK_DRAWABLE(list_tooltip.window->window), NULL, 
			list_tooltip.icon, 0, 0, 4, 4, -1 , -1,
			GDK_RGB_DITHER_NONE, 0, 0);
	
	gtk_paint_layout(list_tooltip.window->style, list_tooltip.window->window, 
			GTK_STATE_NORMAL, TRUE, NULL, list_tooltip.window, 
			"tooltip", 24, 4, list_tooltip.layout);

	return;
}


/* update tooltip position to follow mouse pointer */
void list_tooltip_move()
{
	gint x, y, s_width, s_height, w_width, w_height;
	GdkScreen *screen = NULL;

	if(list_tooltip.window) {
		gdk_display_get_pointer(gdk_display_get_default(), &screen, &x, &y, NULL);
		gtk_window_get_size(GTK_WINDOW(list_tooltip.window), &w_width, &w_height);
		s_width = gdk_screen_get_width(screen);
		s_height = gdk_screen_get_height(screen);
		x = x + 10;
		y = y + 10;

		if(x + w_width > s_width) 
			x = s_width - w_width;
		if(y + w_height > s_height) 
			y = s_height - w_height;
		
		gtk_window_move(GTK_WINDOW(list_tooltip.window), x, y);
	}
}


/* destroy all tooltip window related stuff */
void list_destroy_tooltip()
{
	tleenx_print(DEBUG,"list_destroy_tooltip()\n");
	
	if(!list_tooltip.window) 
		return;
	
	gtk_widget_destroy(list_tooltip.window);
	list_tooltip.window = NULL;
	g_free(list_tooltip.text);
	list_tooltip.text = NULL;
	g_object_unref(list_tooltip.layout);
	list_tooltip.icon = NULL;
}


/* main tooltip function - get data from treeview, create tooltip window */
gboolean list_tooltip_timeout(GtkWidget *widget)
{
	gint w, h;;
	GtkTreePath *path = NULL;
	GtkTreeIter iter;
	GtkTreeModel *model = NULL;
	guint val;
	gchar *jid;
	
	if( (list_tooltip.window) || (_reloading_gui_atm) )
		return FALSE;

	/* Get value from cell under the pointer ang generate markup text*/
	tleenx_print(DEBUG, "list_tooltip_timeout(): get values\n");
	if(!gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget), 
			list_tooltip.cell_rect.x, list_tooltip.cell_rect.y, &path, NULL, NULL, NULL))
		return FALSE;
	
	model = gtk_tree_view_get_model(GTK_TREE_VIEW(widget));
	gtk_tree_model_get_iter(model, &iter, path);
	gtk_tree_model_get(model, &iter, 0, &val, -1);
	if(path) gtk_tree_path_free(path);

	if((val==0) || (val==1))
		return FALSE;
	
	gtk_tree_model_get(model, &iter, 2, &jid, -1);
	
	list_tooltip.text = _get_text(jid);
	list_tooltip.icon = _get_user_icon(jid);
	g_free(jid);
	
	/* Create tooltip window and pango layout*/
	tleenx_print(DEBUG, "list_tooltip_timeout(): create tooltip window\n");
	list_tooltip.window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_widget_set_app_paintable(list_tooltip.window, TRUE);
	gtk_window_set_resizable(GTK_WINDOW(list_tooltip.window), FALSE);
	gtk_widget_set_name(list_tooltip.window, "gtk-tooltips");
	g_signal_connect(G_OBJECT(list_tooltip.window), "expose_event", 
			G_CALLBACK(_paint_tip), NULL);
	gtk_widget_ensure_style(list_tooltip.window);
	
	list_tooltip.layout = gtk_widget_create_pango_layout (list_tooltip.window, NULL);
	pango_layout_set_wrap(list_tooltip.layout, PANGO_WRAP_WORD);
	pango_layout_set_width(list_tooltip.layout, 300000);
	pango_layout_set_markup(list_tooltip.layout, list_tooltip.text, strlen(list_tooltip.text));
	pango_layout_get_size(list_tooltip.layout, &w, &h);
	w = PANGO_PIXELS(w) + 8 + 24;
	h = PANGO_PIXELS(h) + 8;
	gtk_widget_set_size_request(list_tooltip.window, w, h);
	list_tooltip_move();
	gtk_widget_show(list_tooltip.window);
	
	return TRUE;
}

