-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2008  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.Characters.Latin_1;
with Ada.IO_Exceptions;
with Ada.Numerics.Discrete_Random;
with Ada.Strings.Fixed;
with Ada.Strings.Maps.Constants;
with Ada.Text_IO;
with Externals.Mail;
with Externals.Simple;
with Menus;                        use Menus;
with Misc;                         use Misc;

package body Externals.Mail is

   -- We need to figure out what the boundary is.  We don't trust Pine
   --  and the shell to get it for us, so we'll use this heuristic.
   --  Open Infile, read it in a line at a time.  If a line has the
   --  form `--(some characters)--' then save it.  The *last* one is
   --  assumed to be the correct boundary.  Then we just have to slice
   --  off the leading and trailiing `--'.
   function Find_Mime_Boundary (Infile : String) return UBS is
      BF : Ada.Text_IO.File_Type;
      L2 : UBS; -- the line we've read in.
      PB : UBS; -- possible boundary
   begin
      Debug("+Externals.Mail.Find_Mime_Boundary");
      Ada.Text_IO.Open(File => BF,
                       Mode => Ada.Text_IO.In_File,
                       Name => Infile);
  Boundary_Loop:
      loop
         begin
            L2 := Unbounded_Get_Line(BF);
            declare
               L2S : constant String := ToStr(L2);
            begin
               Debug("Considering: " & L2S);
               if L2S'Length > 4 then
                  if L2S(1..2) = "--"
                 and L2S(L2S'Last-1..L2S'Last) = "--" then
                     -- This is a possible boundary.
                     PB := ToUBS(L2S(3..L2S'Last-2));
                     Debug("Possible boundary: " & ToStr(PB));
                  end if;
               end if;
            end;
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit Boundary_Loop; -- Ignore.
         end;
      end loop Boundary_Loop;
      Ada.Text_IO.Close(BF);
      Debug("-Externals.Mail.Find_Mime_Boundary");
      return PB;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Find_Mime_Boundary");
         raise;
   end Find_Mime_Boundary;

   -- I should really generalise this to n part MIME....
   procedure Split_Two_Parts (Infile   : in String;
                              Part_One : in String;
                              Part_Two : in String;
                              Boundary : in UBS) is
      Infile_F   : Ada.Text_IO.File_Type;
      Part_One_F : Character_IO.File_Type;
      Part_Two_F : Character_IO.File_Type;
      L1         : UBS;
      L2         : UBS;
      BS         : constant String := ToStr(Boundary);
   begin
      -- Open files....
      begin
         Ada.Text_IO.Open(File => Infile_F,
                          Mode => Ada.Text_IO.In_File,
                          Name => Infile);
         Character_IO.Create(File => Part_One_F,
                            Mode => Character_IO.Out_File,
                            Name => Part_One);
         Character_IO.Create(File => Part_Two_F,
                            Mode => Character_IO.Out_File,
                            Name => Part_Two);
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Opening files)");
            raise;
      end;
      -- Get the two parts.  The first part, we want to include the MIME
      -- headers, but not the boundaries nor the blank line before the next
      -- part.
      -- So, first, we walk through infile looking for the boundary.
      begin
     Find_First_Boundary:
         loop
            L2 := Unbounded_Get_Line(Infile_F);
            declare
               L2S : constant String := ToStr(L2);
            begin
               exit Find_First_Boundary when L2S = "--" & BS;
            end;
         end loop Find_First_Boundary;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Find_First_Boundary)");
            raise;
      end;
      -- Now, walk through infile, copying complete lines to Part_One
      -- until we find a blank line - boundary pair.
      begin
         L1 := Unbounded_Get_Line(Infile_F);
     Find_Second_Boundary:
         loop
            L2 := Unbounded_Get_Line(Infile_F);
            declare
               L1S : constant String := ToStr(L1);
               L2S : constant String := ToStr(L2);
            begin
               exit Find_Second_Boundary
                  when L1S'Length = 0 and L2S = "--" & BS;
               -- Trap case where data ends without newline.
               if L2S = "--" & BS then
                  Character_IO_Put(Part_One_F, L1S);
                  exit Find_Second_Boundary;
               end if;
               Character_IO_Put_Line(Part_One_F, L1S);
            end;
            L1 := L2;
         end loop Find_Second_Boundary;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Find_Second_Boundary)");
            raise;
      end;
      -- Continue through infile, copying complete lines to Part_Two,
      -- until we find another blank_line - boundary pair.
      begin
         L1 := Unbounded_Get_Line(Infile_F);
     Find_Third_Boundary:
         loop
            L2 := Unbounded_Get_Line(Infile_F);
            declare
               L1S : constant String := ToStr(L1);
               L2S : constant String := ToStr(L2);
            begin
               exit Find_Third_Boundary
                  when L1S'Length = 0 and L2S = "--" & BS & "--";
               -- Trap case where data ends without newline.
               if L2S = "--" & BS & "--" then
                  Character_IO_Put(Part_Two_F, L1S);
                  exit Find_Third_Boundary;
               end if;
               Character_IO_Put_Line(Part_Two_F, L1S);
            end;
            L1 := L2;
         end loop Find_Third_Boundary;
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Find_Third_Boundary)");
            raise;
      end;
      -- Close files.
      begin
         Ada.Text_IO.Close(Infile_F);
         Character_IO.Close(Part_One_F);
         Character_IO.Close(Part_Two_F);
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Externals.Mail.Split_Two_Parts (Closing files)");
            raise;
      end;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Split_Two_Parts");
         raise;
   end Split_Two_Parts;

   procedure Extract_Content_Type_From_Header (Email_Filename  : in String;
                                               Target_Filename : in String;
                                               Ignore_Missing  : in Boolean := False) is
      E1, E2 : Integer;
   begin
      ForkExec2_InOut(Value_Nonempty(Config.Binary(Sed)),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("1,/^[:space:]*$/ ! d")),
                      E1,
                      Value_Nonempty(Config.Binary(Grep)),
                      UBS_Array'(0 => ToUBS("grep"),
                                 1 => ToUBS("-i"),
                                 2 => ToUBS("content-type: ")),
                      E2,
                      Source => Email_Filename,
                      Target => Target_Filename);
      if E1 /= 0 then
         Error("Problem with sed! (ff2a)");
      elsif E2 /= 0 and (not Ignore_Missing) then
         Error("Problem with grep! (ff2b)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Content_Type_From_Header");
         raise;
   end Extract_Content_Type_From_Header;

    procedure Extract_Content_Transfer_Encoding_From_Header (Email_Filename  : in String;
                                               Target_Filename : in String;
                                               Ignore_Missing  : in Boolean := False) is
      E1, E2 : Integer;
   begin
      ForkExec2_InOut(Value_Nonempty(Config.Binary(Sed)),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("1,/^[:space:]*$/ ! d")),
                      E1,
                      Value_Nonempty(Config.Binary(Grep)),
                      UBS_Array'(0 => ToUBS("grep"),
                                 1 => ToUBS("-i"),
                                 2 => ToUBS("content-transfer-encoding: ")),
                      E2,
                      Source => Email_Filename,
                      Target => Target_Filename);
      if E1 /= 0 then
         Error("Problem with sed! (ff2aa)");
      elsif E2 /= 0 and (not Ignore_Missing) then
         Error("Problem with grep! (ff2ba)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Content_Transfer_Encoding_From_Header");
         raise;
   end Extract_Content_Transfer_Encoding_From_Header;

   -- Get the header from a header-body mail into a file.
   procedure Extract_Header (Email_Filename  : in String;
                             Target_Filename : in String) is
   begin
      if ForkExec_InOut(Value_Nonempty(Config.Binary(Sed)),
                        UBS_Array'(0 => ToUBS("sed"),
                                   1 => ToUBS("-e"),
                                   2 => ToUBS("1,/^[:space:]*$/ ! d"),
                                   3 => ToUBS("-e"),
                                   4 => ToUBS("/^[:space:]*$/ d")),
                        Source => Email_Filename,
                        Target => Target_Filename) /= 0 then
         Error("sed failed! (ff11)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Header");
         raise;
   end Extract_Header;

   -- Get the body from a header-body mail into a file.
   procedure Extract_Body (Email_Filename  : in String;
                           Target_Filename : in String) is
      E1, E2 : Integer;
   begin
      ForkExec2_InOut(Value_Nonempty(Config.Binary(Sed)),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("-e"),
                                 2 => ToUBS("/^[:space:]*$/,$ ! d")),
                      E1,
                      Value_Nonempty(Config.Binary(Sed)),
                      UBS_Array'(0 => ToUBS("sed"),
                                 1 => ToUBS("-e"),
                                 2 => ToUBS("1,1 { /^[:space:]*$/ d ; }")),
                      E2,
                      Source => Email_Filename,
                      Target => Target_Filename);
      if E1 /= 0 then
         Error("sed failed! (ff3)");
      elsif E2 /= 0 then
         Error("sed failed! (ff13)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Extract_Body");
         raise;
   end Extract_Body;

   procedure Delete_Trailing_Blank_Lines (Infile  : in String;
                                          Outfile : in String) is
   begin
      if ForkExec_InOut(Value_Nonempty(Config.Binary(Sed)),
                        UBS_Array'(0 => ToUBS("sed"),
                                   1 => ToUBS("-e"),
                                   2 => ToUBS(":a"),
                                   3 => ToUBS("-e"),
                                   4 => ToUBS("/^\n*$/{$d;N;};/\n$/ba")),
                        Source => Infile,
                        Target => Outfile) /= 0 then
         Error("sed failed! (ff5)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Delete_Trailing_Blank_Lines");
         raise;
   end Delete_Trailing_Blank_Lines;

   procedure Formail_Concat_Extract_InOut (Header : in String;
                                           Source : in String;
                                           Target : in String) is
      E : Integer;
      pragma Unreferenced(E);
   begin
      E := ForkExec_InOut(Value_Nonempty(Config.Binary(Formail)),
                          UBS_Array'(0 => ToUBS("formail"),
                                     1 => ToUBS("-c"),
                                     2 => ToUBS("-x"),
                                     3 => ToUBS(Header)),
                          Source => Source,
                          Target => Target);
      -- Ignore the return code of formail.  It seems to return 1 a
      -- lot, and I can't find any documentation telling me what it
      -- _should_ return.
   end Formail_Concat_Extract_InOut;

   procedure Formail_Drop_InOut (Header : in UBS_Array;
                                 Source : in String;
                                 Target : in String) is
      F : UBS_Array(0..2*Header'Length);
      E : Integer;
      pragma Unreferenced(E);
   begin
      F(0) := ToUBS("formail");
      for I in 1..Header'Length loop
         F((2*I)-1) := ToUBS("-I");
         F(2*I) := Header(I+Header'First-1);
      end loop;
      E := ForkExec_InOut(Value_Nonempty(Config.Binary(Formail)),
                          F,
                          Source => Source,
                          Target => Target);
      -- Ignore the return code of formail.  It seems to return 1 a
      -- lot, and I can't find any documentation telling me what it
      -- _should_ return.
   end Formail_Drop_InOut;

   procedure Formail_Action_InOut (Source : in String;
                                   Target : in String;
                                   Action : in String) is
      E : Integer;
      pragma Unreferenced(E);
   begin
      E := ForkExec_InOut(Value_Nonempty(Config.Binary(Formail)),
                          ToUBS("formail -s " & Action),
                          Source => Source,
                          Target => Target);
      -- Ignore the return code of formail.  It seems to return 1 a
      -- lot, and I can't find any documentation telling me what it
      -- _should_ return.
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Formail_Action_InOut");
         raise;
   end Formail_Action_InOut;

   subtype Upper_Letters is Character range 'A'..'Z';

   package Random_Letters
   is new Ada.Numerics.Discrete_Random(Upper_Letters);

   Random_Letters_Generator : Random_Letters.Generator;

   function Get_Boundary return String is
      U : UBS;
      use type UBS;
   begin
      U := ToUBS("MIME_CONTENT_BREAK_"
                 & "=_");
      -- Add some random stuff.
      for I in 1..31 loop
         U := U & Random_Letters.Random(Random_Letters_Generator);
      end loop;
      return ToStr(U);
   end Get_Boundary;

   procedure Mimeconstruct2 (Part1_Filename  : in String;
                             Part2_Filename  : in String;
                             Output_Filename : in String;
                             Content_Type    : in String) is
   begin
      if ForkExec_Out(Value_Nonempty(Config.Binary(Mimetool)),
                      UBS_Array'(0 => ToUBS("mime-tool"),
                                 1 => ToUBS("-B"),
                                 2 => ToUBS(Get_Boundary),
                                 3 => ToUBS("-O"),
                                 4 => ToUBS(Content_Type),
                                 5 => ToUBS("-o"),
                                 6 => ToUBS("-p"),
                                 7 => ToUBS(Part1_Filename),
                                 8 => ToUBS("-o"),
                                 9 => ToUBS("-p"),
                                 10 => ToUBS(Part2_Filename)),
                      Output_Filename) /= 0 then
         Error("mime-tool failed. (mff1)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Mimeconstruct2");
         raise;
   end Mimeconstruct2;

   procedure Mimeconstruct_Subpart (Infile       : in String;
                                    Outfile      : in String;
                                    Content_Type : in String;
                                    Dos2UnixU    : in Boolean;
                                    Use_Header   : in Boolean;
                                    Use_Encoding : in Boolean;
                                    Header       : in String := "";
                                    Encoding     : in String := "") is
      -- Remember, Use_Header and Header are ignored.
      pragma Unreferenced(Use_Header, Header);
      Encode   : String(1..2);
   begin
      -- Figure out which encoding to use.
      if Use_Encoding then
         if Encoding = "base64" then
            Encode := "-x";
         elsif Encoding = "quoted-printable" then
            Encode := "-q";
         elsif Encoding = "7bit" then
            Encode := "-7";
         elsif Encoding = "8bit" then
            Encode := "-8";
         else
            Encode := "-x";
         end if;
         -- Construct a suitable MIME subpart.
         if ForkExec_Out(Value_Nonempty(Config.Binary(Mimetool)),
                         UBS_Array'(0 => ToUBS("mime-tool"),
                                    1 => ToUBS("-s"),
                                    2 => ToUBS(Encode),
                                    3 => ToUBS("-o"),
                                    4 => ToUBS("-c"),
                                    5 => ToUBS(Content_Type),
                                    6 => ToUBS(Infile)),
                         Outfile) /= 0 then
            Error("mime-tool failed. (mff3a)");
         end if;
      else
         -- Construct a suitable MIME subpart.
         if ForkExec_Out(Value_Nonempty(Config.Binary(Mimetool)),
                         UBS_Array'(0 => ToUBS("mime-tool"),
                                    1 => ToUBS("-s"),
                                    2 => ToUBS("-b"),
                                    3 => ToUBS("-m"),
                                    4 => ToUBS("-o"),
                                    5 => ToUBS("-c"),
                                    6 => ToUBS(Content_Type),
                                    7 => ToUBS(Infile)),
                         Outfile) /= 0 then
            Error("mime-tool failed. (mff3b)");
         end if;
      end if;
      -- Do we need to convert the file with dos2unix?
      if Dos2UnixU then
         Externals.Simple.Dos2Unix_U(Outfile);
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Mimeconstruct_Subpart");
         raise;
   end Mimeconstruct_Subpart;

   -- Construct an entire multipart/mixed email.
   procedure Mimeconstruct_Mixed (Filenames : in UBS_Array;
                                  Outfile   : in String) is
      -- Arg array size is ( 3*num-filenames ) +5, but starting at 0.
      A : UBS_Array(0 .. (3 * Filenames'Length)+4);
   begin
        A(0) := ToUBS("mime-tool");
        A(1) := ToUBS("-B");
        A(2) := ToUBS(Get_Boundary);
        A(3) := ToUBS("-O");
        A(4) := ToUBS("multipart/mixed");
        for I in 1..Filenames'Length loop
           -- First one should be 5.
           A((3*I)+2) := ToUBS("-o");
           A((3*I)+3) := ToUBS("-p");
           A((3*I)+4) := Filenames(I+Filenames'First-1);
        end loop;
        if ForkExec_Out(Value_Nonempty(Config.Binary(Mimetool)),
                        A,
                        Outfile) /= 0 then
         Error("mime-tool failed. (mff2)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Mimeconstruct_Mixed");
         raise;
   end Mimeconstruct_Mixed;


   -- From RFC2045: (A good strategy is to choose a boundary that includes
   -- a character sequence such as "=_" which can never appear in a
   -- quoted-printable body.  See the definition of multipart messages in
   -- RFC 2046.)

   -- Because we might build multiple parts, we'll use a boundary of
   --     MIME_CONTENT_BREAK_=_(partno)=_(random stuff)
   -- where partno is a per-topal-run counter.  (Later: removed partno)

   -- Note that mime-tool actually uses --=_%s_= as the boundary.

   -- We don't currently check for boundary collisions in the data.
   --  We should.

   -- Then we call out to our patched version of mime-tool


   -- View a MIME file.
   procedure View_MIME (F : String) is
      Chosen  : Positive;
      MTF     : constant String := Temp_File_Name("vmmt");
      MTFE    : constant String := Temp_File_Name("vmmte");
      MT, MTE : UBS;
      NC      : Boolean         := False;
      NTFE    : Boolean         := False;
      CT      : UBS;
      CTE     : UBS;
   begin
      -- Find out the type of MIME file.
      Externals.Mail.Extract_Content_Type_From_Header(F, MTF, Ignore_Missing => True);
      MT := Read_Fold(MTF);
      Externals.Mail.Extract_Content_Transfer_Encoding_From_Header(F, MTFE, Ignore_Missing => True);
      MTE := Read_Fold(MTFE);
      Ada.Text_IO.New_Line(5);
      declare
         MTFS  : constant String := ToStr(MT);
         MTFES : constant String := ToStr(MTE);
         I, I2 : Integer;
         J, J2 : Integer;
      begin
         I := Ada.Strings.Fixed.Index(MTFS, ": ");
         if I = 0 then
            I := MTFS'First;
            NC := True;
         else
            I := I + 2;
         end if;
         I2 := Ada.Strings.Fixed.Index(MTFS, ";");
         if I2 = 0 then
            I2 := MTFS'Last;
         else
            I2 := I2 - 1;
         end if;
         J := Ada.Strings.Fixed.Index(MTFES, ": ");
         if J = 0 then
            J := MTFES'First;
            NTFE := True;
         else
            J := J + 2;
         end if;
         J2 := Ada.Strings.Fixed.Index(MTFES, ";");
         if J2 = 0 then
            J2 := MTFES'Last;
         else
            J2 := J2 - 1;
         end if;
         if NC then
            Ada.Text_IO.Put_Line("About to view an attachment with no content type");
            CT := ToUBS("text/plain");
         else
            Ada.Text_IO.Put_Line("About to view an attachment with content type `"
                                 & MTFS(I..I2) & "'");
            CT := ToUBS(MTFS(I..I2));
         end if;
         if NTFE then
            Ada.Text_IO.Put_Line("No content transfer encoding");
            CT := ToUBS("7bit");
         else
            CTE := ToUBS(Ada.Strings.Fixed.Translate(MTFES(J..J2),
                                                     Ada.Strings.Maps.Constants.Lower_Case_Map));
            Ada.Text_IO.Put_Line("Content transfer encoding is `"
                                 & ToStr(CTE) & "'");
         end if;
      end;
      -- Do we have to ask which viewer...?
      if Config.MIME_Viewer = 1 then
         -- Ask what we should do.
         Chosen := MV_Values(MIME_Viewer_Menu2);
      end if;
      -- Actually execute the chosen viewer method.
      case Chosen is
         when 2 => -- Metamail
            Ada.Text_IO.Put_Line("Viewing attachment with metamail...");
            if ForkExec(Misc.Value_Nonempty(Config.Binary(Metamail)),
                        UBS_Array'(0 => ToUBS("metamail"),
                                   1 => ToUBS(F))) /= 0 then
               Misc.ErrorNE("metamail failed! (ff5)");
            end if;
         when 3 => -- run-mailcap
                   -- FIXME: Deal with multipart nicely.
            declare
               B  : constant String := Temp_File_Name("vmb");
               B2 : constant String := Temp_File_Name("vmb2");
               use type UBS;
            begin
               if NC then
                  Ada.Text_IO.Put_Line("Hoping that this is text/plain (no content-type!)");
               end if;
               -- Handle the transfer encoding!  If it's QP or base64,
               --  decode it!
               Externals.Mail.Extract_Body(F, B);
               if ToStr(CTE) = "quoted-printable" then
                  QP_Decode(B, B2);
                  Externals.Simple.Cat_Out(B2, B);
               elsif ToStr(CTE) = "base64" then
                  Base64_Decode(B, B2);
                  Externals.Simple.Cat_Out(B2, B);
               end if;
               Ada.Text_IO.Put_Line("Viewing attachment with run-mailcap...");
               if ForkExec(Misc.Value_Nonempty(Config.Binary(Runmailcap)),
                           UBS_Array'(0 => ToUBS("run-mailcap"),
                                      1 => CT & ToUBS(":" & B))) /= 0 then
                  Misc.ErrorNE("run-mailcap failed! (ff5)");
               end if;
            end;
         when 4 => -- Save
                   -- Write out the file to a well-known place.
            declare
               VMF : constant String := ToStr(Topal_Directory) & "/viewmime";
            begin
               Externals.Simple.Echo_Append_N("From MAILER-DAEMON ", VMF);
               Externals.Simple.Date_1_Append("+%a %b %e %H:%M:%S %Y", VMF);
               Externals.Simple.Echo_Append("From: Topal-MIME-view", VMF);
               Externals.Simple.Echo_Append_N("Date: ", VMF);
               Externals.Simple.Date_1_Append("--rfc-2822", VMF);
               Externals.Simple.Echo_Append("MIME-Version: 1.0", VMF);
               Externals.Simple.Cat_Append(F, VMF);
               Ada.Text_IO.Put_Line("Appended email to mbox "
                                    & VMF);
            end;
         when 5 => -- Skip
            Ada.Text_IO.Put_Line("Skipping attachment...");
         when others =>
            Misc.ErrorNE("Bogus value in Externals.Mail.View_MIME");
      end case;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.View_MIME");
         raise;
   end View_MIME;

   -- Base 64 decode from infile to outfile.
   -- Uses openssl internally.
   procedure Base64_Decode (Infile, Outfile : in String) is
      Dummy : Integer;
      pragma Unreferenced(Dummy);
   begin
      Dummy := ForkExec(Value_Nonempty(Config.Binary(Openssl)),
                        UBS_Array'(0 => ToUBS("openssl"),
                                   1 => ToUBS("base64"),
                                   2 => ToUBS("-d"),
                                   3 => ToUBS("-in"),
                                   4 => ToUBS(Infile),
                                   5 => ToUBS("-out"),
                                   6 => ToUBS(Outfile)));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.Base64_Decode");
         raise;
   end Base64_Decode;

   -- Quoted-printable decode from infile to outfile.
   procedure QP_Decode (Infile, Outfile : in String) is
      InF  : Ada.Text_IO.File_Type;
      OutF : Character_IO.File_Type;
   begin
      -- Read, character by character, until end-of-line.
      Ada.Text_IO.Open(File => InF,
                       Mode => Ada.Text_IO.In_File,
                       Name => Infile);
      Character_IO.Create(File => OutF,
                          Mode => Character_IO.Out_File,
                          Name => Outfile);
  Decode_Loop:
      loop
         begin
            -- Simple copy.
            declare
               U       : constant UBS := Unbounded_Get_Line(InF);
               S       : constant String := ToStr(U);
               I       : Integer;
               H       : Integer;
               Need_NL : Boolean := True;
            begin
               -- Walk through S.
               -- If we see `=', then
               --  (i) it's at the end of a line: it's a soft line break, so don't output even a newline.
               --  (ii) it should be followed by two hex characters.  decode and print.
               I := S'First;
           Char_Loop:
               while I <= S'Last loop
                  if S(I) = '=' then
                     if I = S'Last then
                        -- Soft line break.
                        Need_NL := False;
                        exit Char_Loop;
                     elsif I+2 > S'Last then
                        -- Overrunning the end of a line?  Just print it.
                        Character_IO_Put(OutF, S(I..S'Last));
                        I := S'Last + 1;
                     else
                        -- Decode.
                        begin
                           H := Hex_Decode(S(I+1..I+2));
                           Character_IO.Write(OutF, Character'Val(H));
                           I := I + 3;
                        exception
                           when Constraint_Error =>
                              -- Not hex?  Ignoring it.
                              Character_IO.Write(OutF, S(I));
                              I := I + 1;
                        end;
                     end if;
                  else
                     Character_IO.Write(OutF, S(I));
                     I := I + 1;
                  end if;
               end loop Char_Loop;
               if Need_NL then
                  Character_IO.Write(OutF, Ada.Characters.Latin_1.LF);
               end if;
            end;
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit Decode_Loop;
         end;
      end loop Decode_Loop;
      Ada.Text_IO.Close(File => InF);
      Character_IO.Close(File => OutF);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Mail.QP_Decode");
         raise;
   end QP_Decode;


begin
   Random_Letters.Reset(Random_Letters_Generator);
end Externals.Mail;
