-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2008  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Globals; use Globals;

package Keys is

   type Key_List is limited private;

   Key_Not_Found : exception;

   type Key_Property_Types is
     (Invalid, Disabled, Revoked, Expired, Encrypter, Signer);

   type Key_Properties is array(Key_Property_Types) of Boolean;

   -- Both means Encrypter or Signer.  Any means any, even those not
   --  capable of signing or encrypting.
   type Key_Roles is (Encrypter, Signer, Both, Any);

   -- Do the properties and roles match?
   function Match_PR (KP : Key_Properties;
                      KR : Key_Roles) return Boolean;

   -- Is the key usable?
   function Usable (KP : Key_Properties) return Boolean;

   -- Given a line from Findkey, return a processed version.
   procedure Process_Key (L  : in     String;
                          P  :    out UBS;
                          KP :    out Key_Properties);

   -- Add a key to the list....
   procedure Add_Key (Key  : in     UBS;
                      List : in out Key_List;
                      Role : in     Key_Roles);
   Fingerprint_Expected : exception;

   -- Remove a key.  We then shuffle the array down.
   -- It will only remove the last key if there are multiple instances.
   procedure Remove_Key (Key                  : in     UBS;
                         List                 : in out Key_List;
                         Exception_If_Missing : in     Boolean := False);

   -- Turn the list of keys to send into `-r <key>' string.
   function Processed_Recipient_List (List : in Key_List) return String;

   -- Get key fingerprint(s) for a key.  Add them.
   -- The name is a bit odd.  Really, they should be `Search_For_Keys' or similar.
   -- The `By_Fingerprint' bit refers to how the keys are recorded.
   procedure Add_Keys_By_Fingerprint (Key_In : in     UBS;
                                      List   : in out Key_List;
                                      Role   : in     Key_Roles;
                                      Found  :    out Boolean);

   procedure Add_Keys_By_Fingerprint (Key_In : in     UBS;
                                      List   : in out Key_List;
                                      Role   : in     Key_Roles);

   -- Add the secret keys to this keylist.
   procedure Add_Secret_Keys (Key_In : in     UBS;
                              List   : in out Key_List;
                              Role   : in     Key_Roles);

   -- List the keys and edit them as appropriate (include removing a key
   -- from that list, and adding one from the keyring.
   procedure List_Keys (List : in out Key_List);

   -- List the keys.  Either return with Aborted true, or
   -- The_Fingerprint set to the chosen fingerprint.
   procedure Select_Key_From_List (List            : in out Key_List;
                                   The_Fingerprint :    out UBS;
                                   Aborted         :    out Boolean);

   procedure First_Key_From_List (List            : in out Key_List;
                                  The_Fingerprint :    out UBS);

   -- Use keylist.  Given a list of recipients, add the keys, returning a
   -- new key list.
   procedure Use_Keylist (Recipients : in     UBS_Array;
                          List       : in out Key_List);

   procedure Empty_Keylist (List : in out Key_List);

   function Count (List : in Key_List) return Natural;

private

   type Key_List is
      record
         KA    : UBS_Big_Array;
         Count : Natural;
      end record;

end Keys;
