// Copyright (C) 2002 Johnny Mariethoz (Johnny.Mariethoz@idiap.ch)
//                and Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef IOHTK_INC
#define IOHTK_INC

#include "Object.h"

namespace Torch {

/// HTK File Header 
struct HTKhdr {              
    long  nSamples;
    long  sampPeriod;
    short sampSize;
    short sampKind;
};

#define BASEMASK  077         /* Mask to remove qualifiers */
#define HASENERGY  0100       /* _E log energy included */
#define HASNULLE   0200       /* _N absolute energy suppressed */
#define HASDELTA   0400       /* _D delta coef appended */
#define HASACCS   01000       /* _A acceleration coefs appended */
#define HASCOMPX  02000       /* _C is compressed */
#define HASZEROM  04000       /* _Z zero meaned */
#define HASCRCC  010000       /* _K has CRC check */
#define HASZEROC 020000       /* _0 0'th Cepstra included */
#define HASVQ    040000       /* _V has VQ index attached */


enum _BaseParmKind{
  WAVEFORM,            /* Raw speech waveform (handled by HWave) */
  LPC,LPREFC,LPCEPSTRA,LPDELCEP,   /* LP-based Coefficients */
  IREFC,                           /* Ref Coef in 16 bit form */
  MFCC,                            /* Mel-Freq Cepstra */
  FBANK,                           /* Log Filter Bank */
  MELSPEC,                         /* Mel-Freq Spectrum (Linear) */
  USER,                            /* Arbitrary user specified data */
  DISCRETE,                        /* Discrete VQ symbols (shorts) */
  ANON
};

typedef short ParmKind;          /* BaseParmKind + Qualifiers */


/** This class is used to read HTK objects (used for various datasets)
 *
 * @author Samy Bengio (bengio@idiap.ch)
 * @author Johnny Mariethoz (Johnny.Mariethoz@idiap.ch)
 */
class IOHtk : public Object
{
  public:
    
    /// the data itself
    real* data;

    /// number of columns in the data
    int n_cols;
    /// number of lines in the data
    int n_lines;

    /// kind of data
    char* kind;

    /// file name
    char* file_name;

		/// sampling period
		long samp_period;

    /** for each IOHtk object, specify the corresponding filename as well
        as the maximum number of examples to load
    */
    IOHtk(char* file,int max_load=-1);

		/** write data on HTK binary format on the specifing directory
		 */
		void write(char* dir_to_save=NULL);

   /** create a mask to remove energy and corresponding delta, delta-delta
		*/
		virtual void createMaskFromParam(bool* mask);

    virtual ~IOHtk();
    

    /// HTK source code 
    char* parmKind2Str(ParmKind kind, char *buf);
    ParmKind str2ParmKind(char *str);
    ParmKind baseParmKind(ParmKind kind);
    bool hasEnergy(ParmKind kind);
    bool hasDelta(ParmKind kind) ;
    bool hasAccs(ParmKind kind)  ;
    bool hasNulle(ParmKind kind) ;
    bool hasCompx(ParmKind kind) ;
    bool hasCrcc(ParmKind kind)  ;
    bool hasZerom(ParmKind kind) ;
    bool hasZeroc(ParmKind kind) ;
    bool hasVQ(ParmKind kind)    ;      
};

}

#endif


