// Copyright (C) 2002 Samy Bengio (bengio@idiap.ch)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "MultiCriterion.h"

namespace Torch {

MultiCriterion::MultiCriterion(int n_criterions_,Criterion** criterions_,real* weights_)
{
  n_criterions = n_criterions_;
  criterions = criterions_;
  n_inputs = criterions[0]->n_inputs;
  weights = weights_;
  if (weights) {
    allocate_weights = false;
  } else {
    allocate_weights = true;
    weights = (real*)xalloc(sizeof(real)*n_criterions);
    for (int i=0;i<n_criterions;i++)
      weights[i] = 1.;
  }
}

void MultiCriterion::setDataSet(DataSet *data_)
{
  data = data_;
  for (int i=0;i<n_criterions;i++)
    criterions[i]->setDataSet(data_);
}

void MultiCriterion::iterInitialize()
{
  for (int i=0;i<n_criterions;i++)
    criterions[i]->iterInitialize();
}

void MultiCriterion::reset()
{
  for (int i=0;i<n_criterions;i++)
    criterions[i]->reset();
}

void MultiCriterion::forward(List *inputs)
{
  real *ptr_out = (real *)outputs->ptr;
  *ptr_out = 0.;
  real *w = weights;
  for (int i=0;i<n_criterions;i++) {
    criterions[i]->forward(inputs);
    *ptr_out += *((real*)criterions[i]->outputs->ptr) * *w++;
  }
}

void MultiCriterion::backward(List *inputs, real *alpha)
{
  real *b = beta;
  for (int i=0;i<n_inputs;i++)
    *b++ = 0;

  real *w = weights;
  for (int i=0;i<n_criterions;i++,w++) {
    criterions[i]->backward(inputs,alpha);
    b = beta;
    real* bi = criterions[i]->beta;
    for (int j=0;j<n_inputs;j++)
      *b++ += *bi++ * *w;
  }
}

MultiCriterion::~MultiCriterion()
{
  if (allocate_weights)
    free(weights);
}

}

