// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "SparseDataSet.h"

namespace Torch {

SparseDataSet::SparseDataSet()
{
}

void SparseDataSet::init()
{
  DataSet::init();
  addToList(&inputs, n_inputs, NULL);
}

void SparseDataSet::setInputs(sreal **all_inputs_, int n_inputs_, int n_examples_)
{
  if(is_already_initialized)
    error("SparseDataSet: please, call setInputs *before* initialization");

  all_inputs = all_inputs_;
  n_inputs = n_inputs_;
  n_real_examples = n_examples_;
}

void SparseDataSet::setTargets(real **all_targets_, int n_targets_)
{
  if(is_already_initialized)
    error("SparseDataSet: please, call setTargets *before* initialization");

  all_targets = all_targets_;
  n_targets = n_targets_;
}

void SparseDataSet::setRealExample(int t)
{
  current_example = t;

  if(n_targets > 0)
    targets = all_targets[t];
  
  if(n_inputs > 0)
    inputs->ptr = all_inputs[t];
}

real SparseDataSet::realRealDotProduct(int i, int j)
{
  real z = 0;
  sreal *xx = all_inputs[i];
  sreal *yy = all_inputs[j];
  while( (xx->index != -1) && (yy->index != -1) )
  {
    if(xx->index == yy->index)
    {
      z += xx->value * yy->value;
      xx++;
      yy++;
    }
    else
    {
      if(xx->index > yy->index)
        yy++;
      else
        xx++;
    }
  }

  return(z);
}

real SparseDataSet::realDotProduct(int i, List *y)
{
  real z = 0;
  sreal *xx = all_inputs[i];
  sreal *yy = (sreal *)y->ptr;
  while( (xx->index != -1) && (yy->index != -1) )
  {
    if(xx->index == yy->index)
    {
      z += xx->value * yy->value;
      xx++;
      yy++;
    }
    else
    {
      if(xx->index > yy->index)
        yy++;
      else
        xx++;
    }
  }

  return(z);
}

real SparseDataSet::dotProduct(List *x, List *y)
{
  real z = 0;
  sreal *xx = (sreal *)x->ptr;
  sreal *yy = (sreal *)y->ptr;
  while( (xx->index != -1) && (yy->index != -1) )
  {
    if(xx->index == yy->index)
    {
      z += xx->value * yy->value;
      xx++;
      yy++;
    }
    else
    {
      if(xx->index > yy->index)
        yy++;
      else
        xx++;
    }
  }

  return(z);
}

SparseDataSet::~SparseDataSet()
{
  freeList(&inputs);
}

}

