/***********************************************************************
 *
 * Author: Martin Fluch
 *
 *     Copyright (c) 1999 Martin Fluch, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *
 ***********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <linux/unistd.h>
#include <linux/types.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>

#include <curses.h>
#include <menu.h>
#include <signal.h>

#include "ntpctl.h"

#define SZ_NTPCTL_VERSION "4.0"

#define MIN_COLS		40
#define MIN_LINES		15

char *device_name;
int device_fd = -1;
bool dev_rdonly = FALSE;

int (*configmenuraw[])(struct menuinfo *,int) =
{
  pm_mode_ac,
  pm_mode_battery,
  pm_mode_save_suspend,
  pm_sedate_hw_and_sw_event,
  pm_sedate_power_switch,
  pm_sedate_lid,
  pm_sedate_standby_timer,
  pm_sedate_system_timer,
  pm_sedate_hib_from_sup,
  pm_sedate_low_battery,
  pm_sedate_env_exhausted,
  pm_resume_hardware,
  pm_resume_lid,
  pm_resume_serial_RI,
  pm_resume_RTC,
  rt_date,
  rt_time,
  rt_daily,
  tm_ctrl_hib_susp,
  tm_ctrl_standby,
  tm_ctrl_LCD_off,
  tm_ctrl_HDD_off,
  tm_system_high,
  tm_system_auto,
  tm_system_manual_ac,
  tm_system_manual_battery,
  tm_hibernation,
  fn_hotkey,
  pd_internal_ch,
  pd_external_ch,
  pd_internal_CMOS,
  pd_external_CMOS
};

#define CONFIGMENUSIZE	(sizeof(configmenuraw)/sizeof(configmenuraw[1]))

#define DEVICE_NAME_DEFAULT "/dev/thinkpad/thinkpad"

struct menuinfo configmenuinfo[CONFIGMENUSIZE];
ITEM *configmenuitems[CONFIGMENUSIZE+1];
char configmenuname[CONFIGMENUSIZE][MAXNAMESIZE];
char configmenudesc[CONFIGMENUSIZE][MAXDESCSIZE];

MENU *configmenu=NULL;
WINDOW *configwin=NULL;

const char *helptext[]=
{
  "[q]uit, [h]igh, [a]uto, [m]anual",
  "[q]uit, [e]nable, [d]isable",
  "[q]uit, [s], [z], [r], [h], [o], [-]",
  "[q]uit, [return] to change timer, [d]isable",
  "[q]uit, [e]nable, [d]isable, [a]uto",
  "[q]uit, [y]es, [n]o",
  "[q]uit, [return] to change value"
};

struct {byte rc; char *errstr;} rcenc[]= // see TP 770 TechRef page 87
{
    {0x00,"done."},
    {0x53,"SMAPI fuction is not available"},
    {0x81,"Invalid parameter"},
    {0x86,"Function is not supported"},
    {0x90,"System error"},
    {0x91,"System is invalid"},
    {0x92,"System is busy"},
    {0xa0,"Device error (disk read error)"},
    {0xa1,"Device is busy"},
    {0xa2,"Device is not attached"},
    {0xa3,"Device is disbled"},
    {0xa4,"Request parameter is out of range"},
    {0xa5,"Request parameter is not accepted"},
    {0xff,"Unknown return code!"}
};

#define RCENCSIZE ((sizeof(rcenc)/sizeof(rcenc[1]))-1)

int rc_shown=0;            //
smapi_inparm_t  debug_in;  // see do_smapi() and clear_rc() 
smapi_outparm_t debug_out; // 

/***********************************************************************
 *
 * rc2errstr()
 *
 ***********************************************************************/

char *
rc2errstr(byte rc)
{
  int i;

  for(i=0;i<RCENCSIZE;i++) if(rc==rcenc[i].rc) return rcenc[i].errstr;

  return rcenc[RCENCSIZE].errstr;
}

/***********************************************************************
 *
 * clear_rc()
 *
 ***********************************************************************/

void 
clear_rc(void)
{
  int i;
  if(rc_shown) mvhline(LINES-4,3,0,COLS-6);
  for(i=1;i<rc_shown;i++) 
    {
      move(LINES-4+i,3);
      clrtoeol();
    }
}

/***********************************************************************
 *
 * display_debug_info()
 *
 ***********************************************************************/

void 
display_debug_info(void)
{
  static char outstr[100];
  const char *fmt="%s %.2x %.2x %.4x %.4x %.4x %.8lx %.8lx";
  clear_rc();
  snprintf(outstr,100,fmt,"In:  ",(int)debug_in.bFunc,
	   (int)debug_in.bSubFunc,(int)debug_in.wParm1,
	   (int)debug_in.wParm2,  (int)debug_in.wParm3,
	   (long int)debug_in.dwParm4,
	   (long int)debug_in.dwParm5);
  mvaddnstr(LINES-3,3,outstr,COLS-6);
  snprintf(outstr,100,fmt,"Out: ",(int)debug_out.bRc,
	   (int)debug_out.bSubRc, (int)debug_out.wParm1,
	   (int)debug_out.wParm2, (int)debug_out.wParm3,
	   (long int)debug_out.dwParm4,
	   (long int)debug_out.dwParm5);
  mvaddnstr(LINES-2,3,outstr,COLS-6);
  rc_shown=3;
}

/***********************************************************************
 *
 * do_smapi()
 *
 ***********************************************************************/

smapi_outparm_t *
do_smapi(byte func, byte sub_func, word p1, word p2,
	 word p3, dword p4, dword p5, int eflag)
{
  static smapi_ioparm_t parm;
  char *errstr;
  
  parm.in.bFunc=func;
  parm.in.bSubFunc=sub_func;
  parm.in.wParm1=p1;
  parm.in.wParm2=p2;
  parm.in.wParm3=p3;
  parm.in.dwParm4=p4;
  parm.in.dwParm5=p5;

  if(ioctl_smapi(device_fd,&parm)<0) return NULL;

  if(eflag)
    {
      errstr=rc2errstr(parm.out.bRc);
      mvaddnstr(LINES-4,3,errstr,COLS-6);
      hline(0,COLS-6-strlen(errstr));
      if(rc_shown==2)
	{
	  move(LINES-3,3);
	  clrtoeol();
	}
      rc_shown=1;

      debug_in.bFunc=func;
      debug_in.bSubFunc=sub_func;
      debug_in.wParm1=p1;
      debug_in.wParm2=p2;
      debug_in.wParm3=p3;
      debug_in.dwParm4=p4;
      debug_in.dwParm5=p5;

      debug_out=parm.out;
    }

  return &(parm.out);
}

/***********************************************************************
 *
 * handle_action()
 *
 ***********************************************************************/

int 
handle_action(ITEM *item, int action)
{
  int nr;
  if(item==NULL)
    {
      if((item=current_item(configmenu))==NULL) return 1;
    }
  if((nr=item_index(item))==ERR) return 1;
  
  return configmenuinfo[nr].func(&configmenuinfo[nr],action);  
}

/***********************************************************************
 *
 * print_help()
 *
 ***********************************************************************/

int 
print_help(void)
{
  ITEM *item;
  int nr;

  if((item=current_item(configmenu))==NULL) return 1;
  if((nr=item_index(item))==ERR) return 1;

  if(configmenuinfo[nr].help)
    {
      mvaddnstr(LINES-2,3,configmenuinfo[nr].help,COLS-6);
    }
  else
    {
      move(LINES-2,3);
    }

  clrtoeol();
  
  return 0;  
}

/***********************************************************************
 *
 * int handle_resume_cond()
 *
 * see TP 770 technical reference page 123
 *
 ***********************************************************************/

int 
handle_resume_cond(struct menuinfo *mi, int action, int bitnr)
{
  smapi_outparm_t *result;
  int chconfig=FALSE;
  int tmp=0;

  if((result=do_smapi(0x34,0x00,0,0,0,0,0,FALSE))==NULL) return 1;

  switch(action)
    {
    case ACTION_KEY_Y:
    case ACTION_KEY_E:
      tmp=result->wParm2 | (0x0001 << bitnr);
      chconfig=TRUE;
      break;

    case ACTION_KEY_N:
    case ACTION_KEY_D:
      tmp=result->wParm2 & ~(0x0001 << bitnr);
      chconfig=TRUE;
      break;

    case ACTION_NEXT:
    case ACTION_PREV:
      if(result->wParm2 & (0x0001 << bitnr)) 
	{
	  tmp=result->wParm2 & ~(0x0001 << bitnr);
	}
      else
	{
	  tmp=result->wParm2 | (0x0001 << bitnr);
	}
      chconfig=TRUE;
      break;
    }

  if(chconfig)
    {
      if(do_smapi(0x34,0x01,tmp,0,0,0,0,TRUE)==NULL) return 1;
      if((result=do_smapi(0x34,0x00,0,0,0,0,0,FALSE))==NULL) return 1;
    }

  switch(((((result->wParm3 & 0x0f) << 8) | 
	   (result->wParm2 & 0x0f)) >> bitnr) & 0x0101)
    {
    case 0x0000: strncpy(mi->desc,"no        (not capable)",MAXDESCSIZE); break;
    case 0x0100: strncpy(mi->desc,"no        (capable)    ",MAXDESCSIZE); break;
    case 0x0101: strncpy(mi->desc,"yes       (capable)    ",MAXDESCSIZE); break;
    case 0x0001: strncpy(mi->desc,"yes       (not capable)",MAXDESCSIZE); break;
    }

  return 0;
}

/***********************************************************************
 *
 * handle_pm_mode()
 * 
 * see TP 770 technical reference page 105
 *
 * Caution: In order that this algorithm works, bitnr must be either 
 * equal to zero or equal to 8!
 *
 ***********************************************************************/

int
handle_pm_mode(struct menuinfo *mi, int action, int bitnr)
{
  smapi_outparm_t *result;
  int chconfig=FALSE;
  int tmp=0;

  if((bitnr!=0) && (bitnr!=8)) return 1;

  if((result=do_smapi(0x22,0x00,0,0,0,0,0,FALSE))==NULL) return 1;

  switch(action)
    {
    case ACTION_KEY_H: 
      tmp=(result->wParm2 & (0xff00 >> bitnr)) | (0x0000 << bitnr);
      chconfig=TRUE;
      break;
      
    case ACTION_KEY_A:
      tmp=(result->wParm2 & (0xff00 >> bitnr)) | (0x0001 << bitnr);
      chconfig=TRUE;
      break;

    case ACTION_KEY_M:
      tmp=(result->wParm2 & (0xff00 >> bitnr)) | (0x0002 << bitnr);
      chconfig=TRUE;
      break;

    case ACTION_NEXT:
      tmp=(result->wParm2 & (0x0003 << bitnr)) + (0x0001 << bitnr);
      if(tmp> (0x0002 << bitnr)) tmp=0x0000;
      tmp=(result->wParm2 & (0xff00 >> bitnr)) | tmp;
      chconfig=TRUE;
      break;

    case ACTION_PREV:
      tmp=(result->wParm2 & (0x0003 << bitnr));
      if(tmp==0) tmp=(0x0003 << bitnr);
      tmp=(result->wParm2 & (0xff00 >> bitnr)) | (tmp-(0x0001<<bitnr));
      chconfig=TRUE;
      break;
  }

  if(chconfig)
    {
      if(do_smapi(0x22,0x01,tmp,0,0,0,0,TRUE)==NULL) return 1;
      if((result=do_smapi(0x22,0x00,0,0,0,0,0,FALSE))==NULL) return 1;
    }

  switch((result->wParm2>>bitnr) & 0x00ff)
    {
    case 0:  strncpy(mi->desc,"high   ",MAXDESCSIZE); break;
    case 1:  strncpy(mi->desc,"auto   ",MAXDESCSIZE); break;
    case 2:  strncpy(mi->desc,"manual ",MAXDESCSIZE); break;
    default: strncpy(mi->desc,"unknown",MAXDESCSIZE); break;
    }

  return 0;
}

/***********************************************************************
 *
 * pm_mode_ac()
 *
 ***********************************************************************/

int
pm_mode_ac(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM mode AC",MAXNAMESIZE);
      mi->help=helptext[HELP_ahm];
    }

  return handle_pm_mode(mi, action, 0);
}

/***********************************************************************
 *
 * pm_mode_battery()
 *
 ***********************************************************************/

int
pm_mode_battery(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM mode battery",MAXNAMESIZE);
      mi->help=helptext[HELP_ahm];
    }

  return handle_pm_mode(mi, action, 8);
}

/***********************************************************************
 *
 * pm_mode_save_suspend()
 *
 * see TP 770 technical reference page 111
 *
 ***********************************************************************/

int
pm_mode_save_suspend(struct menuinfo *mi, int action)
{
  smapi_outparm_t *result;
  int chconfig=FALSE;
  int tmp=0;

  if((result=do_smapi(0x30,0x00,0,0,0,0,0,FALSE))==NULL) return 1;

  switch(action)
    {
    case ACTION_INIT:
      strncpy(mi->name,"RediSafe suspend (globally)",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
      break;

    case ACTION_KEY_Y:
    case ACTION_KEY_E:
      tmp=0x0001;
      chconfig=TRUE;
      break;

    case ACTION_KEY_N:
    case ACTION_KEY_D:
      chconfig=TRUE;
      tmp=0x0000;
      break;

    case ACTION_NEXT:
    case ACTION_PREV:
      if(result->wParm2 & 0x0001) tmp=result->wParm2 & 0xfffe;
      else                        tmp=result->wParm2 | 0x0001;
      chconfig=TRUE;
      break;
    }

  if(chconfig)
    {
      if(do_smapi(0x30,0x01,tmp,0,0,0,0,TRUE)==NULL) return 1;
      if((result=do_smapi(0x30,0x00,0,0,0,0,0,FALSE))==NULL) return 1;
    }

  snprintf(mi->desc,MAXDESCSIZE,"%s       %s",
	   (result->wParm2 & 0x0001) ? "yes" : "no ",
	   (result->wParm2 & 0x0100) ? "(capable)    " : "(not capable)");

  return 0;
}

/***********************************************************************
 *
 * fn_hotkey()
 *
 * see TP 770 technical reference page 103
 *
 ***********************************************************************/

int
fn_hotkey(struct menuinfo *mi, int action)
{
  smapi_outparm_t *result;
  int chconfig=FALSE;
  int tmp=0;

  if((result=do_smapi(0x13,0x02,0,0,0,0,0,FALSE))==NULL) return 1;

  switch(action)
    {
    case ACTION_INIT:
      strncpy(mi->name,"Fn hotkey sticky/lock",MAXNAMESIZE);
      switch(result->wParm2 & 0x0300)
	{
	case 0x0100: 
	  mi->help="[q]uit, [d]isable, [s]ticky";
	  break;
	case 0x0200: 
	  mi->help="[q]uit, [d]isable, sticky & [l]ock";
	  break;
	case 0x0300:
	  mi->help="[q]uit, [d]isable, [s]ticky, sticky & [l]ock";
	  break;
	case 0x0000: mi->help="[q]uit, [d]isable";
	  break;
	}
      break;
      
    case ACTION_KEY_D:
    case ACTION_KEY_N:
      tmp=0x00;
      chconfig=TRUE;
      break;
      
    case ACTION_KEY_S:
      tmp=0x01;
      chconfig=TRUE;
      break;

    case ACTION_KEY_L:
      tmp=0x03;
      chconfig=TRUE;
      break;
    }

  if(chconfig)
    {
      if(do_smapi(0x13,0x03,tmp,0,0,0,0,TRUE)==NULL) return 1;
      if((result=do_smapi(0x13,0x02,0,0,0,0,0,FALSE))==NULL) return 1;
    }

  switch(result->wParm2 & 0x00ff)
    {
    case 0x00: strncpy(mi->desc,"disabled   ",MAXDESCSIZE); break;
    case 0x01: strncpy(mi->desc,"sticky     ",MAXDESCSIZE); break;
    case 0x03: strncpy(mi->desc,"sticky&lock",MAXDESCSIZE); break;
    default:   strncpy(mi->desc,"unknown    ",MAXDESCSIZE); break;
    }

  return 0;
}

/***********************************************************************
 *
 * pm_resume_hardware()
 *
 ***********************************************************************/

int
pm_resume_hardware(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM resume hardware",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
    }

  return handle_resume_cond(mi, action, 0);
}

/***********************************************************************
 *
 * pm_resume_lid()
 *
 ***********************************************************************/

int
pm_resume_lid(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM resume lid opened",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
    }

  return handle_resume_cond(mi, action, 1);
}

/***********************************************************************
 *
 * pm_resume_RTC()
 *
 ***********************************************************************/

int
pm_resume_RTC(struct menuinfo *mi, int action)
{
  static struct menuinfo *RTCmi=NULL;

  if(mi==NULL)
    {
      if(RTCmi==NULL) return 1;
      mi=RTCmi;
    }
  else RTCmi=mi;

  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM resume RTC appointment",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
    }

  if(handle_resume_cond(mi, action, 2)) return 1;
  if(action!=ACTION_INIT) if(rt_daily(NULL, ACTION_UPDATE)) return 1;
  
  return 0;
}

/***********************************************************************
 *
 * pm_resume_serial_RI()
 *
 ***********************************************************************/

int
pm_resume_serial_RI(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"PM resume serial RI",MAXNAMESIZE);
      mi->help=helptext[HELP_ed];
    }

  return handle_resume_cond(mi, action, 3);
}
  
/***********************************************************************
 *
 * int init_display()
 *
 ***********************************************************************/

int
init_display(void)
{
  if((COLS<MIN_COLS)||(LINES<MIN_LINES)) return 1;

  attrset(A_BOLD);
  mvaddstr(1,3,"Ncurses ThinkPad Configuration Tool v" SZ_NTPCTL_VERSION );
  if(dev_rdonly) { addstr(" (read only)"); }
  attrset(A_NORMAL);
  mvhline(3,0,0,COLS);

  if((configwin=newwin(LINES-10,COLS-10,5,3))==NULL) return 1;

  mvhline(LINES-4,0,0,COLS);

  return 0;
}

/***********************************************************************
 *
 * cleanup()
 *
 ***********************************************************************/

void
cleanup(char *msg)
{
  int i;

  if(device_fd!=-1) close(device_fd);

  if(configwin!=NULL) delwin(configwin);
  if(configmenu!=NULL) free_menu(configmenu);

  for(i=0;configmenuinfo[i].item && i<CONFIGMENUSIZE;i++) 
    {
      free_item(configmenuinfo[i].item);
    }

  resetty();
  endwin();

  if(msg) puts(msg);
}

/***********************************************************************
 *
 * handle_sig_int()
 *
 ***********************************************************************/

void
handle_sig_int( int intDummy )
{
  cleanup("Exit on SIGINT...");
  exit(0);
}

/***********************************************************************
 *
 * handle_sig_winch()
 *
 ***********************************************************************/

void
handle_sig_winch( int intDummy )
{
  int x,y;

  unpost_menu(configmenu);
  delwin(configwin);
  erase();

  endwin();
  refresh();

  if(init_display())
    {
      cleanup("handle_sig_winch(): Error while init_display() - "
	      "Display to small?");
      exit(1);
    }

  set_menu_win(configmenu,configwin);
  set_menu_spacing(configmenu,3,1,1);
  getmaxyx(configwin,y,x);
  set_menu_format(configmenu,y,0);

  if(post_menu(configmenu)!=E_OK)
    {
      cleanup("handle_sig_winch(): Error while post_menu() - "
	      "Display to small?");
      exit(1);
    }

  print_help();
  refresh();
  wrefresh(configwin);
}

/***********************************************************************
 *
 * int startup()
 *
 * Alle ntigen Initalisierungen vornehmen...
 * Liefert 0, wenn alles OK, sonst 1.
 *
 ***********************************************************************/

int fdevice, fhelp;

int
startup(void)
{
  int i,x,y;
  struct stat statBuf;

  initscr();
  savetty();
  nonl();
  cbreak();
  noecho();
  keypad(stdscr,TRUE);

  for(i=0;i<CONFIGMENUSIZE;i++)
    {
      configmenuinfo[i].item=configmenuitems[i]=NULL;
      configmenuinfo[i].name=configmenuname[i];
      configmenuinfo[i].desc=configmenudesc[i];
      configmenuinfo[i].func=configmenuraw[i];
      configmenuinfo[i].help=NULL;
    }

  signal(SIGINT,handle_sig_int);

	if (stat(device_name,&statBuf)) {
		if (strcmp(device_name,DEVICE_NAME_DEFAULT)) {
			if (errno==ENOENT)
				cleanup( "Device file does not exist" );
			else
				cleanup( "stat() of device file failed" );
			exit( EXIT_FAILURE );
		} else {
			if (stat("/dev/thinkpad",&statBuf)) {
				if (errno==ENOENT) 
					cleanup( "Device file does not exist" );
				else
					cleanup( "stat() of device file failed" );
				exit( EXIT_FAILURE );
			} else {
				device_name = "/dev/thinkpad";
			}
		}
	}
	/* stat(device_name,...) successful */
	if ( !(statBuf.st_mode & S_IFCHR) ) {
		cleanup("File is not a character device node." );
		exit( EXIT_FAILURE );
	}
	/* The file named by device_name is a character device node */

  if((device_fd=open(device_name,O_RDWR))==-1) {
      if((device_fd=open(device_name,O_RDONLY))==-1) {
          cleanup("Error while opening device.  Exiting.");
          exit(-1);
      }                                                                       
      dev_rdonly=TRUE;
  }

  for(i=0;i<CONFIGMENUSIZE;i++) 
    {
      if ((*configmenuinfo[i].func)(&configmenuinfo[i],ACTION_INIT))
	{
	  cleanup("Error while setting up configmenu...");
	  return 1;
	}

      if ((configmenuinfo[i].item=configmenuitems[i]=
	   new_item(configmenuinfo[i].name,configmenuinfo[i].desc))==NULL)
	{
	  cleanup("Error while allocating configmenu...");
	  return 1;
	}
    }

  configmenuitems[CONFIGMENUSIZE]=NULL;

  if((configmenu=new_menu(configmenuitems))==NULL)
    {
      cleanup("Error while new_menu()...");
      return 1;
    }

  if(init_display()) 
    {
      cleanup("Error while init_display() ... display to small?");
      return 1;
    }

  set_menu_win(configmenu,configwin);
  set_menu_spacing(configmenu,3,1,1);
  getmaxyx(configwin,y,x);
  set_menu_format(configmenu,y,0);
  
  if(post_menu(configmenu)!=E_OK)
    {
      cleanup("Error: Could not do post_menu() ... display to small?");
      return 1;
    }

  signal(SIGWINCH,handle_sig_winch);

  return 0;
}

/***********************************************************************
 *
 * main()
 *
 * No comment...
 *
 ***********************************************************************/

// int opterr, optopt; /* unused getopt feedback variables */
int optind;   /* getopt feedback variable: index of the option in argv */
char *optarg; /* getopt feedback variable: pointer to the argument string of the option */
struct option asoMyLongopts[] = {
  { "help", no_argument, &fhelp, 1 },
  { "device", required_argument, &fdevice, 1 },
  { 0, 0, 0, 0 }
};


int 
main(int argc, char *argv[])
{
  int running=TRUE;
  int key;
  int display_help=TRUE;
  char *pchOpts="";
  char chRtn;
  int iasoCur;

  device_name = DEVICE_NAME_DEFAULT;

  while(
    (chRtn = getopt_long_only(argc, argv, pchOpts, asoMyLongopts, &iasoCur )) != EOF
  ) {
    if ( chRtn == '?' || chRtn == '-' ) {
      /* unrecognized option or missing argument */
      fprintf( stderr, "Error scanning options.  Try 'ntpctl --help'.\n" );
      exit(-1);
    }
    if ( optarg == NULL ) continue; /* with while loop */
    /* option has an argument */
    if ( asoMyLongopts[iasoCur].flag == &fdevice ) {
      sscanf( optarg, "%as", &device_name );
    }
  }

  if ( fhelp ) {
    printf( "Usage: ntpctl [--help] [--device=<name>]\n" );
    exit(0);
  }

  if(startup()) return 1;

  while(running)
    {
      if(display_help) print_help();

      if(!handle_action(NULL,ACTION_UPDATE))
	{
	  if(unpost_menu(configmenu)!=E_OK) 
	    {
	      cleanup("Error: Could not unpost configmenu!");
	      return 1;
	    }
	  if(post_menu(configmenu)!=E_OK)
	    {
	      cleanup("Error: Could not repost configmenu!");
	      return 1;
	    }
	}

      refresh();
      wrefresh(configwin);

      key=getch();
      
      clear_rc();
      display_help=TRUE;

      handle_action(NULL,ACTION_UPDATE);

      switch(key)
	{
	case 'Q': case 'q':
	  running=FALSE;
	  break;

	case KEY_DOWN:
	  menu_driver(configmenu, REQ_DOWN_ITEM);
	  break;

	case KEY_UP:        
	  menu_driver(configmenu, REQ_UP_ITEM); 
	  break;

	case KEY_NPAGE:
	  if(menu_driver(configmenu, REQ_SCR_DPAGE)==E_REQUEST_DENIED)
	    menu_driver(configmenu, REQ_LAST_ITEM);
	  break;

	case KEY_PPAGE:
	  if(menu_driver(configmenu, REQ_SCR_UPAGE)==E_REQUEST_DENIED) 
	    menu_driver(configmenu,REQ_FIRST_ITEM);
	  break;

	case KEY_LEFT:      handle_action(NULL,ACTION_PREV);  break;
	case KEY_RIGHT:     handle_action(NULL,ACTION_NEXT);  break;
	case 'A': case 'a': handle_action(NULL,ACTION_KEY_A); break;
	case 'H': case 'h': handle_action(NULL,ACTION_KEY_H); break;
	case 'M': case 'm': handle_action(NULL,ACTION_KEY_M); break;
	case 'Y': case 'y': handle_action(NULL,ACTION_KEY_Y); break;
	case 'N': case 'n': handle_action(NULL,ACTION_KEY_N); break;
	case 'E': case 'e': handle_action(NULL,ACTION_KEY_E); break;
	case 'D': case 'd': handle_action(NULL,ACTION_KEY_D); break;
	case 'S': case 's': handle_action(NULL,ACTION_KEY_S); break;
	case 'Z': case 'z': handle_action(NULL,ACTION_KEY_Z); break;
	case 'R': case 'r': handle_action(NULL,ACTION_KEY_R); break;
	case 'O': case 'o': handle_action(NULL,ACTION_KEY_O); break;
	case '0': case '-': handle_action(NULL,ACTION_KEY_0); break;
	case 'L': case 'l': handle_action(NULL,ACTION_KEY_L); break;
	case 10:  case 13:
	case KEY_ENTER:     handle_action(NULL,ACTION_KEY_RET); break;

	case KEY_F(1):
	  display_debug_info();
	  display_help=FALSE;
	  break;
	}
    }

  cleanup(NULL); 

  return 0;
}
