/*
 * This file is part of TraceProto.
 * Copyright 2004 Eric Hope
 * skippysaurus@skippylair.net
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <libnet.h>
#include <pcap.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <ctype.h>

#include "traceproto.h"

/*
 * builds the packet
 * called once per probe
 */
void build_packet()
{
	switch ( packet.protocol_number )
	{
	case IPPROTO_TCP:
		state.tcp_h = libnet_build_tcp(
			packet.src_port,
			packet.dst_port,
			libnet_get_prand(LIBNET_PRu32),	 /* seq number */
			libnet_get_prand(LIBNET_PRu32),	 /* ack number */
			packet.tcp_flags,			/* SYN,ACK,etc */
			1500,				   /* indow */
			0,				      /* checksum */
			0,				      /* urg */
			LIBNET_TCP_H + behavior.payload_size,	    /* length */
			packet.payload,
			behavior.payload_size,
			state.packet,				/* libnet thingie */
			state.tcp_h );			/* !0 = mod packet */
		break;
	case IPPROTO_UDP:
		state.udp_h = libnet_build_udp(
			packet.src_port,
			packet.dst_port,
			LIBNET_UDP_H + behavior.payload_size,	/* length */
			0,					/* checksum */
			packet.payload,
			behavior.payload_size,
			state.packet,				/* libnet thingie */
			state.udp_h );
		break;
	case IPPROTO_ICMP:
		state.icmp_h = libnet_build_icmpv4_echo (
			ICMP_ECHO,				/* itype (8) */
			0,					/* icode (0) */
			0,					/* checksum */
			libnet_get_prand ( LIBNET_PRu16 ),	/* ident */
			libnet_get_prand ( LIBNET_PRu16 ),	/* sequence */
			packet.payload,
			behavior.payload_size,
			state.packet,				/* libnet thingie */
			state.icmp_h );				/* !0 = mod packet */
		break;
	default:
		printf ( "protocol error\n" );
		tixe ( tixe_cleanup, 1 );
		break;
	}
}

/*
 * if its a tcp packet, take that portion
 * and sort it out
 */
int parse_tcp_packet( const u_char * raw_packet )
{
	struct libnet_tcp_hdr * tcp_hdr;

	if ( state.ip_hdr->ip_ttl >= 1 ) 
		state.low_ttl = YES;

	tcp_hdr = ( struct libnet_tcp_hdr * ) & raw_packet[ 0 ];

	switch (  tcp_hdr->th_flags )
	{
	case ( TH_ACK | TH_RST ):
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_ACKRST );
		state.packet_match = TP_PACKET_DONE;
		state.target_response = YES;
		break;
	case ( TH_ACK | TH_SYN ):
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_SYNACK );
		state.packet_match = TP_PACKET_DONE;
		state.target_response = YES;
		break;
	case TH_SYN:
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_SYN );
		state.packet_match = TP_PACKET_DONE;
		state.target_response = YES;
		break;
	case TH_RST:
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_RST );
		state.packet_match = TP_PACKET_DONE;
		state.target_response = YES;
		break;
	case TH_FIN:
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_FIN );
		state.packet_match = TP_PACKET_DONE;
		state.target_response = YES;
		break;
	case TH_ACK:
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_ACK );
		state.packet_match = TP_PACKET_DONE;
		state.target_response = YES;
		break;
	default:
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_OTHER );
		break;
	}


	return ( state.packet_match == TP_PACKET_NO ) ? 0 : 1;
} //end of parse_tcp_packet




/*
 * if its a udp packet, take that portion
 * and sort it out
 */
int parse_udp_packet( const u_char * raw_packet )
{
	struct libnet_udp_hdr * udp_hdr;

	if ( state.ip_hdr->ip_ttl >= 1 ) 
		state.low_ttl = YES;

	udp_hdr = ( struct libnet_udp_hdr * ) & raw_packet[ 0 ];

	behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_UDP );
	state.packet_match = TP_PACKET_DONE;
	state.target_response = YES;

	return ( state.packet_match == TP_PACKET_NO ) ? 0 : 1;
} //end of parse_udp_packet



/*
 * if its an icmp packet, take that portion
 * and sort it out
 */
int parse_icmp_packet( const u_char * raw_packet )
{
	struct libnet_icmpv4_hdr * icmp_hdr;
	struct libnet_ipv4_hdr * icmp_payload;

	int icmp_payload_offset = 8;

	icmp_hdr = ( struct libnet_icmpv4_hdr * ) & raw_packet [ 0 ];
	icmp_payload = ( struct libnet_ipv4_hdr * ) & raw_packet [ icmp_payload_offset ];

//	if ( icmp_payload->ip_id != tp_ntol ( packet.ip_id ) )
//		return ( state.packet_match == TP_PACKET_NO );

	if ( icmp_hdr->icmp_type == ICMP_TIMXCEED )
	{
		behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_TIMXCEED );
		state.packet_match = TP_PACKET_MATCH;
	}
	else if ( icmp_hdr->icmp_type == ICMP_UNREACH )
	{
		switch ( icmp_hdr->icmp_code )
		{
		case ICMP_UNREACH_PORT:
			behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_PORT_UNREACH );
			state.packet_match = TP_PACKET_DONE;
			state.target_response = YES;
			break;
		case ICMP_UNREACH_NET:
			behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_NET_UNREACH );
			state.packet_match = TP_PACKET_DONE;
			state.target_response = YES;
			break;
		case ICMP_UNREACH_HOST:
			behavior.report( TP_OUT_HOP_INFO, & state.ip_hdr->ip_src, TP_TYPE_HOST_UNREACH );
			state.packet_match = TP_PACKET_DONE;
			state.target_response = YES;
			break;
		default:
			break;
		}
	}

	return ( state.packet_match == TP_PACKET_NO ) ? 0 : 1;
} //end of parse_icmp_packet


/*
 * take the raw packet and do stuff with it
 * also figure out what it is
 */
int parse_packet( const u_char * raw_packet )
{
	struct libnet_802_3_hdr * eth_hdr;

	int eth_offset = LINUX_OFFSET + sizeof(struct libnet_802_3_hdr);
	int ipv4_offset = eth_offset + sizeof( struct libnet_ipv4_hdr );
/*
 * peel the ethernet and ip headers
 * off of the packet
 */
	eth_hdr = ( struct libnet_802_3_hdr * ) raw_packet;
	state.ip_hdr = ( struct libnet_ipv4_hdr * ) & raw_packet[ eth_offset ];

	if ( state.ip_hdr->ip_ttl >= 1 ) 
		state.low_ttl = YES;

	state.packet_match = TP_PACKET_NO;
/*
 * now send it to the correct place
 */
	switch( state.ip_hdr->ip_p )
	{
	case IPPROTO_ICMP:
		return parse_icmp_packet ( ( const u_char * ) & raw_packet [ ipv4_offset ] );
		break;
	case IPPROTO_TCP:
		return parse_tcp_packet ( ( const u_char * ) & raw_packet [ ipv4_offset ] );
		break;
	case IPPROTO_UDP:
		return parse_udp_packet ( ( const u_char * ) & raw_packet [ ipv4_offset ] );
		break;
	default:
		break;
	}

	return 0;
} //end of parse_packet


/*
 * minor slight of hand to manipulate
 * the packet ip_id fields
 */
int tp_ntol ( int n_order )
{
	int tmp = n_order & 0xff;
	n_order = n_order >> 8;
	tmp = tmp << 8;
	n_order = n_order | tmp;

	return n_order;
}

/*
 * not really used at this point,
 * here for os that may not be as 
 * good about send the resets for
 * raw sockets
 */
int send_tcp_reset ()
{

	state.tcp_h = libnet_build_tcp(
		packet.src_port,
		packet.dst_port,
		libnet_get_prand ( LIBNET_PRu32 ),	/* seqnumber */
		libnet_get_prand(LIBNET_PRu32),		/* acknumber */
		TH_RST & TH_FIN,
		1500,					/* indow */
		0,					/* checksum */
		0,					/* urg */
		LIBNET_TCP_H + behavior.payload_size,	/* length */
		packet.payload,
		behavior.payload_size,
		state.packet,				/* libnet thingie */
		state.tcp_h );			/* !0 = modpacket */

	state.ip_h = libnet_build_ipv4(
		LIBNET_IPV4_H,		  /* IP hdr len */
		0,			      /* tos */
		packet.ip_id,		    /* frag id */
		packet.frag_bit,		/* frag offset */
		state.current_hop,	      /* ttl */
		packet.protocol_number,
		0,			      /* checksum */
		packet.packed_src,	      /* packed source */
		packet.packed_target,	   /* packed target */
		NULL,			   /* payload */
		0,			      /* payload size */
		state.packet,
		state.ip_h );

	if ( libnet_write( state.packet ) == -1 )
	{
		return 1;
	}
	return 0;
}
